/**********************************************/
/* Rise of the Tribes                         */
/* C version, Take 2                          */
/* Evert Glebbeek 1998, 2002                  */
/* eglebbk@phys.uva.nl                        */
/**********************************************/
#include <allegro.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include "script.h"
#include "global.h"
#include "dialog.h"
#include "map.h"
#include "unit.h"

#define streq(str1, str2) (!strcmp(str1, str2))

typedef struct {
   char *name;
   short location;
} PARSE_LABEL;

static char **ucp = NULL;
static char **ucpname = NULL;   /* Names of unit control programs; used when loading */
static size_t num_ucp = 0;

static int pass = 1;

static PARSE_LABEL *pl = NULL;
static size_t num_label = 0;

static void push_label(char *name, short ptr)
{
   pl = realloc(pl, (num_label + 1) * sizeof(PARSE_LABEL));
   pl[num_label].name = strdup(name);
   pl[num_label].location = ptr;
   num_label++;
}

static short find_label(char *name)
{
   int c;

   for (c = 0; c < num_label; c++)
      if (streq(name, pl[c].name))
         return pl[c].location;
   return 0;
}

/* Clears up all temporary data from the parser */
void clear_parser_stack(void)
{
   int c;

   for (c = 0; c < num_label; c++) {
      free(pl[c].name);
   }
   if (pl)
      free(pl);
   pl = NULL;
   num_label = 0;
}

void pass_one(void)
{
   pass = 1;
   clear_parser_stack();
}

void pass_two(void)
{
   int c;

   /* If we're going to do a second pass, clear everything now */
   if (num_label) {
      for (c = 0; c < num_ucp; c++) {
         free(ucp[c]);
         free(ucpname[c]);
      }
      free(ucp);
      free(ucpname);
      ucp = ucpname = NULL;
      num_ucp = 0;
   }
   pass = 2;
}

/* Returns TRUE if the string s is a (integer) numerical data */
static int is_numeric(char *s)
{
   char *parse;

   for (parse = s; isdigit(parse[0]); parse++);
   
   if (parse[0])
      return FALSE;
   else
      return TRUE;   
}

/* Loads character control programs from file and stores in byte code */
/* Some of the commands may not make sense, but I try to remain compatible */
/*  with the 'mother' language from SGT. This has then the potential to */
/*  become a very powerful general-purpose tool */
void load_unit_control(char *filename)
{
   char *s = malloc(256);
   char *proc_name = NULL;
   enum { NO_CONTEXT, CONTEXT_UNIT, CONTEXT_EFFECT } context = NO_CONTEXT;
   FILE *f;
   int ln = 0;
   int proc_closed = TRUE;
   int cproclen = 0;
   int cprocmax = 0;
   int b;
   int r;
   
   #define RESERVE_PROC_SIZE(n)  while ((cproclen + (n)) >= (cprocmax)) {\
                                    cprocmax += 8;\
                                    ucp[num_ucp - 1] =\
                                       realloc(ucp[num_ucp - 1], \
                                             (cprocmax + 8) * sizeof(char));\
                                 }


   /* No need for a second pass if we encountered no labels */
   if ((!num_label) && (pass == 2))
      return;

   if (s == NULL) {
      set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
      allegro_message("Cannot open file: %s\n(Memory problem)", filename);
      exit(EXIT_FAILURE);
   }
   if (!filename) {
      free(s);
      set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
      allegro_message
         ("Error in game description (fatal):\n no unit control programs");
      exit(EXIT_FAILURE);
   }

   f = fopen(filename, "rt");
   if (f == NULL) {
      free(s);
      set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
      allegro_message("Cannot open file: %s\n(read error)", filename);
      exit(EXIT_FAILURE);
   }

   while (fgets(s, 255, f)) {
      char *eol = strstr(s, ";");

      /* skip end-of-line comments */
      if (eol)
         eol[0] = '\0';

      ln++;
      if (s[0] != ';') {        /* Skip comments */
         eol = strstr(s, "\n");

         if (eol)
            eol[0] = '\0';

         /* clear spaces from the beginning of the string */
         strflushl(s);
         strflushr(s);

         if (s[0] != '\0') {    /* check if line isn't empty */
            /* split into two strings */
            char *opt = s;
            char *arg;
            char *arg2;

            arg = strstr(opt, " ");
            if (arg) {
               arg[0] = '\0';
               arg++;
               /* Remove trailing newlines */
               eol = strstr(arg, "\n");
               if (eol)
                  eol[0] = '\0';
               strflushl(arg);
            } else {
               arg = opt + strlen(opt);
            }

            arg2 = strstr(arg, ",");
            if (arg2) {
               arg2[0] = '\0';
               arg2++;
               /* Remove trailing newlines */
               strflushl(arg2);
            } else {
               arg2 = arg + strlen(arg);
            }

            /* Parse program lines and 'compile' */
            if (streq(opt, "context")) {
               if (streq(arg, "unit")) {
                  context = CONTEXT_UNIT;
               } else if (streq(arg, "none")) {
                  context = NO_CONTEXT;
               } else if (streq(arg, "effect")) {
                  context = CONTEXT_EFFECT;
               } else {
                  popup_message("%s, line %d:\nUnknown context: %s", filename,
                                ln, arg);
               }
            } else if (streq(opt, "#include")) {
               /* recursive call! */
               load_unit_control(arg);
            } else if (streq(opt, "proc") && (context != NO_CONTEXT)) {
               if (proc_name) {
                  popup_message
                     ("%s, line %d:\n Warning: nested proc %s in %s",
                      filename, ln, arg, proc_name);
                  free(proc_name);
                  proc_name = NULL;
               }
               proc_closed = FALSE;
               proc_name = strdup(arg);

               /* Allocate memory for the new control program */
               ucp = realloc(ucp, (num_ucp + 1) * sizeof(char *));
               ucpname = realloc(ucpname, (num_ucp + 1) * sizeof(char *));
               ucp[num_ucp] = NULL;
               ucpname[num_ucp] = strdup(arg);
               num_ucp++;
               /* Set control pointers */
               cproclen = 0;    /* number of bytes written to current cproc */
               cprocmax = 0;    /* max size of current proc */
            } else if (streq(arg, "endp") && (context != NO_CONTEXT)) {
               if (!proc_name) {
                  popup_message
                     ("%s, line %d:\nWarning: %s endp without proc %s",
                      filename, ln, opt, opt);
               } else {
                  if (strcmp(opt, proc_name))
                     popup_message
                        ("%s, line %d:\nWarning: endp %s does not match proc %s",
                         filename, ln, opt, proc_name);

                  /* insert implicid return if not present */
                  RESERVE_PROC_SIZE(1)

                  ucp[num_ucp - 1][cproclen] = CMD_RETURN;
                  cproclen += 1;

                  if (!proc_closed)
                     popup_message
                        ("%s, line %d:\nWarning: proc %s not closed",
                         filename, ln, proc_name);
                  proc_closed = TRUE;
                  free(proc_name);
                  proc_name = NULL;
               }
            } else if ((streq(opt, "load") || (streq(opt, "push") && !is_numeric(arg))) && (context == CONTEXT_UNIT)) {
               /* Store a program (or character) property on the stack */
               RESERVE_PROC_SIZE(5)
               
               ucp[num_ucp - 1][cproclen] = CMD_LOAD;
               
               if (streq(arg, "arg0")) {
                  ucp[num_ucp - 1][cproclen + 1] = 0;
               } else if (streq(arg, "arg1")) {
                  ucp[num_ucp - 1][cproclen + 1] = 1;
               } else if (streq(arg, "arg2")) {
                  ucp[num_ucp - 1][cproclen + 1] = 2;
               } else if (streq(arg, "argc")) {
                  ucp[num_ucp - 1][cproclen + 1] = 3;

               /* Command targets - these should actually be stored as numbers */
               /*  but allowing them to be written like this makes scripts more*/
               /*  readable */
               } else if (streq(arg, "target->self")) {
                  ucp[num_ucp - 1][cproclen + 1] = 0;
                  ucp[num_ucp - 1][cproclen + 2] = 0;
                  ucp[num_ucp - 1][cproclen + 3] = 0;
                  ucp[num_ucp - 1][cproclen + 4] = 0;
                  ucp[num_ucp - 1][cproclen] = CMD_PUSHN;
                  cproclen += 3;
               } else if (streq(arg, "target->parent")) {
                  ucp[num_ucp - 1][cproclen + 1] = 1;
                  ucp[num_ucp - 1][cproclen + 2] = 0;
                  ucp[num_ucp - 1][cproclen + 3] = 0;
                  ucp[num_ucp - 1][cproclen + 4] = 0;
                  ucp[num_ucp - 1][cproclen] = CMD_PUSHN;
                  cproclen += 3;
               } else if (streq(arg, "target->attacker")) {
                  ucp[num_ucp - 1][cproclen + 1] = 2;
                  ucp[num_ucp - 1][cproclen + 2] = 0;
                  ucp[num_ucp - 1][cproclen + 3] = 0;
                  ucp[num_ucp - 1][cproclen + 4] = 0;
                  ucp[num_ucp - 1][cproclen] = CMD_PUSHN;
                  cproclen += 3;
               } else if (streq(arg, "target->aitarget")) {
                  ucp[num_ucp - 1][cproclen + 1] = 3;
                  ucp[num_ucp - 1][cproclen + 2] = 0;
                  ucp[num_ucp - 1][cproclen + 3] = 0;
                  ucp[num_ucp - 1][cproclen + 4] = 0;
                  ucp[num_ucp - 1][cproclen] = CMD_PUSHN;
                  cproclen += 3;

               } else {
                  popup_message("%s, line %d:\nWarning: unknown property: %s",
                                   filename, ln, arg);
               }
               cproclen += 2;
            } else if ((streq(opt, "store") || (streq(opt, "pop")&&arg[0])) && (context == CONTEXT_UNIT)) {
               /* Read a program (or character) property from the stack */
               RESERVE_PROC_SIZE(2)

               ucp[num_ucp - 1][cproclen] = CMD_STORE;
               if (streq(arg, "arg0")) {
                  ucp[num_ucp - 1][cproclen + 1] = 0;
               } else if (streq(arg, "arg1")) {
                  ucp[num_ucp - 1][cproclen + 1] = 1;
               } else if (streq(arg, "arg2")) {
                  ucp[num_ucp - 1][cproclen + 1] = 2;
               } else if (streq(arg, "argc")) {
                  ucp[num_ucp - 1][cproclen + 1] = 3;
               } else {
                  popup_message("%s, line %d:\nWarning: unknown property: %s",
                                filename, ln, arg);
               }
               cproclen += 2;
            } else if (streq(opt, "target") && (context != NO_CONTEXT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(2)

               ucp[num_ucp - 1][cproclen] = CMD_TARGET;
               if (streq(arg, "self")) {
                  ucp[num_ucp - 1][cproclen + 1] = 0;
               } else if (streq(arg, "parent")) {
                  ucp[num_ucp - 1][cproclen + 1] = 1;
               } else if (streq(arg, "attacker")) {
                  ucp[num_ucp - 1][cproclen + 1] = 2;
               } else if (streq(arg, "aitarget")) {
                  ucp[num_ucp - 1][cproclen + 1] = 3;
               } else {
                  popup_message("%s, line %d:\n Warning: Unknown target %s",
                                filename, ln, arg);
               }
               cproclen += 2;
            } else if (streq(opt, "label") && (context != NO_CONTEXT)) {
               if (!proc_name) {
                  popup_message
                     ("%s, line %d:\n Warning: label outside %s proc",
                      filename, ln, arg);
               }
               /* Push label */
               if (pass == 1) {
                  char *s = malloc(strlen(proc_name) + strlen(arg) + 4);

                  sprintf(s, "%s.%s", proc_name, arg);
                  push_label(s, cproclen);
                  free(s);
               }
            } else if (streq(opt, "nop") && (context != NO_CONTEXT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(1)

               ucp[num_ucp - 1][cproclen] = CMD_NOP;
               cproclen += 1;
            } else if (streq(opt, "showcounter") && (context != NO_CONTEXT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(1);

               ucp[num_ucp - 1][cproclen] = CMD_SHOWCNTER;
               cproclen += 1;
            } else if (streq(opt, "hidecounter") && (context != NO_CONTEXT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(1);

               ucp[num_ucp - 1][cproclen] = CMD_HIDECNTER;
               cproclen += 1;
            } else if (streq(opt, "cmdoff") && (context != NO_CONTEXT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(1);

               ucp[num_ucp - 1][cproclen] = CMD_CMDOFF;
               cproclen += 1;
            } else if (streq(opt, "cmdon") && (context != NO_CONTEXT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(1);

               ucp[num_ucp - 1][cproclen] = CMD_CMDON;
               cproclen += 1;
            } else if (streq(opt, "faceto") && (context != NO_CONTEXT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(1);

               ucp[num_ucp - 1][cproclen] = CMD_FACETO;
               cproclen += 1;
            } else if (streq(opt, "chkd") && (context == CONTEXT_UNIT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(1);

               ucp[num_ucp - 1][cproclen] = CMD_CHKD;
               cproclen += 1;
            } else if (streq(opt, "clrd") && (context == CONTEXT_UNIT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(1);

               ucp[num_ucp - 1][cproclen] = CMD_CLRD;
               cproclen += 1;
            } else if (streq(opt, "setd") && (context == CONTEXT_UNIT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(1);

               ucp[num_ucp - 1][cproclen] = CMD_SETD;
               cproclen += 1;
            } else if (streq(opt, "clrc") && (context != NO_CONTEXT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(1);

               ucp[num_ucp - 1][cproclen] = CMD_CLRC;
               cproclen += 1;
            } else if (streq(opt, "setc") && (context != NO_CONTEXT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(1);

               ucp[num_ucp - 1][cproclen] = CMD_SETC;
               cproclen += 1;
            } else if (streq(opt, "chkc") && (context != NO_CONTEXT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(1);

               ucp[num_ucp - 1][cproclen] = CMD_CHKC;
               cproclen += 1;
            } else if (streq(opt, "castspell") && (context == CONTEXT_UNIT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(1);

               ucp[num_ucp - 1][cproclen] = CMD_CASTSPL;
               cproclen += 1;
            } else if (streq(opt, "invsbl") && (context == CONTEXT_UNIT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(1);

               ucp[num_ucp - 1][cproclen] = CMD_INVSBL;
               cproclen += 1;
            } else if (streq(opt, "vsbl") && (context == CONTEXT_UNIT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(1);

               ucp[num_ucp - 1][cproclen] = CMD_VSBL;
               cproclen += 1;
            } else if (streq(opt, "atomic") && (context != NO_CONTEXT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(1);

               ucp[num_ucp - 1][cproclen] = CMD_ATOMIC;
               cproclen += 1;
            } else if (streq(opt, "touch") && (context == CONTEXT_UNIT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(1);

               ucp[num_ucp - 1][cproclen] = CMD_TOUCH;
               cproclen += 1;
            } else if (streq(opt, "untouch") && (context == CONTEXT_UNIT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(1);

               ucp[num_ucp - 1][cproclen] = CMD_UNTOUCH;
               cproclen += 1;
            } else if ((streq(opt, "jmp") || streq(opt, "goto")) &&
                       (context != NO_CONTEXT)) {
               short n;
               char *s = malloc(strlen(proc_name) + strlen(arg) + 4);

               sprintf(s, "%s.%s", proc_name, arg);
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(3);

               n = find_label(s);
               free(s);
               ucp[num_ucp - 1][cproclen + 1] = n & 0xFF;
               ucp[num_ucp - 1][cproclen + 2] = (n >> 8) & 0xFF;
               ucp[num_ucp - 1][cproclen] = CMD_JMP;
               cproclen += 3;
               //popup_message ("%s, line %d [%s]:\nUnimplemented command: %s, %s", filename, ln, proc_name, opt, arg);
            } else if ((streq(opt, "je") || streq(opt, "jz")) &&
                       (context != NO_CONTEXT)) {
               short n;
               char *s = malloc(strlen(proc_name) + strlen(arg) + 4);

               sprintf(s, "%s.%s", proc_name, arg);
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(3);

               n = find_label(s);
               //popup_message ("%s (%d)",s,n);
               free(s);
               ucp[num_ucp - 1][cproclen + 1] = n & 0xFF;
               ucp[num_ucp - 1][cproclen + 2] = (n >> 8) & 0xFF;
               ucp[num_ucp - 1][cproclen] = CMD_JE;
               cproclen += 3;
            } else if (streq(opt, "jcf") && (context != NO_CONTEXT)) {
               /* Jump on Cancel Flag */
               short n;
               char *s = malloc(strlen(proc_name) + strlen(arg) + 4);

               sprintf(s, "%s.%s", proc_name, arg);
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(3);

               n = find_label(s);
               //popup_message ("%s (%d)",s,n);
               free(s);
               ucp[num_ucp - 1][cproclen + 1] = n & 0xFF;
               ucp[num_ucp - 1][cproclen + 2] = (n >> 8) & 0xFF;
               ucp[num_ucp - 1][cproclen] = CMD_JCF;
               cproclen += 3;
            } else if ((streq(opt, "jne") || streq(opt, "jnz")) &&
                       (context != NO_CONTEXT)) {
               short n;
               char *s = malloc(strlen(proc_name) + strlen(arg) + 4);

               sprintf(s, "%s.%s", proc_name, arg);
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(3);

               n = find_label(s);
               free(s);
               ucp[num_ucp - 1][cproclen + 1] = n & 0xFF;
               ucp[num_ucp - 1][cproclen + 2] = (n >> 8) & 0xFF;
               ucp[num_ucp - 1][cproclen] = CMD_JNE;
               cproclen += 3;
            } else if ((streq(opt, "ja") || streq(opt, "jnbe")) &&
                       (context != NO_CONTEXT)) {
               short n;
               char *s = malloc(strlen(proc_name) + strlen(arg) + 4);

               sprintf(s, "%s.%s", proc_name, arg);
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(3);

               n = find_label(s);
               free(s);
               ucp[num_ucp - 1][cproclen + 1] = n & 0xFF;
               ucp[num_ucp - 1][cproclen + 2] = (n >> 8) & 0xFF;
               ucp[num_ucp - 1][cproclen] = CMD_JA;
               cproclen += 3;
            } else if ((streq(opt, "jb") || streq(opt, "jnae")) &&
                       (context != NO_CONTEXT)) {
               short n;
               char *s = malloc(strlen(proc_name) + strlen(arg) + 4);

               sprintf(s, "%s.%s", proc_name, arg);
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(3);

               n = find_label(s);
               free(s);
               ucp[num_ucp - 1][cproclen + 1] = n & 0xFF;
               ucp[num_ucp - 1][cproclen + 2] = (n >> 8) & 0xFF;
               ucp[num_ucp - 1][cproclen] = CMD_JB;
               cproclen += 3;
            } else if ((streq(opt, "jae") || streq(opt, "jnb")) &&
                       (context != NO_CONTEXT)) {
               short n;
               char *s = malloc(strlen(proc_name) + strlen(arg) + 4);

               sprintf(s, "%s.%s", proc_name, arg);
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(3);

               n = find_label(s);
               free(s);
               ucp[num_ucp - 1][cproclen + 1] = n & 0xFF;
               ucp[num_ucp - 1][cproclen + 2] = (n >> 8) & 0xFF;
               ucp[num_ucp - 1][cproclen] = CMD_JBE;
               cproclen += 3;
            } else if (streq(opt, "alert") && (context != NO_CONTEXT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(1);
               
               ucp[num_ucp - 1][cproclen] = CMD_ALERT;
               cproclen += 1;
            } else if (streq(opt, "clrmsg") && (context != NO_CONTEXT)) {
               /* Clear message window */
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(1);
               
               ucp[num_ucp - 1][cproclen] = CMD_CLRMSG;
               cproclen++;
            } else if (streq(opt, "showmsg") && (context != NO_CONTEXT)) {
               int n;

               /* Popup a message */
               /* This is a complex one: we have to store the message in a
                  global array, then store the number (int) in the program */
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(5);
               
               ucp[num_ucp - 1][cproclen] = CMD_SHOWMSG;
               if (arg)
                  n = add_global_msg(arg);
               else {
                  n = add_global_msg("");
               }
               ucp[num_ucp - 1][cproclen + 1] = n & 0xFF;
               ucp[num_ucp - 1][cproclen + 2] = (n >> 8) & 0xFF;
               ucp[num_ucp - 1][cproclen + 3] = (n >> 16) & 0xFF;
               ucp[num_ucp - 1][cproclen + 4] = (n >> 24) & 0xFF;
               cproclen += 5;
            } else if (streq(opt, "return") && (context != NO_CONTEXT)) {
               /* The behavior of this command depends on the contet */
               if (context == CONTEXT_UNIT) {
                  /* Check if enough memory to store program */
                  RESERVE_PROC_SIZE(1);
               
                  ucp[num_ucp - 1][cproclen] = CMD_RETURN;
                  cproclen += 1;
                  proc_closed = TRUE;
               } else if (context == CONTEXT_EFFECT) {
                  /* Spells objects must be destroyed at the end of their
                     program */
                  RESERVE_PROC_SIZE(2);
               
                  ucp[num_ucp - 1][cproclen] = CMD_REMOVE;
                  ucp[num_ucp - 1][cproclen + 1] = CMD_RETURN;
                  cproclen += 2;
                  proc_closed = TRUE;
               }
            } else if (streq(opt, "rmv") && (context == CONTEXT_UNIT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(1);
               
               ucp[num_ucp - 1][cproclen] = CMD_REMOVE;
               cproclen += 1;
            } else if (streq(opt, "chng") && (context == CONTEXT_UNIT)) {
               int n1;
               int n2;

               RESERVE_PROC_SIZE(11);
               
               if (streq(arg, "self")) {
                  n1 = 0;
               } else if (streq(arg, "parent")) {
                  n1 = 1;
               } else if (streq(arg, "attacker")) {
                  n1 = 2;
               } else if (streq(arg, "aitarget")) {
                  n1 = 3;
               } else {
                  n1 = 0;
                  popup_message("%s, line %d:\n Warning: Unknown target %s",
                                filename, ln, arg);
               }

               if (arg2)
                  n2 = key_to_id(arg2);
               else {
                  n2 = key_to_id("");
               }
               ucp[num_ucp - 1][cproclen] = CMD_PUSHN;
               ucp[num_ucp - 1][cproclen + 1] = n2 & 0xFF;
               ucp[num_ucp - 1][cproclen + 2] = (n2 >> 8) & 0xFF;
               ucp[num_ucp - 1][cproclen + 3] = (n2 >> 16) & 0xFF;
               ucp[num_ucp - 1][cproclen + 4] = (n2 >> 24) & 0xFF;
               ucp[num_ucp - 1][cproclen + 5] = CMD_PUSHN;
               ucp[num_ucp - 1][cproclen + 6] = n1 & 0xFF;
               ucp[num_ucp - 1][cproclen + 7] = (n1 >> 8) & 0xFF;
               ucp[num_ucp - 1][cproclen + 8] = (n1 >> 16) & 0xFF;
               ucp[num_ucp - 1][cproclen + 9] = (n1 >> 24) & 0xFF;
               ucp[num_ucp - 1][cproclen + 10] = CMD_CHANGE;
               cproclen += 11;
               //popup_message ("%s, line %d [%s]:\nUnsupported command: %s, %s", filename, ln, proc_name, opt, arg);
            } else if (streq(opt, "sound") && (context != NO_CONTEXT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(1);
               
               ucp[num_ucp - 1][cproclen] = CMD_SOUND;
               cproclen += 1;
            } else if (streq(opt, "newdir") && (context == CONTEXT_UNIT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(1);
               
               ucp[num_ucp - 1][cproclen] = CMD_NEWDIR;
               cproclen += 1;
            } else if (streq(opt, "fire") && (context == CONTEXT_UNIT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(1);
               
               ucp[num_ucp - 1][cproclen] = CMD_FIRE;
               cproclen += 1;
            } else if (streq(opt, "clrzf") && (context != NO_CONTEXT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(1);
               
               ucp[num_ucp - 1][cproclen] = CMD_CLRZF;
               cproclen += 1;
            } else if (streq(opt, "advcounter") && (context != NO_CONTEXT)) {
               unsigned int n;

               RESERVE_PROC_SIZE(6);
               
               if (arg)
                  sscanf(arg, "%d", &n);
               else {
                  n = 1;
               }
               ucp[num_ucp - 1][cproclen] = CMD_PUSHN;
               ucp[num_ucp - 1][cproclen + 1] = n & 0xFF;
               ucp[num_ucp - 1][cproclen + 2] = 0;//(n >> 8) & 0xFF;
               ucp[num_ucp - 1][cproclen + 3] = 0;//(n >> 16) & 0xFF;
               ucp[num_ucp - 1][cproclen + 4] = 0;//(n >> 24) & 0xFF;
               ucp[num_ucp - 1][cproclen + 5] = CMD_ADVCOUNTER;
               cproclen += 6;
            } else if ((streq(opt, "pushn")||(streq(opt, "push")&&is_numeric(arg))) && (context != NO_CONTEXT)) {
               unsigned int n;

               RESERVE_PROC_SIZE(5);
               
               if (arg)
                  sscanf(arg, "%d", &n);
               else {
                  n = 1;
               }
               ucp[num_ucp - 1][cproclen] = CMD_PUSHN;
               ucp[num_ucp - 1][cproclen + 1] = n & 0xFF;
               ucp[num_ucp - 1][cproclen + 2] = 0;//(n >> 8) & 0xFF;
               ucp[num_ucp - 1][cproclen + 3] = 0;//(n >> 16) & 0xFF;
               ucp[num_ucp - 1][cproclen + 4] = 0;//(n >> 24) & 0xFF;
               cproclen += 5;
            } else if ((streq(opt, "popn")||(streq(opt, "pop") && !arg[0])) && (context != NO_CONTEXT)) {
               RESERVE_PROC_SIZE(1);
               
               ucp[num_ucp - 1][cproclen] = CMD_PUSHN;
               cproclen ++;
            } else if (streq(opt, "chkcounter") && (context != NO_CONTEXT)) {
               unsigned int n;

               RESERVE_PROC_SIZE(6);
               
               if (arg)
                  sscanf(arg, "%d", &n);
               else {
                  n = 1;
               }
               ucp[num_ucp - 1][cproclen] = CMD_PUSHN;
               ucp[num_ucp - 1][cproclen + 1] = n & 0xFF;
               ucp[num_ucp - 1][cproclen + 2] = (n >> 8) & 0xFF;
               ucp[num_ucp - 1][cproclen + 3] = (n >> 16) & 0xFF;
               ucp[num_ucp - 1][cproclen + 4] = (n >> 24) & 0xFF;
               ucp[num_ucp - 1][cproclen + 5] = CMD_CHKCOUNTER;
               cproclen += 6;
            } else if ((streq(opt, "create")||streq(opt, "spawn")||streq(opt, "build")) && (context == CONTEXT_UNIT)) {
               unsigned int n;

               /* We must get the object ID from the global list of objects */
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(6);
               
               if (arg) {
                  n = key_to_id(arg);
               } else {
                  n = key_to_id("");
               }
               ucp[num_ucp - 1][cproclen] = CMD_PUSHN;
               ucp[num_ucp - 1][cproclen + 1] = n & 0xFF;
               ucp[num_ucp - 1][cproclen + 2] = (n >> 8) & 0xFF;
               ucp[num_ucp - 1][cproclen + 3] = (n >> 16) & 0xFF;
               ucp[num_ucp - 1][cproclen + 4] = (n >> 24) & 0xFF;
               ucp[num_ucp - 1][cproclen + 5] = CMD_SPAWN;
               cproclen += 6;
            } else if (streq(opt, "chkres") && (context == CONTEXT_UNIT)) {
               unsigned int n;

               /* We must get the object ID from the global list of objects */
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(6);
               
               if (arg) {
                  n = key_to_id(arg);
               } else {
                  n = key_to_id("");
               }
               ucp[num_ucp - 1][cproclen] = CMD_PUSHN;
               ucp[num_ucp - 1][cproclen + 1] = n & 0xFF;
               ucp[num_ucp - 1][cproclen + 2] = (n >> 8) & 0xFF;
               ucp[num_ucp - 1][cproclen + 3] = (n >> 16) & 0xFF;
               ucp[num_ucp - 1][cproclen + 4] = (n >> 24) & 0xFF;
               ucp[num_ucp - 1][cproclen + 5] = CMD_CHKRES;
               cproclen += 6;
            } else if (streq(opt, "unload") && (context == CONTEXT_UNIT)) {
               RESERVE_PROC_SIZE(1);
               
               ucp[num_ucp - 1][cproclen] = CMD_UNLOAD;
               cproclen++;
            } else if (streq(opt, "setsw") && (context != NO_CONTEXT)) {
               /* Check switch */
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(5);
               
               ucp[num_ucp - 1][cproclen] = CMD_SETSW;
               sscanf(arg, "%d", &b);
               ucp[num_ucp - 1][cproclen + 1] = b & 0xFF;
               ucp[num_ucp - 1][cproclen + 2] = (b >> 8) & 0xFF;
               ucp[num_ucp - 1][cproclen + 3] = (b >> 16) & 0xFF;
               ucp[num_ucp - 1][cproclen + 4] = (b >> 24) & 0xFF;
               cproclen += 5;
            } else if (streq(opt, "clrsw") && (context != NO_CONTEXT)) {
               /* Check switch */
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(5);
               
               ucp[num_ucp - 1][cproclen] = CMD_CLRSW;
               sscanf(arg, "%d", &b);
               ucp[num_ucp - 1][cproclen + 1] = b & 0xFF;
               ucp[num_ucp - 1][cproclen + 2] = (b >> 8) & 0xFF;
               ucp[num_ucp - 1][cproclen + 3] = (b >> 16) & 0xFF;
               ucp[num_ucp - 1][cproclen + 4] = (b >> 24) & 0xFF;
               cproclen += 5;
            } else if (streq(opt, "chksw") && (context != NO_CONTEXT)) {
               /* Check switch */
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(5);
               
               ucp[num_ucp - 1][cproclen] = CMD_CHKSW;
               sscanf(arg, "%d", &b);
               ucp[num_ucp - 1][cproclen + 1] = b & 0xFF;
               ucp[num_ucp - 1][cproclen + 2] = (b >> 8) & 0xFF;
               ucp[num_ucp - 1][cproclen + 3] = (b >> 16) & 0xFF;
               ucp[num_ucp - 1][cproclen + 4] = (b >> 24) & 0xFF;
               cproclen += 5;
            } else if (streq(opt, "hpadd") && (context == CONTEXT_UNIT)) {
               RESERVE_PROC_SIZE(5);
               
               ucp[num_ucp - 1][cproclen] = CMD_HPADD;
               sscanf(arg, "%d", &b);
               ucp[num_ucp - 1][cproclen + 1] = b & 0xFF;
               ucp[num_ucp - 1][cproclen + 2] = (b >> 8) & 0xFF;
               ucp[num_ucp - 1][cproclen + 3] = (b >> 16) & 0xFF;
               ucp[num_ucp - 1][cproclen + 4] = (b >> 24) & 0xFF;
               cproclen += 5;
            } else if (streq(opt, "hpbound") && (context == CONTEXT_UNIT)) {
               RESERVE_PROC_SIZE(1);
               
               ucp[num_ucp - 1][cproclen] = CMD_HPBOUND;
               cproclen += 1;
            } else if (streq(opt, "cmphp") && (context == CONTEXT_UNIT)) {
               RESERVE_PROC_SIZE(5);
               
               ucp[num_ucp - 1][cproclen] = CMD_CMPHP;
               sscanf(arg, "%d", &b);
               ucp[num_ucp - 1][cproclen + 1] = b & 0xFF;
               ucp[num_ucp - 1][cproclen + 2] = (b >> 8) & 0xFF;
               ucp[num_ucp - 1][cproclen + 3] = (b >> 16) & 0xFF;
               ucp[num_ucp - 1][cproclen + 4] = (b >> 24) & 0xFF;
               cproclen += 5;
            } else if (streq(opt, "lint") && (context == CONTEXT_UNIT)) {
               RESERVE_PROC_SIZE(5);
               
               ucp[num_ucp - 1][cproclen] = CMD_LINT;
               sscanf(arg, "%d", &b);
               ucp[num_ucp - 1][cproclen + 1] = b & 0xFF;
               ucp[num_ucp - 1][cproclen + 2] = (b >> 8) & 0xFF;
               ucp[num_ucp - 1][cproclen + 3] = (b >> 16) & 0xFF;
               ucp[num_ucp - 1][cproclen + 4] = (b >> 24) & 0xFF;
               cproclen += 5;
            } else if (streq(opt, "lrad") && (context == CONTEXT_UNIT)) {
               RESERVE_PROC_SIZE(5);
               
               ucp[num_ucp - 1][cproclen] = CMD_LRAD;
               sscanf(arg, "%d", &b);
               ucp[num_ucp - 1][cproclen + 1] = b & 0xFF;
               ucp[num_ucp - 1][cproclen + 2] = (b >> 8) & 0xFF;
               ucp[num_ucp - 1][cproclen + 3] = (b >> 16) & 0xFF;
               ucp[num_ucp - 1][cproclen + 4] = (b >> 24) & 0xFF;
               cproclen += 5;
            } else if (streq(opt, "mpadd") && (context == CONTEXT_UNIT)) {
               RESERVE_PROC_SIZE(5);
               
               ucp[num_ucp - 1][cproclen] = CMD_MPADD;
               sscanf(arg, "%d", &b);
               ucp[num_ucp - 1][cproclen + 1] = b & 0xFF;
               ucp[num_ucp - 1][cproclen + 2] = (b >> 8) & 0xFF;
               ucp[num_ucp - 1][cproclen + 3] = (b >> 16) & 0xFF;
               ucp[num_ucp - 1][cproclen + 4] = (b >> 24) & 0xFF;
               cproclen += 5;
            } else if (streq(opt, "mpbound") && (context == CONTEXT_UNIT)) {
               RESERVE_PROC_SIZE(1);
               
               ucp[num_ucp - 1][cproclen] = CMD_MPBOUND;
               cproclen += 1;
            } else if (streq(opt, "cmpmp") && (context == CONTEXT_UNIT)) {
               RESERVE_PROC_SIZE(5);
               
               ucp[num_ucp - 1][cproclen] = CMD_CMPMP;
               sscanf(arg, "%d", &b);
               ucp[num_ucp - 1][cproclen + 1] = b & 0xFF;
               ucp[num_ucp - 1][cproclen + 2] = (b >> 8) & 0xFF;
               ucp[num_ucp - 1][cproclen + 3] = (b >> 16) & 0xFF;
               ucp[num_ucp - 1][cproclen + 4] = (b >> 24) & 0xFF;
               cproclen += 5;
            } else if (streq(opt, "damage0") && (context == CONTEXT_UNIT)) {
               /* Check switch */
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(9);
               
               ucp[num_ucp - 1][cproclen] = CMD_DAMAGE0;
               sscanf(arg, "%d,%d", &b, &r);
               ucp[num_ucp - 1][cproclen + 1] = b & 0xFF;
               ucp[num_ucp - 1][cproclen + 2] = (b >> 8) & 0xFF;
               ucp[num_ucp - 1][cproclen + 3] = (b >> 16) & 0xFF;
               ucp[num_ucp - 1][cproclen + 4] = (b >> 24) & 0xFF;
               ucp[num_ucp - 1][cproclen + 5] = r & 0xFF;
               ucp[num_ucp - 1][cproclen + 6] = (r >> 8) & 0xFF;
               ucp[num_ucp - 1][cproclen + 7] = (r >> 16) & 0xFF;
               ucp[num_ucp - 1][cproclen + 8] = (r >> 24) & 0xFF;

               cproclen += 9;
            } else if (streq(opt, "tlprtl") && (context == CONTEXT_UNIT)) {
               popup_message("Local teleports not yet implemented!");
            } else if (streq(opt, "tlprtf") && (context == CONTEXT_UNIT)) {
               popup_message("Far teleports not yet implemented!");
            } else if (streq(opt, "showf") && (context == CONTEXT_UNIT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(2);
               
               ucp[num_ucp - 1][cproclen] = CMD_SHOWF;
               sscanf(arg, "%d", &b);
               ucp[num_ucp - 1][cproclen + 1] = b;
               cproclen += 2;
            } else if (streq(opt, "move") && (context == CONTEXT_UNIT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(2);
               
               ucp[num_ucp - 1][cproclen] = CMD_MOVE;
               sscanf(arg, "%d", &b);
               ucp[num_ucp - 1][cproclen + 1] = b;
               cproclen += 2;
            } else if (streq(opt, "reset") && (context != NO_CONTEXT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(1);
               
               ucp[num_ucp - 1][cproclen] = CMD_RESET;
               cproclen++;
            } else if (streq(opt, "left") && (context == CONTEXT_UNIT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(2);
               
               ucp[num_ucp - 1][cproclen] = CMD_CHDIR;
               ucp[num_ucp - 1][cproclen + 1] = LEFT;
               cproclen += 2;
            } else if (streq(opt, "right") && (context == CONTEXT_UNIT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(2);
               
               ucp[num_ucp - 1][cproclen] = CMD_CHDIR;
               ucp[num_ucp - 1][cproclen + 1] = RIGHT;
               cproclen += 2;
            } else if (streq(opt, "backward") && (context == CONTEXT_UNIT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(2);

               ucp[num_ucp - 1][cproclen] = CMD_CHDIR;
               ucp[num_ucp - 1][cproclen + 1] = BACKWARD;
               cproclen += 2;
            } else if (streq(opt, "north") && (context == CONTEXT_UNIT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(2);
               
               ucp[num_ucp - 1][cproclen] = CMD_CHDIR;
               ucp[num_ucp - 1][cproclen + 1] = NORTH;
               cproclen += 2;
            } else if (streq(opt, "east") && (context == CONTEXT_UNIT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(2);
               
               ucp[num_ucp - 1][cproclen] = CMD_CHDIR;
               ucp[num_ucp - 1][cproclen + 1] = EAST;
               cproclen += 2;
            } else if (streq(opt, "south") && (context == CONTEXT_UNIT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(2);
               
               ucp[num_ucp - 1][cproclen] = CMD_CHDIR;
               ucp[num_ucp - 1][cproclen + 1] = SOUTH;
               cproclen += 2;
            } else if (streq(opt, "west") && (context == CONTEXT_UNIT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(2);
               
               ucp[num_ucp - 1][cproclen] = CMD_CHDIR;
               ucp[num_ucp - 1][cproclen + 1] = WEST;
               cproclen += 2;
            } else if (streq(opt, "northeast") && (context == CONTEXT_UNIT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(2);
               
               ucp[num_ucp - 1][cproclen] = CMD_CHDIR;
               ucp[num_ucp - 1][cproclen + 1] = NORTH_EAST;
               cproclen += 2;
            } else if (streq(opt, "southeast") && (context == CONTEXT_UNIT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(2);
               
               ucp[num_ucp - 1][cproclen] = CMD_CHDIR;
               ucp[num_ucp - 1][cproclen + 1] = SOUTH_EAST;
               cproclen += 2;
            } else if (streq(opt, "northwest") && (context == CONTEXT_UNIT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(2);
               
               ucp[num_ucp - 1][cproclen] = CMD_CHDIR;
               ucp[num_ucp - 1][cproclen + 1] = NORTH_WEST;
               cproclen += 2;
            } else if (streq(opt, "southwest") && (context == CONTEXT_UNIT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(2);
               
               ucp[num_ucp - 1][cproclen] = CMD_CHDIR;
               ucp[num_ucp - 1][cproclen + 1] = SOUTH_WEST;
               cproclen += 2;
            } else if (streq(opt, "randdir") && (context == CONTEXT_UNIT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(2);
               
               ucp[num_ucp - 1][cproclen] = CMD_CHDIR;
               ucp[num_ucp - 1][cproclen + 1] = RANDOM;
               cproclen += 2;
            } else if (streq(opt, "wait1") && (context != NO_CONTEXT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(6);
               
               sscanf(arg, "%d", &b);
               ucp[num_ucp - 1][cproclen] = CMD_PUSHN;
               ucp[num_ucp - 1][cproclen + 1] = b;
               ucp[num_ucp - 1][cproclen + 2] = 0;
               ucp[num_ucp - 1][cproclen + 3] = 0;
               ucp[num_ucp - 1][cproclen + 4] = 0;
               ucp[num_ucp - 1][cproclen + 5] = CMD_WAIT1;
               cproclen += 6;
            } else if (streq(opt, "wait2") && (context != NO_CONTEXT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(6);
               
               ucp[num_ucp - 1][cproclen] = CMD_PUSHN;
               sscanf(arg, "%d", &b);
               ucp[num_ucp - 1][cproclen + 1] = b;
               ucp[num_ucp - 1][cproclen + 2] = 0;
               ucp[num_ucp - 1][cproclen + 3] = 0;
               ucp[num_ucp - 1][cproclen + 4] = 0;
               ucp[num_ucp - 1][cproclen + 5] = CMD_WAIT2;
               cproclen += 6;
            } else if (streq(opt, "wait3") && (context != NO_CONTEXT)) {
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(6);
               
               ucp[num_ucp - 1][cproclen] = CMD_PUSHN;
               sscanf(arg, "%d", &b);
               ucp[num_ucp - 1][cproclen + 1] = b;
               ucp[num_ucp - 1][cproclen + 2] = 0;
               ucp[num_ucp - 1][cproclen + 3] = 0;
               ucp[num_ucp - 1][cproclen + 4] = 0;
               ucp[num_ucp - 1][cproclen + 5] = CMD_WAIT3;
               cproclen += 6;
               
            /* Bare opcodes/binary output */
            } else if ((streq(opt, "$create")||streq(opt, "$spawn")||streq(opt, "$build")) && (context == CONTEXT_UNIT)) {
               /* We must get the object ID from the global list of objects */
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(1);
               
               ucp[num_ucp - 1][cproclen] = CMD_SPAWN;
               cproclen ++;
            } else if (streq(opt, "$advcounter") && (context == CONTEXT_UNIT)) {
               /* We must get the object ID from the global list of objects */
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(1);
               
               ucp[num_ucp - 1][cproclen] = CMD_ADVCOUNTER;
               cproclen ++;
            } else if (streq(opt, "$wait1") && (context == CONTEXT_UNIT)) {
               /* We must get the object ID from the global list of objects */
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(1);
               
               ucp[num_ucp - 1][cproclen] = CMD_WAIT1;
               cproclen ++;
            } else if (streq(opt, "$chkres") && (context == CONTEXT_UNIT)) {
               /* We must get the object ID from the global list of objects */
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(1);
               
               ucp[num_ucp - 1][cproclen] = CMD_CHKRES;
               cproclen ++;
            } else if (streq(opt, "$chng") && (context == CONTEXT_UNIT)) {
               /* We must get the object ID from the global list of objects */
               /* Check if enough memory to store program */
               RESERVE_PROC_SIZE(1);
               
               ucp[num_ucp - 1][cproclen] = CMD_CHANGE;
               cproclen ++;
            } else {
               if (proc_name)
                  popup_message("%s, line %d [%s]:\nUnknown command: %s %s",
                                filename, ln, proc_name, opt, arg);
               else
                  popup_message("%s, line %d:\nUnknown command: %s %s",
                                filename, ln, opt, arg);

            }
         }
      }
   }
   fclose(f);
   free(s);
}

/* Get a pointer to the start of the unit controlprogram 'name' */
unsigned char *get_prog_ptr(char *name)
{
   int c;

   for (c = 0; c < num_ucp; c++) {
      if (streq(name, ucpname[c])) {
         return ucp[c];
      }
   }
   return NULL;
}
