/**********************************************/
/* Rise of the Tribes                         */
/* C version, Take 2                          */
/* Evert Glebbeek 1998, 2002                  */
/* eglebbk@phys.uva.nl                        */
/**********************************************/
#include <allegro.h>
#include "opcodes.h"
#include "dialog.h"
#include "playgame.h"
#include "map.h"
#include "maptile.h"
#include "unit.h"
#include "unitprog.h"
#include "ugrp.h"
#include "script.h"
#include "udraw.h"
#include "global.h"
#include "unitmap.h"
#include "gamedraw.h"
#include "gamemsg.h"
#include "cheats.h"

/* Helper functions */
static int pop_script_stack(UNITDATA *c)
{
   ASSERT(c->sp<8);

   return c->ss[c->sp++];
}

static void push_script_stack(UNITDATA *c, int n)
{
   ASSERT(c->sp);

   c->ss[--c->sp] = n;
}


static int oc_nop(UNITDATA *c, int *loop)
{
   c->ip++;
   return 0;
}

/* update unit status and orders */
static int oc_reset(UNITDATA *c, int *loop)
{
   int n;
   
   c->script_target = c;
   c->cprog->progress = 0;
   c->cprog->flags &= ~(SCRIPT_CANCEL|SCRIPT_SIGNAL);
   c->flags &= ~(CF_INVISIBLE | CF_PROGRESS);
   if (c->draw_dir != c->direction) {
      unregister_unit(c);
      c->draw_dir = c->direction;
      register_unit(c);
      mark_unit(c);
      set_screen_changed();
   }
   /* Building sites should take their icon/hit points/etc from what they'll */
   /*  become when they're done. The end product is in the arg1 variable */
   if (c->flags&CC_BUILDSITE) {
      c->icon = get_char_icon(c->cprog->argv[1]);
      c->name = get_char_name(c->cprog->argv[1]);
      c->maxhp = get_char_maxhp(c->cprog->argv[1]);
      c->maxmp = get_char_maxmp(c->cprog->argv[1]);
      for (n=0; n<8; n++) {
         int r = get_char_resource_cost(c->cprog->argv[1], n);
         if (r>=0) {
            c->resource_cost[n] = r;
            mark_interface_area(INT_RESOURCE);
         }
      }
   }
   c->sp = 8;
   c->ip++;
   (*loop)--;
   return 0;
}

/* make unit invisible; atomic */
static int oc_invsbl(UNITDATA *c, int *loop)
{
   c->flags |= CF_INVISIBLE;
   c->ip++;
   (*loop)--;
   return 0;
}

/* Treat a wait command as a nop command (that is, ignore arguments) */
static int oc_wait_is_nop(UNITDATA *c, int *loop)
{
   pop_script_stack(c);
   *loop = 0;
   c->ip++;
   return 0;
}

/* Take the time needed to execute the 'wait' command into account */
static int oc_wait1(UNITDATA *c, int *loop)
{
   c->wait_counter = pop_script_stack(c) - 1;
   c->ip ++;
   *loop = 0;
   return 0;
}

/* Take the time needed to execute the 'wait' command into account */
/* This version of the WAIT command adjusts the wait time to the unit's */
/*  speed characteristic */
static int oc_wait3(UNITDATA *c, int *loop)
{
   c->wait_counter = (pop_script_stack(c)) * 10 / c->speed;
   if (c->wait_counter)
      c->wait_counter--;
   c->ip ++;
   *loop = 0;
   return 0;
}

static int oc_showf(UNITDATA *c, int *loop)
{
   /* Is this a normal object or a map object? */
   if (is_mapobject(c->script_target)) {
      /* We store the tile pointer in the gfx structure */
      /* The base tile (the bottom layer, so to speak, is in the */
      /* c->tiles structure */
      /*
         MAP_TILE *t = (MAP_TILE *)c->script_target->icon;
         t->gfx_index = prog->prog[c->ip+1];
         t->gfx = get_tile_bitmap(t->gfx_index);
         mark_tile (c->script_target->tiles->tile);
       */
   } else {
      /* parameter: frame number */
      set_screen_changed();
      unregister_unit(c->script_target);
      c->script_target->current_frame = c->cprog->prog[c->ip + 1];
      c->script_target->flags &= ~CF_INVISIBLE;
      register_unit(c->script_target);
      mark_unit(c);
   }
   c->ip += 2;
   return 0;
}

static int oc_move(UNITDATA *c, int *loop)
{
   int x, y;

   /* parameter: number of pixels */
   set_screen_changed();
   x = c->script_target->x +
      dir_to_dx(c->script_target->draw_dir) * c->cprog->prog[c->ip + 1];
   y = c->script_target->y -
      dir_to_dy(c->script_target->draw_dir) * c->cprog->prog[c->ip + 1];
   if ((x >= 0) && (y >= 0) && (x < TILE_WIDTH * (get_map_width() - 1)) &&
       (y < TILE_HEIGHT * (get_map_height() - 1))) {
      unregister_unit(c->script_target);
      c->script_target->x = x;
      c->script_target->y = y;
      register_unit(c->script_target);
      mark_unit(c);
   }
   c->ip += 2;
   return 0;
}

static int oc_chdir(UNITDATA *c, int *loop)
{
   if (c->cprog->prog[c->ip + 1] <= NORTH_WEST) {
      c->script_target->direction = c->cprog->prog[c->ip + 1];
   } else {
      switch (c->cprog->prog[c->ip + 1]) {
         case RANDOM:
            break;
         case LEFT:
            c->script_target->direction -= 2;
            c->script_target->direction %= 8;
            break;
         case RIGHT:
            c->script_target->direction += 2;
            c->script_target->direction %= 8;
            break;
         case BACKWARD:
            c->script_target->direction += 4;
            c->script_target->direction %= 8;
            break;
         default:
      }                         /* End of switch */
   }
   c->ip += 2;
   return 0;
}

static int oc_alert(UNITDATA *c, int *loop)
{
   popup_message("Alert!");
   c->ip++;
   return 0;
}

static int oc_return(UNITDATA *c, int *loop)
{
   /* Remove current control program, prepare for next one */
   release_prog_head(c);
   *loop = 0;                   // We're absolutely really done now
   return 0;
}

static int prog_cmd_jmp(UNITDATA *c, int *loop)
{
   unsigned int n = 0;

   n = 0;
   n |= c->cprog->prog[c->ip + 1] + (c->cprog->prog[c->ip + 2] << 8);
   c->ip = n;
   (*loop)--;                   // Don't return yet
   return 0;
}

/* Jump if Cancel flag is set */
static int prog_cmd_jcf(UNITDATA *c, int *loop)
{
   unsigned int n = 0;

   /* Conditional jump (CF=1) */
   if (c->cprog->flags & SCRIPT_CANCEL) {
      n = 0;
      n |= c->cprog->prog[c->ip + 1] + (c->cprog->prog[c->ip + 2] << 8);
      c->ip = n;
   } else {
      c->ip += 3;
   }
   (*loop)--;                   // Don't return yet
   return 0;
}

static int prog_cmd_je(UNITDATA *c, int *loop)
{
   unsigned int n = 0;

   /* Conditional jump (ZF=1) */
   if (c->cprog->flags & SCRIPT_ZF) {
      n = 0;
      n |= c->cprog->prog[c->ip + 1] + (c->cprog->prog[c->ip + 2] << 8);
      c->ip = n;
   } else {
      c->ip += 3;
   }
   (*loop)--;                   // Don't return yet
   return 0;
}

static int prog_cmd_jne(UNITDATA *c, int *loop)
{
   unsigned int n = 0;

   /* Conditional jump (ZF=0) */
   if (c->cprog->flags & SCRIPT_ZF) {
      c->ip += 3;
   } else {
      n = 0;
      n |= c->cprog->prog[c->ip + 1] + (c->cprog->prog[c->ip + 2] << 8);
      c->ip = n;
   }
   (*loop)--;                   // Don't return yet
   return 0;
}

static int prog_cmd_ja(UNITDATA *c, int *loop)
{
   unsigned int n = 0;

   /* Conditional jump (SF=0, ZF=0) */
   if (!(c->cprog->flags & (SCRIPT_ZF + SCRIPT_SF))) {
      n = 0;
      n |= c->cprog->prog[c->ip + 1] + (c->cprog->prog[c->ip + 2] << 8);
      c->ip = n;
   } else {
      c->ip += 3;
   }
   (*loop)--;                   // Don't return yet
   return 0;
}

static int prog_cmd_jb(UNITDATA *c, int *loop)
{
   unsigned int n = 0;

   /* Conditional jump (SF=1, ZF=0) */
   if (c->cprog->flags & SCRIPT_ZF && !(c->cprog->flags & SCRIPT_SF)) {
      n = 0;
      n |= c->cprog->prog[c->ip + 1] + (c->cprog->prog[c->ip + 2] << 8);
      c->ip = n;
   } else {
      c->ip += 3;
   }
   (*loop)--;                   // Don't return yet
   return 0;
}

static int prog_cmd_jae(UNITDATA *c, int *loop)
{
   unsigned int n = 0;

   /* Conditional jump (SF=0 or ZF=1) */
   if (!(c->cprog->flags & SCRIPT_SF) || (c->cprog->flags & SCRIPT_ZF)) {
      n = 0;
      n |= c->cprog->prog[c->ip + 1] + (c->cprog->prog[c->ip + 2] << 8);
      c->ip = n;
   } else {
      c->ip += 3;
   }
   (*loop)--;                   // Don't return yet
   return 0;
}

static int prog_cmd_jbe(UNITDATA *c, int *loop)
{
   unsigned int n = 0;

   /* Conditional jump (SF=1 or ZF=1) */
   if (c->cprog->flags & (SCRIPT_ZF + SCRIPT_SF)) {
      n = 0;
      n |= c->cprog->prog[c->ip + 1] + (c->cprog->prog[c->ip + 2] << 8);
      c->ip = n;
   } else {
      c->ip += 3;
   }
   (*loop)--;                   // Don't return yet
   return 0;
}

static int prog_cmd_clrzf(UNITDATA *c, int *loop)
{
   c->cprog->flags &= ~SCRIPT_ZF;
   c->ip++;
   (*loop)--;                   // Don't return yet
   return 0;
}

static int prog_cmd_chkd(UNITDATA *c, int *loop)
{
   UNIT *chr;
   int dx, dy;
   unsigned int tx, ty;
   unsigned int x, y;

   dx = dir_to_dx(c->script_target->draw_dir);
   dy = dir_to_dy(c->script_target->draw_dir);

   tx = c->script_target->x / TILE_WIDTH;
   ty = c->script_target->y / TILE_HEIGHT;
   x = tx + dx;
   y = ty - dy;

   if (tile_flags(x, y, c->script_target->layer) & TILE_BARRIER) {
      c->cprog->flags |= SCRIPT_ZF;
   } else {
      chr = get_block_char(x, y);
      while (chr) {
         if ((chr->data != c) && !(chr->data->flags & CF_NOBLOCK) &&
             (chr->data->layer == c->script_target->layer)) {
            c->cprog->flags |= SCRIPT_ZF;
            break;
         }
         chr = chr->next;
      }
   }


   c->ip++;
   (*loop)--;                   // Don't return yet
   return 0;
}

static int prog_cmd_chkc(UNITDATA *c, int *loop)
{
   unsigned int tx, ty;

   tx = c->script_target->x / TILE_WIDTH;
   ty = c->script_target->y / TILE_HEIGHT;

   if (tile_blocked(tx, ty, c->script_target->layer, NULL))
      c->cprog->flags |= SCRIPT_ZF;

   c->ip++;
   (*loop)--;                   // Don't return yet
   return 0;
}

/* Clear current tile occupied flag */
static int prog_cmd_clrc(UNITDATA *c, int *loop)
{
   clear_unit_block(c);

   c->ip++;
   (*loop)--;                   // Don't return yet
   return 0;
}

/* Set current tile occupied flag */
static int prog_cmd_setc(UNITDATA *c, int *loop)
{
   set_unit_block(c);

   c->ip++;
   (*loop)--;                   // Don't return yet
   return 0;
}

/* Clear destination tile occupied flag */
static int prog_cmd_clrd(UNITDATA *c, int *loop)
{
   clear_unit_dest_block(c);

   c->ip++;
   (*loop)--;                   // Don't return yet
   return 0;
}

/* Set destination tile occupied flag */
static int prog_cmd_setd(UNITDATA *c, int *loop)
{
   set_unit_dest_block(c);

   c->ip++;
   (*loop)--;                   // Don't return yet
   return 0;
}

/* Check the progress indicator */
static int prog_cmd_chkcounter(UNITDATA *c, int *loop)
{
   int n;

   n = pop_script_stack(c);//c->cprog->prog[c->ip + 1] | c->cprog->prog[c->ip + 2] << 8;

   if (c->cprog->progress >= n)
      c->cprog->flags |= SCRIPT_ZF;

   c->ip++;
   (*loop)--;                   // Don't return yet
   return 0;
}

static int prog_cmd_remove(UNITDATA *c, int *loop)
{
   set_screen_changed();
   c->ip++;
   return 1;
}


/* Check if enough resources to create a certain unit */
/* Set ZF if ok */
static int prog_cmd_chkres(UNITDATA *c, int *loop)
{
   int n;
   int r;
   int rc;
   int ok = TRUE;
   
   n = pop_script_stack(c);

   if (!(cheat_flags & CHEAT_CONTROL)) {
      for (r=1; r<3; r++) {

         rc = get_char_resource_cost(n, r);
         
         if ((rc>0) && (rc+get_player_upkeep(c->player, r) > get_player_provision(c->player, r))) {
            if (!(c->cprog->flags&SCRIPT_SIGNAL)) {
               broadcast_message(GAMEMSG_NOR0+r,c->player,c->player);
               c->cprog->flags|=SCRIPT_SIGNAL;
               set_screen_changed();
            }
            ok = FALSE;
         }
      }
      /* Check resource 0 (morale) */
      if ((get_char_resource_cost(n, 0)>0) && (get_char_resource_cost(n, 0)+get_player_upkeep(c->player, 0) > get_player_provision(c->player, 0))) {
         if (!(c->cprog->flags&SCRIPT_SIGNAL) && ok) {
            broadcast_message(GAMEMSG_NOR0+0,c->player,c->player);
            c->cprog->flags|=SCRIPT_SIGNAL;
            set_screen_changed();
         }
      }
   }
   
   if (ok)
      c->cprog->flags |= SCRIPT_ZF;

   c->ip ++;
   (*loop)--;
   return 0;
}

/* Create a child unit loaded into the current structure, or a structure with */
/*  the current unit loaded inside it */
static int prog_cmd_spawn(UNITDATA *c, int *loop)
{
   UNIT *u;
   unsigned int n = 0;

   /* Check cancel flag */
   /* We check that here because this instruction can wait until upkeep is */
   /*  available for the unit to build */
   if (c->cprog->flags&SCRIPT_CANCEL) {
      c->ip++;
      return 0;
   }

   /* Get unit type (first argument) */
   n = pop_script_stack(c);
   
   //popup_message("unit-type = %d", n);

   u = create_unit(n);

   /* If there is still data on the stack, we pass this as arguments to the */
   /*  init script */
   for (n=c->sp; n<8; n++) {
      pass_script_argument(u->data, c->ss[n]);
      //popup_message("argv[%d] = %d", 7-n, c->ss[n]);
   }

   /* Now we need to know what to do next: if the unit we just `spawned' is */
   /*  in fact a unit, then load it into the parent. */
   /* Otherwise, if the unit is a structure, then load the parent into the */
   /*  structure. */
   if (is_structure(u->data)) {
      /* Set coordinates of structure: here */
      set_unit_coors(u->data, c->x, c->y);

      if (c->flags & CF_SELECTED)
         mark_interface_area(INT_COMMANDS|INT_GROUP|INT_CAPTAIN|INT_INFO|INT_QUEUE);

      /* Remove worker from the map and put him inside the structure */
      mark_unit(c);
      unregister_unit(c);
      clear_unit_block(c);
      load_unit(c, u->data);
      c->flags |= (CF_NOORDERS + CF_NOSELECT);
      remove_from_active_selections(c);
      set_command_panel();
      free_unit(take_from_player(c->player, c));

      /* Give the new structure to the player and register it */
      give_to_player(c->player, u);
      mark_unit(u->data);
      register_unit(u->data);

      set_screen_changed();
   } else {
      load_unit(u->data, c);
      free_unit(u);
   }

   if (c->player==get_local_player()) {
      mark_interface_area(INT_RESOURCE);
      set_screen_changed();
   }

   c->ip ++;
   return 0;
}

/* Unload the first of the loaded_units() */
static int prog_cmd_unload(UNITDATA *c, int *loop)
{
   UNIT *u;
   int dx;
   int dy;
   int r = -1;

   u = c->loaded_units;
   /* Remove unit from the list of loaded units */
   if (u) {
      c->loaded_units = c->loaded_units->next;
      if (c->loaded_units)
         c->loaded_units->prev = NULL;
      u->prev = u->next = NULL;
   } else {
      c->ip++;
      return 0;
   }

   for (r = 0; r < 12; r++)
      for (dy = -r; dy < r + c->gfx_y_size; dy++)
         for (dx = -r; dx < r + c->gfx_x_size; dx++) {
            if (in_rect
                (c->script_target->x / TILE_WIDTH + dx,
                 c->script_target->y / TILE_HEIGHT + dy, 0, 0,
                 get_map_width() - 1, get_map_height() - 1))
               if (!tile_blocked
                   (c->script_target->x / TILE_WIDTH + dx,
                    c->script_target->y / TILE_HEIGHT + dy,
                    c->script_target->layer, NULL)) {
                  set_unit_coors(u->data,
                                 c->script_target->x + dx * TILE_WIDTH,
                                 c->script_target->y + dy * TILE_HEIGHT);
                  r = -1;
                  goto unit_placed;
               }
         }
 unit_placed:

   if (r < 0) {
      /* Check rally points */
      /*
         if (is_structure(c->script_target)) {
         u->data->ai_target.x = c->script_target->ai_target.x;
         u->data->ai_target.y = c->script_target->ai_target.y;
         u->data->ai_target.unit = c->script_target->ai_target.unit;
         }
       */

      register_unit(u->data);
      mark_unit(u->data);
      set_unit_block(u->data);
      u->data->flags &= ~(CF_NOORDERS + CF_NOSELECT);
      give_to_player(c->script_target->player, u);
      set_screen_changed();
   } else {
      release_unit(u);
   }

   c->ip++;
   return 0;
}

static int prog_cmd_cmdoff(UNITDATA *c, int *loop)
{
   c->flags |= CF_NOORDERS;
   c->ip++;
   return 0;
}

static int prog_cmd_cmdon(UNITDATA *c, int *loop)
{
   c->flags &= ~CF_NOORDERS;
   c->ip++;
   return 0;
}

static int prog_cmd_target(UNITDATA *c, int *loop)
{
   SCRIPT_STACK *prog = c->cprog;

   switch (prog->prog[c->ip + 1]) {
      case 3:                  /* Ai-target */
         if (c->ai_target.unit)
            c->script_target = c->ai_target.unit;
         break;
      case 2:                  /* Attacker */
         if (c->attacker)
            c->script_target = c->attacker;
         break;
      case 1:                  /* Parent */
         if (c->parent)
            c->script_target = c->parent;
         break;
      case 0:                  /* Self */
      default:                 /* Assume self */
         c->script_target = c;
         break;
   }                            /* end of switch */
   c->ip += 2;
   return 0;
}

static int prog_cmd_change(UNITDATA *c, int *loop)
{
   UNITDATA *cd = NULL;
   int n;
   int w;

   w = pop_script_stack(c);
   n = pop_script_stack(c);

   switch (w) {
      case 3:                  /* Ai-target */
         if (c->ai_target.unit)
            cd = c->ai_target.unit;
         break;
      case 2:                  /* Attacker */
         if (c->attacker)
            cd = c->attacker;
         break;
      case 1:                  /* Parent */
         if (c->parent)
            cd = c->parent;
         break;
      case 0:                  /* Self */
      default:                 /* Assume self */
         cd = c;
         break;
   }                            /* end of switch */

   if (cd) {
      unregister_unit(cd);
      change_unit(cd, n);
      set_command_panel();
      if (c->flags & CF_SELECTED)
         mark_interface_area(INT_COMMANDS|INT_GROUP|INT_CAPTAIN|INT_INFO|INT_QUEUE);
      register_unit(cd);
      mark_unit(cd);
      set_screen_changed();
   }

   if (c->player==get_local_player()) {
      mark_interface_area(INT_RESOURCE);
      set_screen_changed();
   }

   c->ip++;
   return 0;
}

static int prog_cmd_atomic(UNITDATA *c, int *loop)
{
   c->ip++;
   (*loop) -= 2;                // Don't return yet, even after next command
   return 0;
}

/* Advance progress indicator in unit construction */
static int prog_cmd_advcounter(UNITDATA *c, int *loop)
{
   int n;

   n = pop_script_stack(c);

   c->cprog->progress += n;

   c->ip++;
   (*loop)--;
   return 0;
}

/* Advance progress indicator in unit construction */
static int prog_cmd_showcounter(UNITDATA *c, int *loop)
{
   c->flags |= CF_PROGRESS;

   if (c->flags & CF_SELECTED) {
      mark_interface_area(INT_QUEUE);
      set_screen_changed();
   }
   c->ip++;
   (*loop)--;
   return 0;
}
/* Advance progress indicator in unit construction */
static int prog_cmd_hidecounter(UNITDATA *c, int *loop)
{
   if (get_queuecount(c)<=1) {
      c->flags &= ~CF_PROGRESS;

      if (c->flags & CF_SELECTED) {
         mark_interface_area(INT_QUEUE);
         set_screen_changed();
      }
   }
   c->ip ++;
   (*loop)--;
   return 0;
}

/* Elementaty pushn and popn instructions */
/* These are (supposedly!) the only instructions that have their immediate */
/*  arguments inlined in the program code */

/* Push an immediate value onto the stack */
static int prog_cmd_pushn(UNITDATA *c, int *loop)
{
   int n;

   n = c->cprog->prog[c->ip + 1] | c->cprog->prog[c->ip + 2] << 8 |
         c->cprog->prog[c->ip + 3] << 16 | c->cprog->prog[c->ip + 4] << 24;

   push_script_stack(c, n);

   c->ip += 5;
   (*loop)--;
   return 0;
}

/* Pop a number off the stack (and throw it away) */
static int prog_cmd_popn(UNITDATA *c, int *loop)
{
   pop_script_stack(c);

   c->ip ++;
   (*loop)--;
   return 0;
}

/* Store a game-state variable on the stack */
static int prog_cmd_load(UNITDATA *c, int *loop)
{
   int n = c->cprog->prog[c->ip+1];
   
   switch(n) {
      case 0:
      case 1:
      case 2:
         push_script_stack(c, c->cprog->argv[n]);
         break;
      case 3:
         push_script_stack(c, c->cprog->argc);
         break;
      default:
         break;
   }
   
   c->ip +=2;
   (*loop)--;
   return 0;
}

/* Retrieve a game-state variable off the stack */
static int prog_cmd_store(UNITDATA *c, int *loop)
{
   int n = c->cprog->prog[c->ip+1];
   
   switch(n) {
      case 0:
      case 1:
      case 2:
         c->cprog->argv[n] = pop_script_stack(c);
         break;
      case 3:
         c->cprog->argc = pop_script_stack(c);
         break;
      default:
         break;
   }
   
   c->ip +=2;
   (*loop)--;
   return 0;
}

void register_opcodes(void)
{
   register_opcode(CMD_RESET, oc_reset);
   register_opcode(CMD_WAIT1, oc_wait1);
   register_opcode(CMD_WAIT2, oc_wait1);
   register_opcode(CMD_WAIT3, oc_wait3);
   register_opcode(CMD_WAIT4, oc_wait1);
   register_opcode(CMD_SHOWF, oc_showf);
   register_opcode(CMD_MOVE, oc_move);
   register_opcode(CMD_CHDIR, oc_chdir);
   register_opcode(CMD_ALERT, oc_alert);
   register_opcode(CMD_RETURN, oc_return);
   register_opcode(CMD_NOP, oc_nop);
   register_opcode(CMD_INVSBL, oc_invsbl);
   register_opcode(CMD_CLRZF, prog_cmd_clrzf);
   register_opcode(CMD_JMP, prog_cmd_jmp);
   register_opcode(CMD_JBE, prog_cmd_jbe);
   register_opcode(CMD_JAE, prog_cmd_jae);
   register_opcode(CMD_JB, prog_cmd_jb);
   register_opcode(CMD_JA, prog_cmd_ja);
   register_opcode(CMD_JNE, prog_cmd_jne);
   register_opcode(CMD_JE, prog_cmd_je);
   register_opcode(CMD_JCF, prog_cmd_jcf);
   register_opcode(CMD_CHKD, prog_cmd_chkd);
   register_opcode(CMD_CLRC, prog_cmd_clrc);
   register_opcode(CMD_SETC, prog_cmd_setc);
   register_opcode(CMD_SETD, prog_cmd_setd);
   register_opcode(CMD_CLRD, prog_cmd_clrd);
   register_opcode(CMD_CHKC, prog_cmd_chkc);
   register_opcode(CMD_SPAWN, prog_cmd_spawn);
   register_opcode(CMD_UNLOAD, prog_cmd_unload);
   register_opcode(CMD_CMDON, prog_cmd_cmdon);
   register_opcode(CMD_CMDOFF, prog_cmd_cmdoff);
   register_opcode(CMD_REMOVE, prog_cmd_remove);
   register_opcode(CMD_TARGET, prog_cmd_target);
   register_opcode(CMD_CHANGE, prog_cmd_change);
   register_opcode(CMD_ATOMIC, prog_cmd_atomic);
   register_opcode(CMD_ADVCOUNTER, prog_cmd_advcounter);
   register_opcode(CMD_CHKCOUNTER, prog_cmd_chkcounter);
   register_opcode(CMD_SHOWCNTER, prog_cmd_showcounter);
   register_opcode(CMD_HIDECNTER, prog_cmd_hidecounter);
   register_opcode(CMD_PUSHN, prog_cmd_pushn);
   register_opcode(CMD_POPN, prog_cmd_popn);
   register_opcode(CMD_STORE, prog_cmd_store);
   register_opcode(CMD_LOAD, prog_cmd_load);
   register_opcode(CMD_CHKRES, prog_cmd_chkres);
}

void disable_waittimes(void)
{
   register_opcode(CMD_WAIT1, oc_wait_is_nop);
   register_opcode(CMD_WAIT2, oc_wait_is_nop);
   register_opcode(CMD_WAIT3, oc_wait_is_nop);
   register_opcode(CMD_WAIT4, oc_wait_is_nop);
}
