/**********************************************/
/* Rise of the Tribes                         */
/* C version, Take 2                          */
/* Evert Glebbeek 1998, 2002                  */
/* eglebbk@phys.uva.nl                        */
/**********************************************/
#include <allegro.h>
#include <string.h>
#include <stdio.h>
#include "dialog.h"
#include "map.h"
#include "maptile.h"
#include "gamegfx.h"
#include "global.h"
#include "gfx.h"

#define TILE_INDEX(x,y) ((x+1)+(y+1)*map.width)

#define MAX_MAPTILES    524288

#define CORNER_FOG      1
#define CORNER_HIDDEN   2

typedef struct MAP_TILE {
   struct MAP_TILE *next;       // Next tile in list or NULL
   BITMAP *gfx;                 // Tile graphic for this tile
   int gfx_index;               // Tile graphic index (in case it is needed)
   void *ovru;                  // Tag list of overlapping units
   void *blocku;                // Tag list of blocking units
   int flags;                   // Tile flags
   int old_fw_flags;            // Old fog-of-war flags
   unsigned short mapx;         // Tile map coordinates
   unsigned short mapy;         // Tile map coordinates
   char height;                 // Tile 'level' or height (layer)
} MAP_TILE;

typedef struct WORLDMAP {
   int width;                   // Map width, in tiles
   int height;                  // Map height, in tiles
   int viewport_x;              // Map location of viewport UL corner, in
   int viewport_y;              //  pixels. (The viewport moves like a unit)
   int num_layers;              // Number tile layers
   MAP_TILE *map;               // The actual map datastructure
} WORLDMAP;

static int dir_to_dx_arr[8] = { 0, 1, 1, 1, 0, -1, -1, -1 };
static int dir_to_dy_arr[8] = { 1, 1, 0, -1, -1, -1, 0, 1 };

static WORLDMAP map = {0, 0, 0, 0, 0, NULL};

/* Tile corner flags */
static const int tf_ul = FOG_UL + HIDDEN_UL;
static const int tf_ur = FOG_UR + HIDDEN_UR;
static const int tf_ll = FOG_LL + HIDDEN_LL;
static const int tf_lr = FOG_LR + HIDDEN_LR;

/* Corner <--> tile coordinet conversions */
/* Corner coordinates are aranged so that the upper left corner of the map is */
/*  at (-1, -1). In this way, the TILE_INDEX() macro can be used to access */
/*  both the map and the corner array. dx and dy are the offsets of the tiles */
/*  that share a corner (cx, cy) */
static const int corner_dx[4] = {-1, -1, 0,  0};
static const int corner_dy[4] = {-1,  0, 0, -1};

/* Tile flags to set for each border tile at this corner */
static const int corner_fog_flags[4] = {
   FOG_LR + HIDDEN_LR, 
   FOG_UR + HIDDEN_UR, 
   FOG_UL + HIDDEN_UL, 
   FOG_LL + HIDDEN_LL
};

/* The memory for all UNITs/UNITDATA is allocated in one chunk */
static MAP_TILE_LIST *maptile_chunk = NULL;
static MAP_TILE_LIST **maptile_stack = NULL;
static int maptile_stack_counter = 0;

int dir_to_dx(const int dir)
{
   ASSERT(dir<8)
   ASSERT(dir>=0)
   
   return dir_to_dx_arr[dir];
}

int dir_to_dy(const int dir)
{
   ASSERT(dir<8)
   ASSERT(dir>=0)

   return dir_to_dy_arr[dir];
}

void dir_to_dx_dy(const int dir, int *dx, int *dy)
{
   ASSERT(dir<8)
   ASSERT(dir>=0)
   ASSERT(dx)
   ASSERT(dy)

   *dx = dir_to_dx(dir);
   *dy = dir_to_dy(dir);
}


MAP_TILE_LIST *alloc_maptile(void)
{
   if (maptile_stack_counter >= MAX_MAPTILES) {
      game_message("Too many tiles!");
      return NULL;
   }
   /* Advance stack counter */
   maptile_stack_counter++;
   /* return the address of the last free UNITDATA in the list */
   return maptile_stack[maptile_stack_counter - 1];
}

void free_maptile(MAP_TILE_LIST *c)
{

   if (maptile_stack_counter) {
      maptile_stack_counter--;
      maptile_stack[maptile_stack_counter] = c;
   }
}

/* reset the MAP_TILE global chunks */
static void reset_maptile_chunk(void)
{
   int c;

   if (!maptile_chunk)
      maptile_chunk = malloc(sizeof(MAP_TILE) * MAX_MAPTILES);
   if (!maptile_stack)
      maptile_stack = malloc(sizeof(MAP_TILE *) * MAX_MAPTILES);

   maptile_stack_counter = 0;
   maptile_stack_counter = 0;
   for (c = 0; c < MAX_MAPTILES; c++)
      maptile_stack[c] = &(maptile_chunk[c]);
}

void map_init(void)
{
   reset_maptile_chunk();
}

int map_loaded(void)
{
   return map.map!=NULL;
}

/* Convert a (dx, dy) vector to a scalar direction */
int dx_dy_to_dir(const int dx, const int dy)
{
   return ((dx < 0) && (dy < 0) ? NORTH_WEST :
           (dx < 0) && (dy == 0) ? WEST :
           (dx < 0) && (dy > 0) ? SOUTH_WEST :
           (dx == 0) && (dy > 0) ? SOUTH :
           (dx > 0) && (dy > 0) ? SOUTH_EAST :
           (dx > 0) && (dy == 0) ? EAST :
           (dx > 0) && (dy < 0) ? NORTH_EAST : NORTH);
}

/* Push Fog-of-War state and cover the map with the Fog of War */
void shroud_map(void)
{
   MAP_TILE *tile;
   int x, y;
   int c;
   
   ASSERT(map.map);

   for (y = 0; y < map.height; y++)
      for (x = 0; x < map.width; x++) {
         c = TILE_INDEX(x, y);
         tile = &(map.map[c]);
         tile->old_fw_flags = tile->flags & (tf_ul | tf_ur | tf_ll | tf_lr);
         tile->flags |= (FOG_UL | FOG_UR | FOG_LL | FOG_LR);
      }
}

/* Removes the fog-of-war from the entire map */
void unshroud_map(void)
{
   MAP_TILE *tile;
   int x, y;
   int c;
   
   ASSERT(map.map);

   for (y = 0; y < map.height; y++)
      for (x = 0; x < map.width; x++) {
         c = TILE_INDEX(x, y);
         tile = &(map.map[c]);
         tile->flags &= ~(FOG_UL | FOG_UR | FOG_LL | FOG_LR);
      }
}


/* Removes the black shroud from the map */
void reveal_map(void)
{
   MAP_TILE *tile;
   int x, y;
   int c;
   
   ASSERT(map.map);

   for (y = 0; y < get_map_height(); y++)
      for (x = 0; x < get_map_width(); x++) {
         c = TILE_INDEX(x, y);
         tile = &(map.map[c]);
         tile->flags &= ~(HIDDEN_UL|HIDDEN_UR|HIDDEN_LL|HIDDEN_LR);
      }
}


/* Reveals a tile corner */
/* Corner coordinates are shared between the lower right tile and the corner */
static void reveal_corner(const int tx, const int ty)
{
   MAP_TILE *tile;
   int c;

   ASSERT(map.map);

   for (c=0; c<4; c++) {   
      if (in_rect(tx + corner_dx[c], ty + corner_dy[c], 0, 0, map.width, map.height)) {      
         tile = &(map.map[TILE_INDEX(tx + corner_dx[c], ty + corner_dy[c])]);
         tile->flags &= ~corner_fog_flags[c];
      }
   }
}

/* Lift the Fog of War over a radius r around the location (tx, ty) */
void lift_fog(const int tx, const int ty, const int r)
{
   int rsqr = (r+1) * (r+1);
   int dx;
   int dy;
   
   /* First mark all corners */
   for (dx = -r; dx < r+1; dx++)
      for (dy = -r; dy < r+1; dy++) {
         if (in_rect(tx + dx, ty + dy, -1, -1, map.width, map.height)) {
            if ((dx * dx + dy * dy) <= rsqr) {         
               /* Mark and all tiles that are adjacent to the corner */
               reveal_corner(tx + dx, ty + dy);
            }
         }
      }
}

/* Mark fog-of-war change state */
/* Only bother for tiles that are on-screen */
void mark_fog_tiles(void)
{
   MAP_TILE *tile;
   int px = 0;
   int py = 0;
   int x;
   int y;
   int ix, iy;
   int sx, sy;
   int lx, ly;
   int flags;
   
   ASSERT(map.map);

   /* Compute the first tile visible and get it's screen coordinates */
   get_viewport_tiles(&ix, &iy, &sx, &sy, &px, &py);

   ix = iy = 0;
   sx = map.width;
   sy = map.height;
   
   lx = ix + sx;
   ly = iy + sy;
   flags = (tf_ul | tf_ur | tf_ll | tf_lr);

   for (y = iy; y < ly; y++)
      for (x = ix; x < lx; x++) {
         tile = &(map.map[TILE_INDEX(x, y)]);
         if (tile->old_fw_flags !=
             (tile->flags & flags))
            mark_tile(x, y);
      }
      
   for (y = 0; y < map.height; y++)
      for (x = 0; x < map.width; x++) {
         tile = &(map.map[TILE_INDEX(x, y)]);
         if (tile->old_fw_flags !=
             (tile->flags & flags))
            mark_mmtile(x, y);
      }
}

void load_map(const char *filename)
{
   int x, y;
   int c;
   MAP_TILE *tile;

   reset_maptile_chunk();
   
   /* Load the tileset for this map */
   load_tileset(NULL);

   map.width = 64;
   map.height = 64;
   map.viewport_x = map.viewport_y = 0;
   map.num_layers = 1;
   /* Allocate extra `border' tiles tahat are not used */
   map.map =
      realloc(map.map, sizeof(MAP_TILE) * (map.width + 2) * (map.height + 2));
      
   /* Set `extra' border tiles to `blocked' */
   for (y = -1; y < map.height + 1; y++) {
      c = TILE_INDEX(-1, y);
      tile = &(map.map[c]);
      
      tile->next = NULL;
      tile->gfx = NULL;
      tile->gfx_index = -1;
      tile->ovru = NULL;
      tile->blocku = NULL;
      tile->flags = TILE_BARRIER | TILE_BASE | tf_ul | tf_ur | tf_ll | tf_lr;
      tile->old_fw_flags = tile->flags & (tf_ul | tf_ur | tf_ll | tf_lr);

      tile = &(map.map[TILE_INDEX(map.width, y)]);
      tile->next = NULL;
      tile->gfx = NULL;
      tile->gfx_index = -1;
      tile->ovru = NULL;
      tile->blocku = NULL;
      tile->flags = TILE_MMCHANGED | TILE_BARRIER | TILE_BASE | tf_ul | tf_ur | tf_ll | tf_lr;
      tile->old_fw_flags = tile->flags & (tf_ul | tf_ur | tf_ll | tf_lr);
   }

   for (x = -1; x < map.width + 1; x++) {
      c = TILE_INDEX(x, -1);
      tile = &(map.map[c]);

      tile->next = NULL;
      tile->gfx = NULL;
      tile->gfx_index = -1;
      tile->ovru = NULL;
      tile->blocku = NULL;
      tile->flags = TILE_MMCHANGED | TILE_BARRIER | TILE_BASE | tf_ul | tf_ur | tf_ll | tf_lr;
      tile->old_fw_flags = tile->flags & (tf_ul | tf_ur | tf_ll | tf_lr);

      tile = &(map.map[TILE_INDEX(x, map.height)]);
      tile->next = NULL;
      tile->gfx = NULL;
      tile->gfx_index = -1;
      tile->ovru = NULL;
      tile->blocku = NULL;
      tile->flags = TILE_BARRIER | TILE_BASE | tf_ul | tf_ur | tf_ll | tf_lr;
      tile->old_fw_flags = tile->flags & (tf_ul | tf_ur | tf_ll | tf_lr);
   }

   for (y = 0; y < map.height; y++)
      for (x = 0; x < map.width; x++) {
         c = TILE_INDEX(x, y);
         tile = &(map.map[c]);
         
         tile->next = NULL;
         tile->gfx = get_tile_bitmap(0);
         tile->gfx_index = 0;
         tile->ovru = NULL;
         tile->blocku = NULL;
         tile->flags = TILE_MMCHANGED | TILE_BASE | tf_ul | tf_ur | tf_ll | tf_lr;
         tile->old_fw_flags = tile->flags & (tf_ul | tf_ur | tf_ll | tf_lr);

         tile->height = 0;
         tile->mapx = x;
         tile->mapy = y;
      }
}

int get_map_width(void)
{
   return map.width;
}

int get_map_height(void)
{
   return map.height;
}

int get_viewport_x(void)
{
   return map.viewport_x;
}

int get_viewport_y(void)
{
   return map.viewport_y;
}

int get_map_layers(void)
{
   return map.num_layers;
}

int scrx_to_tilex(const int scrx)
{
   return (scrx + get_viewport_x() - get_commandmap_x()) / TILE_WIDTH;
}

int scry_to_tiley(const int scry)
{
   return (scry + get_viewport_y() - get_commandmap_y()) / TILE_HEIGHT;
}

/******************************/
/* Tile registering functions */
/******************************/
/* Mark tile for next update session */
void mark_tile(const int x, const int y)
{
   MAP_TILE *tile = &(map.map[TILE_INDEX(x, y)]);

   while (tile) {
      tile->flags |= (TILE_CHANGED+TILE_MMCHANGED);
      tile = tile->next;
   }
}

/* Mark tile for next minimap update session */
void mark_mmtile(const int x, const int y)
{
   MAP_TILE *tile = &(map.map[TILE_INDEX(x, y)]);

   while (tile) {
      tile->flags |= TILE_MMCHANGED;
      tile = tile->next;
   }
}

/* Unmark tile */
void unmark_tile(const int x, const int y)
{
   MAP_TILE *tile = &(map.map[TILE_INDEX(x, y)]);

   while (tile) {
      tile->flags &= ~(TILE_CHANGED+TILE_MMCHANGED);
      tile = tile->next;
   }
}

/* Unmark tile for minimap */
void unmark_mmtile(const int x, const int y)
{
   MAP_TILE *tile = &(map.map[TILE_INDEX(x, y)]);

   while (tile) {
      tile->flags &= ~TILE_MMCHANGED;
      tile = tile->next;
   }
}

/* Retrieve the linked list of registered characters */
void *get_ovr_char(const int x, const int y)
{
   ASSERT(x >= -1);
   ASSERT(y >= -1);
   ASSERT(x <= map.width);
   ASSERT(y <= map.height);

   return map.map[TILE_INDEX(x, y)].ovru;
}

/* Set the linked list of registered characters */
void set_ovr_char(const int x, const int y, void *ovru)
{
   ASSERT(x >= -1);
   ASSERT(y >= -1);
   ASSERT(x <= map.width);
   ASSERT(y <= map.height);

   map.map[TILE_INDEX(x, y)].ovru = ovru;
}

/* Retrieve the linked list of blocking characters */
void *get_block_char(const int x, const int y)
{
   ASSERT(x >= -1);
   ASSERT(y >= -1);
   ASSERT(x <= map.width);
   ASSERT(y <= map.height);

   return map.map[TILE_INDEX(x, y)].blocku;
}

/* Set the linked list of blocking characters */
void set_block_char(const int x, const int y, void *blocku)
{
   ASSERT(x >= -1);
   ASSERT(y >= -1);
   ASSERT(x <= map.width);
   ASSERT(y <= map.height);

   map.map[TILE_INDEX(x, y)].blocku = blocku;
}

/* Returns TRUE if the tile is (partially) obscured by the Fog of War */
int tile_fog(const int tx, const int ty)
{
   ASSERT(tx >= -1);
   ASSERT(ty >= -1);
   ASSERT(tx <= map.width);
   ASSERT(ty <= map.height);

   return map.map[TILE_INDEX(tx, ty)].flags & (FOG_UL | FOG_UR | FOG_LL | FOG_LR);
}

int tile_hidden(const int tx, const int ty)
{
   ASSERT(tx >= -1);
   ASSERT(ty >= -1);
   ASSERT(tx <= map.width);
   ASSERT(ty <= map.height);

   return map.map[TILE_INDEX(tx, ty)].
      flags & (HIDDEN_UL | HIDDEN_UR | HIDDEN_LL | HIDDEN_LR);
}

/* Read tile flags */
int tile_flags(const int tx, const int ty, const int layer)
{
   ASSERT(tx >= -1);
   ASSERT(ty >= -1);
   ASSERT(tx <= map.width);
   ASSERT(ty <= map.height);

   return map.map[TILE_INDEX(tx, ty)].flags;
}

/* Returns TRUE if the tile is marked */
int tile_marked(const int tx, const int ty)
{
   return tile_flags(tx, ty, 0) & TILE_CHANGED;
}

/* Read tile type index */
int tile_type(const int tx, const int ty, const int layer)
{
   return map.map[TILE_INDEX(tx, ty)].gfx_index;
}

/* Get the tiles that are on the viewport, as well as the tile off-set */
/*  tw and th are the width and height of the map section visible */
void get_viewport_tiles(int *start_tx, int *start_ty, int *tw, int *th,
                        int *dx, int *dy)
{
   /* Compute the first tile visible and get it's screen coordinates */
   *start_tx = map.viewport_x / TILE_WIDTH;
   *start_ty = map.viewport_y / TILE_HEIGHT;
   /* Get offset in screen coordinates */
   *dx = -map.viewport_x % TILE_WIDTH;
   *dy = -map.viewport_y % TILE_HEIGHT;
   /* Get the tile coordinates of the last tile to draw */
   *tw = (get_commandmap_width() - *dx + TILE_WIDTH - 1) / TILE_WIDTH;
   *th = (get_commandmap_height() - *dy + TILE_HEIGHT - 1) / TILE_HEIGHT;
}

/* Marks all tiles in the viewport */
void mark_viewport(void)
{
   int px = 0;
   int py = 0;
   int x;
   int y;
   int ix, iy;
   int sx, sy;
   
   /* For fixing a crash bug from main menu - but must handle this in a 
      better way */
   if (!map_loaded())
      return;

   /* Compute the first tile visible and get it's screen coordinates */
   get_viewport_tiles(&ix, &iy, &sx, &sy, &px, &py);

   for (y = 0; y < sy; y++)
      for (x = 0; x < sx; x++)
         mark_tile(ix + x, iy + y);
}

/* Unmarks all tiles within the viewport */
void unmark_viewport(void)
{
   int px = 0;
   int py = 0;
   int x;
   int y;
   int ix, iy;
   int sx, sy;

   /* Compute the first tile visible and get it's screen coordinates */
   get_viewport_tiles(&ix, &iy, &sx, &sy, &px, &py);

   for (y = 0; y < sy; y++)
      for (x = 0; x < sx; x++) {
         unmark_tile(ix + x, iy + y);
      }

}

/* Scrolls the map, ie, moves the viewport in map space by (dx, dy) */
/* Returns TRUE if the viewport coordinates were actually changed */
int scroll_map(const int dx, const int dy)
{
   int x = map.viewport_x;
   int y = map.viewport_y;

   map.viewport_x += dx;
   map.viewport_y += dy;
   if (map.viewport_x < 0)
      map.viewport_x = 0;
   if (map.viewport_y < 0)
      map.viewport_y = 0;
   if (map.viewport_x > (map.width - 1) * TILE_WIDTH - get_commandmap_width())
      map.viewport_x = (map.width - 1) * TILE_WIDTH - get_commandmap_width();
   if (map.viewport_y >
       (map.height - 1) * TILE_HEIGHT - get_commandmap_height())
      map.viewport_y =
         (map.height - 1) * TILE_HEIGHT - get_commandmap_height();
   if (x != map.viewport_x || y != map.viewport_y) {
      mark_viewport();
      return TRUE;
   }
   return FALSE;
}

/* Centres the command map at the specified coordinates */
int centre_map(const int cx, const int cy)
{
   int x = map.viewport_x;
   int y = map.viewport_y;

   map.viewport_x =
      (cx - get_commandmap_width() / TILE_WIDTH / 2) * TILE_WIDTH;
   map.viewport_y =
      (cy - get_commandmap_height() / TILE_HEIGHT / 2) * TILE_HEIGHT;
   if (map.viewport_x < 0)
      map.viewport_x = 0;
   if (map.viewport_y < 0)
      map.viewport_y = 0;
   if (map.viewport_x > (map.width - 1) * TILE_WIDTH - get_commandmap_width())
      map.viewport_x = (map.width - 1) * TILE_WIDTH - get_commandmap_width();
   if (map.viewport_y >
       (map.height - 1) * TILE_HEIGHT - get_commandmap_height())
      map.viewport_y =
         (map.height - 1) * TILE_HEIGHT - get_commandmap_height();
   if (x != map.viewport_x || y != map.viewport_y) {
      mark_viewport();
      return TRUE;
   }
   return FALSE;
}

/* Returns true if the tile (tx, ty) is in the list */
int tile_in_list(const MAP_TILE_LIST *tlst, const int tx, const int ty)
{
   MAP_TILE_LIST *t = (MAP_TILE_LIST *)tlst;

   while (t) {
      if ((t->tx == tx) && (t->ty == ty))
         return TRUE;
      t = t->next;
   }
   return FALSE;
}
