/**********************************************/
/* Rise of the Tribes                         */
/* C version, Take 2                          */
/* Evert Glebbeek 1998, 2002                  */
/* eglebbk@phys.uva.nl                        */
/**********************************************/
#include <allegro.h>
#include <stdio.h>
#include "gamegfx.h"
#include "map.h"
#include "maptile.h"
#include "gfx.h"
#include "unit.h"
#include "ugfx.h"
#include "ugrp.h"
#include "udraw.h"
#include "global.h"
#include "dialog.h"
#include "upkeep.h"
#include "icpanel.h"
#include "unitprog.h"
#include "gamedraw.h"
#include "unitmap.h"
#include "mapdraw.h"

static BITMAP *command_panel = NULL;
static int redraw_area;

/****************************/
/* Floating windows         */
/****************************/
/* Draw (ie, blit) a floating window to the screen */
void draw_floatwin(FLOATWIN *flw)
{
   BITMAP *bmp;
   int x, y;

   if (flw->flags & FWIN_HIDDEN)
      return;

   if (!(flw->flags & FWIN_DIRTY))
      return;

   if ((flw->flags & FWIN_MARK)) {
      mark_rect(flw->x, flw->y, flw->w, flw->h);
      flw->flags &= ~FWIN_MARK;
   }

   if (flw->flags & FWIN_STICKMAP) {
      if (flw->y - get_viewport_y() < get_commandmap_height()) {
         x = flw->x;
         y = flw->y;
      } else {
         x = -flw->w;
         y = -flw->h;
      }
      bmp = get_commandmap_bmp();
   } else {
      x = flw->x;
      y = flw->y;
      bmp = get_screen_bmp();
   }

   if (flw->flags & FWIN_FADETPDN) {
      draw_gouraud_sprite(bmp, flw->content, x, y, 64,
                          64, 255, 255);
   } else if (flw->flags & FWIN_TRANS) {
      draw_trans_sprite(bmp, flw->content, x, y);
   } else {
      draw_sprite(bmp, flw->content, x, y);
   }
   flw->flags &= ~FWIN_DIRTY;
}

/* Draw (ie, blit) all floating windows to the screen */
void draw_floatwins(void)
{
   FLOATWIN_LIST *flwl = flwin;

   while (flwl) {
      draw_floatwin (flwl->win);
      flwl = flwl->next;
   }
}

/* Floating strings are special floating windows: they have no border and */
/*  have a fixed location on the world map */
/* (x,y) are the coordinate on the world map */
FLOATWIN *make_floatstr(int x, int y, int bgcolor, int fgcolor, int lifetime,
                        const FONT *textfont, char *msg, ...)
{
   char *s = malloc(4096);
   FLOATWIN *flw;
   int w;
   int h;


   /* Expand the parameter into a full-length string */
   va_list ap;

   va_start(ap, msg);
   uvsprintf(s, msg, ap);
   va_end(ap);

   w = text_length(textfont, s) + 2;
   h = text_height(textfont) + 2;

   /* Now, create the window */
   x /= TILE_WIDTH;
   y /= TILE_HEIGHT;
   flw =
      make_floatwin(x * TILE_WIDTH + TILE_WIDTH / 2 - w / 2,
                    y * TILE_HEIGHT + 2 * TILE_HEIGHT / 3 - h / 2, w, h,
                    FWIN_STICKMAP + FWIN_FADETPDN, lifetime, NULL,
                    create_bitmap(w, h));

   /* Clear the bitmap */
   rectfill(flw->content, 0, 0, w, h, bitmap_mask_color(flw->content));

   /* Now, begin writing the text lines */
   text_mode(-1);

   textout(flw->content, textfont, s, 0, 0, bgcolor);
   textout(flw->content, textfont, s, 2, 2, bgcolor);
   textout(flw->content, textfont, s, 1, 1, fgcolor);

   /* And we're done */
   free(s);

   return flw;
}

/* Marks all tiles under a floating window, effectively erasing it */
void undraw_floatwin(FLOATWIN *flw)
{
   int tx, ty;
   int dx, dy;
   int ix, iy;
   int sx, sy;
   int lx, ly;
   int w, h;

   if (flw->flags & FWIN_CMDNAME) {
      mark_interface_area(INT_DESCR);
   }

   /* Get tile coordinates of the floating window left-top */
   if (!(flw->flags & FWIN_STICKMAP)) {
      tx = scrx_to_tilex(flw->x);
      ty = scry_to_tiley(flw->y);
      w = (flw->w + flw->x % TILE_WIDTH + TILE_WIDTH - 1);
      h = (flw->h + flw->y % TILE_HEIGHT + TILE_HEIGHT - 1);
   } else {
      tx = flw->x / TILE_WIDTH;
      ty = flw->y / TILE_HEIGHT;
      w = (flw->w + flw->x % TILE_WIDTH + TILE_WIDTH - 1);
      h = (flw->h + flw->y % TILE_HEIGHT + TILE_HEIGHT - 1);
   }

   if (w%TILE_WIDTH)
      w+=TILE_WIDTH-1;
   if (h%TILE_HEIGHT)
      h+=TILE_HEIGHT-1;
   w /= TILE_WIDTH;
   h /= TILE_HEIGHT;


   /* Compute the first tile visible and get it's screen coordinates */
   get_viewport_tiles(&ix, &iy, &sx, &sy, &lx, &ly);

   for (dx = 0; dx < w; dx++)
      for (dy = 0; dy < h; dy++) {
         lx = tx + dx;
         ly = ty + dy;

         if (in_rect(lx, ly, ix, iy, sx, sy))
            mark_tile(lx, ly);
      }
}

/* Draw the tile ok/not ok mask for structure (of tw x th) placement */
/* Allocates a new bitmap, or draws on the passed bitmap if that is not NULL */
BITMAP *draw_structure_mask(BITMAP *bmp, int tx, int ty, int tw, int th, UNIT *ignore)
{
   BITMAP *draw_bmp;
   int x, y;
   int xm = MIN(tx+tw+1, get_map_width());
   int ym = MIN(ty+th+1, get_map_height());

   ASSERT(tx >= -1);
   ASSERT(ty >= -1);
   ASSERT(tx <= get_map_width());
   ASSERT(ty <= get_map_height());

   if (!bmp) {
      draw_bmp = create_bitmap((tw+2)*TILE_WIDTH, (th+2)*TILE_HEIGHT);
   } else {
      draw_bmp = bmp;
   }

   clear_to_color(draw_bmp, darkgreen);

   for (x = tx; x<tx+tw; x++) {
      for (y = ty; y<ty+th; y++) {
         if (tile_blocked(x, y, 0, ignore) || tile_hidden(x, y)) {
            rectfill (draw_bmp, (x-tx+1)*TILE_WIDTH, (y-ty+1)*TILE_HEIGHT,
                      (x-tx+2)*TILE_WIDTH-1, (y-ty+2)*TILE_HEIGHT-1, red);
         }
      }
   }

   for (x = MAX(tx-1, -1); x<xm; x++) {
      for (y = MAX(ty-1, -1); y<ym; y++) {
         if (count_structures(get_block_char(x, y)) || tile_hidden(x, y))
            rectfill (draw_bmp, (x-tx+1)*TILE_WIDTH, (y-ty+1)*TILE_HEIGHT,
                      (x-tx+2)*TILE_WIDTH-1, (y-ty+2)*TILE_HEIGHT-1, red);
      }
   }

   return draw_bmp;
}

/* Mark an area of the interface for redraws */
void mark_interface_area(int area)
{
   redraw_area |= area;
}

/* Mark interface rectangles */
static inline void mark_interface_rects(void)
{
   int x, y;
   int dx;
   int c;

   /* Mark minimap. Do so in two rectangles: if the entire interface frame is */
   /*  drawn, the second one is dropped; if not, the two are merged anyway */
   if (redraw_area & INT_MINIMAP) {
      mark_rect(SCREEN_W - 128 - 4, SCREEN_H - 132, 128, 12);
      mark_rect(SCREEN_W - 128 - 4, SCREEN_H - 120, 128, 116);
   }

   /* Draw map border frame */
   if (redraw_area & (INT_BORDER|INT_ALL)) {
      mark_rect(0, 0, SCREEN_W, 18);
      mark_rect(0, 18, 8, SCREEN_H - 120);
      mark_rect(SCREEN_W - 8, 18, 8, SCREEN_H - 120);
   }

   /* If we update everything, then we don't have to mark everything twice */
   if (redraw_area & INT_ALL) {
      mark_rect(0, SCREEN_H - 120, SCREEN_W, 120);
      return;
   }

   /* Mark command panel */
   if (redraw_area & INT_COMMANDS) {
      mark_rect(0, SCREEN_H - 96 - 16, 128, 96);
   }

   /* Command description */
   if (redraw_area & INT_DESCR) {
      mark_rect(0, SCREEN_H - 16, 128+64, 16);
   }


   /* Menu button */
   if (redraw_area & (INT_ALL|INT_BUTTON)) {
      mark_rect(game_butt->x, game_butt->y, game_butt->w, game_butt->h);
   }

   /* Draw captain information */
   if (redraw_area & INT_CAPTAIN) {
      x = game_butt->x + 8;
      y = SCREEN_H - 128 + 44;

      /* Portrait */
      mark_rect(x, y, 34, 38);

      /* Name */
      mark_rect(x + 40, y, SCREEN_W - 128 - 4 - (x + 40)-4,
                text_height(get_normal_font()));

      /* Health points/mana */
      mark_rect(x + 40, y + 12, SCREEN_W - 128 - 4 - (x + 40)-4, 24);

      /* Production icons */
      y += 40;
      x += 2;

      mark_rect(x, y, SCREEN_W - 128 - 8 - x, 2*16);
   }


   if (redraw_area & INT_INFO) {
      mark_rect(136, SCREEN_H - 128 + 16, game_butt[0].x-136-1, 60);
   }

   if (redraw_area & INT_RESOURCE) {
      mark_rect(0, 0, SCREEN_W, 18);
   }

   if (redraw_area & INT_GROUP) {
      for (c = 0; c < 12; c++) {
         x = 136 + (c / 2) * 36;
         y = SCREEN_H - 128 + 28 + 42 * (c % 2);

         mark_rect(x, y, 34, 38);
      }
   }

   if (redraw_area & INT_QUEUE) {
      mark_rect(cancel_butt->x, cancel_butt->y, 32, 32);

      for (dx = 1; dx < 7; dx++) {
         mark_rect(cancel_butt->x + dx * 32, cancel_butt->y, 32, 32);
      }

      mark_rect(64 + 16 + 136, SCREEN_H - 16, 124, 16);
   }
}

/* Draw the border around the command map */
static inline void draw_commandmap_border(void)
{
   int c;
   int maxc;
   BITMAP *bmp;
   bmp = get_interface_bmp();

   /* Left and right */
   maxc = ((SCREEN_H - 120) / 120);
   if ((SCREEN_H - 120) % 120)
      maxc++;
   for (c = 0; c < maxc; c++) {
      blit(get_left_panel_bitmap(), get_screen_bmp(), 0, 0,
           0, SCREEN_H - 120 - 120 - c * 120, 8, 120);
      blit(get_right_panel_bitmap(), get_screen_bmp(), 0, 0,
           SCREEN_W - 8, SCREEN_H - 120 - 120 - c * 120, 8, 120);
   }

   /* Top and bottom */
   maxc = SCREEN_W / 640;
   if (SCREEN_W % 640)
      maxc++;
   for (c = 0; c < maxc; c++) {
      blit(get_menubg_bitmap(), bmp, 0, 0,
           128 + c * 640, bmp->h - 120, 640, 120);
      blit(get_top_panel_bitmap(), get_screen_bmp(), 0, 0,
           c * 640, 0, 640, 120);
   }
}

/* Draw resource icons and status */
/* The amount of resources should be in the player structure, instead of */
/*  being recalculated at every stage */
static inline void draw_resource_icons(void)
{
   int udrw = get_local_player();

   /* Upkeep-type resources */
   /* Upper limit for display: 999/999 */
   //masked_blit (intrface_dat[RESOURCE1].dat, get_screen_bmp(), 0,0, SCREEN_W-250, SCREEN_H-86, 17, 17);
   masked_blit(get_resource_icon(0), get_screen_bmp(), 0, 0, 16, 1, 17, 17);
   textprintf(get_screen_bmp(), get_menu_font(), 16 + 24 + 1, 3 + 1, black,
              "%d/%d", get_player_upkeep(udrw, 0),
              get_player_provision(udrw, 0));
   textprintf(get_screen_bmp(), get_menu_font(), 16 + 24, 3, white, "%d/%d",
              get_player_upkeep(udrw, 0), get_player_provision(udrw, 0));

   //masked_blit (intrface_dat[RESOURCE2].dat, get_screen_bmp(), 0,0, SCREEN_W-250, SCREEN_H-66, 17, 17);
   //textprintf (get_screen_bmp(), get_menu_font(), SCREEN_W-250+24, SCREEN_H-66+2, white, "%d/%d", get_player_upkeep(udrw, 1), get_player_provision(udrw, 1));
   masked_blit(get_resource_icon(1), get_screen_bmp(), 0, 0,
               16 + 24 + 56, 1, 17, 17);
   textprintf(get_screen_bmp(), get_menu_font(), 16 + 24 + 56 + 1 + 24, 3 + 1,
              black, "%d/%d", get_player_upkeep(udrw, 1),
              get_player_provision(udrw, 1));
   textprintf(get_screen_bmp(), get_menu_font(), 16 + 24 + 56 + 24, 3, white,
              "%d/%d", get_player_upkeep(udrw, 1),
              get_player_provision(udrw, 1));

   //masked_blit (intrface_dat[RESOURCE3].dat, get_screen_bmp(), 0,0, SCREEN_W-250, SCREEN_H-46, 17, 17);
   //textprintf (get_screen_bmp(), get_menu_font(), SCREEN_W-250+24, SCREEN_H-46+2, white, "%d/%d", get_player_upkeep(udrw, 2), get_player_provision(udrw, 2));
   masked_blit(get_resource_icon(2), get_screen_bmp(), 0, 0,
               16 + 2 * (24 + 56), 1, 17, 17);
   textprintf(get_screen_bmp(), get_menu_font(), 16 + 2 * (24 + 56) + 1 + 24,
              3 + 1, black, "%d/%d", get_player_upkeep(udrw, 2),
              get_player_provision(udrw, 2));
   textprintf(get_screen_bmp(), get_menu_font(), 16 + 2 * (24 + 56) + 24, 3,
              white, "%d/%d", get_player_upkeep(udrw, 2),
              get_player_provision(udrw, 2));

   /* Trade-good like resources */
   /* Upper limit for display: 9999999  */
   //masked_blit (intrface_dat[RESOURCE4].dat, get_screen_bmp(), 0,0, SCREEN_W-240+56, SCREEN_H-86, 17, 17);
   //textprintf (get_screen_bmp(), get_menu_font(), SCREEN_W-240+24+56, SCREEN_H-86+2, white, "%d", get_resource(get_local_player(), 3));
   masked_blit(get_resource_icon(3), get_screen_bmp(), 0, 0, 320 + 16,
               1, 17, 17);
   textprintf(get_screen_bmp(), get_menu_font(), 320 + 16 + 1 + 24, 3 + 1,
              black, "%d", get_resource(get_local_player(), 3));
   textprintf(get_screen_bmp(), get_menu_font(), 320 + 16 + 24, 3, white,
              "%d", get_resource(get_local_player(), 3));

   //masked_blit (intrface_dat[RESOURCE5].dat, get_screen_bmp(), 0,0, SCREEN_W-240+56, SCREEN_H-66, 17, 17);
   //textprintf (get_screen_bmp(), get_menu_font(), SCREEN_W-240+24+56, SCREEN_H-66+2, white, "%d", get_resource(get_local_player(), 4));
   masked_blit(get_resource_icon(4), get_screen_bmp(), 0, 0,
               320 + 16 + 56 + 24, 1, 17, 17);
   textprintf(get_screen_bmp(), get_menu_font(), 320 + 16 + 1 + 56 + 24 + 24,
              3 + 1, black, "%d", get_resource(get_local_player(), 4));
   textprintf(get_screen_bmp(), get_menu_font(), 320 + 16 + 56 + 24 + 24, 3,
              white, "%d", get_resource(get_local_player(), 4));
}

/* Update all game graphics. Supports dirty-rectangles as needed */
void do_screen_update(void)
{
   FLOATWIN *flw = find_floatwin(FWIN_PLACESTRUCTURE);
   int c;
   int x, y;
   int dx, dy;
   BITMAP *bmp;
   UNIT *udrw = get_selected_units(get_local_player());
   int num_sel_units = get_num_selected_units(get_local_player());
   int colour;
   int n;
   FONT *f;

   TRACE("Drawing map...\n");

   /* Draw terrain and units */
   draw_map();
   draw_all_units();

   /* Draw fog-of-war and map shrouds */
   if (!(settings.flags & LIT_INLINE))
      do_render_job();
   if ((settings.flags & FOG_OF_WAR_BITMAP))
      draw_bitmapped_fog_of_war();

   /* Draw the structure area floatwin, if it exists */
   if (flw) {
      flw->flags &= ~FWIN_HIDDEN;
      draw_floatwin(flw);
      flw->flags |= FWIN_HIDDEN;
   }

   /* Clear the viewport markings */
   unmark_viewport();

   TRACE("Map drawn.\n");
   TRACE("Drawing interface...");

   /* Draw the frame around the commandmap (always) */
   draw_commandmap_border();

   /* Draw the command-panel background */
   bmp = get_cmdpanel_bitmap();
   draw_sprite(get_screen_bmp(), bmp, 0, SCREEN_H - bmp->h);

   /* Draw the corners */
   blit(get_ll_panel_corner_bitmap(), get_screen_bmp(), 0, 0,
        0, SCREEN_H - 120 - 32, 8, 32);
   blit(get_lr_panel_corner_bitmap(), get_screen_bmp(), 0, 0,
        SCREEN_W - 8, SCREEN_H - 120 - 32 + 16, 8, 32);

   /* Draw the mini-map */
   bmp = get_mmframe_bitmap();

   get_viewport_tiles(&x, &y, &dx, &dy, &n, &c);

   stretch_blit(get_minimap(), get_screen_bmp(),
                0, 0, get_map_width(), get_map_height(),
                SCREEN_W - 128 - 4, SCREEN_H - 128 - 4, 128, 128);
   rect(get_screen_bmp(),
        SCREEN_W - 128 - 4 + x * 128 / get_map_width(),
        SCREEN_H - 128 - 4 + y * 128 / get_map_height(),
        SCREEN_W - 128 - 4 + (x + get_commandmap_width()/TILE_WIDTH + 1) * 128 / get_map_width(),
        SCREEN_H - 128 - 4 + (y + get_commandmap_height()/TILE_HEIGHT + 1) * 128 / get_map_height(), grey);
   masked_blit(bmp, get_screen_bmp(), 0, 0, SCREEN_W - bmp->w,
               SCREEN_H - bmp->h, bmp->w, bmp->h);

   /* Draw game-menu button */
   game_butt->proc(MSG_DRAW, game_butt, 0);

   bmp = get_empty_slot_bitmap();

   x = game_butt[0].x + 8;
   y = SCREEN_H - 128 + 44;
   //x = 160;
   //y = SCREEN_H-128+28;

   blit(bmp, get_screen_bmp(), 0, 0, x, y, 34, 34);


   /* Draw selected unit slots */
   if (num_sel_units) {
      /* Draw detailed info for the group leader selected */
      /* Maybe add an animated icon here also */

      if (udrw->data->icon && (redraw_area & INT_CAPTAIN)) {
         select_palette(*get_char_palette(udrw->data->gfxidx, udrw->data->player));
         blit(udrw->data->icon, get_screen_bmp(), 0, 0, x + 1, y + 1, 34, 34);
         n = udrw->data->maxhp / udrw->data->hp;
         if (n >= 4) {
            colour = red;
         } else if (n >= 2) {
            colour = yellow;
         } else {
            colour = darkgreen;
         }
         rectfill(get_screen_bmp(), x, y + 32, x + 33, y + 37, black);
         rectfill(get_screen_bmp(), x + 1, y + 33,
                  x + 32 * udrw->data->hp / udrw->data->maxhp, y + 36,
                  colour);
         if (udrw->data->maxmp) {
            rectfill(get_screen_bmp(), x + 1, y + 35,
                     x + 32 * udrw->data->mp / udrw->data->maxmp, y + 36,
                     bright_blue);
         }
         unselect_palette();
      }

      textprintf(get_screen_bmp(), get_normal_font(), x + 40, y, white, "%s",
                 udrw->data->name);

      if (udrw->data->player == get_local_player()) {
         select_palette(*get_char_palette(udrw->data->gfxidx, udrw->data->player));

         //textprintf_centre (get_screen_bmp(), get_small_font(), x+17, y+32, white, "%d/%d", udrw->data->hp, udrw->data->maxhp);
         textprintf(get_screen_bmp(), get_menu_font(), x + 48, y + 12, white,
                    "HP: %d/%d", udrw->data->hp, udrw->data->maxhp);

         if (udrw->data->maxmp) {
            textprintf(get_screen_bmp(), get_menu_font(), x + 48, y + 24,
                       white, "MP: %d/%d", udrw->data->mp, udrw->data->maxmp);
         }

         /* Draw progress indicator and command queue */
         if ((udrw->data->flags & CF_PROGRESS) && get_queuecount(udrw->data)) {
            if (num_sel_units == 1) {
               cancel_butt->x = 134;
               cancel_butt->y = SCREEN_H - 16 - 32;

               blit(udrw->data->unit_command[12]->icon, get_screen_bmp(),
                    0, 0, cancel_butt->x, cancel_butt->y, 32, 32);
               masked_blit(get_cmdpanel_button_bitmap(), get_screen_bmp(),
                           0, 0, cancel_butt->x, cancel_butt->y, 32, 32);

               for (dx = 1; dx < get_queuecount(udrw->data); dx++) {
                  c = (udrw->data->queuestart+dx-1) % udrw->data->queuesize;
                  /*
                  printf("%d\t%08x\t%08x\n", c, udrw->data->cqueue[c].icon, udrw->data->cqueue[udrw->data->queuestart%udrw->data->queuesize].icon);
                  fflush(stdout);
                  */
                  draw_sprite(get_screen_bmp(), udrw->data->cqueue[udrw->data->queuestart%udrw->data->queuesize].icon,
                              cancel_butt->x + dx * 32, cancel_butt->y);
                  masked_blit(get_cmdpanel_button_bitmap(), get_screen_bmp(),
                              0, 0, cancel_butt->x + dx * 32, cancel_butt->y,
                              32, 32);
               }
               for (dx = get_queuecount(udrw->data); dx < udrw->data->queuesize; dx++) {
                  rectfill(get_screen_bmp(), cancel_butt->x + dx * 32,
                           cancel_butt->y, cancel_butt->x + dx * 32 + 31,
                           cancel_butt->y + 31, black);
                  masked_blit(get_cmdpanel_button_bitmap(), get_screen_bmp(),
                              0, 0, cancel_butt->x + dx * 32, cancel_butt->y,
                              32, 32);
               }
               f = (FONT *)get_small_font();
               for (dx = 0; dx < udrw->data->queuesize; dx++) {
                  textprintf_right(get_screen_bmp(), f,
                                    cancel_butt->x + dx * 32+28,
                                    cancel_butt->y+28-text_height(f),
                                    white, "%d", dx+1);
               }
            }

            /* Filling of the progress bar */
            n = udrw->data->cprog ? udrw->data->cprog->progress:0;

            blit(get_proind_bitmap(), get_screen_bmp(), 0, 0, 64 + 16 + 136,
                 SCREEN_H - 16, 128, 16);
            dx = n * 116 / 1000;
            rectfill(get_screen_bmp(), 64 + 16 + 136 + 4, SCREEN_H - 13,
                     64 + 16 + 136 + 4 + dx, SCREEN_H - 3, darkgreen);
            textprintf_centre(get_screen_bmp(), get_menu_font(),
                              64 + 16 + 136 + 64, SCREEN_H - 14, white,
                              "Progress %d%%",
                              n / 10);
            cancel_butt->flags &= ~D_HIDDEN;
         } else {
            cancel_butt->flags |= D_HIDDEN;
         }

         /* Draw production icons */
         y += 40;
         x += 2;
         dx = 0;
         dy = 0;
         for (c = 0; c < 8; c++) {
            if (udrw->data->resource_cost[c] < 0) {
               blit(get_resource_icon(c), get_screen_bmp(), 0, 0,
                    x + dx, y + dy, 16, 16);
               textprintf(get_screen_bmp(), get_menu_font(), x + 24 + dx,
                          y + 2 + dy, white, "%d",
                          -udrw->data->resource_cost[c]);
               dy += 17;
               if (dy >= 34) {
                  dy = 0;
                  dx += 40;
               }
            }
         }
         unselect_palette();
      }

      if (num_sel_units > 1) {  /* More than one unit selected */
         for (c = 0; c < 12; c++) {
            x = 136 + (c / 2) * 36;
            y = SCREEN_H - 128 + 28 + 42 * (c % 2);
            blit(bmp, get_screen_bmp(), 0, 0, x, y, 34, 34);
            if (udrw) {
               if (udrw->data->icon) {
                  select_palette(*get_char_palette(udrw->data->gfxidx, udrw->data->player));

                  blit(udrw->data->icon, get_screen_bmp(), 0, 0,
                       x + 1, y + 1, 34, 34);
                  /* Draw health bar */
                  rectfill(get_screen_bmp(), x, y + 32, x + 33, y + 37,
                           black);
                  rectfill(get_screen_bmp(), x + 1, y + 33,
                           x + 32 * udrw->data->hp / udrw->data->maxhp,
                           y + 36, darkgreen);
                  if (udrw->data->maxmp) {
                     rectfill(get_screen_bmp(), x + 1, y + 35,
                              x + 32 * udrw->data->mp / udrw->data->maxmp,
                              y + 36, bright_blue);
                  }

                  /* Mark the icon region */
                  unselect_palette();
               }
               udrw = udrw->next;
            }
         }
      } else {                  /* Detailed info about this unit */
         if (udrw->data->player == get_local_player()) {
            x = 152;
            y = SCREEN_H - 128 + 16;

            /* Global info for all unit types */
            if (udrw->data->radius) {
               textprintf(get_screen_bmp(), get_menu_font(), x, y, white,
                          "Sight: %d", udrw->data->radius);
               y += 12;
            }
            if (udrw->data->speed) {
               textprintf(get_screen_bmp(), get_menu_font(), x, y, white,
                          "Speed: %d", udrw->data->speed);
               y += 12;
            }
            textprintf(get_screen_bmp(), get_menu_font(), x, y, white,
                       "Range: 0");
            y += 12;
            textprintf(get_screen_bmp(), get_menu_font(), x, y, white,
                       "Armour: 0");
            y += 12;
            textprintf(get_screen_bmp(), get_menu_font(), x, y, white,
                       "Damage: 0-0");
            y += 12;

            /* Specific info */
            x = 256;
            y = SCREEN_H - 128 + 16;

            if (udrw->data->flags & CC_COMMANDPOST) {
               textprintf(get_screen_bmp(), get_menu_font(), x, y, white,
                          "Units: %d",
                          count_characters(get_player_units
                                           (get_local_player())));
               y += 12;

               textprintf(get_screen_bmp(), get_menu_font(), x, y, white,
                          "Structures: %d",
                          count_structures(get_player_units(get_local_player())));
               y += 12;
            }
         }

      }

      /* Now draw the command buttons on the command panel; first set the palette */
      udrw = get_selected_units(get_local_player());
      if (udrw->data->player == get_local_player()) {
         select_palette(*get_char_palette(udrw->data->gfxidx, udrw->data->player));
         draw_command_panel(get_screen_bmp(), 0, SCREEN_H - 96 - 16);
         unselect_palette();
      }
   }

   /* Draw player's resource icons and resource status */
   draw_resource_icons();

   /* Now mark all interface area's just drawn for dirty-rectangle updates */
   mark_interface_rects();

   redraw_area = 0;
   TRACE("done.\n");
   TRACE("Drawing floating windows...");
   draw_floatwins();
   TRACE("done.\n");
   TRACE("Screenupdate complete.\n");
   TRACE("======================\n");
}

void game_draw_initialize(void)
{
   map_draw_initialize();

   command_panel = get_cmdpanel_bitmap();

   redraw_area = INT_ALL;
}
