/**********************************************/
/* Rise of the Tribes                         */
/* C version, Take 2                          */
/* Evert Glebbeek 1998, 2002                  */
/* eglebbk@phys.uva.nl                        */
/**********************************************/
#include <allegro.h>
#include <stdio.h>
#include <string.h>
#include "playgame.h"
#include "global.h"
#include "dialog.h"
#include "maptile.h"
#include "map.h"
#include "unit.h"
#include "unitprog.h"
#include "gfx.h"
#include "ugfx.h"
#include "script.h"
#include "gmeparse.h"

/* Maximum number of units total */
/* (About 200 max. per player) */
#define MAX_UNITDATA 300
#define MAX_CHARS    (10*MAX_UNITDATA)
#define MAX_CMDS     (12*MAX_UNITDATA)

/* We need a structure to store UNIT data and a string `key' to */
/*  retrieve it. This is used when parsing .gme files */
typedef struct CHAR_KEY_TMP {
   UNIT *cdta;
   char *key;
   int dummy;
   struct CHAR_KEY_TMP *next;
} CHAR_KEY_TMP;

/* Store data needed for parsing unit command buttons */
typedef struct UCMD_KEY_TMP {
   char *key;
   UCMDS cmddata;
   struct UCMD_KEY_TMP *next;
} UCMD_KEY_TMP;

static CHAR_KEY_TMP *char_key_tmp = NULL;
static UCMD_KEY_TMP *ucmd_list = NULL;

/* These are all linear linked lists */

/* these are actually mini-programs not associated with a weapon or graphic */
UNIT *active = NULL;

/* Active tiles: tiles that have an associated mini-program built in */
UNIT *mapobj = NULL;

/* UNIT types */
static UNITDATA **chartps = NULL;
static int num_char_tps = 0;

/* The memory for all UNITs/UNITDATA is allocated in one chunk */
static UNITDATA *cdta_chunk = NULL;
static UNIT *char_chunk = NULL;
static UCMD_KEY_TMP *cmd_chunk = NULL;
static UNITDATA **cdta_stack = NULL;
static UNIT **char_stack = NULL;
static UCMD_KEY_TMP **cmd_stack = NULL;
static int cdta_stack_counter = 0;
static int char_stack_counter = 0;
static int cmd_stack_counter = 0;

/* UNIT data allocation/release commands */
/* For now, these ar just wrappers to malloc()/free(), but that will change */
UNITDATA *alloc_unitdata(void)
{
   if (cdta_stack_counter >= MAX_UNITDATA) {
      game_message("Too much unit data!");
      return NULL;
   }
   /* Advance stack counter */
   cdta_stack_counter++;
   /* return the address of the last free UNITDATA in the list */
   return cdta_stack[cdta_stack_counter - 1];
}

void free_unitdata(UNITDATA *c)
{
   if (cdta_stack_counter) {
      cdta_stack_counter--;
      cdta_stack[cdta_stack_counter] = c;
      c->flags = 0;
   }
}

UNIT *alloc_unit(void)
{
   if (char_stack_counter >= MAX_CHARS) {
      game_message("Too many UNITs!");
      return NULL;
   }
   TRACE("Unit alloc'ed: stack_counter: %d/%d\n", cdta_stack_counter,
         MAX_CHARS);
   /* Advance stack counter */
   char_stack_counter++;
   /* return the address of the last free UNIT in the list */
   return char_stack[char_stack_counter - 1];
}

void free_unit(UNIT *c)
{
   if (char_stack_counter) {
      char_stack_counter--;
      TRACE("Unit free'ed: stack_counter: %d/%d\n", cdta_stack_counter,
            MAX_CHARS);
      char_stack[char_stack_counter] = c;
   }
}

static UCMD_KEY_TMP *alloc_ucmdb(void)
{
   if (char_stack_counter >= MAX_CMDS) {
      game_message("Too many COMMANDSs!");
      return NULL;
   }

   TRACE("Command alloc'ed: stack_counter: %d/%d\n", cmd_stack_counter,
         MAX_CMDS);

   /* Advance stack counter */
   cmd_stack_counter++;
   /* return the address of the last free UNIT in the list */
   return cmd_stack[cmd_stack_counter - 1];
}

static __attribute__ ((unused)) void free_ucmdb(UCMD_KEY_TMP * c)
{
   if (cmd_stack_counter) {
      cmd_stack_counter--;
      TRACE("Command free'ed: stack_counter: %d/%d\n", cmd_stack_counter,
            MAX_CMDS);
      cmd_stack[char_stack_counter] = c;
   }
}

int get_unit_count(void)
{
   return char_stack_counter;
}

int get_udta_count(void)
{
   return cdta_stack_counter;
}

/* reset the UNIT/UNITDATA global chunks */
static void reset_char_chunk(void)
{
   int c;

   /* Clear the command queues if they hadn't been cleared before */
   if (char_stack) {
      for (c = 0; c < MAX_UNITDATA; c++) {
         if (cdta_chunk[c].cqueue) {
            free(cdta_chunk[c].cqueue);
            cdta_chunk[c].cqueue = NULL;
         }
      }
   }

   if (!cdta_chunk)
      cdta_chunk = malloc(sizeof(UNITDATA) * MAX_UNITDATA);
   if (!char_chunk)
      char_chunk = malloc(sizeof(UNIT) * MAX_CHARS);
   if (!cmd_chunk)
      cmd_chunk = malloc(sizeof(UCMD_KEY_TMP) * MAX_CMDS);
   if (!cdta_stack)
      cdta_stack = malloc(sizeof(UNITDATA *) * MAX_UNITDATA);
   if (!char_stack)
      char_stack = malloc(sizeof(UNIT *) * MAX_CHARS);
   if (!cmd_stack)
      cmd_stack = malloc(sizeof(UCMD_KEY_TMP *) * MAX_CMDS);

   cdta_stack_counter = 0;
   char_stack_counter = 0;
   cmd_stack_counter = 0;
   for (c = 0; c < MAX_UNITDATA; c++)
      cdta_stack[c] = &(cdta_chunk[c]);
   for (c = 0; c < MAX_CHARS; c++)
      char_stack[c] = &(char_chunk[c]);
   for (c = 0; c < MAX_CMDS; c++)
      cmd_stack[c] = &(cmd_chunk[c]);

   /* Clear fields that need to be in a clean state */
   for (c = 0; c < MAX_UNITDATA; c++)
      cdta_chunk[c].cqueue = NULL;

   ucmd_list = NULL;
   //popup_message ("UNITDATA: %d bytes\n", sizeof(UNITDATA));
}

/* Add a UNIT with identifier key to the list */
void push_char_def(const int id, const char *key)
{
   CHAR_KEY_TMP *ckt = NULL;

   ckt = malloc(sizeof(CHAR_KEY_TMP));
   ckt->cdta = create_unit(id);
   ckt->cdta->data->hp = ckt->cdta->data->maxhp;
   ckt->cdta->data->mp = ckt->cdta->data->maxmp;
   ckt->key = strdup(key);
   ckt->next = char_key_tmp;
   char_key_tmp = ckt;
}

/* Retrieve a UNIT by key reference */
/* Returns NULL if the key could not be found */
UNIT *get_char_def(const char *key)
{
   CHAR_KEY_TMP *ckt = char_key_tmp;

   while (ckt) {
      if (!strcmp(ckt->key, key))
         return ckt->cdta;
      ckt = ckt->next;
   }
   return NULL;
}

/* Returns the value of the integer DUMMY associated with key */
int get_char_dummy(const char *key)
{
   CHAR_KEY_TMP *ckt = char_key_tmp;

   while (ckt) {
      if (!strcmp(ckt->key, key))
         return ckt->dummy;
      ckt = ckt->next;
   }
   return 0;
}

/* Set the temporary dummy register for UNIT */
void set_char_dummy(const char *key, int c)
{
   CHAR_KEY_TMP *ckt = char_key_tmp;

   while (ckt) {
      if (!strcmp(ckt->key, key)) {
         ckt->dummy = c;
         return;
      }
      ckt = ckt->next;
   }
}

/* Destroy the temporary UNIT array */
void destroy_char_def(void)
{
   CHAR_KEY_TMP *ckt;

   while (char_key_tmp) {
      ckt = char_key_tmp->next;
      free(char_key_tmp->key);
      if (!(char_key_tmp->cdta->data->flags & CF_UNIQUE))
         release_unit(char_key_tmp->cdta);
      free(char_key_tmp);
      char_key_tmp = ckt;
   }
}

UCMDS *find_ucmdb(char *key)
{
   UCMD_KEY_TMP *ucmd = ucmd_list;
   ASSERT(key);

   while (ucmd) {
      /* This can fail if there is no proper key= statement in the .gme file */
      ASSERT(ucmd->key);
      
      if (!strcmp(ucmd->key, key))
         return &(ucmd->cmddata);
      ucmd = ucmd->next;
   }

   return NULL;
}

static UCMD_KEY_TMP *parse_next_command(PACKFILE * inf)
{
   char *s = malloc(256 * sizeof(char));
   UCMD_KEY_TMP *c;

   c = alloc_ucmdb();

   if (!c)
      return NULL;

   /* default settings */
   c->cmddata.name = "";
   c->cmddata.icon = NULL;
   c->cmddata.script = NULL;
   c->cmddata.uprog = 0;
   c->cmddata.flags = 0;
   c->cmddata.argc = 0;
   c->key = NULL;
   c->next = NULL;

   while (pack_fgets(s, 256, inf)) {
      if (s[0] != ';') {        /* Skip comments */
         char *eol = strstr(s, "\n");

         if (eol)
            eol[0] = '\0';

         /* clear spaces from the beginning of the string */
         strflushl(s);
         strflushr(s);

         if (s[0] != 0) {       /* check if line isn't empty */
            /* split into two strings */
            char *opt = s;
            char *arg;

            arg = strstr(opt, "=");
            if (arg) {
               arg[0] = '\0';
               arg++;
               /* Remove trailing newlines */
               eol = strstr(arg, "\n");
               if (eol)
                  eol[0] = '\0';
               strflushl(arg);
            } else {
               arg = opt + strlen(opt);
            }

            /* Parse lines */
            if (!strcmp(opt, "name")) {
               c->cmddata.name = strdup(arg);
            } else if (!strcmp(opt, "key")) {
               c->key = strdup(arg);
            } else if (!strcmp(opt, "select_target")) {
               c->cmddata.flags |= UCMD_TARGET;
            } else if (!strcmp(opt, "place_structure")) {
               c->cmddata.flags |= UCMD_TARGET+UCMD_PLACE;
            } else if (!strcmp(opt, "exclusive")) {
               c->cmddata.flags |= UCMD_EXCLUSIVE;
            } else if (!strcmp(opt, "clear_queue")) {
               c->cmddata.flags |= UCMD_CLR_QUEUE;
            } else if (!strcmp(opt, "clear_target")) {
               c->cmddata.flags |= UCMD_CLR_TARGET;
            } else if (!strcmp(opt, "cancel_command")) {
               c->cmddata.flags |= UCMD_CANCEL;
            } else if (!strcmp(opt, "progress")) {
               c->cmddata.flags |= UCMD_PROGRESS;
            } else if (!strcmp(opt, "buildargs")) {
               /* Arguments: */
               /*  initial site, delay time, final state */
               eol = strstr(arg,",");
               eol[0] = '\0';
               c->cmddata.argv[0] = key_to_id(arg);
               arg = eol+1;
               sscanf(arg, "%d,", &(c->cmddata.argv[1]));
               eol = strstr(arg,",");
               arg = eol+1;
               c->cmddata.argv[2] = key_to_id(arg);
               c->cmddata.argc = 3;
            } else if (!strcmp(opt, "upgradeargs")) {
               /* Arguments: */
               /*  initial unit-type, delay time, final unit-type */
               /* For now, the first argument is ignored */
               eol = strstr(arg,",");
               eol[0] = '\0';
               c->cmddata.argv[0] = key_to_id(arg);
               arg = eol+1;
               sscanf(arg, "%d,", &(c->cmddata.argv[1]));
               eol = strstr(arg,",");
               arg = eol+1;
               c->cmddata.argv[2] = key_to_id(arg);
               c->cmddata.argc = 3;
            } else if (!strcmp(opt, "icon")) {
               int b;

               b = find_icon_name(arg);
               if (b < 0) {
                  popup_message("Cannot find icon:\n[%s]", arg);
               } else {
                  c->cmddata.icon = get_icon_gfx(b);
               }
            } else if (!strcmp(opt, "prog")) {
               c->cmddata.script = get_prog_ptr(arg);
            } else if (!strcmp(opt, "newcommand")) {
               /* this baby is only needed as a bug work-around */
            } else {
               if (arg)
                  popup_message("Unknown property:\n%s=%s", opt, arg);
               else
                  popup_message("Unknown property:\n%s", opt);
            }
         } else {               /* Empty line ==> we're done */
            break;
         }
      }
   }
   
   /* Make sure all things that need to be asigned are */
   /* Do so via a number of assertions rather than giving an error message */
   ASSERT(c->cmddata.icon);
   ASSERT(c->key);

   free(s);
   return c;
}

/* Parse the file from the current position to the next empty line */
/*  Interpret data as UNIT data and return it */
static UNITDATA *parse_next_char(PACKFILE * inf)
{
   char *s = malloc(256 * sizeof(char));
   UNITDATA *c;
   int n;

   c = alloc_unitdata();
   ASSERT(c);
   /* default settings */
   c->name = "";
   c->idkey = NULL;
   c->icon = NULL;
   c->flags = 0;
   c->maxhp = c->hp = 0;
   c->maxmp = c->mp = 0;
   c->layer = 0;
   c->size = 1;
   c->draw_dir = c->direction = 4;
   c->gfx_x_size = 1;
   c->gfx_y_size = 1;
   c->speed = 0;
   c->path = NULL;
   c->gfxidx = 0;
   c->current_frame = 0;
   c->cprog = NULL;
   c->stack_size = 0;
   c->ip = 0;
   c->wait_counter = 0;
   c->cqueue = NULL;
   c->queuesize = 1;
   c->queuecount = 0;
   c->script_target = c;
   c->ai_target.unit = NULL;
   c->parent = NULL;
   c->loaded_units = NULL;
   c->attacker = NULL;
   c->radius = 0;
   c->gfx_dx = 0;
   c->gfx_dy = 0;
   c->gfx_w = 0;
   c->gfx_h = 0;

   for (n = 0; n < 13; n++) {
      c->unit_command[n] = NULL;
   }

   for (n = 0; n < 8; n++) {
      c->resource_cost[n] = 0;
   }

   for (n = 0; n < NUM_PROGS; n++)
      c->prog[n] = NULL;
   c->tiles = NULL;
   c->occ_tiles = NULL;

   while (pack_fgets(s, 256, inf)) {
      if (s[0] != ';') {        /* Skip comments */
         char *eol = strstr(s, "\n");
         
         /*
         printf("%s\n", s);
         fflush(stdout);
         */

         if (eol)
            eol[0] = '\0';

         /* clear spaces from the beginning of the string */
         strflushl(s);
         strflushr(s);

         if (s[0] != 0) {       /* check if line isn't empty */
            /* split into two strings */
            char *opt = s;
            char *arg;

            arg = strstr(opt, "=");
            if (arg) {
               arg[0] = '\0';
               arg++;
               /* Remove trailing newlines */
               eol = strstr(arg, "\n");
               if (eol)
                  eol[0] = '\0';
               strflushl(arg);
            } else {
               arg = opt + strlen(opt);
            }

            //popup_message ("|%s = %s|", opt, arg);
            /* Parse lines */
            if (!strcmp(opt, "key")) {
               /* Must warn for `#self', `#parent', `#target' ! */
               c->idkey = str_to_lower(arg);
            } else if (!strcmp(opt, "name")) {
               /* Must warn for `#self', `#parent', `#target' ! */
               c->name = strdup(arg);
               if (!c->idkey)
                  c->idkey = str_to_lower(c->name);
            } else if (!strcmp(opt, "icon")) {
               int b;

               b = find_icon_name(arg);
               if (b < 0) {
                  popup_message("Cannot find icon:\n[%s]", arg);
               } else {
                  c->icon = get_icon_gfx(b);
               }
            } else if (!strcmp(opt, "gfx")) {
               int b;

               b = find_cgfx_name(arg);
               if (b < 0) {
                  popup_message("Cannot find cgfx:\n%s", arg);
               } else {
                  c->gfxidx = b;
                  if (get_num_gfx(b) == 1)
                     c->draw_dir = c->direction = 0;
               }
            } else if (!strcmp(opt, "hp")) {
               int b;

               sscanf(arg, "%d", &b);
               c->hp = c->maxhp = b;
            } else if (!strcmp(opt, "mp")) {
               int b;

               sscanf(arg, "%d", &b);
               c->mp = c->maxmp = b;
            } else if (!strcmp(opt, "x")) {
               int b;

               sscanf(arg, "%d", &b);
               c->x = b * TILE_WIDTH;
            } else if (!strcmp(opt, "y")) {
               int b;

               sscanf(arg, "%d", &b);
               c->y = b * TILE_HEIGHT;
            } else if (!strcmp(opt, "speed")) {
               int b;

               sscanf(arg, "%d", &b);
               c->speed = b;
            } else if (!strcmp(opt, "lightradius")) {
               int b;

               sscanf(arg, "%d", &b);
               c->radius = b;
            } else if (!strcmp(opt, "layer")) {
               int b;

               sscanf(arg, "%d", &b);
               c->layer = b;
            } else if (!strcmp(opt, "queue")) {
               int b;

               sscanf(arg, "%d", &b);
               //c->queuesize = b;
            } else if (!strcmp(opt, "gfx_x_size")) {
               int b;

               sscanf(arg, "%d", &b);
               c->gfx_x_size = b;
            } else if (!strcmp(opt, "gfx_y_size")) {
               int b;

               sscanf(arg, "%d", &b);
               c->gfx_y_size = b;
            } else if (!strcmp(opt, "prog_built")) {
               c->prog[PROG_BUILT] = get_prog_ptr(arg);
            } else if (!strcmp(opt, "prog_move")) {
               c->prog[PROG_MOVE] = get_prog_ptr(arg);
            } else if (!strcmp(opt, "prog_attack")) {
               c->prog[PROG_ATTACK] = get_prog_ptr(arg);
            } else if (!strcmp(opt, "prog_turn_north")) {
               c->prog[PROG_TURN_NORTH] = get_prog_ptr(arg);
            } else if (!strcmp(opt, "prog_turn_ne")) {
               c->prog[PROG_TURN_NE] = get_prog_ptr(arg);
            } else if (!strcmp(opt, "prog_turn_east")) {
               c->prog[PROG_TURN_EAST] = get_prog_ptr(arg);
            } else if (!strcmp(opt, "prog_turn_se")) {
               c->prog[PROG_TURN_SE] = get_prog_ptr(arg);
            } else if (!strcmp(opt, "prog_turn_south")) {
               c->prog[PROG_TURN_SOUTH] = get_prog_ptr(arg);
            } else if (!strcmp(opt, "prog_turn_sw")) {
               c->prog[PROG_TURN_SW] = get_prog_ptr(arg);
            } else if (!strcmp(opt, "prog_turn_west")) {
               c->prog[PROG_TURN_WEST] = get_prog_ptr(arg);
            } else if (!strcmp(opt, "prog_turn_nw")) {
               c->prog[PROG_TURN_NW] = get_prog_ptr(arg);
            } else if (!strcmp(opt, "prog_dies")) {
               c->prog[PROG_DIES] = get_prog_ptr(arg);
/*            } else if (!strcmp(opt, "prog_cast")) {
               c->prog[PROG_CAST] = get_prog_ptr (arg);*/
            } else if (!strcmp(opt, "prog_idle")) {
               c->prog[PROG_IDLE] = get_prog_ptr(arg);
/*            } else if (!strcmp(opt, "prog_touch")) {
               c->prog[PROG_TOUCH] = get_prog_ptr (arg);
            } else if (!strcmp(opt, "prog_untouch")) {
               c->prog[PROG_UNTOUCH] = get_prog_ptr (arg);
            } else if (!strcmp(opt, "prog_attacked")) {
               c->prog[PROG_ATTACKED] = get_prog_ptr (arg);*/
            } else if (!strcmp(opt, "groundunit")) {
               /* check the switches */
               c->flags |= CC_GUNIT;
            } else if (!strcmp(opt, "structure")) {
               /* check the switches */
               c->flags |= CC_STRUCTURE;
               /* Structures, in general, have only one orientation */
               c->draw_dir = c->direction = 0;
            } else if (!strcmp(opt, "commandcentre")) {
               /* check the switches */
               c->flags |= CC_COMMANDPOST;
            } else if (!strcmp(opt, "warehouse")) {
               /* check the switches */
               c->flags |= CC_WAREHOUSE;
            } else if (!strcmp(opt, "buildsite")) {
               /* check the switches */
               c->flags |= CC_BUILDSITE;
            } else if (!strcmp(opt, "object")) {
               /* check the switches */
               c->flags |= CC_OBJECT;
            } else if (!strcmp(opt, "mapobject")) {
               /* check the switches */
               c->flags |= CC_MAPOBJECT;
            } else if (!strcmp(opt, "spell")) {
               /* check the switches */
               c->flags |= CC_SPELL;
            } else if (!strcmp(opt, "book")) {
               /* check the switches */
               c->flags |= CC_BOOK;
            } else if (!strcmp(opt, "weapon")) {
               /* check the switches */
               c->flags |= CC_WEAPON;
            } else if (!strcmp(opt, "helmet")) {
               /* check the switches */
               c->flags |= CC_HELMET;
            } else if (!strcmp(opt, "bracers")) {
               /* check the switches */
               c->flags |= CC_BRACERS;
            } else if (!strcmp(opt, "jewelry")) {
               /* check the switches */
               c->flags |= CC_JEWELRY;
            } else if (!strcmp(opt, "shield")) {
               /* check the switches */
               c->flags |= CC_SHIELD;
            } else if (!strcmp(opt, "armor")) {
               /* check the switches */
               c->flags |= CC_ARMOR;
            } else if (!strcmp(opt, "projectile")) {
               /* check the switches */
               c->flags |= CC_PROJECTILE;
            } else if (!strcmp(opt, "effect")) {
               /* check the switches */
               c->flags |= CC_EFFECT;
            } else if (!strcmp(opt, "lightsource")) {
               /* check the switches */
               c->flags |= CF_LIGHTSOURCE;
            } else if (!strcmp(opt, "selectmulti")) {
               /* check the switches */
               c->flags |= CF_SEL_MUL;
            } else if (!strcmp(opt, "unique")) {
               /* check the switches */
               c->flags |= CF_UNIQUE;
            } else if (!strcmp(opt, "noselect")) {
               /* check the switches */
               c->flags |= CF_NOSELECT;
            } else if (!strcmp(opt, "newchar")) {
               /* this baby is only needed as a bug work-around */
            } else if (!strcmp(opt, "button")) {
               /* Add a command-button for this unit */
               char *key;

               sscanf(arg, "%d,", &n);
               key = strstr(arg, ",");

               if (key) {
                  key++;
                  while (key[0] && (key[0] == ' '))
                     key++;
                     
                  c->unit_command[n] = find_ucmdb(key);
                  /*
                     if (c->unit_command[n])
                     popup_message ("%d, %s", n, key);
                     // */
               } else {
                  c->unit_command[n] = NULL;
               }
            } else if (!strcmp(opt, "resource")) {
               /* Set resources for this unit */
               char *key;
               int cost;

               sscanf(arg, "%d,%d", &n, &cost);
               key = strstr(arg, ",");
               if (key) {
                  key++;
                  while (key[0] && (key[0] == ' '))
                     key++;
                  c->resource_cost[n] = cost;
               } else {
                  c->resource_cost[n] = 0;
               }
            } else {
               if (arg)
                  popup_message("Unknown property:\n%s=%s", opt, arg);
               else
                  popup_message("Unknown property:\n%s", opt);
            }
         } else {               /* Empty line ==> we're done */
            break;
         }
      }
   }

   /* Command number 13 is special: it's the global cancel command */
   c->unit_command[12] = find_ucmdb("cancel_command");

   free(s);
   return c;
}

/* Load unit data */
/* f points to an open file, ie, it must be of type FILE* */
void load_unit_data(PACKFILE * f)
{
   UNITDATA *c;
   
   ASSERT(f);

   c = parse_next_char(f);
   if (c) {
      chartps = realloc(chartps, (num_char_tps + 1) * sizeof(UNITDATA *));
      chartps[num_char_tps] = c;
      num_char_tps++;
   }
}


/* load unit command button data */
/* f points to an open file, ie, it must be of type FILE* */
void load_cmdb_data(PACKFILE * f)
{
   UCMD_KEY_TMP *c;

   c = parse_next_command(f);
   
   if (ucmd_list) {
      if (c) {
         c->next = ucmd_list;
         ucmd_list = c;
      }
   } else {
      ucmd_list = c;
   }
}

/* Retrieve a UNIT from a list (by ID-key) */
/* returns NULL if the UNIT couldn't be found */
UNITDATA *find_char_idkey(UNIT *clst, char *charname)
{
   UNIT *c = clst;
   char *cname = str_to_lower(charname);

   while (c) {
      if (!strcmp(c->data->idkey, cname)) {
         free(cname);
         return c->data;
      }
      c = c->next;
      if (c == clst) {
         free(cname);
         return NULL;
      }
   }
   free(cname);
   return NULL;
}

/* Retrieve a UNIT from a list (by flags) */
UNITDATA *find_char_flags(UNIT *clst, int flags)
{
   UNIT *c = clst;

   while (c) {
      if ((c->data->flags & flags) == flags)
         return c->data;
      c = c->next;
      if (c == clst)
         return NULL;
   }
   return NULL;
}

/* matches a UNIT key to an ID number */
unsigned int key_to_id(char *charname)
{
   int n;
   char *cname;
   
   ASSERT(charname);
   
   cname = str_to_lower(charname);

   ASSERT(cname);

   for (n = 0; n < num_char_tps; n++)
      if (!strcmp(chartps[n]->idkey, cname)) {
         break;
      }
   free(cname);
   
   ASSERT(n<num_char_tps);
   
   return n;
}

/* matches a UNIT name to an ID number */
unsigned int name_to_id(char *char_name)
{
   int n;

   for (n = 0; n < num_char_tps; n++)
      if (!strcmp(chartps[n]->name, char_name)) {
         break;
      }
   return n;
}


/* creates a new UNIT/object of the specified type and returns a pointer
   to its structure. Handles unique UNITs/objects differently */
UNIT *create_unit(unsigned int n)
{
   UNIT *c = NULL;

   ASSERT(n < num_char_tps);

   c = alloc_unit();
   if (!c)
      return NULL;
   c->next = c->prev = NULL;
   if (chartps[n]->flags & CF_UNIQUE) {
      c->data = chartps[n];
   } else {
      /* copy specs for a new instance of this UNIT */
      c->data = alloc_unitdata();
      if (c->data) {
         memcpy(c->data, chartps[n], sizeof(UNITDATA));
      } else {
         popup_message("Out of memory creating UNIT!");
         exit(EXIT_FAILURE);
      }
   }
   c->data->flags &= ~CF_CHANGED;
   c->data->tiles = NULL;
   c->data->script_target = c->data;
   c->data->ai_target.unit = NULL;
   c->data->parent = NULL;
   c->data->loaded_units = NULL;
   c->data->attacker = NULL;
   c->data->path_tries = 0;
   c->data->cqueue = malloc(c->data->queuesize*sizeof(SCRIPT_QUEUE));
   c->data->queuecount = 0;
   /* Activate UNIT creation mini-prog */
   if (c->data->prog[PROG_BUILT])
      push_command(c->data, c->data->prog[PROG_BUILT], 0);
   return c;
}

/* Changes a unit to a new identity */
void change_unit(UNITDATA *udta, const int n)
{
   int c;

   ASSERT(n < num_char_tps);
   ASSERT(udta);

   udta->flags = chartps[n]->flags | (udta->flags & CF_SELECTED);
   udta->name = chartps[n]->name;
   udta->icon = chartps[n]->icon;
   udta->hp = udta->hp * chartps[n]->maxhp / udta->maxhp;
   udta->maxhp = chartps[n]->maxhp;
   if (udta->maxmp)
      udta->mp = udta->mp * chartps[n]->maxmp / udta->maxmp;
   udta->maxmp = chartps[n]->maxmp;
   udta->speed = chartps[n]->speed;
   udta->radius = chartps[n]->radius;
   udta->size = chartps[n]->size;
   udta->gfxidx = chartps[n]->gfxidx;
   udta->queuesize = chartps[n]->queuesize;
   udta->direction = chartps[n]->direction;
   udta->draw_dir = chartps[n]->draw_dir;
   udta->current_frame = chartps[n]->current_frame;

   for (c = 0; c < NUM_PROGS; c++)
      udta->prog[c] = chartps[n]->prog[c];
   for (c = 0; c < 8; c++)
      udta->resource_cost[c] = chartps[n]->resource_cost[c];
   for (c = 0; c < 13; c++)
      udta->unit_command[c] = chartps[n]->unit_command[c];
}

/* Sets a unit's (x, y) coorinates. Note: x and y are in pixel coordinates! */
void set_unit_coors(UNITDATA *u, const int x, const int y)
{
   ASSERT(u);
   
   u->x = x;
   u->y = y;
   u->ai_target.x = x / TILE_WIDTH;
   u->ai_target.y = y / TILE_HEIGHT;
   u->ai_target.unit = NULL;
}

/* Sets a unit's target to the specified target */
void set_unit_target(UNITDATA *u, const TARGET target)
{
   ASSERT(u);
   
   u->ai_target.unit = target.unit;
   u->ai_target.x = target.x;
   u->ai_target.y = target.y;
}

/* Compares two targets. Expands the unit field if nescessary */
int targets_are_equal(const TARGET t1, const TARGET t2)
{
   int x1, y1;
   int x2, y2;

   if (t1.unit && t2.unit) {
      return t1.unit == t2.unit;
   }

   if (t1.unit) {
      x1 = t1.unit->x/TILE_WIDTH;
      y1 = t1.unit->y/TILE_HEIGHT;
   } else {
      x1 = t1.x;
      y1 = t1.y;
   }

   if (t2.unit) {
      x2 = t2.unit->x/TILE_WIDTH;
      y2 = t2.unit->y/TILE_HEIGHT;
   } else {
      x2 = t2.x;
      y2 = t2.y;
   }

   if ((x1 == -1) || (y1 == -1) || (x2 == -1) || (y2 == -1))
      return TRUE;

   return (x1 == x2) && (y1 == y2);
}

inline BITMAP *get_char_icon(unsigned int n)
{
   ASSERT(n <= num_char_tps);
   return chartps[n]->icon;
}

inline char *get_char_name(unsigned int n)
{
   ASSERT(n <= num_char_tps);
   return chartps[n]->name;
}

inline int get_char_maxhp(unsigned int n)
{
   ASSERT(n <= num_char_tps);
   return chartps[n]->hp;
}

inline int get_char_maxmp(unsigned int n)
{
   ASSERT(n <= num_char_tps);
   return chartps[n]->mp;
}

inline int get_char_xsize(unsigned int n)
{
   ASSERT(n <= num_char_tps);
   return chartps[n]->gfx_x_size;
}

inline int get_char_ysize(unsigned int n)
{
   ASSERT(n <= num_char_tps);
   return chartps[n]->gfx_y_size;
}

inline int get_char_resource_cost(int c, int r)
{
   ASSERT(c <= num_char_tps);
   ASSERT(r < 8);
   
   return chartps[c]->resource_cost[r];
}

/* Duplicate char - creates a copy of the passed UNIT, unless it is */
/*  unique, in which case the original UNIT is returned */
UNIT *duplicate_unit(UNIT *c)
{
   UNIT *cc;

   ASSERT(c);

   if (c->data->flags & CF_UNIQUE)
      return c;

   cc = alloc_unit();
   if (!cc)
      return NULL;
   cc->next = c->next;
   cc->prev = c->prev;
   cc->data = alloc_unitdata();
   memcpy(cc->data, c->data, sizeof(UNITDATA));
   cc->data->script_target = cc->data;
   cc->data->ai_target.unit = NULL;
   cc->data->attacker = NULL;
   cc->data->loaded_units = NULL;
   cc->data->cprog = duplicate_commands(c->data->cprog);
   return cc;
}

/* Clones a UNIT. This even copies unique UNITs, althoug the */
/*  clone doesn't have the UNIQUE flag set */
UNIT *clone_unit(UNIT *c)
{
   UNIT *cc;

   ASSERT(c);

   cc = alloc_unit();
   if (!cc)
      return NULL;
   cc->next = c->next;
   cc->prev = c->prev;
   cc->data = alloc_unitdata();
   memcpy(cc->data, c->data, sizeof(UNITDATA));
   cc->data->script_target = cc->data;
   cc->data->ai_target.unit = NULL;
   cc->data->attacker = NULL;
   cc->data->loaded_units = NULL;
   /* This clone obviously isn't unique */
   cc->data->flags &= ~CF_UNIQUE;

   return cc;
}

/* Initialize all chracters at the start of the game */
void ini_chars(void)
{
   int n;

   reset_char_chunk();
   destroy_char_def();

   /* Clear previous character descriptions (if any) */
   for (n = 0; n < num_char_tps; n++) {
      free_unitdata(chartps[n]);
   }
   free(chartps);
   chartps = NULL;
   num_char_tps = 0;

   /* load UNIT descriptions */
   /* this can't be done properly yet, because we haven't got the
      control programs yet, but we need the UNIT names now */
   parse_gme_file(get_game_path("rise.gme"), UNITS);

   /* Initiaize mini-programs */
   pass_one();
   load_unit_control(get_game_path("rise.rup"));
   pass_two();
   load_unit_control(get_game_path("rise.rup"));
   clear_parser_stack();

   /* Load command button descriptions */
   parse_gme_file(get_game_path("rise.gme"), COMMANDS);

   /* now clear the descriptions we had and reload */
   for (n = 0; n < num_char_tps; n++) {
      free_unitdata(chartps[n]);
   }
   free(chartps);
   chartps = NULL;
   num_char_tps = 0;

   /* load UNIT descriptions again, this time with control programs and buttons */
   parse_gme_file(get_game_path("rise.gme"), UNITS);
   /*
   printf ("Alloc'ed %d bytes (%dkB) for units\n",
           sizeof(UNITDATA) * MAX_UNITDATA+
           sizeof(UNIT) * MAX_CHARS+
           sizeof(UCMD_KEY_TMP) * MAX_CMDS+
           sizeof(UNITDATA *) * MAX_UNITDATA+
           sizeof(UNIT *) * MAX_CHARS+
           sizeof(UCMD_KEY_TMP *) * MAX_CMDS,
           (sizeof(UNITDATA) * MAX_UNITDATA+
            sizeof(UNIT) * MAX_CHARS+
            sizeof(UCMD_KEY_TMP) * MAX_CMDS+
            sizeof(UNITDATA *) * MAX_UNITDATA+
            sizeof(UNIT *) * MAX_CHARS+
            sizeof(UCMD_KEY_TMP *) * MAX_CMDS)/1024);
   // */            
}

/* Destroy all UNITs */
/* Currently does nothing! */
/* Must also free inventories/tile lists! */
void destroy_all_chars(void)
{
}

/* Check if UNIT is a mapobject */
int is_mapobject(UNITDATA *obj)
{
   if (!obj)
      return 0;
   return (obj->flags & CC_MAPOBJECT) == CC_MAPOBJECT;
}

int is_structure(UNITDATA *obj)
{
   if (!obj)
      return 0;
   return (obj->flags & CC_STRUCTURE) == CC_STRUCTURE;
}

int can_be_selected(UNITDATA *u)
{
   ASSERT(u);
   return !((u->flags & CF_NOSELECT) == CF_NOSELECT);
}

int count_characters(UNIT *ulst)
{
   int count = 0;
   UNIT *u;

   for (u = ulst; u; u = u->next) {
      if (!is_structure(u->data))
         count++;
      count += count_characters(u->data->loaded_units);
   }

   return count;
}

int count_structures(UNIT *ulst)
{
   int count = 0;
   UNIT *u;

   for (u = ulst; u; u = u->next) {
      if (is_structure(u->data))
         count++;
   }
   return count;
}

/* Returns TRUE if UNIT chr appears in the list */
/* Handles both circular and linear lists */
int in_list(const UNIT *list, const UNITDATA *chr)
{
   UNIT *lst = (UNIT *)list;

   while (lst) {
      if (lst->data == chr)
         return TRUE;
      lst = lst->next;
      if (lst == list)
         break;
   }
   return FALSE;
}

/* Duplicates a linked list */
UNIT *duplicate_list(UNIT *list)
{
   UNIT *u = NULL;

   if (list) {
      u = alloc_unit();
      if (!u)
         return NULL;
      u->next = NULL;
      u->prev = NULL;
      u->data = list->data;
      u->next = duplicate_list(list->next);
      if (u->next)
         u->next->prev = u;
   }
   return u;
}

/* Destroys a linked list. Doesn't free the data */
void destroy_list(UNIT *list)
{
   if (list) {
      if (list->next)
         destroy_list(list->next);
      free_unit(list);
   }
}

/* Add element to an existing circular linked UNIT list (at the END) */
void add_circ_list(UNIT *list, UNIT *c)
{
   c->next = list;
   c->prev = list->prev;
   c->prev->next = c;
   c->next->prev = c;
}

/* Remove element from an existing circular linked UNIT list */
/* It cannot be the only element in the list without causing trouble later */
void remove_circ_list(UNIT *list)
{
   list->prev->next = list->next;
   list->next->prev = list->prev;
}

/* Add a unit to a linked list. Returns the new head of the list. */
UNIT *add_to_list(UNITDATA *udta, UNIT *ulst)
{
   UNIT *newunit = alloc_unit();

   if (!newunit)
      return ulst;

   newunit->data = udta;
   newunit->prev = NULL;
   newunit->next = ulst;
   if (newunit->next)
      newunit->next->prev = newunit;

   return newunit;
}

/* Remove a unit from a linear linked list. If there are multiple instances */
/*  of the unit in the list, it will delete all of them. */
/* Returns the new head of the list */
UNIT *remove_from_list(UNITDATA *udta, UNIT *ulst)
{
   UNIT *nul = ulst;
   UNIT *ul = ulst;
   UNIT *u;

   while (ul) {
      if (ul->data == udta) {
         u = ul;
         ul = ul->next;
         if (u == nul)
            nul = nul->next;
         if (u->next)
            u->next->prev = u->prev;
         if (u->prev)
            u->prev->next = u->next;
         free_unit(u);
      } else {
         ul = ul->next;
      }
   }
   return nul;
}

/* Add unit to the list of loaded units in container */
void load_unit(UNITDATA *unit, UNITDATA *container)
{
   ASSERT(container);
   ASSERT(unit);
   
   container->loaded_units = add_to_list(unit, container->loaded_units);
}

/* Remove unit from the list of loaded units in container */
void unload_unit(UNITDATA *unit, UNITDATA *container)
{
   ASSERT(container);
   ASSERT(unit);

   container->loaded_units = remove_from_list(unit, container->loaded_units);
}

/* Unload all units */
void unload_all_units(UNITDATA *container)
{
   ASSERT(container);

   while (container->loaded_units) {
      unload_unit(container->loaded_units->data, container);
   }
}

/* Releases a UNIT; assumes that it is already removed from whatever */
/*  linked list it was part of. */
void release_unit(UNIT *c)
{
   UNIT *cc;

   cc = active;
   while (cc) {
      if (cc->data->parent == c->data)
         cc->data->parent = NULL;
      if (cc->data->attacker == c->data)
         cc->data->attacker = NULL;
      if (cc->data->script_target == c->data)
         cc->data->script_target = cc->data;
      cc = cc->next;
   }

   cc = mapobj;
   while (cc) {
      if (cc->data->parent == c->data)
         cc->data->parent = NULL;
      if (cc->data->attacker == c->data)
         cc->data->attacker = NULL;
      if (cc->data->script_target == c->data)
         cc->data->script_target = cc->data;
      cc = cc->next;
   }
   
   free(c->data->cqueue);
   c->data->cqueue = NULL;
   c->data->queuesize = 1;
   c->data->queuecount = 0;

   remove_commands(c->data);
   unload_all_units(c->data);
   if (!(c->data->flags & CF_UNIQUE))
      free_unitdata(c->data);
   free_unit(c);
}

/* remove an active tile (mapobj) from the list */
/* Note that the tile itself will also have to be removed, something */
/* this baby doesn't do */
void remove_mapobj(UNIT *c)
{
   ASSERT(c);

   if (c == mapobj) {
      mapobj = mapobj->next;
      if (mapobj)
         mapobj->prev = NULL;
   } else {
      c->prev->next = c->next;
      c->next->prev = c->prev;
   }
   /* Free the tiles list (not normally done for mapobj's) */
   free_maptile(c->data->tiles);
   c->data->tiles = NULL;
}


/* Add a UNIT to the activation list */
void activate(UNIT *c)
{
   ASSERT(c);

   if (active)
      active->prev = c;
   c->next = active;
   c->prev = NULL;
   active = c;
   c->data->flags |= CF_EFFECT;
   c->data->flags &= ~CF_PLAYER;
}

/* remove a UNIT from the active list */
void deactivate(UNIT *c)
{
   ASSERT(c);

   if (c == active) {
      active = active->next;
      if (active)
         active->prev = NULL;
   } else {
      c->prev->next = c->next;
      c->next->prev = c->prev;
   }
}
