/**********************************************/
/* Rise of the Tribes                         */
/* C version, Take 2                          */
/* Evert Glebbeek 1998, 2002                  */
/* eglebbk@phys.uva.nl                        */
/**********************************************/
#include <allegro.h>
#include "dialog.h"
#include "global.h"
#include "gfx.h"
#include "map.h"
#include "maptile.h"
#include "unit.h"
#include "udraw.h"
#include "ugfx.h"
#include "ugrp.h"
#include "gamedraw.h"
#include "mapdraw.h"

/* Obtain the correct source graphic for a character (can still be flipped) */
/* This is either the graphic to be draw, or it's inverse */
#define GET_BASE_CGFX(chr) ((chr).draw_dir<5?(chr).draw_dir+(chr).current_frame:8-(chr).draw_dir+(chr).current_frame)

/* Linked list of units that need to be drawn */
static UNIT *registered_units = NULL;
static int num_reg_units = 0;

/*********************************/
/* Unit register/unregister code */
/*********************************/

/* QuickSort callback function: specifies the order of two characters */
/* Ensure the y coordinates are handled properly, as well as the layers */
/* Negative: e1 comes before e2 */
static int sort_charsbyy_cb(const void *e1, const void *e2)
{
   UNITDATA *c1 = (UNITDATA *)e1;
   UNITDATA *c2 = (UNITDATA *)e2;
   int flag;

   if (c1->layer < c2->layer)
      return -1;
   if (c2->layer < c1->layer)
      return 1;
   /* Same layer - sort by secundary criteria */
   /*  Objects that can be taken should appear below characters */
   if ((c1->flags & ALL_OBJECT) && (c2->flags & CC_ALL))
      return -1;
   if ((c2->flags & ALL_OBJECT) && (c1->flags & CC_ALL))
      return 1;
   flag = c1->y - c2->y;
   if (flag)
      return flag;
   if (is_structure(c1) && !is_structure(c2))
      return -1;
   if (is_structure(c2) && !is_structure(c1))
      return 1;
   return 0;
}


/* Add a character to the linked list, in such a way that the list is */
/*  sorted. Returns the new head of the list. */
/* Similar to MergeSort algorithm */
static UNIT *merge_list(UNIT *c, UNIT *list)
{
   if (!list) {
      /* If the list is NULL, then a new list is created */
      c->next = NULL;
      return c;
   } else {
      /* Check if we must insert the character here */
      if (sort_charsbyy_cb(c->data, list->data) <= 0) {
         /* Insert character at this node */
         c->prev = list->prev;
         c->next = list;
         if (c->prev)
            c->prev->next = c;
         if (c->next)
            c->next->prev = c;
         return c;
      } else {
         /* Insert character after this node */
         list->next = merge_list(c, list->next);
         if (list->next)
            list->next->prev = list;
         return list;
      }
   }
}

void u_add_to_reg(UNITDATA *udta)
{
   UNIT *c = registered_units;
   int n = 0;

   /* Make sure unit wasn't registered before */
   while (c && (n < num_reg_units)) {
      if (udta == c->data) {
         return;
      }
      c = c->next;
      n++;
   }
   c = NULL;
   c = alloc_unit();
   if (c) {
      //udta->flags |= CF_CHANGED;
      c->data = udta;
      c->prev = c->next = NULL;
      registered_units = merge_list(c, registered_units);
      num_reg_units++;
   } else {
      game_message("Failed to register unit");
   }
}

/* Remove unit from list of units that need to be redrawn */
void u_rmv_from_reg(UNITDATA *udta)
{
   UNIT *c = registered_units;

   if (udta)
      while (c) {
         if (udta == c->data) {
            /* Disable unit flags */
            udta->flags &= ~CF_CHANGED;
            if (c == registered_units)
               registered_units = c->next;
            if (c->next)
               c->next->prev = c->prev;
            if (c->prev)
               c->prev->next = c->next;
            free_unit(c);
            num_reg_units--;
            return;
         }
         c = c->next;
      }
}

/* Clear the registered unit list */
void clear_reg(void)
{
   UNIT *c = registered_units;

   while (c) {
      registered_units = c->next;
      free_unit(c);
      c = registered_units;
   }
   num_reg_units = 0;
   /* And now, registered_units is automatically NULL */
}

/* Register a list of units */
static int mark_units(UNIT *u)
{
   int c = 0;

   if (u) {
      c = mark_unit(u->data);
      c += mark_units(u->next);
   }
   return c;
}

/* Mark both the tile and all units it overlaps */
void register_tile(const int tx, const int ty)
{
   int ix, iy, sx, sy, px, py;
   UNIT *u;

   TRACE("Registering tile (%d, %d)\n", tx, ty);
   mark_mmtile(tx, ty);
   if (!in_rect(tx, ty, 0, 0, get_map_width() - 1, get_map_height() - 1)) {
      TRACE("Tile not on map - closing\n\n");
      return;
   }

   /* Only mark tiles that appear on-screen */
   get_viewport_tiles(&ix, &iy, &sx, &sy, &px, &py);
   if (!in_rect(tx, ty, ix - 1, iy - 1, sx + 1, sy + 1)) {
      TRACE("Tile not in viewport - closing\n\n");
      return;
   }

   /* Don't bother if already registered */
   if (tile_flags(tx, ty, 0) & TILE_CHANGED) {
      TRACE("Tile already registered\n\n");
      return;
   }

   mark_tile(tx, ty);
   TRACE("Registering units at (%d, %d)...\n", tx, ty);
   u = duplicate_list(get_ovr_char(tx, ty));
   mark_units(u);
   destroy_list(u);
   TRACE("Tile (%d, %d) registered.\n\n", tx, ty);
}

static void add_ovr_char(const int x, const int y, UNITDATA *udta)
{
   UNIT *u;
   UNIT *ulst;

   u = ulst = get_ovr_char(x, y);

   if (u) {
      while (u) {
         if (u->data == udta)
            break;
         /* Add unit to the end of the list */
         if (u->next == NULL) {
            u->next = alloc_unit();
            if (u->next) {
               u->next->prev = u;
               u->next->next = NULL;
               u->next->data = udta;
            } else {
               game_message("Failed to add unit overlap");
            }
            break;
         }
         u = u->next;
      }
   } else {
      ulst = alloc_unit();
      if (ulst) {
         ulst->prev = NULL;
         ulst->next = NULL;
         ulst->data = udta;
      } else {
         game_message("Failed to add unit overlap");
      }
   }

   set_ovr_char(x, y, ulst);
}

/* Mark a unit and all the tiles it overlaps */
/* Returns the number of tiles that were marked */
int mark_unit(UNITDATA *udta)
{
   int sg;
   BITMAP *frame;
   MAP_TILE_LIST *m;
   int c = 0;

   /* Effects should never be drawn */
   if (udta->flags & CC_EFFECT)
      return c;

   /* Break if already marked */
   if (udta->flags & CF_CHANGED) {
      TRACE("Unit %s [%d, %d] already marked\n", udta->name, udta->x,
            udta->y);
      return 0;
   }

   /* Break if there's no point in marking this unit because all of it's tiles */
   /*  are under the black shroud */
   for (m = udta->tiles; m; m = m->next) {
      if (!tile_hidden(m->tx, m->ty)) {
         udta->flags |= CF_CHANGED;
         break;
      }
   }

   if (!udta->flags & CF_CHANGED)
      return 0;


   sg = GET_BASE_CGFX(*udta);
   //popup_message ("%s [%d], %d", udta->name, udta->gfxidx, sg);
   frame = get_char_gfx(udta->gfxidx, sg);

   if (!frame) {
      popup_message("No frame!\n Dir: %d", udta->draw_dir);
      return c;
   }

   /* Do nothing further if already registered */
   if (udta->tiles) {
      TRACE("Unit %s [%d, %d] previously registered\n", udta->name, udta->x,
            udta->y);
      m = udta->tiles;
      while (m) {
         mark_tile(m->tx, m->ty);
         m = m->next;
         c++;
      }
      //return;
   }

   TRACE("Registered %s [%d, %d]\n", udta->name, udta->x, udta->y);

   return c;
}

/* Register a unit as overlapping tiles it covers */
void register_unit(UNITDATA *udta)
{
   int dx;
   int dy;
   int minx;
   int miny;
   int maxx;
   int maxy;
   int sg;
   BITMAP *frame;


   /* Effects should never be drawn */
   if (udta->flags & CC_EFFECT)
      return;

   sg = GET_BASE_CGFX(*udta);
   frame = get_char_gfx(udta->gfxidx, sg);

   if (!frame) {
      popup_message("No frame!\n Dir: %d", udta->draw_dir);
      return;
   }

   /* Add the char-data to the array of characters that need updating */
//   u_add_to_reg (udta);

   if (udta->draw_dir < 5) {
      dx = get_gfx_dx(udta->gfxidx, sg);
      dy = get_gfx_dy(udta->gfxidx, sg);
   } else {                     /* Flip unit graphic */
      dx = (frame->w - 3 * TILE_WIDTH / 2) / 2 - get_gfx_dx(udta->gfxidx, sg);
      //dx = (frame->w-TILE_WIDTH)/2-get_gfx_dx(udta->gfxidx, sg);
      dy = get_gfx_dy(udta->gfxidx, sg);
   }

   /* If the unit is selected, we may need to mark additional tiles */
   /*
      if (udta->flags&CF_SELECTED) {
      minx = udta->x - TILE_WIDTH/2;
      miny = udta->y - TILE_HEIGHT*1/3;
      maxx = minx + TILE_WIDTH;
      maxy = miny + TILE_HEIGHT;

      for (dx=minx; dx<maxx; dx++)
      for (dy=miny; dy<maxy; dy++)
      if (in_rect(dx,dy, 0,0,get_map_width()-1,get_map_height()-1))
      register_tile (dx, dy);
      }
    */

   /* Right now, this doesn't work entierly accurate, but it'll do for now */
   /* This is to ensure that the selection outline is removed properly */
   if (udta->flags & CF_SELECTED) {
      minx = udta->x - dx - TILE_WIDTH / 2;
      miny = udta->y - dy - TILE_HEIGHT / 3;
      maxx = minx + frame->w + TILE_WIDTH;
      maxy = miny + frame->h + TILE_HEIGHT;
   } else {
      minx = udta->x - dx;
      miny = udta->y - dy;
      maxx = minx + frame->w;
      maxy = miny + frame->h;
   }
   /* Now convert pixel coordinates to tile coordinates */
   minx /= TILE_WIDTH;
   miny /= TILE_HEIGHT;
   maxx = (maxx + TILE_WIDTH - 1) / TILE_WIDTH;
   maxy = (maxy + TILE_HEIGHT - 1) / TILE_HEIGHT;

   for (dx = minx; dx < maxx; dx++)
      for (dy = miny; dy < maxy; dy++)
         if (in_rect(dx, dy, 0, 0, get_map_width() - 1, get_map_height() - 1)) {
            MAP_TILE_LIST *m;
            UNIT *ulst;

            /* It could be that unit was already registered at this tile */
            /* Ok, so it shouldn't. Let's make sure. */
            if (!tile_in_list(udta->tiles, dx, dy)) {
               m = alloc_maptile();
               m->next = udta->tiles;
               m->tx = dx;
               m->ty = dy;
               udta->tiles = m;
               /* Add character to the list of characters registered to this tile */
               ulst = get_ovr_char(dx, dy);
               /* Make sure unit isn't in this list */
               /* (it shouldn't but better check) */
               if (!in_list(ulst, udta)) {
                  add_ovr_char(dx, dy, udta);
               } else {
                  //game_message ("Internal error:\nUnit multiple registered:\n%s at (%d, %d)", udta->name, dx, dy);
               }
            } else {
               //game_message ("Internal error:\nTile multiple registered");
            }
         }

   for (dx = minx; dx < maxx; dx++)
      for (dy = miny; dy < maxy; dy++)
         /* Mark tiles and the units that overlap them */
         mark_tile(dx, dy);

   for (dx = minx-1; dx < maxx; dx++)
      for (dy = miny-1; dy < maxy; dy++)
         /* Mark tiles and the units that overlap them */
         mark_mmtile(dx, dy);
   TRACE("Registered %s [%d, %d]\n", udta->name, udta->x, udta->y);
}

/* Remove a unit from the registration array and all it's tiles */
void unregister_unit(UNITDATA *udta)
{
   MAP_TILE_LIST *m;
   UNIT *ulst;

   u_rmv_from_reg(udta);

   m = udta->tiles;
   while (m) {
      /* First of all, remove this unit from the list of units overlapping */
      /*  this tile */
      ulst = get_ovr_char(m->tx, m->ty);
      ulst = remove_from_list(udta, ulst);
      if (in_list(ulst, udta)) {
         game_message
            ("Internal error:\nUnit not properly unregistered:\n%s at (%d, %d)",
             udta->name, m->tx, m->ty);
      }
      set_ovr_char(m->tx, m->ty, ulst);
      //set_ovr_char (m->tx, m->ty, NULL);

      register_tile(m->tx, m->ty);
      m = m->next;
      free_maptile(udta->tiles);
      udta->tiles = m;
   }
   /* udta->tiles is now automatically NULL */

}

/*************************/
/* Unit drawing code     */
/*************************/

/* Obtain the correct source graphic for a character (can still be flipped) */
/* This is either the graphic to be draw, or it's inverse */
#define GET_BASE_CGFX(chr) ((chr).draw_dir<5?(chr).draw_dir+(chr).current_frame:8-(chr).draw_dir+(chr).current_frame)

/* Draw unit sprite, gouraud shaded */
__attribute__ ((unused)) static void draw_gouraud_unit(UNITDATA *unit)
{
   int c1, c2, c3, c4;
   int dx, dy;
   int x, y;
   BITMAP *frame = NULL;
   int sg;

   /* Map objects, ie, active tiles, are not drawn here */
   if ((unit->flags & CC_MAPOBJECT) == CC_MAPOBJECT)
      return;

   if (unit->flags & CF_INVISIBLE)
      return;

   if (settings.flags & (IMMEDIATE + DIRTY_RECTANGLE))
      if (!(unit->flags & CF_CHANGED)) {  // No need to redraw character
         return;
      }

   unit->draw_dir %= 8;
   unit->direction %= 8;
   sg = GET_BASE_CGFX(*unit);

   select_palette(*get_char_palette(unit->gfxidx, unit->player));

   if (unit->draw_dir < 5) {
      frame = get_char_gfx(unit->gfxidx, sg);
      dx = get_gfx_dx(unit->gfxidx, sg);
      dy = get_gfx_dy(unit->gfxidx, sg);
   } else {                     /* Flip unit graphic */
      frame = get_char_gfx(unit->gfxidx, sg);
      frame = create_bitmap_ex(bitmap_color_depth(frame), frame->w, frame->h);
      clear_to_color(frame, bitmap_mask_color(frame));
      draw_sprite_h_flip(frame, get_char_gfx(unit->gfxidx, sg), 0, 0);
      dx = (frame->w - 3 * TILE_WIDTH / 2) / 2 - get_gfx_dx(unit->gfxidx, sg);
      dy = get_gfx_dy(unit->gfxidx, sg);
   }

   unit->gfx_dx = dx;
   unit->gfx_dy = dy;
   unit->gfx_w = frame->w;
   unit->gfx_h = frame->h;

   x = unit->x - dx;
   y = unit->y - dy;
   /*
      if (settings.flags & LIT_INLINE) {
      c1 = get_lum(x, y);
      c2 = get_lum(x+frame->w, y);
      c3 = get_lum(x+frame->w, y+frame->h);
      c4 = get_lum(x, y+frame->h);

      } else {
      c1 = c2 = c3 = c4 = 256;
      }
    */

   c1 = c2 = c3 = c4 = 255;
   x -= get_viewport_x();
   y -= get_viewport_y();

   if (c1 < 0)
      c1 = 0;
   if (c1 > 255)
      c1 = 255;
   if (c2 < 0)
      c2 = 0;
   if (c2 > 255)
      c2 = 255;
   if (c3 < 0)
      c3 = 0;
   if (c3 > 255)
      c3 = 255;
   if (c4 < 0)
      c4 = 0;
   if (c4 > 255)
      c4 = 255;

   /* Draw a circle below the unit if it is selected */
   /* Modify the marked tiles in the register_unit routine to mark any */
   /*  additional tiles this makes dirty */
   if (unit->flags & CF_SELECTED) {
#define DRAW_SELECT_ELLIPSE
#ifdef DRAW_SELECT_ELLIPSE
      ellipse(get_commandmap_bmp(),
              x + dx + TILE_WIDTH / 2, y + dy + TILE_HEIGHT * 2 / 3,
              TILE_WIDTH / 2 - 1, TILE_HEIGHT / 3, green);
#else
      rect(get_commandmap_bmp(),
           x + dx, y + dy, x + dx + TILE_WIDTH, y + dy + TILE_HEIGHT, green);
#endif
   }

   if (c1 == c2 && c2 == c3 && c3 == c4 && c4 == 255) {
      /* Now, the frame should still be converted to the proper colour scheme */
      /* The select_palette() will do this automatically for high and truecolour modes, but not for 8 bit colour */
      draw_sprite(get_commandmap_bmp(), frame,
                  unit->x - get_viewport_x() - dx,
                  unit->y - get_viewport_y() - dy);
   } else if ((c1 == c2) && (c2 == c3) && (c3 == c4) && (c4 == 0)) {
   } else {
      draw_gouraud_sprite(get_commandmap_bmp(), frame,
                          unit->x - get_viewport_x() - dx,
                          unit->y - get_viewport_y() - dy, c1, c2, c3, c4);
   }
   if (frame != get_char_gfx(unit->gfxidx, sg))
      destroy_bitmap(frame);

   /* Return to previous palette */
   unselect_palette();
}

/* Draw unit sprite, unshaded */
static void draw_plain_unit(UNITDATA *unit)
{
   int dx, dy;
   int x, y;
   BITMAP *ugfx;
   BITMAP *frame = NULL;
   int sg;

   /* Map objects, ie, active tiles, are not drawn here */
   if ((unit->flags & CC_MAPOBJECT) == CC_MAPOBJECT)
      return;

   if (unit->flags & CF_INVISIBLE) {
      popup_message("Hi!");
      return;
   }

   if (settings.flags & (IMMEDIATE + DIRTY_RECTANGLE))
      if (!(unit->flags & CF_CHANGED)) {  // No need to redraw character
         return;
      }

   unit->draw_dir %= 8;
   unit->direction %= 8;
   sg = GET_BASE_CGFX(*unit);

   ugfx = get_char_gfx(unit->gfxidx, sg);
   select_palette(*get_char_palette(unit->gfxidx, unit->player));

   if (unit->draw_dir < 5) {
      frame = ugfx;
      dx = get_gfx_dx(unit->gfxidx, sg);
      dy = get_gfx_dy(unit->gfxidx, sg);
   } else {                     /* Flip unit graphic */
      frame = create_bitmap_ex(bitmap_color_depth(ugfx), ugfx->w, ugfx->h);
      clear_to_color(frame, bitmap_mask_color(frame));
      draw_sprite_h_flip(frame, ugfx, 0, 0);
      dx = (frame->w - 3 * TILE_WIDTH / 2) / 2 - get_gfx_dx(unit->gfxidx, sg);
      dy = get_gfx_dy(unit->gfxidx, sg);
   }

   unit->gfx_dx = dx;
   unit->gfx_dy = dy;
   unit->gfx_w = frame->w;
   unit->gfx_h = frame->h;

   x = unit->x - dx;
   y = unit->y - dy;
   x -= get_viewport_x();
   y -= get_viewport_y();

   /* Draw a circle below the unit if it is selected */
   /* Modify the marked tiles in the register_unit routine to mark any */
   /*  additional tiles this makes dirty */
   if (unit->flags & CF_SELECTED) {
#define DRAW_SELECT_ELLIPSE
#ifdef DRAW_SELECT_ELLIPSE
      ellipse(get_commandmap_bmp(),
              x + dx + unit->gfx_x_size * TILE_WIDTH / 2,
              y + dy + unit->gfx_y_size * TILE_HEIGHT * 2 / 3 - 1,
              unit->gfx_x_size * TILE_WIDTH / 2 - 1,
              unit->gfx_y_size * TILE_HEIGHT / 3, green);
#else
      rect(get_commandmap_bmp(),
           x + dx, y + dy, x + dx + unit->gfx_x_size * TILE_WIDTH, y + dy + unit->gfx_y_size * TILE_HEIGHT, green);
#endif
   }

   /* Now, the frame should still be converted to the proper colour scheme */
   /* The select_palette() will do this automatically for high and truecolour modes, but not for 8 bit colour */
   draw_sprite(get_commandmap_bmp(), frame,
               unit->x - get_viewport_x() - dx,
               unit->y - get_viewport_y() - dy);

   if (frame != ugfx)
      destroy_bitmap(frame);

   /* Return to previous palette */
   unselect_palette();
}


static void char_check_touch(UNITDATA *c)
{
   MAP_TILE_LIST *tl = c->tiles;

   while (tl) {
      if ((tile_flags(tl->tx, tl->ty, 0) & TILE_CHANGED) &&
          (!tile_fog(tl->tx, tl->ty))) {
         //game_message ("Unit touched: %s", c->name);
         register_unit(c);
         return;
      }
      tl = tl->next;
   }
}

static __attribute__ ((unused))
     void chars_check_touch(UNIT *ulst)
{
   UNIT *u = ulst;

   while (u) {
      char_check_touch(u->data);
      u = u->next;
   }
}

/* Now, we mark all units that need updating, until no new updates are */
/*  found. Then, we add units to the list of units that must be redrawn. */
static void prepare_redraw_list(void)
{
   UNIT *ulst;
   int px = 0;
   int py = 0;
   int x;
   int y;
   int ix, iy;
   int sx, sy;
   int registered = 0;
   int last_registered = 0;

   /* Compute the first tile visible and get it's screen coordinates */
   get_viewport_tiles(&ix, &iy, &sx, &sy, &px, &py);

   do {
      last_registered = registered;
      registered = 0;
      for (y = iy; y < iy + sy; y++)
         for (x = ix; x < ix + sx; x++) {
            if (tile_marked(x, y))  //&&(!tile_fog(x, y)))
               registered += mark_units(get_ovr_char(x, y));
         }
   } while (registered != last_registered);

   /* Now add all units in the map viewport to the redraw-needed list */
   for (y = iy - 1; y < iy + sy; y++)
      for (x = ix - 1; x < ix + sx; x++)
         for (ulst = get_ovr_char(x, y); ulst; ulst = ulst->next) {
            /* Check if tile is marked `foggy', if so, don't draw the  */
            /*  unless it is a structure */
            if ((!tile_fog(x, y)) || (is_structure(ulst->data)))
               u_add_to_reg(ulst->data);
         }
}

void draw_all_units(void)
{
   UNIT *c;
   int n;

   prepare_redraw_list();

   for (n = 0; n < 9; n++) {
      draw_minimap_units(get_player_units(n));
   }


   c = registered_units;
   for (c = registered_units; c; c = c->next) {
      draw_plain_unit(c->data);
      u_rmv_from_reg(registered_units->data);
   }
   registered_units = NULL;
}
