/**********************************************/
/* Rise of the Tribes                         */
/* C version, Take 2                          */
/* Evert Glebbeek 1998, 2002                  */
/* eglebbk@phys.uva.nl                        */
/**********************************************/
#include <allegro.h>
#include <string.h>
#include <stdio.h>
#include "mapdraw.h"
#include "map.h"
#include "unit.h"
#include "unitmap.h"
#include "maptile.h"
#include "gamegfx.h"
#include "gamedraw.h"
#include "ugfx.h"
#include "global.h"
#include "gfx.h"

#define FOG_INTENS      160

static BITMAP *minimap = NULL;

/* Tile corner flags */
static const int tf_ul = FOG_UL + HIDDEN_UL;
static const int tf_ur = FOG_UR + HIDDEN_UR;
static const int tf_ll = FOG_LL + HIDDEN_LL;
static const int tf_lr = FOG_LR + HIDDEN_LR;

void (*draw_tile) (int tilex, int tiley, int scrx, int scry, 
                   int flag, int layer) = NULL;

BITMAP *get_minimap(void)
{
   return minimap;
}

/*
   Draws a tile struct at the specified screen coordinates. Behavior
   depends on flag:
    0 - Draw only the base tile in the linked list
    1 - Draw only 'under' tiles in the linked list
    2 - Draw only 'over' tiles in the linked list
    3 - Draw all tiles
   This routine pushes the gouraud shading onto a stack for later rendering.
*/
static void draw_plain_tile(int tilex, int tiley, int scrx, int scry,
                             int flag, int layer)
{
   int draw_this_tile = FALSE;
   int flags = tile_flags(tilex, tiley, layer);
   BITMAP *bmp = get_tile_bitmap (tile_type(tilex, tiley, layer));
   int c1, c2, c3, c4;

   /* Need to check if a tile in this layer actually exists! */

   /* Don't redraw tile if it's not needed */
   if (settings.flags & (IMMEDIATE + DOUBLE_BUFFER + DIRTY_RECTANGLE))
      if (!(flags & TILE_CHANGED))
         return;

   ASSERT(!(settings.flags & LIT_INLINE));
   ASSERT(!(settings.flags & FOG_OF_WAR_BITMAP));

   /* Get fog-of-war shading */
   if (flags & tf_ul) {
      if (flags & HIDDEN_UL) {
         c1 = 0;
      } else {
         c1 = FOG_INTENS;
      }
   } else {
      c1 = 255;
   }

   if (flags & tf_ur) {
      if (flags & HIDDEN_UR) {
         c2 = 0;
      } else {
         c2 = FOG_INTENS;
      }
   } else {
      c2 = 255;
   }

   if (flags & tf_lr) {
      if (flags & HIDDEN_LR) {
         c3 = 0;
      } else {
         c3 = FOG_INTENS;
      }
   } else {
      c3 = 255;
   }

   if (flags & tf_ll) {
      if (flags & HIDDEN_LL) {
         c4 = 0;
      } else {
         c4 = FOG_INTENS;
      }
   } else {
      c4 = 255;
   }

   if (c1 < 0)
      c1 = 0;
   if (c1 > 255)
      c1 = 255;
   if (c2 < 0)
      c2 = 0;
   if (c2 > 255)
      c2 = 255;
   if (c3 < 0)
      c3 = 0;
   if (c3 > 255)
      c3 = 255;
   if (c4 < 0)
      c4 = 0;
   if (c4 > 255)
      c4 = 255;
   
   draw_this_tile = FALSE;
   /* Check if we want to draw anything */
   switch (flag) {
      case TILE_UNDER:      // Draw only 'under' tiles
         if ((flags & TILE_UNDER) == TILE_UNDER)
            draw_this_tile = TRUE;
         break;
      case TILE_OVER:       // Draw only 'over' tiles
         if ((flags & TILE_OVER) == TILE_OVER)
            draw_this_tile = TRUE;
         break;
      case 0:
         if ((flags & TILE_BASE) == TILE_BASE)
            draw_this_tile = TRUE;
         break;
      case 3:               // Draw all tiles
         draw_this_tile = TRUE;
         break;
   }

   if (draw_this_tile) {
      if (((c1 == c2) && (c2 == c3) && (c3 == c4) && (c4 == 9))) {
         rectfill(get_commandmap_bmp(), scrx, scry, 
                  scrx + TILE_WIDTH, scry + TILE_HEIGHT, black);
      } else {
         /* Draw masked tiles only when not in ground layer and mask-bit is set */
         if (flag || (flags & TILE_MASK)) {
            draw_sprite(get_commandmap_bmp(), bmp, scrx, scry);
         } else {
            blit(bmp, get_commandmap_bmp(), 0, 0, scrx, scry,
                 TILE_WIDTH, TILE_HEIGHT);
         }
         /* Stack the coordinates of the map tile we just drew for */
         /* later rendering. Each map tile should only be marked for */
         /* rendering once, so let's do it when the `base' is drawn */
         if (!((c1 == c2) && (c2 == c3) && (c3 == c4) && (c4 == 255))) {
            if (flag == 0) {
               push_render_job(get_commandmap_bmp(), scrx, scry,
                               TILE_WIDTH, TILE_HEIGHT, c1, c2, c3, c4);
            }
         }
      }
   }
   if (!flag) {
      if (draw_this_tile) {
            int x = scrx + get_commandmap_x();
            int y = scry + get_commandmap_y();
            int w = TILE_WIDTH;
            int h = TILE_HEIGHT;
            
            /* Clip rectangle to commandmap viewport */
            if (x<get_commandmap_x()) {
               w -= get_commandmap_x() - x;
               x = get_commandmap_x();
            }
            
            if (y<get_commandmap_y()) {
               h -= get_commandmap_y() - y;

               y = get_commandmap_y();
            }
            
            if (x + w > get_commandmap_x() + get_commandmap_width()) {
               w = get_commandmap_x() + get_commandmap_width() - x;
            }
            
            if (y + h > get_commandmap_y() + get_commandmap_height()) {
               h = get_commandmap_y() + get_commandmap_height() - y;
            }
            
            if (w>0 && h>0)
               mark_rect(x, y, w, h);
         }
   }
}

/*
   Draws a tile struct at the specified screen coordinates. Behavior
   depends on flag:
    0 - Draw only the base tile in the linked list
    1 - Draw only 'under' tiles in the linked list
    2 - Draw only 'over' tiles in the linked list
    3 - Draw all tiles
   This routine assumes bitmapped fow and doesn't do gouraud shading at all
*/
static void draw_bmpfow_tile(int tilex, int tiley, int scrx, int scry,
                             int flag, int layer)
{
   int draw_this_tile = FALSE;
   int flags = tile_flags(tilex, tiley, layer);
   BITMAP *bmp = get_tile_bitmap (tile_type(tilex, tiley, layer));
   
   /* Need to check if a tile in this layer actually exists! */
   
   /* Don't redraw tile if it's not needed */
   if (settings.flags & (IMMEDIATE + DOUBLE_BUFFER + DIRTY_RECTANGLE))
      if (!(flags & TILE_CHANGED))
         return;

   ASSERT(settings.flags & FOG_OF_WAR_BITMAP);

   draw_this_tile = FALSE;
   /* Check if we want to draw anything */
   switch (flag) {
      case TILE_UNDER:      // Draw only 'under' tiles
         if ((flags & TILE_UNDER) == TILE_UNDER)
            draw_this_tile = TRUE;
         break;
      case TILE_OVER:       // Draw only 'over' tiles
         if ((flags & TILE_OVER) == TILE_OVER)
            draw_this_tile = TRUE;
         break;
      case 0:
         if ((flags & TILE_BASE) == TILE_BASE)
            draw_this_tile = TRUE;
         break;
      case 3:               // Draw all tiles
         draw_this_tile = TRUE;
         break;
   }

   if (draw_this_tile) {
      /* Draw masked tiles only when not in ground layer and mask-bit is set */
      if (flag || (flags & TILE_MASK)) {
         draw_sprite(get_commandmap_bmp(), bmp, scrx, scry);
      } else {
         blit(bmp, get_commandmap_bmp(), 0, 0, scrx, scry,
              TILE_WIDTH, TILE_HEIGHT);
      }
   }
   if (!flag) {
      if (draw_this_tile) {
            int x = scrx + get_commandmap_x();
            int y = scry + get_commandmap_y();
            int w = TILE_WIDTH;
            int h = TILE_HEIGHT;
            
            /* Clip rectangle to commandmap viewport */
            if (x<get_commandmap_x()) {
               w -= get_commandmap_x() - x;
               x = get_commandmap_x();
            }
            
            if (y<get_commandmap_y()) {
               h -= get_commandmap_y() - y;

               y = get_commandmap_y();
            }
            
            if (x + w > get_commandmap_x() + get_commandmap_width()) {
               w = get_commandmap_x() + get_commandmap_width() - x;
            }
            
            if (y + h > get_commandmap_y() + get_commandmap_height()) {
               h = get_commandmap_y() + get_commandmap_height() - y;
            }
            
            if (w>0 && h>0)
               mark_rect(x, y, w, h);
         }
   }
}

/* (re) initializes the entire map drawing code. */
/* Must also be called whenever map drawing settings change */
void map_draw_initialize(void)
{
   /* Create a bitmap used for drawing onto the minimap */
   if (!minimap) {
      minimap = create_bitmap(256, 256);
   }
   clear_bitmap(minimap);

   if (settings.flags & FOG_OF_WAR_BITMAP)
      draw_tile = draw_bmpfow_tile;
   else if (settings.flags & LIT_INLINE)
   //   draw_tile = draw_gouraud_tile;
      draw_tile = draw_plain_tile;
   else
      draw_tile = draw_plain_tile;
}

/*************************/
/* Map drawing code      */
/*************************/

/* Draws the map */
void draw_map(void)
{
   int px = 0;
   int py = 0;
   int x;
   int y;
   int ix, iy;
   int sx, sy;
   int layer;
   int num_layers = get_map_layers();
   int flags;
   int mark_mm = FALSE;

   /* Compute the first tile visible and get it's screen coordinates */
   get_viewport_tiles(&ix, &iy, &sx, &sy, &px, &py);

   /* For now, just draw the base layer of tiles */
   for (layer = 0; layer < num_layers-1; layer++)
      for (y = 0; y < sy; y++)
         for (x = 0; x < sx; x++) {
            draw_tile(ix + x, iy + y,
                      x * TILE_WIDTH + px, y * TILE_HEIGHT + py, 0, layer);
            draw_tile(ix + x, iy + y,
                      x * TILE_WIDTH + px, y * TILE_HEIGHT + py, TILE_UNDER,
                      layer);
         }
   /* Draw top layer */
   for (y = 0; y < sy; y++)
      for (x = 0; x < sx; x++) {
         draw_tile(ix + x, iy + y,
                   x * TILE_WIDTH + px, y * TILE_HEIGHT + py, 0,
                   num_layers-1);
         draw_tile(ix + x, iy + y,
                   x * TILE_WIDTH + px, y * TILE_HEIGHT + py, TILE_UNDER,
                   num_layers-1);
      }

   /* For debugging purposes */
   //clear_to_color(minimap, white);
   for (y = 0; y < get_map_height(); y++)
      for (x = 0; x < get_map_width(); x++) {
         flags = tile_flags(x,y,0);
         if (flags & TILE_MMCHANGED) {
            draw_minimap_tile(x, y, tile_type(x, y, 0));
            unmark_mmtile(x,y);
            mark_mm = TRUE;
         }
      }
   if (mark_mm)
      mark_interface_area(INT_MINIMAP);
}

/* Draw the bitmapped fog-of-war/obscuring */
void draw_bitmapped_fog_of_war(void)
{
   BITMAP *bmp = get_commandmap_bmp();
   RLE_SPRITE *fsprt = NULL;
   int px = 0;
   int py = 0;
   int x;
   int y;
   int ix, iy;
   int sx, sy;
   int flags;

   /* Compute the first tile visible and get it's screen coordinates */
   get_viewport_tiles(&ix, &iy, &sx, &sy, &px, &py);

   /* For now, just draw the base layer of tiles */
   for (y = 0; y < sy; y++)
      for (x = 0; x < sx; x++) {
         //if (in_rect (ix+x,iy+y, 0,0, map.width-1, map.height-1)) {
         if (tile_marked(ix + x, iy + y)) {
            flags = tile_hidden(ix + x, iy + y);
            if (flags) {
               fsprt = get_fog_sprite(flags);
               if (fsprt)
                  draw_rle_sprite(bmp, fsprt, x * TILE_WIDTH + px,
                                  y * TILE_HEIGHT + py);
            }
            if (flags != ALL_FOG) {
               flags = tile_fog(ix + x, iy + y);
               if (flags) {
                  fsprt = get_fog_sprite(flags);
                  if (fsprt)
                     draw_rle_sprite(bmp, fsprt, x * TILE_WIDTH + px,
                                     y * TILE_HEIGHT + py);
               }
            }
         }
         //}
      }
}

void draw_minimap_tile(const int tx, const int ty, const int tile_index)
{
   if (tile_hidden(tx, ty)) {
      putpixel(minimap, tx, ty, black);
   } else if (tile_fog(tx, ty)) {
      putpixel(minimap, tx, ty, get_tile_fog_colour(tile_index));
   } else {
      putpixel(minimap, tx, ty, get_tile_colour(tile_index));
   }
}

void draw_minimap_unit(UNITDATA *udta)
{
   int tx;
   int ty;
   int x, y;
   int draw = FALSE;

   tx = udta->x / TILE_WIDTH;
   ty = udta->y / TILE_WIDTH;

   for (x = tx; x < tx + udta->gfx_x_size; x++)
      for (y = ty; y < ty + udta->gfx_y_size; y++)
         if (!tile_hidden(x, y)) {
            if (!tile_fog(x, y)) {
               draw = TRUE;
               break;
            } else if (is_structure(udta)) {
               draw = TRUE;
               break;
            }
         }

   if (draw) {
      for (x = tx; x < tx + udta->gfx_x_size; x++)
         for (y = ty; y < ty + udta->gfx_y_size; y++) {
            putpixel(minimap, x, y,
                     palette_color[get_player_colour(udta->player, 0)]);
         }
   }
}

void draw_minimap_units(UNIT *ulst)
{
   if (ulst) {
      draw_minimap_unit(ulst->data);
      draw_minimap_units(ulst->next);
   }
}
