#ifndef UNIT_H
#define UNIT_H
#include "map.h"

/* mini program command identifiers */
#define PROG_BUILT         0
#define PROG_MOVE          1
#define PROG_ATTACK        2
#define PROG_TURN_NORTH    3
#define PROG_TURN_NE       4
#define PROG_TURN_EAST     5
#define PROG_TURN_SE       6
#define PROG_TURN_SOUTH    7
#define PROG_TURN_SW       8
#define PROG_TURN_WEST     9
#define PROG_TURN_NW       10
#define PROG_DIES          11
#define PROG_IDLE          12
//#define PROG_TOUCH         13
//#define PROG_UNTOUCH       14
//#define PROG_ATTACKED      15
#define NUM_PROGS          13

typedef struct TARGET {
   int x;
   int y;
   struct UNITDATA *unit;    /* Overrides the (x, y) target location */
} TARGET;

typedef struct SCRIPT_STACK {
   struct SCRIPT_STACK *next;
   struct SCRIPT_STACK *prev;
   unsigned char *prog;
   int argc;
   int argv[3];
   int progress;
   int flags;
} SCRIPT_STACK;

typedef struct SCRIPT_QUEUE {
   unsigned char *prog;
   int argv[3];              /* Arguments */
   int argc;                 /* Argument count for the script */
   int flags;
   BITMAP *icon;
   TARGET target;
   int ucmds;
} SCRIPT_QUEUE;

/* Rewrite this: in the unit structure, store a global identifier that */
/*  identifies the command in question and get the name and bitmap pointer */
/*  from a global array (saves memory) */
typedef struct UCMDS {
   char *name;               /* Tagline for what the command does */
   BITMAP *icon;             /* Command button icon */
   char *script;             /* Script associated with the command, or NULL */
   int uprog;                /* Unit program to call if prog is NULL */
   int flags;                /* Flags for the command */
   int argc;                 /* Argument count for the script */
   int argv[3];              /* Arguments */
} UCMDS;

typedef struct UNITDATA {
   /* Data that is identical for all units of a specific type */
   char *name;
   char *idkey;
   BITMAP *icon;
   int maxhp;
   int maxmp;

   /* Unit size */
   int gfx_x_size;
   int gfx_y_size;

   /* Resource costs (want special one for yielded resources?) */
   int resource_cost[8];
   
   unsigned char *prog[NUM_PROGS];
   UCMDS *unit_command[13];
   
   /* Individual specific data fields */
   unsigned int flags;
   int hp;
   int mp;
   int x;
   int y;
   int direction;
   int speed;
   int radius;
   int layer;
   int size;
   int player;

   /* Animation variables */
   int gfx_dx;
   int gfx_dy;
   int gfx_w;
   int gfx_h;
   int gfxidx;
   int current_frame;
   int draw_dir;

   MAP_TILE_LIST *path;
   int path_tries;

   /* Character control programs, for movement/attacking/idle/dying */
   SCRIPT_STACK *cprog;       /* Current stack of programs */
   int ss[8];                 /* Stack for use by the scripts */
   int ip;
   int sp;
   int wait_counter;
   /* Control queue, used for stacking commands */
   SCRIPT_QUEUE *cqueue;
   int queuestart;
   int queuesize;
   int queuecount;
   int stack_size;            /* number of programs on the stack */

   /* Character registered tiles (graphic overlaps these tiles) */
   MAP_TILE_LIST *tiles;

   /* Tiles marked `occupied' by this unit */
   MAP_TILE_LIST *occ_tiles;

   struct UNITDATA *parent;
   struct UNITDATA *script_target;       /* target of script commands */

   /* Units that are loaded into this one (bunkers, transports) */
   struct UNIT *loaded_units;

   /* Special: this is a pointer to the LAST character that */
   /*  touched/attacked a character */
   struct UNITDATA *attacker;
   
   /* Data shared between units in a specific group */   
   TARGET ai_target;                     /* target of AI actions */
} UNITDATA;

typedef struct UNIT {
   struct UNIT *next;
   struct UNIT *prev;
   UNITDATA *data;
} UNIT;

/* these are actually mini-programs not associated with a weapon or graphic */
extern UNIT *active;

/* Active tiles: tiles that have an associated mini-program built in */
extern UNIT *mapobj;

/* Basic unit types */
#define CC_STRUCTURE       1
#define CC_GUNIT           2
#define CC_COMMANDPOST     4
#define CC_WAREHOUSE       8
#define CC_BUILDSITE       16

/* Advanced character classes/combined classes */
/*  determines if a character has chosen magic or weapons as secundary goal */

#define CC_OBJECT          32

/* Object classes (same flags) */
#define CC_SPELL           64
#define CC_BOOK            128
#define CC_WEAPON          256
#define CC_SHIELD          512
#define CC_HELMET          1024
#define CC_BRACERS         2048
#define CC_JEWELRY         4096
#define CC_ARMOR           8192

#define ALL_OBJECT         (64+128+256+512+1024+2048+4096+8192)

/* Special character classes */
#define CC_PROJECTILE      16384
#define CC_EFFECT          32768
#define CC_MAP             65536
#define CC_PARTICLE        131072
/* Mixed classes. Note: OBJECTs can be picked up, MAPOBJECTs cannot */
#define CC_MAPOBJECT       (CC_OBJECT|CC_MAP)
#define CC_ALL             (1+2+4+8+16)
/* CC_MAP+CC_OBJECT is a map object; see the ideas.txt for description  */

/* Character flags */
#define CF_CHANGED   (65536*4)      /* Graphics/state has changed; redraw */
#define CF_LIGHTSOURCE (65536*8)    /* This character acts as a lightsource */
#define CF_UNIQUE    (65536*16)     /* This character is unique */
#define CF_INVISIBLE (65536*32)     /* This object is invisible */
#define CF_PLAYER    (65536*64)     /* Character is under player control */
#define CF_NPC       (65536*128)    /* Character is not under player control */
#define CF_EFFECT    (65536*256)    /* Character is an effect */
#define CF_SELECTED  (65536*512)    /* Unit is selected */
#define CF_SEL_MUL   (65536*1024)   /* Unit can be selected with others */
#define CF_NOSELECT  (65536*2048)   /* Unit can't be selected at all */
#define CF_NOBLOCK   (65536*4096)   /* Unit doesn't block the tile it occupies */
#define CF_NOORDERS  (65536*8192)   /* Unit can't accept commands */
#define CF_PROGRESS  (65536*16384)  /* Show progress indicator */

/* Player flags */
#define PLAYER_LOCAL  1    /* Player is local, ie, on this machine */
#define PLAYER_REMOTE 2    /* Player is remote, ie, on an other machine */
#define PLAYER_COMPUTER 4  /* Player is computer controlled */
#define PLAYER_PASSIVE 8   /* Player is passive */
#define PLAYER_HUMAN 16    /* Player is human controlled */

/* Unit command-button flags */
#define UCMD_ACTIVE       1024  /* Unit is performing this action */
#define UCMD_EXCLUSIVE    2048  /* Command specific for leader in a group */
#define UCMD_PROGRESS     4096  /* Display progress indicator */
#define UCMD_TARGET       8192  /* Request map target for command */
#define UCMD_PLACE       16384  /* Request map area for a structure */
#define UCMD_CLR_TARGET  32768  /* Clear unit/group target */
#define UCMD_CLR_QUEUE   65536  /* Clear unit/group command queue */
#define UCMD_CANCEL   (2*65536) /* Raise the cancel flag */
#define UCMD_ALL (UCMD_ACTIVE|UCMD_EXCLUSIVE|UCMD_PROGRESS|UCMD_TARGET|UCMD_CLR_TARGET|UCMD_CLR_QUEUE|UCMD_CANCEL|UCMD_OTHERTARGET)

extern UNITDATA *alloc_unitdata(void);
extern void free_unitdata(UNITDATA *c);
extern UNIT *alloc_unit(void);
extern void free_unit(UNIT *c);
extern int get_unit_count(void);
extern int get_udta_count(void);

extern void push_char_def (const int id, const char *key);
extern UNIT *get_char_def (const char *key);
extern int get_char_dummy (const char *key);
extern void set_char_dummy(const char *key, int c);
extern void destroy_char_def (void);

extern void load_unit_data (PACKFILE *f);
extern void load_cmdb_data (PACKFILE *f);

extern UNITDATA *find_char_name(UNIT *clst, char *charname);
extern UNITDATA *find_char_flags(UNIT *clst, int flags);
extern unsigned int key_to_id (char *char_name);
extern unsigned int name_to_id (char *char_name);

extern UNIT *create_unit (unsigned int n);
extern void change_unit(UNITDATA *udta, const int new_type);
extern void set_unit_coors(UNITDATA *u, const int x, const int y);

extern inline BITMAP *get_char_icon(unsigned int n);
extern inline char *get_char_name(unsigned int n);
extern inline int get_char_maxhp(unsigned int n);
extern inline int get_char_maxmp(unsigned int n);
extern inline int get_char_xsize(unsigned int n);
extern inline int get_char_ysize(unsigned int n);
extern inline int get_char_resource_cost(int c, int r);

extern void set_unit_target(UNITDATA *u, const TARGET target);
extern int targets_are_equal(const TARGET t1, const TARGET t2);

extern UNIT *duplicate_unit (UNIT *c);
extern UNIT *clone_unit (UNIT *c);

extern void ini_chars(void);
extern void destroy_all_chars(void);

extern int is_mapobject(UNITDATA *obj);
extern int is_structure(UNITDATA *obj);

extern int can_be_selected(UNITDATA *u);

extern int in_list(const UNIT *list, const UNITDATA *chr);
extern UNIT *duplicate_list(UNIT *list);
extern void destroy_list(UNIT *list);

extern int count_characters(UNIT *ulst);
extern int count_structures(UNIT *ulst);

extern UNIT *add_to_list(UNITDATA *udta, UNIT *ulst);
extern UNIT *remove_from_list(UNITDATA *udta, UNIT *ulst);

extern void load_unit(UNITDATA *unit, UNITDATA *container);
extern void unload_unit(UNITDATA *unit, UNITDATA *container);
extern void unload_all_units(UNITDATA *container);

extern void release_unit (UNIT *c);

extern void remove_mapobj(UNIT *c);
extern void activate(UNIT *c);
extern void deactivate(UNIT *c);

#endif
