#include <allegro.h>
#include "ticTacToe.h"

void clearGridCell (int row, int col); //Clears a game grid cell, including the selector
void clearIconSelectCell (int row, int col); //Clears a cell from the icon select screen
void displayDemonischLogo ();
void displayFeedback ();
void displayPlayers (); //Displays the player icons
void displayScore (); //Display the current scores
void drawBackground ();
void drawGridCell (int row, int col);
void drawGridPiece (int row, int col); //Draws a piece on the game grid, or nothing if the cell is empty
void drawIconSelectPiece (int row, int col); //Draws an icon on the icon select screen
void drawPiece (int spriteX, int spriteY, int screenX, int screenY); //Draws the icon at the exact position on the screen
void drawScreen (); //Double buffer finally is draw to the screen
void drawSelect (); //Draws the selector on the grid at its current location
void gameLoop (); //The actual game loop
bool hasWon (int player); //Checks if this player has 3 matching pieces anywhere in the grid
void iconSelect (); //The icon select screen
bool moveD (int pressedKey, int numRows, int numCols); //Checks if a movement key was pressed, and if the selector can move
void moveSelect (int newRow, int newCol); //Clears the logical cell the selector is in, draws the piece, moves the selector and then draws the selector
void resetGrid (); //Clears the grid
void switchPlayer (); //Inverts the current player

void clearGridCell (int row, int col)
{
	
	int x = GRID_START_X + (col * (GRID_SPACE + CELL_SIZE));
	int y = GRID_START_Y + (row * (GRID_SPACE + CELL_SIZE));
	
	rectfill(imgBuffer, x, y, x + CELL_SIZE - 1, y + CELL_SIZE - 1, COLOUR_BACKGROUND);

}

void clearIconSelectCell (int row, int col)
{
	
	int x = ICON_SELECT_START_X + (col * (ICON_SELECT_SPACE + CELL_SIZE));
	int y = ICON_SELECT_START_Y + (row * (ICON_SELECT_SPACE + CELL_SIZE));
	
	rectfill(imgBuffer, x - 1, y - 1, x + CELL_SIZE, y + CELL_SIZE + ICON_SELECT_SPACE - 5, COLOUR_BACKGROUND);
	
}	

void displayDemonischLogo ()
{
	
	BITMAP* imgDemonisch;
	
	imgDemonisch = load_bitmap("imgDemonisch.bmp", NULL);
	
	if (imgDemonisch)
	{
		
		rectfill(imgBuffer, 0, 0, WIDTH, HEIGHT, makecol(0, 0, 0));
		draw_sprite(imgBuffer, imgDemonisch, (WIDTH / 2) - (imgDemonisch->w / 2), (HEIGHT / 2) - (imgDemonisch->h / 2));
		drawScreen();
		
		rest(2000);
		
	}
	
	destroy_bitmap(imgDemonisch);
	
}	

void displayFeedback ()
{	
	
	rectfill(imgBuffer, FEEDBACK_START_X, FEEDBACK_START_Y, SCORE_START_X, FEEDBACK_START_Y + text_height(font), COLOUR_BACKGROUND);
	textprintf_ex(imgBuffer, font, FEEDBACK_START_X, FEEDBACK_START_Y, COLOUR_FEEDBACK, -1, "Player %u turn", currentPlayer + 1);
	
}

void displayPlayers ()
{
	
	unsigned short textMiddle = ((CELL_SIZE / 2) - (text_height(font) / 2));
	unsigned short iconSpace = 20;
	
	drawPiece(players[PLAYER_1].pieceCol * CELL_SIZE, players[PLAYER_1].pieceRow * CELL_SIZE, PLAYER_ICONS_START_X, PLAYER_ICONS_START_Y);
	textout_ex(imgBuffer, font, "Player 1", PLAYER_ICONS_START_X + CELL_SIZE + 10, PLAYER_ICONS_START_Y + textMiddle, COLOUR_PLAYER_ICONS, -1);
	drawPiece(players[PLAYER_2].pieceCol * CELL_SIZE, players[PLAYER_2].pieceRow * CELL_SIZE, PLAYER_ICONS_START_X, PLAYER_ICONS_START_Y + CELL_SIZE + iconSpace);
	textout_ex(imgBuffer, font, "Player 2", PLAYER_ICONS_START_X + CELL_SIZE + 10, PLAYER_ICONS_START_Y + CELL_SIZE + iconSpace + textMiddle, COLOUR_PLAYER_ICONS, -1);
	
}	

void displayScore ()
{
	
	int heightSpace = text_height(font);
	
	rectfill(imgBuffer, SCORE_START_X, SCORE_START_Y, WIDTH, HEIGHT, COLOUR_BACKGROUND);
	textprintf_ex(imgBuffer, font, SCORE_START_X, SCORE_START_Y, COLOUR_SCORES, -1, "Player 1: %02u", players[PLAYER_1].score);
	textprintf_ex(imgBuffer, font, SCORE_START_X, SCORE_START_Y + heightSpace, COLOUR_SCORES, -1, "Player 2: %02u", players[PLAYER_2].score);
	textprintf_ex(imgBuffer, font, SCORE_START_X, SCORE_START_Y + (2 * heightSpace), COLOUR_SCORES, -1, "Draws: %02u", numDraws);
	
}

void drawBackground ()
{
	rectfill(imgBuffer, 0, 0, WIDTH, HEIGHT, COLOUR_BACKGROUND);
}

void drawGridCell (int row, int col)
{
	
	int x = GRID_START_X + (col * (GRID_SPACE + CELL_SIZE));
	int y = GRID_START_Y + (row * (GRID_SPACE + CELL_SIZE));
	
	rect(imgBuffer, x - 1, y - 1, x + CELL_SIZE, y + CELL_SIZE, COLOUR_GRID_CELL);
	
}

void drawGridPiece (int row, int col)
{
	
	if (grid[row][col] != EMPTY)
	{
		
		int x = GRID_START_X + (col * (GRID_SPACE + CELL_SIZE));
		int y = GRID_START_Y + (row * (GRID_SPACE + CELL_SIZE));

		drawPiece(players[grid[row][col]].pieceCol * CELL_SIZE, players[grid[row][col]].pieceRow * CELL_SIZE, x, y);

	}
	
}

void drawIconSelectPiece (int row, int col)
{
	
	int x = ICON_SELECT_START_X + (col * (ICON_SELECT_SPACE + CELL_SIZE));
	int y = ICON_SELECT_START_Y + (row * (ICON_SELECT_SPACE + CELL_SIZE));
	
	rect(imgBuffer, x - 1, y - 1, x + CELL_SIZE, y + CELL_SIZE, COLOUR_ICON_SELECT_CELL);
	
	for (int i = 0; i < 2; i++)
	{
		//Don't show the icon if it has been selected
		if (players[i].pieceSet && players[i].pieceRow == row && players[i].pieceCol == col)
		{
			return;
		}
		
	}
	
	drawPiece(col * CELL_SIZE, row * CELL_SIZE, x, y);
	
}

void drawPiece (int spriteX, int spriteY, int screenX, int screenY)
{//Masked blit is for sprites with transparent spots
	masked_blit(imgSprites, imgBuffer, spriteX, spriteY, screenX, screenY, CELL_SIZE, CELL_SIZE);
}	

void drawScreen ()
{	
	
	acquire_screen(); //Waits for vsync
    draw_sprite(screen, imgBuffer, 0, 0);  //Draws the entire double buffer onto the screen
    release_screen();  //Gives back control of the screen  
    
}

void drawSelect ()
{
	
	int screenX, screenY;
	
	if (iconSelectMode)
	{
		
		screenX = ICON_SELECT_START_X + (selector.col * (ICON_SELECT_SPACE + CELL_SIZE));
		screenY = ICON_SELECT_START_Y + (selector.row * (ICON_SELECT_SPACE + CELL_SIZE));
		
		textprintf_ex(imgBuffer, font, screenX, screenY + CELL_SIZE + 5, COLOUR_PLAYER_ICON_SELECT, -1, "P%u", (currentPlayer + 1));
		
	}
	else
	{
		
		screenX = GRID_START_X + (selector.col * (GRID_SPACE + CELL_SIZE));
		screenY = GRID_START_Y + (selector.row * (GRID_SPACE + CELL_SIZE));
		
	}
		
	draw_trans_sprite(imgBuffer, imgSelector, screenX, screenY);

}

void gameLoop ()
{
    
    clear_keybuf(); //Remove any pending keys 
    rest(100); //If this was not present, the user would be able to move across the screen in the flash of an eye
    //A tap of a key can last for 30ms (varies)
    int pressedKey = readkey(); //Waits for ever until a key is pressed
    //The >> 8 gets the lower 8 bits, which is the scan code
    if (pressedKey >> 8 == KEY_ESC)
    {
	    
	    gameActive = false;
	    return;
	    
    }
    else if (pressedKey >> 8 == KEY_ENTER && grid[selector.row][selector.col] == EMPTY)
    {	    
	    
	    grid[selector.row][selector.col] = currentPlayer;
	    numMoves++;
	    
	    clearGridCell(selector.row, selector.col); //Clears the selector
		drawGridPiece(selector.row, selector.col); //Draws the piece
	    
	    if (hasWon(currentPlayer)) //Returns true if the player has 3 symbols in a row
	    {
		    
		    players[currentPlayer].score++;
		    displayScore();
			drawScreen(); //Draw the screen showing the winning move
	
			rest(750); //Let them see it for a while

		    resetGrid();
		    drawSelect();
		    switchPlayer(); //The loser goes first next found
		    roundStartPlayer = currentPlayer;
		    
	    }
	    else if (numMoves >= 9) //The grid is full and no one has won
	    {
		    
		    numDraws++;
		    displayScore();
			drawScreen();
	
			rest(500); 

		    resetGrid();		    
		    roundStartPlayer = !roundStartPlayer; //Alternate the player who started first the previous round
		    currentPlayer = roundStartPlayer;
		    
		    
	    }
	    else //Another move is possible
	    {
		    		    
		    drawSelect(); //Draw the selector over the just placed piece   
		    switchPlayer();
		    
	    }
	    
	    displayFeedback();
	    drawScreen();
	    	    
    }
    else
    {
	    moveD(pressedKey >> 8, 3, 3); //Check the direction buttons (key, num rows, num cols)
    }
    
}

bool hasWon (int player)
{
	
	if (numMoves < 3) //No point checking if less then 3 symbols have been placed
	{
		return false;
	}
	//Check both digs (both digs use the middle cell)
	if (grid[1][1] == player && ((grid[0][0] == player && grid[2][2] == player) || (grid[0][2] == player && grid[2][0] == player)))
	{
		return true;
	}
	//Check the columns
	for (int col = 0; col < 3; col++)
	{
		
		if (grid[0][col] == player && grid[1][col] == player && grid[2][col] == player)
		{
			return true;
		}
		
	}
	//Check the rows
	for (int row = 0; row < 3; row++)
	{
		
		if (grid[row][0] == player && grid[row][1] == player && grid[row][2] == player)
		{
			return true;
		}
		
	}
	
	return false;
	
}

void iconSelect ()
{
	
	for (int row = 0; row < 2; row++)
    {

	    for (int col = 0; col < 4; col++)
	    {
		    drawIconSelectPiece(row, col); //Draw all the available icons			    
	    }
	    
    }
    
    selector.row = 0; //Move the selector to the logical north west corner
    selector.col = 0;	    
    drawSelect(); //Show the selector on top of the first icon
    
    textout_centre_ex(imgBuffer, font, "Select player icons", WIDTH / 2, 10, COLOUR_ICON_SELECT_TITLE, -1);

    drawScreen(); //Draw everything
    
    players[PLAYER_1].pieceSet = false;
    players[PLAYER_2].pieceSet = false;
	
	while (1)
	{
	
		clear_keybuf();
	    rest(100);
	    
	    int pressedKey = readkey();
	    //Enter key pressed and the icon isn't taken by the other player
	    if (pressedKey >> 8 == KEY_ENTER && (!players[!currentPlayer].pieceSet || !(players[!currentPlayer].pieceRow == selector.row && players[!currentPlayer].pieceCol == selector.col)))
	    {
		    
		    players[currentPlayer].pieceRow = selector.row;
		    players[currentPlayer].pieceCol = selector.col;
		    players[currentPlayer].pieceSet = true;  	    
		    
		    switchPlayer();
		    
		    if (players[PLAYER_1].pieceSet && players[PLAYER_2].pieceSet)
		    {
			    break;
		    }
		    
			clearIconSelectCell(selector.row, selector.col);
	    	drawIconSelectPiece(selector.row, selector.col);
		    drawSelect();
		    drawScreen();		    
		    
	    }
	    else if (pressedKey >> 8 == KEY_ESC)
	    {
		    
		    gameActive = false;	    
		    return;
		    
	    }
	    else
	    {
		    moveD(pressedKey >> 8, 2, 4);
	    }
	    
    }
		    
}

bool moveD (int pressedKey, int numRows, int numCols)
{
	
	if (pressedKey == KEY_UP && selector.row > 0)
    {
	    moveSelect(selector.row - 1, selector.col);
    }
    else if (pressedKey == KEY_RIGHT && selector.col < (numCols - 1))
    {
	    moveSelect(selector.row, selector.col + 1);
    }
    else if (pressedKey == KEY_DOWN && selector.row < (numRows - 1))
    {
	    moveSelect(selector.row + 1, selector.col);
    }
    else if (pressedKey == KEY_LEFT && selector.col > 0)
    {
	    moveSelect(selector.row, selector.col - 1);
    }
    else
    {
	    return false;
    }
    
    return true;
    
}	

void moveSelect (int newRow, int newCol)
{
	
	if (iconSelectMode)
	{
	
		clearIconSelectCell(selector.row, selector.col);
		drawIconSelectPiece(selector.row, selector.col);
		
	}
	else
	{
		
		clearGridCell(selector.row, selector.col); //Clear the selector
		drawGridPiece(selector.row, selector.col); //Draw a piece if one was present (does nothing if the cell is empty)
		
	}
	
	selector.row = newRow; //Move the selector
	selector.col = newCol;
	
	drawSelect(); //Display the selector
	drawScreen();
	
}

void resetGrid ()
{	
	//Clear the grid
	for (int row = 0; row < 3; row++)
	{
		
		for (int col = 0; col < 3; col++)
		{
			
			grid[row][col] = EMPTY;
			clearGridCell(row, col);			
			
		}
		
	}
	
	numMoves = 0;
	
}

void switchPlayer ()
{
	currentPlayer = !currentPlayer; //Simple invert, 1 > 0, 0 > 1
}

int main ()
{
 
    allegro_init();
    install_keyboard();
    install_timer();
    
    set_color_depth(16);
    
    if (set_gfx_mode(GFX_AUTODETECT_WINDOWED, WIDTH, HEIGHT, 0, 0) < 0)
    {
	    
		allegro_message("Couldn't set gfx mode: %s\n", allegro_error);
		exit(1);
		
    }    
  
    imgSprites = load_bitmap("imgTicTacToe.bmp", NULL);
    
    if (!imgSprites) //Could not load the sprite sheet
    {
	    allegro_message("Could not load image \"imgTicTacToe.bmp\"");
    }
    else
    {
    
	    imgBuffer = create_bitmap(WIDTH, HEIGHT); //The double buffer
	    imgSelector = create_bitmap(32, 32);
	    
	    displayDemonischLogo();	    
	    drawBackground();
	    
	    rectfill(imgSelector, 0, 0, 31, 31, COLOUR_SELECTOR); //Make the selector white	    
	    set_trans_blender(255, 255, 255, 100); //The selector will be transparent
	    
	    iconSelect();

	    if (gameActive) //User might press escape during icon select screen
	    {	  
		      
		    iconSelectMode = false; //We are now into game mode
		    
		    drawBackground(); //Clear the screen ready for the game
		    	    
		    displayPlayers(); //Show the player icons
		    
		    for (int row = 0; row < 3; row++)
		    {
			    
			    for (int col = 0; col < 3; col++)
			    {
				    
				    drawGridCell(row, col); //Draw the grid cells
				    grid[row][col] = EMPTY;
				    
			    }
			    
		    }		    
		    
		    selector.row = 0; //Put the selector to the logical north west corner
		    selector.col = 0;
		    
		    drawSelect(); //Draw the selector
		    displayScore();
		    displayFeedback();
		    drawScreen(); //Finally draw everything
		    
		    while (gameActive) //The game begins
		    {
		        gameLoop();        
		    }
		    
	    }
	    
    }
    //Always clean up after yourself to prevent memory leaks
    destroy_bitmap(imgSprites);
    destroy_bitmap(imgBuffer);
    destroy_bitmap(imgSelector);
    
    allegro_exit();
    
    return 0;
    
}   

END_OF_MAIN();
