//  This file handles the loading of 3d objects
//
//  It reads 3D Studio Max ascii scenes files
//
//  It finds and loads one object from the scene file.
//  It ignores all color and texture mapping infomation and just
//  reads the vertexs and the faces.
//
//  If I had more time it would load all the objects from each
//  scene and merge them.
//

#include <allegro.h>
#include <string.h>

#include "obj3d.h"
#include "util.h"

enum symbol_type {CLOSE_STRUCT,OPEN_OBJECT,OPEN_MESH,OPEN_VERTEX_LIST,
OPEN_POLY_LIST,OPEN_OBJECT_INFO,OPEN_UNKNOWN_STRUCT,SYMBOL_NAME,
SYMBOL_NUMVERTEX,SYMBOL_NUMPOLY,SYMBOL_VERTEX,SYMBOL_POLY,
SYMBOL_UNKNOWN};

enum symbol_state {NONE,OBJECT,MESH,VLIST,PLIST,OINFO,USTRUCT};


#define MAX_STATES 20
class sstack 
{
  private:
    symbol_state states[MAX_STATES];
    int cur_state;
  public:
    sstack() {cur_state=0;}
    void push(symbol_state state) 
    {
      states[cur_state++]=state;
    }
    symbol_state pop()
    {
      if(cur_state<=0) return NONE;
      return states[--cur_state];
    }
};

symbol_type decode_line(char *line)
{
  char fc,ec,*fw;
  fc=get_first_char(line);
  ec=get_last_char(line);
  fw=get_word(line,0);

/*  clear(screen);
  int c=makecol(255,255,255);
  textprintf(screen,font,0,0,c,"Line: %s",line);
  textprintf(screen,font,0,8,c,"Fc: %c",fc);
  textprintf(screen,font,0,16,c,"Ec: %c",ec);
  textprintf(screen,font,0,24,c,"Fw: %s",fw);
  readkey();
*/
  switch(fc) {
    case '}':
      return CLOSE_STRUCT;
    case '*':
      if(ec=='{') {
        if(strcmp(fw,"*GEOMOBJECT")==0) 
          return OPEN_OBJECT;
        if(strcmp(fw,"*MESH")==0) 
          return OPEN_MESH;
        if(strcmp(fw,"*MESH_VERTEX_LIST")==0) 
          return OPEN_VERTEX_LIST;
        if(strcmp(fw,"*MESH_FACE_LIST")==0) 
          return OPEN_POLY_LIST;
        if(strcmp(fw,"*NODE_TM")==0) 
          return OPEN_OBJECT_INFO;
        return OPEN_UNKNOWN_STRUCT;
      }
      if(strcmp(fw,"*NODE_NAME")==0) 
        return SYMBOL_NAME;
      if(strcmp(fw,"*MESH_NUMVERTEX")==0) 
        return SYMBOL_NUMVERTEX;
      if(strcmp(fw,"*MESH_NUMFACES")==0) 
        return SYMBOL_NUMPOLY;
      if(strcmp(fw,"*MESH_VERTEX")==0) 
        return SYMBOL_VERTEX;
      if(strcmp(fw,"*MESH_FACE")==0) 
        return SYMBOL_POLY;
    default: 
      return SYMBOL_UNKNOWN;
  }
}

void make_normal(V3D_f *n,V3D_f *a,V3D_f *b, V3D_f *c)
{
  static V3D_f v1,v2;
  
  v1.x = (b->x - a->x);
  v1.y = (b->y - a->y);
  v1.z = (b->z - a->z);
  v2.x = (c->x - a->x);
  v2.y = (c->y - a->y);
  v2.z = (c->z - a->z);

  cross_product_f(v1.x, v1.y, v1.z, v2.x, v2.y, v2.z, &n->x, &n->y, &n->z);

  normalize_vector_f(&n->x, &n->y, &n->z);
  
  
}

OBJ3D *load_3dp(PACKFILE *file,int length)  // ignore length
{
  if(!file) return NULL;

  sstack s;

  symbol_state current=NONE;

  symbol_type next;

  char *name=0;
  char *tmp=0;

  int num_poly=0;
  int num_vertex=0;
  int i;
//  int r,g,b;
  int vnum,pnum;
  int aa,bb,cc;
  
  V3D_f *vertex_list=NULL,*ovl=NULL;
  V3D_f *vtmp=NULL,*ovt=NULL;
  V3D_f *vnormal=NULL,*ovn=NULL;
  int *vtol=NULL,*ovtl=NULL;
  POLY3D *poly_list=NULL,*opl=NULL;
  OBJ3D *obj=NULL;
  int vp=0,pp=0,ov=0,op=0;

  static char line[1024];
  while(!pack_feof(file)) {
    pack_fgets(line,1023,file);
    next = decode_line(line);

    switch(current) {
      case NONE:
        switch(next) {
          case OPEN_MESH:
            s.push(current);
            current=MESH;
            break;
          case OPEN_UNKNOWN_STRUCT:
            s.push(current);
            current=USTRUCT;
            break;
          default: break; // ignore everything else
        }
        break;      
      case OBJECT:
        switch(next) {
          case SYMBOL_NAME:
            tmp=get_word(line,1);
            tmp++;
            tmp[strlen(tmp)-1]=0;
            name=new char[strlen(tmp)+1];
            strcpy(name,tmp);
            break;
          case OPEN_MESH:
            s.push(current);
            current=MESH;
            break;
          case OPEN_OBJECT_INFO:
            s.push(current);
            current=OINFO;
            break;
          case CLOSE_STRUCT:
            current=s.pop();
            break;
          case OPEN_UNKNOWN_STRUCT:
            s.push(current);
            current=USTRUCT;
            break;
          default: break;
        }
        break;
      case USTRUCT:
        switch(next) {
          case OPEN_UNKNOWN_STRUCT:
            s.push(current);
            current=USTRUCT;
            break;
          case CLOSE_STRUCT:
            current=s.pop();
            break;
          default: break;
        }
        break;
      case OINFO:
        switch(next) {
          case OPEN_UNKNOWN_STRUCT:
            s.push(current);
            current=USTRUCT;
            break;
          case CLOSE_STRUCT:
            current=s.pop();
            break;
          default: break;
        }
        break;
      case MESH:
        switch(next) {
          case SYMBOL_NUMVERTEX:
            if(vertex_list!=NULL) {
              ov=num_vertex;
              vp+=ov;
              ovl=vertex_list;
              ovt=vtmp;
              ovn=vnormal;
              ovtl=vtol;
            }
            num_vertex=get_int(line,1);
            vertex_list = new V3D_f[num_vertex];
            vnormal = new V3D_f[num_vertex];
            memset(vnormal,0,sizeof(V3D_f)*num_vertex);
            vtmp = new V3D_f[num_vertex];
            vtol = new int[num_vertex];
            memset(vtol,0,sizeof(int)*num_vertex);
            if(ovl!=NULL) {
              memcpy(ovl,vertex_list,sizeof(V3D_f)*ov);
              memcpy(ovn,vnormal,sizeof(V3D_f)*ov);
              memcpy(ovt,vtmp,sizeof(V3D_f)*ov);
              memcpy(ovtl,vtol,sizeof(int)*ov);
              delete []ovl;
              delete []ovn;
              delete []ovt;
              delete []ovtl;
              ovl=NULL;
            }
            break;
          case SYMBOL_NUMPOLY:
            if(poly_list!=NULL) {
              op=num_poly;
              opl=poly_list;
              pp+=op;
            }
            num_poly=get_int(line,1);
            poly_list=new POLY3D[num_poly];    
            if(opl!=NULL) {
              memcpy(opl,poly_list,sizeof(POLY3D)*op);
              delete []opl;
              opl=NULL;
            }
            break;
          case OPEN_VERTEX_LIST:
            s.push(current);
            current=VLIST;
            break;
          case OPEN_POLY_LIST:
            s.push(current);
            current=PLIST;
            break;
          case OPEN_UNKNOWN_STRUCT:
            s.push(current);
            current=USTRUCT;
            break;
          case CLOSE_STRUCT:
            current=s.pop();
            break;
          default: break;
        }
        break;
      case VLIST:
        switch(next) {
          case SYMBOL_VERTEX:
            if(!vertex_list) break;
            vnum=get_int(line,1)+vp;
            vtmp[vnum].x=vertex_list[vnum].x=get_float(line,2);
            vtmp[vnum].y=vertex_list[vnum].y=get_float(line,3);
            vtmp[vnum].z=vertex_list[vnum].z=get_float(line,4);
            break;
          case OPEN_UNKNOWN_STRUCT:
            s.push(current);
            current=USTRUCT;
            break;
          case CLOSE_STRUCT:
            current=s.pop();
            break;
          default: break;
        }
        break;
      case PLIST:
        switch(next) {
          case SYMBOL_POLY:
            if(!poly_list) break;
            pnum=get_int(line,1)+pp;
            aa=get_int(line,3)+vp;
            bb=get_int(line,5)+vp;
            cc=get_int(line,7)+vp;
//            poly_list[pnum].old[0]=&vertex_list[a];
//            poly_list[pnum].old[1]=&vertex_list[b];
//            poly_list[pnum].old[2]=&vertex_list[c];
            poly_list[pnum].v[0]=&vtmp[aa];
            poly_list[pnum].v[1]=&vtmp[bb];
            poly_list[pnum].v[2]=&vtmp[cc];
//            poly_list[pnum].vnormal[0]=&vnormal[a];
//            poly_list[pnum].vnormal[1]=&vnormal[b];
//            poly_list[pnum].vnormal[2]=&vnormal[c];
            make_normal(&poly_list[pnum].normal,poly_list[pnum].v[0],poly_list[pnum].v[1],poly_list[pnum].v[2]);
//            for(i=0;i<3;i++) {
//              poly_list[pnum].vnormal[i]->x+=poly_list[pnum].normal.x; 
//              poly_list[pnum].vnormal[i]->y+=poly_list[pnum].normal.y; 
//              poly_list[pnum].vnormal[i]->z+=poly_list[pnum].normal.z; 
//            }
            #define av(_x) vnormal[_x].x+=poly_list[pnum].normal.x; \
                           vnormal[_x].y+=poly_list[pnum].normal.y; \
                           vnormal[_x].z+=poly_list[pnum].normal.z; \
                           vtol[_x]++;

            av(aa); av(bb); av(cc);
            #undef av
//            vtol[a]++; vtol[b]++; vtol[c]++;
            break;
          case OPEN_UNKNOWN_STRUCT:
            s.push(current);
            current=USTRUCT;
            break;
          case CLOSE_STRUCT:
            current=s.pop();
            break;
          default: break;
        }
        break;
      default: break;
    }
  }
  for(i=0;i<num_vertex;i++) {
     if(vtol[i]>0) {
       vnormal[i].x/=vtol[i];
       vnormal[i].y/=vtol[i];
       vnormal[i].z/=vtol[i];
     }
  }

  delete []vtol;

  obj = new OBJ3D;
  obj->name=name;
  obj->num_poly=num_poly;
  obj->num_vertex=num_vertex;
  obj->v=vertex_list;
  obj->p=poly_list;
  obj->tmp=vtmp;
  obj->vnormal=vnormal;
  obj->poly_type=POLYTYPE_GRGB;
  obj->tex=NULL;
  memcpy(&obj->pos,&identity_matrix_f,sizeof(MATRIX_f));

  return obj;
}

OBJ3D *load_3d(char *filename)
{
  PACKFILE *p;
  OBJ3D *ret;

  p = pack_fopen(filename,F_READ);
  ret = load_3dp(p,0);
  pack_fclose(p); 

  return ret;
}

