#include "data.h"
#include "map.h"
#include "robo.h"

/* Hexagon map. The tile format is like this:
 *
 *      0____1...2   
 *      /\  /\  .
 *    1/__\/__\.
 *    .\  /\  /
 *  2...\/__\/
 *      /\  /\  .
 *    3/__\/__\.
 *    .\  /\  /
 *  4...\/__\/
 *  0   1   2
 */

float view_x;
float view_y;
float view_a = 0;
int map[65536];
BITMAP *satmap;

void
convertWorldPosToScreenPos (float wx, float wy, float *sx, float *sy)
{
  wx -= view_x;
  wy -= view_y;
  *sx = wx * cos (view_a) - wy * sin (view_a);
  *sy = wy * cos (view_a) + wx * sin (view_a);
}

void
convertScreenPosToWorldPos (float sx, float sy, float *wx, float *wy)
{
  *wx = sx * cos (view_a) + sy * sin (view_a);
  *wy = sy * cos (view_a) - sx * sin (view_a);
  *wx += view_x;
  *wy += view_y;
}

void
convertWorldPosToTilePos (float wx, float wy, int *tx, int *ty)
{
  *tx = (int)wx >> 5;
  *ty = (int)wy >> 5;
}

void
convertTilePosToWorldPos (int tx, int ty, float *wx, float *wy)
{
  *wx = tx << 5;
  *wy = ty << 5;
}

static int
is_free (float x, float y)
{
  return !checkTile (x, y) &&
    !checkTile (x + 16, y) &&
    !checkTile (x - 16, y) &&
    !checkTile (x + 8, y - 8) &&
    !checkTile (x - 8, y - 8) &&
    !checkTile (x + 8, y + 8) &&
    !checkTile (x - 8, y + 8);
}

static int
is_around (int tx, int ty, int c)
{
  int t1, t2, t3, t5, t6, t7;
  tx--;
  ty--;
  if (ty & 1)
    return 0;
  t1 = map[(ty & 255) * 256 + (tx & 255)] & 255;
  t2 = map[(ty & 255) * 256 + ((tx + 1) & 255)] & 255;
  t3 = map[((ty + 1) & 255) * 256 + tx] & 255;
  //t4 = map[((ty + 1) & 255) * 256 + ((tx + 1) & 255)] & 255;
  t7 = map[((ty + 1) & 255) * 256 + ((tx + 2) & 255)] & 255;
  t5 = map[((ty + 2) & 255) * 256 + (tx & 255)] & 255;
  t6 = map[((ty + 2) & 255) * 256 + ((tx + 1) & 255)] & 255;
  return t1 == c && t2 == c && t3 == c && t5 == c && t6 == c && t7 == c;
}

static void
change_to (int tx, int ty, int c)
{
  map[(ty & 255) * 256 + (tx & 255)] |= c << 8;
  map[(ty & 255) * 256 + (tx & 255)] |= c << 16;
  map[((ty - 1) & 255) * 256 + ((tx - 1) & 255)] |= c << 8;
  map[((ty - 1) & 255) * 256 + ((tx - 1) & 255)] |= c << 16;
  map[((ty - 1) & 255) * 256 + (tx & 255)] |= c << 16;
  map[(ty & 255) * 256 + ((tx - 1) & 255)] |= c << 8;
}

void
generateMap (void)
{
  int i;
  int x, y;
  float passages[5];

  for (i = 0; i < 5; i++)
  {
    float rn = (float)(rand () & 255) / 255.0;
    passages[i] = AL_PI / 2 + (i & 1) * AL_PI;
    passages[i] += rn * AL_PI;
    if (passages[i] > AL_PI)
      passages[i] -= AL_PI * 2;
    if (passages[i] < -AL_PI)
      passages[i] += AL_PI * 2;
  }
  
  for (y = 0; y < 256; y++)
  {
    for (x = 0; x < 256; x++)
    {
      /* First, clear everything. */
      map[y * 256 + x] = 1;
      collision_map[y * 256 + x] = 0;
    }
  }

  for (y = 0; y < 256; y++)
  {
    for (x = 0; x < 256; x++)
    {
      float r = sqrt ((y - 128) * (y - 128) + (x - 128) * (x - 128));
      float a = atan2 (x - 128, y - 128);

      /* Outer radius: 120 */
      if (r >= 120)
      {
        map[y * 256 + x] = 0;
      }
      /* 5 stripes of 10 tiles at 100, 80, 60, 40, 20, forming the 6 rings. */
      else if (r >= 20 && (int)r % 20 < 10)
      {
        int p = r / 20 - 1;
        /* Make some ring passages. */
        if (a < passages[p] - 0.1 || a > passages[p] + 0.1)
          map[y * 256 + x] = 0;
      }
    }
  }
  
  clearRobos ();

  for (y = 0; y < 256; y++)
  {
    for (x = 0; x < 256; x++)
    {
      if (!(rand () & 7))
      /* Dissolve walls a bit. */
        map[y * 256 + x] = 1;
    }
  }
  
  for (y = 0; y < 256; y++)
  {
    for (x = 0; x < 256; x++)
    {
      /* Place some random walls. */
      if (!(rand () & 15))
      {
        map[y * 256 + x] = 0;
      }
    }
  }
  
  /* Clear middle. */
  for (y = 0; y < 256; y++)
  {
    for (x = 0; x < 256; x++)
    {
      float r = sqrt ((y - 128) * (y - 128) + (x - 128) * (x - 128));

      /* Outer radius: 120 */
      if (r < 5)
      {
        map[y * 256 + x] = 1;
      }
    }
  }
  /* Place the crystal. */
  change_to (129, 129, 1);
  map[129 * 256 + 129] &= 0xfff0;
  map[128 * 256 + 128] += BITMAP_CRYSTAL << 24;
  
  /* Make some podestals. */
  for (y = 0; y < 256; y++)
  {
    for (x = 0; x < 256; x++)
    {
      if (map[y * 256 + x] == 0 && is_around (x, y, 1))
      {
        if (!(rand () & 7))
        {
          change_to (x, y, 1);
          map[(y - 1) * 256 + x - 1] += BITMAP_LIFE << 24;
        }
      }
    }
  }
  
  /* Add player. */
  addRobo (128 * 32, 246 * 16, 0);
  robos[1].life = 64;
  /* Clear starting spot */
  map[245 * 256 + 127] = 1;
  map[246 * 256 + 127] = 1;
  map[247 * 256 + 127] = 1;
  map[245 * 256 + 128] = 1;
  map[246 * 256 + 128] = 1;
  map[247 * 256 + 128] = 1;
  map[245 * 256 + 129] = 1;
  map[246 * 256 + 129] = 1;
  map[247 * 256 + 129] = 1;
  
  /* Add some enemies, for more fun. */
  for (i = 2; i < ROBOCOUNT; i++)
  {
    x = rand () & 255;
    y = rand () & 255;
    if (y > 128 + 100 && x > 128 - 20 && x < 128 + 20)
      continue;
    if (is_free (x * 32, y * 16))
    {
      int lev = sqrt ((y - 128) * (y - 128) + (x - 128) * (x - 128));
      int rob = addRobo (x * 32, y * 16, (float)(rand() % 360) * AL_PI * 2.0 / 360.0);
      lev = 6 - lev / 20;
      robos[rob].level = lev;
    }
    else
    {
      i--;
    }
  }
  
  /* Let's party. */
  wakeUp ();

  i = 1;
  view_x = robos[i].x - 160 * cos (view_a) - 200 * sin (view_a);
  view_y = robos[i].y - 200 * cos (view_a) + 160 * sin (view_a);
  
  satmap = create_bitmap (256, 128);
  clear_to_color (satmap, bitmap_mask_color (satmap));
  for (y = 0; y < 128; y++)
  {
    for (x = 0; x < 256; x++)
    {
      int m = map[y * 2 * 256 + x];
      if ((m & 255) == 0)
      {
        putpixel (satmap, x, y, makecol (120, 80, 0));
      }
      if ((m >> 24) == BITMAP_LIFE)
        putpixel (satmap, x, y, makecol (155, 0, 0));
      if ((m >> 24) == BITMAP_CRYSTAL)
      {
        putpixel (satmap, x - 1, y, makecol (255, 255, 0));
        putpixel (satmap, x + 1, y, makecol (255, 255, 0));
        putpixel (satmap, x, y, makecol (255, 255, 255));
        putpixel (satmap, x, y - 1, makecol (255, 255, 0));
        putpixel (satmap, x, y + 1, makecol (255, 255, 0));
      }
    }
  }
}

static struct {int x, y, s;} bugs[256];
static int hacks = 0;
static void
hack (int x, int y, int s)
{
  bugs[hacks].x = x;
  bugs[hacks].y = y;
  bugs[hacks].s = s;
  hacks++;
}
static void
hackbugs (void)
{
  int i;
  for (i = 0; i < hacks; i++)
    draw_sprite (page, bitmap[bugs[i].s], bugs[i].x - 16, bugs[i].y - 16);
  hacks = 0;
}

/* Vertex tags.
 *
 *      1____2
 *      /\  /
 *    3/__\/4
 *     \  /\
 *      \/__\
 *      5    6
 */
static void
drawTile (float x, float y, float ax, float ay, int tx, int ty)
{
  V3D_f v1, v2, v3, v4, v5, v6;
  int t1, t2, t3, t4, t5, t6;
  int t142, t134, t354, t456;
  int b142, b134, b354, b456;
  int bmps[] = {BITMAP_SOIL, BITMAP_VOID};
  int u_top[8] = {48, 16, 64, 80, 32, 112,  0, 96};
  int v_top[8] = {16, 16,  0, 16,  0,  16,  0,  0};
  int u_bot[8] = {48, 16, 64, 80, 32, 112,  0, 96}; 
  int v_bot[8] = { 0,  0, 16,  0, 16,   0, 16, 16};
  int sprite;
  ty += ty;

  t1 = map[ty * 256 + tx];
  t2 = map[ty * 256 + ((tx + 1) & 255)];
  t3 = map[((ty + 1) & 255) * 256 + tx];
  t4 = map[((ty + 1) & 255) * 256 + ((tx + 1) & 255)];
  t5 = map[((ty + 2) & 255) * 256 + tx];
  t6 = map[((ty + 2) & 255) * 256 + ((tx + 1) & 255)];
  
  b142 = bmps[(t1 >> 8) & 255];
  b134 = bmps[(t1 >> 16) & 255];
  b354 = bmps[(t3 >> 8) & 255];
  b456 = bmps[(t4 >> 16) & 255];
  
  sprite = t1 >> 24;
  
  t1 &= 255;
  t2 &= 255;
  t3 &= 255;
  t4 &= 255;
  t5 &= 255;
  t6 &= 255;

  t142 = t4 + t2 * 2 + t1 * 4;
  t134 = t1 + t4 * 2 + t3 * 4;
  t354 = t5 + t4 * 2 + t3 * 4;
  t456 = t4 + t6 * 2 + t5 * 4;

  v1.x = x;
  v1.y = y;

  v2.x = x + ax * 2;
  v2.y = y + ay * 2;
  
  v3.x = x - ax - ay;
  v3.y = y - ay + ax;
  
  v4.x = x + ax - ay;
  v4.y = y + ay + ax;
  
  v5.x = x - ay * 2;
  v5.y = y + ax * 2;
  
  v6.x = x + ax * 2 - ay * 2;
  v6.y = y + ay * 2 + ax * 2;

  v1.u = u_bot[t142];
  v1.v = v_bot[t142];
  v2.u = v1.u + 32;
  v2.v = v1.v;
  v4.u = v1.u + 16;
  v4.v = v1.v + 16;
  triangle3d_f (page, POLYTYPE_ATEX, bitmap[b142], &v1, &v4, &v2);
  
  v1.u = u_top[t134] + 16;
  v1.v = v_top[t134];
  v3.u = v1.u - 16;
  v3.v = v1.v + 16;
  v4.u = v3.u + 32;
  v4.v = v3.v;
  triangle3d_f (page, POLYTYPE_ATEX, bitmap[b134], &v1, &v3, &v4);
  
  v3.u = u_bot[t354];
  v3.v = v_bot[t354];
  v4.u = v3.u + 32;
  v4.v = v3.v;
  v5.u = v3.u + 16;
  v5.v = v3.v + 16;
  triangle3d_f (page, POLYTYPE_ATEX, bitmap[b354], &v3, &v5, &v4);
  
  v4.u = u_top[t456] + 16;
  v4.v = v_top[t456];
  v5.u = v4.u - 16;
  v5.v = v4.v + 16;
  v6.u = v5.u + 32;
  v6.v = v5.v;
  triangle3d_f (page, POLYTYPE_ATEX, bitmap[b456], &v4, &v5, &v6);
  
  if (sprite)
    hack (v4.x, v4.y, sprite);
  
  /*line (page, v1.x, v1.y, v2.x, v2.y, makecol (255, 0, 0));
  line (page, v1.x, v1.y, v5.x, v5.y, makecol (255, 0, 0));
  if (collision_map[ty * 256 + tx])
    line (page, v1.x, v1.y, v6.x, v6.y, makecol (255, 0, 0));
  textprintf_ex (page, font, v1.x, v1.y, 0, -1, "%i", collision_map[ty * 256 + tx]);
  
  if (tx & 1)
  {
  }
  else
  {
    line (page, x, y, x + ax, y + ay, makecol (255, 255, 255));
    line (page, x, y, x - ax - ay, y - ay + ax, makecol (255, 255, 255));
    line (page, x - ax - ay, y - ay  + ax, x - ay * 2, y + ax * 2, makecol (255, 255, 255));
    line (page, x + ax, y + ay, x + ax * 2 - ay, y + ay * 2 + ax, makecol (255, 255, 255));
    line (page, x + ax - ay * 2, y + ay + ax * 2, x + ax * 2 - ay, y + ay * 2 + ax, makecol (255, 255, 255));
    line (page, x + ax * 2 - ay, y + ay * 2 + ax, x + ax * 3 - ay, y + ay * 3 + ax, makecol (255, 255, 255));
  }
  */
}

void
drawMap (void)
{
  float x, y;
  float ax, ay;
  float wx, wy;
  int tx, ty;

  int L = 0;
  int T = 0;
  int R = page->w - 1;
  int B = page->h - 1;

  /* sqrt (2) * 32 would make sure every square is in, 64 also adds another
   * layer of tiles around, since we don't exactly fill squares in the tile
   * drawing.
   */
  int S = sqrt (2) * 64;

  float rx, ry;
  int dx, dy;

  convertWorldPosToTilePos (view_x, view_y, &tx, &ty);
  convertTilePosToWorldPos (tx, ty, &wx, &wy);
  convertWorldPosToScreenPos (wx, wy, &x, &y);

  //rect (page, L, T, R, B, makecol (100, 100, 100));

  ax = cos (view_a) * 16;
  ay = sin (view_a) * 16;

  if (ax >= 0 && ay <= 0)
  {
    rx = ax * 2;
    ry = ay * 2;
    dx = 1;
    dy = 0;
  }
  if (ax >= 0 && ay > 0)
  {
    rx = ay * 2;
    ry = -ax * 2;
    dx = 0;
    dy = -1;
  }
  if (ax < 0 && ay <= 0)
  {
    rx = -ay * 2;
    ry = ax * 2;
    dx = 0;
    dy = 1;
  }
  if (ax < 0 && ay > 0)
  {
    rx = -ax * 2;
    ry = -ay * 2;
    dx = -1;
    dy = 0;
  }  

  x = L + x;
  y = T + y;
  
  /* Go 'up'. */
  x += ry;
  y -= rx;
  tx += dy;
  ty -= dx;

  while (1)
  {
    float ox, oy;
    int otx, oty;
    /* Go 'left' until out. */
    while (x >= L - S && y <= B + S)
    {
      x -= rx;
      y -= ry;
      tx -= dx;
      ty -= dy;
    }
    /* Go 'right' until in. */
    while (x < L - S || y > B + S)
    {
      x += rx;
      y += ry;
      tx += dx;
      ty += dy;
      if (x > R + S || y < T - S)
        goto done;
    }
    ox = x;
    oy = y;
    otx = tx;
    oty = ty;
    do
    {
      tx &= 255;
      ty &= 127;
      drawTile (x, y, ax, ay, tx, ty);
     
      /* Go 'right' until out. */
      x += rx;
      y += ry;
      tx += dx;
      ty += dy;
    }
    while (x <= R + S && y >= T - S);
    x = ox;
    y = oy;
    tx = otx;
    ty = oty;

    /* Go 'down'. */
    x -= ry;
    y += rx;
    tx -= dy;
    ty += dx;
  }
done:
  hackbugs ();
}

void
drawSat (void)
{
  int satw = 64, sath = 64;
  hline (page, 0, sath, satw, makecol (0, 0, 0));
  vline (page, satw, 0, sath, makecol (0, 0, 0));
  set_clip (page, 0, 0, satw - 1, sath - 1);
  pivot_sprite (page, satmap, satw / 2, sath / 2, robos[1].x / 32, robos[1].y / 32,
    fixmul (ftofix (view_a), radtofix_r));
  set_clip (page, 0, 0, page->w - 1, page->h - 1);
  circle (page, satw / 2, sath / 2, 3, makecol (0, 0, 0));
}
