#include "robo.h"
#include "data.h"
#include "map.h"
#include "run.h"

Robo robos[ROBOCOUNT] = {};
Laser lasers[256] = {};
int collision_map[65536];
int active[256];
  
static int wake_time = 0;

static void
leaveTile (int i)
{
  int tx, ty;
  convertWorldPosToTilePos (robos[i].x, robos[i].y, &tx, &ty);
  tx &= 255;
  ty &= 127;
  ty += ty;
  collision_map[ty * 256 + tx] = 0;
}
  
void
clearRobos (void)
{
  int i;
  for (i = 0; i < ROBOCOUNT; i++)
  {
    robos[i].alive = 0;
  }
  for (i = 0; i < 256; i++)
  {
    active[i] = 0;
  }
  wake_time = 0;
}

static void
enterActiveTile (int a)
{
  int tx, ty;
  int i = active[a];
  convertWorldPosToTilePos (robos[i].x, robos[i].y, &tx, &ty);
  tx &= 255;
  ty &= 127;
  ty += ty;
  collision_map[ty * 256 + tx] = a;
}

int
addRobo (float x, float y, float a)
{
  int i;
  for (i = 1; i < ROBOCOUNT; i++)
  {
    if (!robos[i].alive)
    {
      memset (&robos[i], 0, sizeof (Robo));
      robos[i].x = x;
      robos[i].y = y;
      robos[i].a = a;
      robos[i].alive = 1;
      return i;
    }
  }
  return 0;
}

static int
wakeRobo (int i)
{
  int a;
  for (a = 1; a < 256; a++)
  {
    if (!active[a])
    {
      active[a] = i;
      robos[i].awake = 1;
      enterActiveTile (a);
      return a;
    }
  }
  return 0;
}

static void
sleepActiveRobo (int a)
{
  int i = active[a];
  active[a] = 0;
  robos[i].awake = 0;
}

void
wakeUp (void)
{
  int i, a;
  int xr, yr;
  for (i = 1; i < ROBOCOUNT; i++)
  {
    if (!robos[i].alive)
      continue;
    xr = robos[i].x - robos[1].x;
    yr = robos[i].y - robos[1].y;
    if (xr * xr + yr * yr < 300 * 300)
    {
      if (!robos[i].awake)
      {
        wakeRobo (i);
      }
    }
  }
  for (a = 1; a < 256; a++)
  {
    i = active[a];
    if (!i)
      continue;
    xr = robos[i].x - robos[1].x;
    yr = robos[i].y - robos[1].y;
    if (xr * xr + yr * yr > 300 * 300)
    {
      if (robos[i].awake)
      {
        leaveTile (i);
        sleepActiveRobo (a);
      }
    }
  }
}

static void
delActiveRobo (int a)
{
  int i = active[a];
  sleepActiveRobo (a);
  leaveTile (i);
  robos[i].alive = 0;
}

static int
addLaser (float x, float y, float a, int player, int level)
{
  int i;
  for (i = 1; i < 256; i++)
  {
    if (!lasers[i].alive)
    {
      memset (&lasers[i], 0, sizeof lasers[i]);
      lasers[i].x = x;
      lasers[i].y = y;
      lasers[i].a = a;
      lasers[i].alive = 1;
      lasers[i].player = player;
      lasers[i].level = level;
      lasers[i].d = a;
      return i;
    }
  }
  return 0;
}

static void
delLaser (int i)
{
  lasers[i].alive = 0;
}

float bouncex, bouncey;
int
checkTile (float x, float y)
{
  int tx, ty, hit;
  convertWorldPosToTilePos (x, y, &tx, &ty);
  tx &= 255;
  ty &= 127;
  ty += ty;
  hit = collision_map[ty * 256 + tx];
  if (hit)
    return hit;
  {
    float rx, ry;
    int t1, t2, t4, t5, t6;
    t1 = map[((ty) & 255) * 256 + ((tx) & 255)] & 255;
    t2 = map[((ty) & 255) * 256 + ((tx + 1) & 255)] & 255;
    t4 = map[((ty + 1) & 255) * 256 + ((tx + 1) & 255)] & 255;
    t5 = map[((ty + 2) & 255) * 256 + ((tx) & 255)] & 255;
    t6 = map[((ty + 2) & 255) * 256 + ((tx + 1) & 255)] & 255;
    if (t1 == 0)
    {
      rx = tx * 32;
      ry = ty * 16;
      rx -= x;
      ry -= y;
      if (rx * rx + ry * ry < 16 * 16)
      {
        bouncex = rx;
        bouncey = ry;
        return 256;
      }
    }
    if (t2 == 0)
    {
      rx = tx * 32 + 32;
      ry = ty * 16;
      rx -= x;
      ry -= y;
      if (rx * rx + ry * ry < 16 * 16)
      {
        bouncex = rx;
        bouncey = ry;
        return 256;
      }
    }
    if (t4 == 0)
    {
      rx = tx * 32 + 16;
      ry = ty * 16 + 16;
      rx -= x;
      ry -= y;
      if (rx * rx + ry * ry < 16 * 16)
      {
        bouncex = rx;
        bouncey = ry;
        return 256;
      }
    }
    if (t5 == 0)
    {
      rx = tx * 32;
      ry = ty * 16 + 32;
      rx -= x;
      ry -= y;
      if (rx * rx + ry * ry < 16 * 16)
      {
        bouncex = rx;
        bouncey = ry;
        return 256;
      }
    }
    if (t6 == 0)
    {
      rx = tx * 32 + 32;
      ry = ty * 16 + 32;
      rx -= x;
      ry -= y;
      if (rx * rx + ry * ry < 16 * 16)
      {
        bouncex = rx;
        bouncey = ry;
        return 256;
      }
    }
    
    return hit;
  }
}

static BITMAP_ID
getSprite (float x, float y)
{
  int tx, ty;
  convertWorldPosToTilePos (x, y, &tx, &ty);
  tx &= 255;
  ty &= 127;
  ty += ty;
  return map[ty * 256 + tx] >> 24;
}

static void
killSprite (float x, float y)
{
  int tx, ty;
  convertWorldPosToTilePos (x, y, &tx, &ty);
  tx &= 255;
  ty &= 127;
  ty += ty;
  map[ty * 256 + tx] &= 0x00ffffff;
}

static void
fire (int i)
{
  float x = robos[i].x;
  float y = robos[i].y;
  robos[i].fire_side = !robos[i].fire_side;
  if (robos[i].fire_side)
  {
    x += cos (robos[i].a) * 12;
    y -= sin (robos[i].a) * 12;
  }
  else
  {
    x -= cos (robos[i].a) * 12;
    y += sin (robos[i].a) * 12;
  }
  x -= sin (robos[i].a) * 12;
  y -= cos (robos[i].a) * 12;
  addLaser (x, y, robos[i].a, i == 1 ? 1 : 0, robos[i].level);
}

static void
movePlayer (void)
{
  int move = 0;
  int a = 1, i;
  float nx, ny;

  i = active[a];

  if (robos[i].fire)
    robos[i].fire--;
  
  nx = robos[i].x;
  ny = robos[i].y;

  if (key[KEY_LEFT])
  {
    robos[i].a += 0.02;
    move = 1;
  }
  if (key[KEY_RIGHT])
  {
    robos[i].a -= 0.02;
    move = 1;
  }
  if (key[KEY_UP])
  {
    nx -= sin (view_a);
    ny -= cos (view_a);
    move = 1;
  }
  if (key[KEY_DOWN])
  {
    nx += sin (view_a);
    ny += cos (view_a);
    move = -1;
  }
  if (key[KEY_RCONTROL] || key[KEY_LCONTROL])
  {
    if (!robos[i].fire)
    {
      robos[i].fire = 16;
      fire (i);
      play_sample (sample[SAMPLE_FIRE], 255, 128, 1000 + rand() % 100, 0);
    }
  }
  
  if (move)
  {
    int what;
    int f = robos[i].fire > 12 ? 5 : 0;
    leaveTile (i);

    what = checkTile (nx, ny);
    if (what == 256)
    {
      float bounce = sqrt (bouncex * bouncex + bouncey * bouncey);
      nx -= 5 * bouncex / bounce;
      ny -= 5 * bouncey / bounce;
      what = checkTile (nx, ny);
    }
    if (!what)
    {
      robos[i].x = nx;
      robos[i].y = ny;
    }
    
    if (robos[i].step == 0)
      play_sample (sample[SAMPLE_STEP], 155, 100, 1000 + rand() % 50, 0);
    robos[i].step += move;
    robos[i].step &= 31;
    robos[i].anim = f + 1 + robos[i].step / 8;
    
    view_a = robos[i].a;
    view_x = robos[i].x - 160 * cos (view_a) - 200 * sin (view_a);
    view_y = robos[i].y - 200 * cos (view_a) + 160 * sin (view_a);

    enterActiveTile (a);
    
    what = getSprite (robos[i].x, robos[i].y);
    if (what == BITMAP_LIFE)
    {
      if (robos[i].life < 64)
      {
        robos[i].life += 33;
        if (robos[i].life > 64)
          robos[i].life = 64;
        killSprite (robos[i].x, robos[i].y);
      }
    }
    if (what == BITMAP_CRYSTAL)
    {
      robos[i].win = 1;
    }
  }
  else
  {
    int f = robos[i].fire > 12 ? 5 : 0;
    robos[i].anim = f;
    robos[i].step = 0;
  }
}

static float
angle_dist (float a1, float a2)
{
  float d = fabs (a1 - a2);
  if (d >= AL_PI * 2)
    d -= AL_PI * 2;
  if (d > AL_PI)
    return AL_PI * 2 - d;
  return d;
}

void
moveRobos (void)
{
  int a, i;

  if (wake_time)
    wake_time--;
  else
  {
    wakeUp ();
    wake_time = FPS;
  }

  movePlayer ();
  for (a = 2; a < 256; a++)
  {
    float nx, ny;
    float d;
    int hit;
    int f;
    
    i = active[a];

    if (!i)
      continue;
    
    if (!robos[i].alive)
      continue;
    
    f = robos[i].fire > 12 ? 5 : 0;
    
    if (robos[i].fire)
      robos[i].fire--;
    
    if (robos[i].wander)
    {
      robos[i].wander--;
    }
    else
    {
      float dist = robos[i].dir - robos[i].a;
      if ((dist > -AL_PI && dist < 0) || dist > AL_PI)
      {
        robos[i].clockwise = 1;
      }
      else
      {
        robos[i].clockwise = 0;
      }
      if (!robos[i].fire && angle_dist (robos[i].dir, robos[i].a) < AL_PI / 8)
      {
        robos[i].fire = 24;
        fire (i);
        play_sample (sample[SAMPLE_FIRE], 155, 128, (600 + rand() % 100) / robos[i].level , 0);
      }
    }

    d = robos[i].clockwise ? -0.01 : 0.01;
    d *= robos[i].level;
    
    leaveTile (i);

    nx = robos[i].x - sin (robos[i].a);
    ny = robos[i].y - cos (robos[i].a);
    
    hit = checkTile (nx, ny);
    if (hit == 256)
    {
      float bounce = sqrt (bouncex * bouncex + bouncey * bouncey);
      nx -= 5 * bouncex / bounce;
      ny -= 5 * bouncey / bounce;
      hit = checkTile (nx, ny);
    }
    if (!hit)
    {
      robos[i].x = nx;
      robos[i].y = ny;
    }
    if (hit)
    {
      /* If looks into right dir, but still collision: wander. */
      if (angle_dist (robos[i].dir, robos[i].a) < AL_PI / 4)
        robos[i].wander = (rand () % FPS) * (7 - robos[i].level);
    }

    robos[i].a += d;
    if (robos[i].a > AL_PI)
      robos[i].a -= AL_PI * 2;
    if (robos[i].a < -AL_PI)
      robos[i].a += AL_PI * 2;

    if (!robos[i].wander && !(robos[i].step % (3 - robos[i].level / 3)))
    {
      robos[i].dir = atan2 (robos[i].x - robos[1].x, robos[i].y - robos[1].y);
    }

    robos[i].step++;
    robos[i].step &= 31;
    robos[i].anim = f + 1 + robos[i].step / 8;
    enterActiveTile (a);
  }

  for (i = 1; i < 256; i++)
  {
    int hit;
    if (!lasers[i].alive)
      continue;
    
    lasers[i].a += 0.2;
    
    if (lasers[i].splash)
    {
      lasers[i].splash--;
      if (lasers[i].splash == 0)
      {
        delLaser (i);
        continue;
      }
    }
    else
    {
      lasers[i].x -= sin (lasers[i].d) * 5;
      lasers[i].y -= cos (lasers[i].d) * 5;
      
      hit = checkTile (lasers[i].x, lasers[i].y);
      if (hit)
      {
        if (hit < 256)
        {
          if (hit > 1 && lasers[i].player)
          {
            delActiveRobo (hit);
            lasers[i].z = 300;
          }
          if (hit == 1 && !lasers[i].player)
          {
            /* Uh oh. */
            robos[hit].life -= 3;
            lasers[i].z = 300;
          }
        }
        else
        {
          lasers[i].z = 300;
        }
      }
      
      lasers[i].z++;
      if (lasers[i].z > 300)
      {
        lasers[i].splash = FPS / 4;
      }
    }
  }
}

void
drawRobos (void)
{
  int i, a, na;
  na = 0;
  for (i = 0; i < 256; i++)
  {
    float x, y;
    BITMAP *sprite = laser[lasers[i].level];
    if (!lasers[i].alive)
      continue;
    if (lasers[i].splash)
      sprite = bitmap[BITMAP_SPLASH];
    convertWorldPosToScreenPos (lasers[i].x, lasers[i].y, &x, &y);
    pivot_sprite (page, sprite, x, y, 8, 8,
      fixmul (ftofix (view_a - lasers[i].a), radtofix_r));
  }
  for (a = 1; a < 256; a++)
  {
    float x, y;
    i = active[a];
    if (!i)
      continue;
    na++;
    if (!robos[i].alive)
      continue;
    convertWorldPosToScreenPos (robos[i].x, robos[i].y, &x, &y);
    pivot_sprite (page, robo[robos[i].level * 10 + robos[i].anim], x, y, 24, 24,
      fixmul (ftofix (view_a - robos[i].a), radtofix_r));
  }
}
