/*
 * ALSPC - SPC playback library for Allegro
 * get the latest version at my website: http://www.student.wau.nl/~martijni/
 *
 * (c) Copyright 2003 Martijn van Iersel (amarillion@yahoo.com) 
 * version 0.91
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "alspc.h"
#include "libspc.h"
#include <assert.h>
#include <string.h>
#include <stdio.h>
#include "snes9x.h"
#include "port.h"
#include "apu.h"
#include "soundux.h"

extern SAPURegisters BackupAPURegisters;
extern unsigned char BackupAPURAM[65536];
extern unsigned char BackupAPUExtraRAM[64];
extern unsigned char BackupDSPRAM[128];

int _alspc_players_installed = 0;

// packfile functions
void *load_spcdata(PACKFILE *f, long size);
void destroy_spcdata(void *data);

// create an alspc_data struct from preloaded memory
ALSPC_DATA *alspc_load_from_memory (char *src)
{
    ALSPC_DATA *data = NULL;
    bool error = FALSE;
        
    data = (ALSPC_DATA*)malloc (sizeof (ALSPC_DATA));
    if (!data)
    {
        error = TRUE;
    }
        
    if (!error)
    {           
        
        memcpy(&data->BackupAPURegisters.PC, src + 0x25, 2);
        
        data->BackupAPURegisters.YA.B.A = src[0x27];
        data->BackupAPURegisters.X = src[0x28];
        data->BackupAPURegisters.YA.B.Y = src[0x29];
        data->BackupAPURegisters.P = src[0x2A];
        data->BackupAPURegisters.S = src[0x2B];
    
        
        memcpy (data->BackupAPURAM, src + 0x100, 0x10000);
        memcpy (data->BackupDSPRAM, src + 0x10100, 128);
        memcpy (data->BackupAPUExtraRAM, src + 0x10100 + 192, 64);
    
        // read ID666 tags
                
        if (src[0x23] != 27) 
        {                    
            memcpy (data->id.songname, src + 0x2E, 32);
            data->id.songname[33] = '\0';
            
            memcpy (data->id.gametitle, src + 0x2E + 32, 32);
            data->id.gametitle[33] = '\0';
            
            memcpy (data->id.dumper, src + 0x2E + 64, 16);
            data->id.dumper[17] = '\0';
            
            memcpy (data->id.comments, src + 0x2E + 80, 32);
            data->id.comments[33] = '\0';
            
            memcpy (data->id.author, src + 0xB0, 32);
            data->id.author[33] = '\0';
                        
            switch (src[0xD0]) {
            case 1:
              data->id.emulator = SPC_EMULATOR_ZSNES;
              break;
            case 2:
              data->id.emulator = SPC_EMULATOR_SNES9X;
              break;
            case 0:
            default:
              data->id.emulator = SPC_EMULATOR_UNKNOWN;
              break;
            }            
        }
        else
        {
            memset (&(data->id), 0, sizeof (SPC_ID666));
        }
    }
    
    if (error)
    {
        free (data);
        return NULL;
    }
    else
    {
        return data;
    }    
}

ALSPC_DATA *alspc_load (const char *fname)
{
    FILE *fp = NULL;
    ALSPC_DATA *data = NULL;
    char *src = NULL;
    bool error = FALSE;
    
    fp = fopen (fname, "rb");
    if (!fp)
    {
        error = TRUE;
    }
    
    if (!error)
    {
        src = (char *)malloc (0x10200);
        if (!src)
        {
            error = TRUE;
        }
    }
        
    if (!error)
    {        
        fread(src, 1, 0x10200, fp);
        data = alspc_load_from_memory (src);
        if (!data) error = true;
    }
    
    if (fp) fclose(fp);    
    
    if (src) 
    {
        free (src);
        src = NULL;
    }
    
    if (error)
    {
        free (data);
        return NULL;
    }
    else
    {
        return data;
    }
}

/*
ALSPC_DATA *alspc_load (const char *fname)
{
    FILE *fp = NULL;
    char temp[64];
    ALSPC_DATA *data = NULL;
    bool error = FALSE;
    
    fp = fopen (fname, "rb");
    if (!fp)
    {
        error = TRUE;
    }
    
    if (!error)
    {
        data = (ALSPC_DATA*)malloc (sizeof (ALSPC_DATA));
        if (!data)
        {
            error = TRUE;
        }
    }
        
    if (!error)
    {        
        fseek(fp, 0x25, SEEK_SET);
        fread(&data->BackupAPURegisters.PC, 1, 2, fp);
        fread(&data->BackupAPURegisters.YA.B.A, 1, 1, fp);
        fread(&data->BackupAPURegisters.X, 1, 1, fp);
        fread(&data->BackupAPURegisters.YA.B.Y, 1, 1, fp);
        fread(&data->BackupAPURegisters.P, 1, 1, fp);
        fread(&data->BackupAPURegisters.S, 1, 1, fp);
    
        fseek(fp, 0x100, SEEK_SET);
        fread(data->BackupAPURAM, 1, 0x10000, fp);
        fread(data->BackupDSPRAM, 1, 128, fp);
        fread(temp, 1, 64, fp);
        fread(data->BackupAPUExtraRAM, 1, 64, fp);
    
        // read ID666 tags
        
        fseek(fp, 0x23, SEEK_SET);
        if (fgetc(fp) != 27) 
        {        
            fseek(fp, 0x2E, SEEK_SET);
            fread(data->id.songname, 1, 32, fp);
            data->id.songname[33] = '\0';
            
            fread(data->id.gametitle, 1, 32, fp);
            data->id.gametitle[33] = '\0';
            
            fread(data->id.dumper, 1, 16, fp);
            data->id.dumper[17] = '\0';
            
            fread(data->id.comments, 1, 32, fp);
            data->id.comments[33] = '\0';
            
            fseek(fp, 0xD0, SEEK_SET);
            switch (fgetc (fp)) {
            case 1:
              data->id.emulator = SPC_EMULATOR_ZSNES;
              break;
            case 2:
              data->id.emulator = SPC_EMULATOR_SNES9X;
              break;
            case 0:
            default:
              data->id.emulator = SPC_EMULATOR_UNKNOWN;
              break;
            }
            
            fseek(fp, 0xB0, SEEK_SET);
            fread(data->id.author, 1, 32, fp);
            data->id.author[33] = '\0';
        }
        else
        {
            memset (&(data->id), 0, sizeof (SPC_ID666));
        }
    }
    
    if (fp) fclose(fp);    
    
    if (error)
    {
        free (data);
        return NULL;
    }
    else
    {
        return data;
    }
}
*/

void *load_spcdata(PACKFILE *f, long size)
{
    ALSPC_DATA *data = NULL;
    char *src = NULL;
    bool error = FALSE;
       
    src = (char *)malloc (0x10200);
    if (!src)
    {
        error = TRUE;
    }
        
    if (!error)
    {        
        pack_fread(src, 0x10200, f);
        data = alspc_load_from_memory (src);
        if (!data) error = true;
    }
    
    if (src) 
    {
        free (src);
        src = NULL;
    }
    
    if (error)
    {
        free (data);
        return NULL;
    }
    else
    {
        return data;
    }
}

void destroy_spcdata(void *data)
{    
    alspc_unload ((ALSPC_DATA *)data);
}

void alspc_install ()
{
    register_datafile_object(DAT_ALSPC, load_spcdata, destroy_spcdata);
}

void alspc_uninstall ()
{
    // nothing for now...
}

void alspc_unload (ALSPC_DATA *alspc)
{
    free (alspc);
}

ALSPC_PLAYER *alspc_start (ALSPC_DATA *alspc, int sampling_rate, int volume, int pan, int stereo, int is_interpolation)
{
    SPC_Config spc_config;
    ALSPC_PLAYER *result;
    
    assert (alspc);
    
    if (_alspc_players_installed > 0) return NULL;
    
    spc_config.sampling_rate = sampling_rate;
    spc_config.resolution = 16;
    spc_config.channels = (stereo ? 2 : 1);
    spc_config.is_interpolation = is_interpolation;
    spc_config.is_echo = 1;
    
    result = (ALSPC_PLAYER*) malloc (sizeof (ALSPC_PLAYER));
    if (!result) return NULL;
    
    result->buf_size = SPC_init (&spc_config);    
    
    result->bytes_per_sample = spc_config.channels * 2;
    result->sample_count = result->buf_size / result->bytes_per_sample;
    
    result->stream = play_audio_stream (
        result->sample_count * ALSPC_BUFFER_MULTIPLIER, 
        16, 
        stereo,
        sampling_rate, 
        volume, 
        pan);
    
    assert (result->stream);
    // this will create a stream buffer 8 times larger than the spc buffer.
            
    S9xInitAPU();
    S9xResetAPU();

    BackupAPURegisters.PC = alspc->BackupAPURegisters.PC;
    BackupAPURegisters.YA.B.A = alspc->BackupAPURegisters.YA.B.A;
    BackupAPURegisters.X = alspc->BackupAPURegisters.X;
    BackupAPURegisters.YA.B.Y = alspc->BackupAPURegisters.YA.B.Y;
    BackupAPURegisters.P = alspc->BackupAPURegisters.P;
    BackupAPURegisters.S = alspc->BackupAPURegisters.S;
    memcpy (BackupAPURAM, alspc->BackupAPURAM, 65536);
    memcpy (BackupAPUExtraRAM, alspc->BackupAPUExtraRAM, 64);
    memcpy (BackupDSPRAM, alspc->BackupDSPRAM, 128);

    RestoreSPC();
    IAPU.OneCycle = ONE_APU_CYCLE;
    
    _alspc_players_installed = 1;
    
    return result;
}

void alspc_stop (ALSPC_PLAYER *player)
{
    if (!player) return;
        
    _alspc_players_installed = 0;
    
    SPC_close ();
    
    if (player->stream) stop_audio_stream (player->stream);
    player->stream = NULL;
    
    free (player);
}

void alspc_poll (ALSPC_PLAYER *player)
{
    void *temp;
    if ((temp = get_audio_stream_buffer (player->stream)))     
    {
        // remember the stream buffer is 8 times larger, so we have to update 8 times now.
        
        int j;
        for (j = 0; j < ALSPC_BUFFER_MULTIPLIER; ++j)
        {
            SPC_update ((unsigned char*)(temp) + j * player->buf_size);
        }                   
        
        short *temp2 = (short *)(temp);
        int i;
        for (i = 0; i < player->buf_size / 2 * ALSPC_BUFFER_MULTIPLIER; ++i)
        {
            temp2[i] ^= 0x8000;
        }
        
        free_audio_stream_buffer (player->stream);
    }            
}

