/**
at the moment gui.h and gui.cpp are part of the hexagon project
but they are entirely independant of it and can be used by other projects
as well, without modification.

It is designed to be a simple gui, similar to the allegro gui but with two
major differences: it is in C++, and the drawing and message handling
are entirely separated, so it can be taken up in a game loop without problems,
and background processes can continue to run.

At the moment the following widgets are available:
GuiButton - a button
GuiText - a line of text
GuiGroupItem - a item that is part of a group - only one can
    be selected at a time
*/

#include <allegro.h>
#include <string>
#include <list>

#define GUIMSG_CLICK 1
#define GUIMSG_UNSELECT 2
#define GUIMSG_SELECT 3
#define GUIMSG_GOT_FOCUS 4
#define GUIMSG_LOST_FOCUS 5

#define GUI_OK 0

class Gui;

class GuiItem
{
protected:
    int x, y, w, h;
    bool dirty;
    Gui *parent;
    bool hasFocus ();
public:
    virtual void draw(BITMAP *target) = 0;
    virtual int message (int msg, int c) = 0;
    virtual bool wantsFocus () = 0;
    bool contains (int cx, int cy);
    void setParent (Gui *newParent) { parent = newParent; }
    GuiItem (int newx, int newy, int neww, int newh)
    {
        x = newx;
        y = newy;
        w = neww;
        h = newh;
        dirty = true;
        parent = NULL;
    }
    bool isDirty () { return dirty; }
    void setDirty ();
};

class GuiText : public GuiItem
{
private:
     std::string text;
     int color;
     FONT *f;
public:
     GuiText (std::string newtext, int newx, int newy, int neww, int newh, int newcolor, FONT *newf)
        : GuiItem (newx, newy, neww, newh)
     {
         text = newtext;
         color = newcolor;
         f = newf;
     }
     virtual ~GuiText() {}
     virtual bool wantsFocus () { return false; }
     virtual void draw(BITMAP *target);
     virtual int message (int msg, int c);
};

class GuiButton : public GuiItem
{
private:
    BITMAP* icon;
    int action;
public:
    GuiButton (BITMAP *newicon, int newaction, int newx, int newy, int neww, int newh)
     : GuiItem (newx, newy, neww, newh)
    {
        icon = newicon;
        action = newaction;
    }
    virtual bool wantsFocus () { return true; }
    virtual void draw(BITMAP *target);
    virtual int message (int msg, int c);
};

class GuiTextButton : public GuiButton
{
private:    
    int action;
    std::string text;
public:
    GuiTextButton (std::string _text, int newaction, int newx, int newy, int neww, int newh)
     : GuiButton (NULL, newaction, newx, newy, neww, newh), text (_text) {}
    virtual void draw(BITMAP *target);    
    virtual ~GuiTextButton() {}
};

class GuiGroupItem : public GuiItem
{
private:
    BITMAP *icon;
    int group;
    int index;
protected:
    bool selected;    
public:
    virtual void draw(BITMAP *target);
    virtual int message (int msg, int c);
    virtual bool wantsFocus () { return selected; }
    GuiGroupItem (BITMAP *newIcon, int newGroup, int newIndex, int newx,
        int newy, int neww, int newh);
    bool isSelected () { return selected; }
    int getGroup () { return group; }
    int getIndex () { return index; }
};

class GuiTextGroupItem : public GuiGroupItem
{
private:
    std::string text;
public:
    GuiTextGroupItem (std::string _text, int newGroup, int newIndex, int newx,
        int newy, int neww, int newh) : 
        GuiGroupItem (NULL, newGroup, newIndex, newx, newy, neww, newh), text (_text) {}
    virtual void draw(BITMAP *target);    
    virtual ~GuiTextGroupItem() {}
};

class Gui
{
    std::list<GuiItem*> guiItems;
    int last_mouse_b;
    bool dirty;
    bool childDirty;
    GuiItem *focusItem;
public:
    GuiItem *getFocusItem () { return focusItem; }
    Gui() { guiItems.clear(); last_mouse_b = 0;
        dirty = true; childDirty = false; focusItem = NULL; }
    ~Gui();
    void add(GuiItem *i);
    int runGui(); // if doesn't return 0, gui is finished.
    void drawGui(BITMAP *target);
    void broadcastMessage(int msg, int c);
    void setGroupSelection (int group, int index);
    int getGroupSelection (int group);
    void searchNextFocusItem();
    void searchPrevFocusItem();
    void setFocus (GuiItem *m);
    void setChildDirty () { childDirty = true; } // called by children to request redraw
    bool isDirty () { return dirty || childDirty; }
    void setDirty () { dirty = true; }
};

