#include <assert.h>
#include "main.h"
#include "misc.h"

// needed for "Sleep()"
#ifdef ALLEGRO_WINDOWS
#include <winalleg.h>
#endif

volatile int Main::counter = 0;

void timer_handler()
{
    Main::counter += 10;
}

END_OF_FUNCTION(timer_handler);

void Main::playMusic (const char *id)
{
    if (!(music && soundOn)) return;
    ALSPC_DATA *alspc_data = resources.getSPC (id);
    assert (alspc_data); // fails if requested SPC is not in datafile.    
    if (alspc_player)
    {
        alspc_stop(alspc_player);
        alspc_player = NULL;
    }
    alspc_player = alspc_start (alspc_data, hifi ? 44100 : 22050, 255, 128, stereo, hifi);
}

void Main::playSample (SAMPLE *s)
{
    if (!soundOn) return;
    play_sample (s, 127, 127, 1000, 0);
}

Main::Main()  : menu (this), engine (this), alspc_player (NULL)
{
    music = true;
    hifi = true;
    stereo = true;
    soundOn = true;
    fpsOn = false;
    buffer = NULL;
    windowed = false;
    doublesize = false;
    
    // use these values during debugging:
    //~ windowed = true;
    //~ doublesize = true;
    
    last_fps = 0;
    update_counter = 0;
    frame_count = 0;
    frame_counter = 0;
}

int Main::init(int argc, const char *const *argv)
// returns 1 on success, 0 on failure
{
    LOCK_VARIABLE(Main::counter);
    LOCK_FUNCTION(timer_handler);
    if (allegro_init () < 0)
    {
        allegro_message ("init failed");
        return 0;
    }
    
    // parse command line arguments
    int i;
    for (i = 1; i < argc; i++)
    {
        if (strcmp (argv[i], "-nosound") == 0)
        {
            soundOn = false;
        }
        else if (strcmp (argv[i], "-windowed") == 0)
        {
            windowed = true;
        }
        else if (strcmp (argv[i], "-doublesize") == 0)
        {
            doublesize = true;
        }
        else if (strcmp (argv[i], "-fullscreen") == 0)
        {
            windowed = false;
        }
        else if (strcmp (argv[i], "-normalsize") == 0)
        {
            doublesize = false;
        }
        else if (strcmp (argv[i], "-showfps") == 0)
        {
            fpsOn = true;
        }
        else
        {
            allegro_message (
                "error parsing command line argument %i\n%s",        
                i, argv[i]);
        }            
    }   
    
    if (install_keyboard () < 0)
    {
        allegro_message ("install keyboard failed");
        return 0;
    }
    if (install_timer() < 0)
    {
        allegro_message ("install timer failed");
        return 0;
    }
    set_volume_per_voice (1);    
    if (soundOn && install_sound(DIGI_AUTODETECT, MIDI_NONE, NULL) < 0)
    {
        // could not get sound to work
        soundOn = false;
        allegro_message ("Could not initialize sound. Sound is turned off.\n%s\n", allegro_error);
    }
    alspc_install();    
    if (install_int (timer_handler, 10) < 0)
    {
        allegro_message ("installation of timer handler failed");
        return 0;
    }
    set_color_depth (16);
    if (windowed)
    {
        if (doublesize)
        {
            if (set_gfx_mode(GFX_AUTODETECT_WINDOWED,640,480,0,0)!=0)
            {
                allegro_message("Unable initialize graphics module\n%s\n", allegro_error);
                return 0;
            }
        }
        else
        {
            if (set_gfx_mode(GFX_AUTODETECT_WINDOWED,320,240,0,0)!=0)
            {
                allegro_message("Unable initialize graphics module\n%s\n", allegro_error);
                return 0;
            }
        }
    }
    else
    {
        if (doublesize)
        {
            if (set_gfx_mode(GFX_AUTODETECT_FULLSCREEN,640,480,0,0)!=0)
            {
                allegro_message("Unable initialize graphics module\n%s\n", allegro_error);
                return 0;
            }
        }
        else
        {
            if (set_gfx_mode(GFX_AUTODETECT_FULLSCREEN,320,240,0,0)!=0)
            {
                allegro_message("Unable initialize graphics module\n%s\n", allegro_error);
                return 0;
            }
        }
    }
    if (doublesize)
    {
        buffer = create_bitmap(SCREEN_W / 2, SCREEN_H / 2);
    }
    else
    {
        buffer = create_bitmap(SCREEN_W, SCREEN_H);
    }
    if (!buffer)
    {
        allegro_message ("Error creating background buffer");
        return 0;
    }
    if (install_mouse() == -1)
    {
        allegro_message ("could not install mouse");
        set_gfx_mode (GFX_TEXT, 0, 0, 0, 0);
        allegro_exit();
        return 0;
    }        
    set_window_title ("Mars! - TINS 2003 - Martijn van Iersel");
    if (!resources.init())
    {
        allegro_message ("Could not load all resources!");
        return 0;
    }
    engine.init();   
    menu.init(); 
    text_mode (-1);
    return 1;
}

void Main::setSoundOn(bool value)    
{
    if (value != soundOn)
    {
        soundOn = value;
        if (soundOn && music)
            playMusic ("MMENU");
        else
        {
            if (alspc_player)
            {
                alspc_stop(alspc_player);
                alspc_player = NULL;
            }    
        }        
    }            
}

void Main::setMusicOn(bool value)    
{
    if (value != music)
    {
        music = value;
        if (soundOn && music)
            playMusic ("MMENU");
        else
        {
            if (alspc_player)
            {
                alspc_stop(alspc_player);
                alspc_player = NULL;
            }    
        }        
    }            
}

void Main::setHifiOn(bool value)    
{
    if (value != hifi)
    {
        hifi = value;
        if (soundOn && music)
            playMusic ("MMENU");
    }            
}

void Main::setStereoOn(bool value)    
{
    if (value != stereo)
    {
        stereo = value;
        if (soundOn && music)
            playMusic ("MMENU");
    }            
}

void Main::run()
{
    menu.initStart();
    state = 1;    
    while (state != 2) // state 2 is quit state
    {
        while ((counter - update_counter) > 40) // update 25 times/sec
        {
            if (alspc_player) alspc_poll (alspc_player);
            update_counter += 40;
            int update_start = counter;            
            update ();
            
            // check to see if update takes too long
            if (counter - update_start > 40)
            {
                // we probably can't catch up anymore.
                // move update_counter ahead, or we have to
                // catch up even more next loop
                update_counter = counter;
                break;
            }
        }
        
        draw();
        if (alspc_player) alspc_poll (alspc_player);
        if (fpsOn)
        {
            text_mode (-1);            
            textprintf (buffer, font, 0, 0, 
                  WHITE, "fps: %d", last_fps);
        }
        
        if (doublesize)
            stretch_blit (buffer, screen, 0, 0, buffer->w, buffer->h, 0, 0, SCREEN_W, SCREEN_H);
        else
            blit (buffer, screen, 0, 0, 0, 0, buffer->w, buffer->h);
        
        if ((counter - frame_counter) > 1000)
        {
            last_fps = frame_count;
            frame_count = 0;
            frame_counter = counter;
        }
        frame_count++;

#ifdef ALLEGRO_LINUX
        struct timeval tv = {0, 1};
        select (0, NULL, NULL, NULL, &tv);
#elif defined(ALLEGRO_WINDOWS)
        Sleep(1);
#else
        yield_time_slice();
#endif   
    }            
}

void Main::update()
{
    int result;
    switch (state)
    {
        case 1: // run menu
            menu.update();
            result = menu.getState();
            switch (result)
            {
                case 0: state = 2; break; // quit
                case 1:
                     engine.initGame(); 
                     state = 4; 
                     break; // start new game
                case 2: 
                     engine.resume();
                     state = 4; 
                     break; // resume game
                case -1: break; // continue showing menu
                default: assert (false); // shouldn't occur
            }
            break;
        case 2: // quit
            // do nothing
            break;
        case 4: // game is playing, no matter if at intro screen or play screen
            engine.update();
            result = engine.getState();
            switch (result)
            {
                case 0:
                case 1:
                case 2: 
                case 5:
                case 6:
                     break; // do nothing
                case 3: // resume menu
                     menu.initResume();
                     state = 1;
                     break;
                case 4: // start menu
                     menu.initStart();
                     state = 1;
                     break;
                default:
                     assert (false);
            }
            break;
        default:
            assert (false); // shouldn't occur
    }
    if (key[KEY_F10]) state = 2; // emergency bail out            
    static bool last_key_f12 = true;
    if (key[KEY_F12] && !last_key_f12) { screenshot(); }
    last_key_f12 = key[KEY_F12];

}

void Main::draw()
{
    switch (state)
    {
        case 1:
            menu.draw(buffer);
            break;
        case 2:
            break;
        case 4:
            // draw game
            engine.draw(buffer);
            break;
        default:
            assert (false);
    }
}

Main::~Main()
{
    menu.done();
    if (alspc_player)
    {
        alspc_stop(alspc_player);
        alspc_player = NULL;
    }    
    alspc_uninstall();
    if (buffer) destroy_bitmap (buffer);    
}

int main(int argc, const char *const *argv)
{
    Main m;
    if (m.init(argc, argv))
    {        
        m.run();
    }
    return 0;
} END_OF_MAIN();
