// see gui.h for more detailed comments.
#include "gui.h"
#include <algorithm>
#include "pattern.h"

using namespace std;

/**
constructor of an empty gui.
*/
Gui::~Gui()
{
    list<GuiItem*>::iterator i;
    for (i = guiItems.begin(); i != guiItems.end(); i++)
    {
        delete (*i);
        (*i) = NULL;
    }
}

/**
set the selected item of a group of GuiGroupItem-s.
group is the group of items
index is the index within the group of items to be selected.
The function searches for a groupitem that matches group and index, and selects it.
the previously selected one is automatically deselected.
*/
void Gui::setGroupSelection (int group, int index)
{
    list<GuiItem*>::iterator i;
    for (i = guiItems.begin(); i != guiItems.end(); i++)
    {
        GuiGroupItem *gi = dynamic_cast <GuiGroupItem *>(*i);
        if (gi != NULL &&
            gi->getGroup() == group &&
            gi->getIndex() == index)
        {
            gi->message (GUIMSG_SELECT, 0);
            break; // break from loop
        }
    }
}

/**
find out which item of a group is selected.
*/
int Gui::getGroupSelection (int group)
{
    int result = -1;
    list<GuiItem*>::iterator i;
    for (i = guiItems.begin(); i != guiItems.end(); i++)
    {
        GuiGroupItem *gi = dynamic_cast <GuiGroupItem *>(*i);
        if (gi != NULL &&
            gi->getGroup() == group && gi->isSelected())
        {
            result = gi->getIndex();
            break; // break from loop
        }
    }
    return result;
}

/**
run gui. It examines the input and passes it on to the gui items.
if one of the gui items doesn't return gui_ok, the result is returned.
this can be used to determine if the gui needs to quit.
this has to be called repeatedly.
*/
int Gui::runGui()
{
    int current_mouse_b = mouse_b;
    int current_mouse_x = mouse_x;
    int current_mouse_y = mouse_y;
    int result = GUI_OK;
    if ((current_mouse_b & 1) == 1 && (last_mouse_b & 1) == 0)
    {
        // we have to send a click message
        list<GuiItem*>::iterator i;
        for (i = guiItems.begin(); i != guiItems.end(); i++)
        {
            if ((*i)->contains(current_mouse_x, current_mouse_y))
            {
               int temp = (*i)->message (GUIMSG_CLICK, 0);
               if ((*i)->wantsFocus())
                   setFocus (*i);
               if (temp != GUI_OK) result = temp;
               break; // break from loop
            }
        }
    }
    last_mouse_b = current_mouse_b;
    if (focusItem == NULL) searchNextFocusItem();
    if (keypressed())
    {
        int i = readkey();
        if ((i >> 8) == KEY_TAB)
        {
             searchNextFocusItem();
        }
        if ((i >> 8) == KEY_DOWN)
        {
             searchNextFocusItem();
        }
        if ((i >> 8) == KEY_UP)
        {
             searchPrevFocusItem();
        }
        if (((i >> 8) == KEY_SPACE) || ((i >> 8) == KEY_ENTER))
        {
            int temp = focusItem->message (GUIMSG_CLICK, 0);
            if (temp != GUI_OK) result = temp;
        }        
    }
    return result;
}

void Gui::searchNextFocusItem()
{
    list<GuiItem*>::iterator i;
    int counter = guiItems.size();
    i = find (guiItems.begin(), guiItems.end(), focusItem);
    do
    {
        // go to the next gui item in the sequence
        if (i != guiItems.end()) i++;
        if (i == guiItems.end()) i = guiItems.begin();
        counter--;
    }
    // continue doing that until you find one that wants focus,
    // or you have had them all
    while (counter > 0 && (!(*i)->wantsFocus()));
    if ((*i)->wantsFocus()) setFocus (*i);
}

void Gui::searchPrevFocusItem()
{
    list<GuiItem*>::reverse_iterator i;
    int counter = guiItems.size();
    i = find (guiItems.rbegin(), guiItems.rend(), focusItem);
    do
    {
        // go to the next gui item in the sequence
        if (i != guiItems.rend()) i++;
        if (i == guiItems.rend()) i = guiItems.rbegin();
        counter--;
    }
    // continue doing that until you find one that wants focus,
    // or you have had them all
    while (counter > 0 && (!(*i)->wantsFocus()));
    if ((*i)->wantsFocus()) setFocus (*i);
}

void Gui::setFocus (GuiItem *m)
{
    GuiItem *temp = focusItem;
    focusItem = m;
    if (focusItem != NULL) focusItem->message (GUIMSG_GOT_FOCUS, 0);
    if (temp != NULL) temp->message (GUIMSG_LOST_FOCUS, 0);
}
bool GuiItem::hasFocus ()
{
   if (parent != NULL)
   {
       return (parent->getFocusItem() == this);
   }
   else
       return false;
}

/**
draw gui draw the entire gui, but only items that are dirty.
call repeatedly
*/
void Gui::drawGui(BITMAP *target)
{
    list<GuiItem*>::iterator i;
    if (dirty)
    {
        clear_bitmap (target);
    }
    for (i = guiItems.begin(); i != guiItems.end(); i++)
    {
        // send draw messages to all objects
        // they will redraw if dirty.
        if ((*i)->isDirty() || dirty)
        {
            (*i)->draw (target);
        }
    }
    dirty = false;
    childDirty = false;
}

/**
send a message to all gui items.
c can be used to send an int together with the message
*/
void Gui::broadcastMessage(int msg, int c)
{
    list<GuiItem*>::iterator i;
    for (i = guiItems.begin(); i != guiItems.end(); i++)
    {
        (*i)->message(msg, c);
    }
}

/**
guigroupitem constructor.
*/
GuiGroupItem::GuiGroupItem (BITMAP *newIcon, int newGroup, int newIndex, int newx,
    int newy, int neww, int newh) : GuiItem (newx, newy, neww, newh)
{
     icon = newIcon;
     group = newGroup;
     index = newIndex;
     setDirty();
     selected = false;
     parent = NULL;
}

/**
draw a guigroupitem
*/
void GuiGroupItem::draw(BITMAP *target)
{
    rectfill (target, x, y, x + w - 1, y + h - 1, makecol (0, 0, 0));
    draw_sprite (target, icon, (w - icon->w) / 2 + x, (h - icon->h) / 2 + y);
    if (hasFocus())
    {
        dotted_rect (target, x, y, x + w - 1, y + h - 1, makecol (255, 255, 255));
    }
    else if (selected)
    {
        rect (target, x, y, x + w - 1, y + h - 1, makecol (255, 255, 255));
    }
    dirty = false;
}

void GuiTextGroupItem::draw(BITMAP *target)
{
    rectfill (target, x, y, x + w - 1, y + h - 1, makecol (0, 0, 0));
    textout_centre (target, font, text.c_str(), 
        x + w / 2, y + (h - text_height (font)) / 2, 
        makecol (255, 255, 255));
    if (hasFocus())
    {
        dotted_rect (target, x, y, x + w - 1, y + h - 1, makecol (255, 255, 255));
    }
    else if (selected)
    {
        rect (target, x, y, x + w - 1, y + h - 1, makecol (255, 255, 255));
    }
    dirty = false;
}

/**
message handler of guigroupitem
*/
int GuiGroupItem::message (int msg, int c)
{
    int result = GUI_OK;
    switch (msg)
    {
    case GUIMSG_GOT_FOCUS:
    case GUIMSG_LOST_FOCUS:
        setDirty();
        break;
    case GUIMSG_CLICK:
        if (!selected)
        {
            parent->broadcastMessage(GUIMSG_UNSELECT, group);
            selected = true;
            setDirty();
        }
        break;
    case GUIMSG_UNSELECT:
        if (selected == true && c == group)
        {
            selected = false;
            setDirty();
        }
        break;
    case GUIMSG_SELECT:
        if (selected == false)
        {
            parent->broadcastMessage(GUIMSG_UNSELECT, group);
            selected = true;
            setDirty();
        }
        break;
    }
    return result;
}

/**
add a guiitem to this gui.
*/
void Gui::add(GuiItem *i)
{
    guiItems.push_back (i);
    i->setParent (this);
}

/**
check if the point cx, cy is within the bounding box of the gui item.
usefull for checking if the mouse is over the gui item.
*/
bool GuiItem::contains (int cx, int cy)
{
    bool result = (cx >= x && cy >= y && cx < (x + w) && cy < (y + h));
    return result;
}

/**
draw the guibutton
*/
void GuiButton::draw(BITMAP *target)
{
    draw_sprite (target, icon, x, y);
    if (hasFocus())
        dotted_rect (target, x, y, x + w - 1, y + h - 1, makecol (255, 255, 255));
    else
        rect (target, x, y, x + w - 1, y + h - 1, makecol (0, 0, 0));
    dirty = false;
}

void GuiTextButton::draw(BITMAP *target)
{
    textout_centre (target, font, text.c_str(), 
        x + w / 2, y + (h - text_height (font)) / 2, 
        makecol (255, 255, 255));
    if (hasFocus())
        dotted_rect (target, x, y, x + w - 1, y + h - 1, makecol (255, 255, 255));
    else
        rect (target, x, y, x + w - 1, y + h - 1, makecol (0, 0, 0));
    dirty = false;
}

/**
guibutton message handler
*/
int GuiButton::message (int msg, int c)
{
    switch (msg)
    {
      case GUIMSG_CLICK:
        return action;
      case GUIMSG_GOT_FOCUS:
      case GUIMSG_LOST_FOCUS:
      default:
        setDirty();
        return GUI_OK;
    }
    
}

/**
draw a guitext
*/
void GuiText::draw(BITMAP *target)
{
    textout (target, f, text.c_str(), x, y, color);
    dirty = false;
}

/**
guitext message handler
*/
int GuiText::message (int msg, int c)
{
    return GUI_OK;
}


void GuiItem::setDirty ()
{
    dirty = true; 
    if (parent) parent->setChildDirty(); 
}
