#include <stdio.h>

#define ALLEGRO_INCLUDE_MATH_H

#include "allegro.h"
#include "scrshot.h"

#ifndef M_PI
   #define M_PI   3.14159
#endif

/* parameters controlling the camera and projection state */
int fov = 48;
float aspect = 1;
float distance = 400;
float xtarget;
float ytarget;
float base_offset = 80;
int pitch = 64; // look right at the top of the board
int rotation = 0;

const int board_max = 3;

const int BOARD_TRANSLATE = 1, BOARD_ROTATE = 2,
    CAMERA_ROTATE = 4, CAMERA_PITCH = 8;

typedef struct ANIMATION_INFO {
    int flags;
    int board;
    int dx, dy, ex, ey; // transformation board
    int brot_d, brot_e; // rotation board
    int crot_d, crot_e; // camera rotation
    int pitch_d, pitch_e; // pitch change
} ANIMATION_INFO;

ANIMATION_INFO a_info;

typedef struct BOARD
{
    bool enabled;
    int w, h;
    int x, y;
    int tx, ty; // texture coords
    int r;
} BOARD;

BOARD board [board_max];


void getCoords (BOARD *board, MATRIX_f *camera,
    int &vc, V3D_f *vultimate[8])
{
   V3D_f _v[4], _vout[8], _vtmp[8];
   V3D_f *v[4], *vout[8], *vtmp[8];
   int flags[4], out[8];
   int c;

   for (c=0; c<4; c++)
      v[c] = &_v[c];

   for (c=0; c<8; c++) {
      vtmp[c] = &_vtmp[c];
      vout[c] = &_vout[c];
   }

   v[0]->x = - board->w / 2;
   v[0]->y = - board->h / 2;
   v[0]->z = 0;
   v[0]->u = board->tx;
   v[0]->v = board->ty;

   v[3]->x = board->w / 2;
   v[3]->y = - board->h / 2;
   v[3]->z = 0;
   v[3]->u = board->tx + board->w;
   v[3]->v = board->ty;

   v[2]->x = board->w / 2;
   v[2]->y = board->h / 2;
   v[2]->z = 0;
   v[2]->u = board->tx + board->w;
   v[2]->v = board->ty + board->h;

   v[1]->x = -board->w / 2;
   v[1]->y = board->h / 2;
   v[1]->z = 0;
   v[1]->u = board->tx;
   v[1]->v = board->ty + board->h;

   MATRIX_f rot_matrix;
   get_z_rotate_matrix_f (&rot_matrix, board->r);
   qtranslate_matrix_f (&rot_matrix,
       board->x + board->w / 2,
       board->y + board->h / 2, 0);
   for (c = 0; c < 4; c++)
   {
       apply_matrix_f(&rot_matrix,
           v[c]->x, v[c]->y, v[c]->z, &v[c]->x, &v[c]->y, &v[c]->z);
   }

   for (c=0; c<4; c++) {
      apply_matrix_f(camera, v[c]->x, v[c]->y, v[c]->z, &v[c]->x, &v[c]->y, &v[c]->z);

      flags[c] = 0;

      if (v[c]->x < -v[c]->z)
	 flags[c] |= 1;
      else if (v[c]->x > v[c]->z)
	 flags[c] |= 2;

      if (v[c]->y < -v[c]->z)
	 flags[c] |= 4;
      else if (v[c]->y > v[c]->z)
	 flags[c] |= 8;

      if (v[c]->z < 0.1)
	 flags[c] |= 16;
   }

   if (flags[0] & flags[1] & flags[2] & flags[3])
      return;

   if (flags[0] | flags[1] | flags[2] | flags[3]) {
      vc = clip3d_f(POLYTYPE_PTEX, 0.1, 0.1, 4, (AL_CONST V3D_f **)v, vout, vtmp, out);

      if (vc <= 0)
	 return;
   }
   else {
      vout[0] = v[0];
      vout[1] = v[1];
      vout[2] = v[2];
      vout[3] = v[3];

      vc = 4;
   }

   for (c=0; c<vc; c++)
      persp_project_f(vout[c]->x, vout[c]->y, vout[c]->z, &vout[c]->x, &vout[c]->y);

   for (c = 0; c < vc; c++)
   {
       *(vultimate[c]) = *(vout[c]);
   }
}


/* draw everything */
void render(BITMAP *bmp, BITMAP *texture)
{
//   char buf[80];
   MATRIX_f camera;
   int x, y, w, h;
   float xfront, yfront, zfront;
   float xup, yup, zup;
   float xpos, ypos, zpos;

   /* clear the background */
//   clear(bmp);

   /* set up the viewport region */
   x = 0;
   y = 0;
   w = SCREEN_W;
   h = SCREEN_H;

   set_projection_viewport(x, y, w, h);
   set_clip(bmp, x, y, x+w-1, y+h-1);

   float base = distance * cos(pitch);

   xpos = - base * sin (rotation);
   ypos = base * cos (rotation);
   zpos = - distance * sin (pitch);

   xfront = -xpos;
   yfront = -ypos;
   zfront = -zpos;

   xpos += xtarget;
   ypos += ytarget;

   base = - sin (pitch);
   xup = - base * sin (rotation);
   yup = base * cos (rotation);
   zup = - cos (pitch);

/*   get_vector_rotation_matrix_f(&roller, xfront, yfront, zfront, roll*128.0/M_PI);
   apply_matrix_f(&roller, 0, -1, 0, &xup, &yup, &zup);
  */

   /* build the camera matrix */
   get_camera_matrix_f(&camera,
		       xpos, ypos, zpos,        /* camera position */
		       xfront, yfront, zfront,  /* in-front vector */
		       xup, yup, zup,           /* up vector */
		       fov,                     /* field of view */
		       aspect);                 /* aspect ratio */


   int i;
   float maxx = bmp->w / 2, minx = maxx,
        maxy = bmp->h / 2, miny = maxy;
   for (i = 0; i < board_max; i++)
   {

       if (board[i].enabled)
       {
           V3D_f _vultimate[8];
           V3D_f *vultimate[8];
           int vc;
        
           int c;
           for (c=0; c<8; c++) {
              vultimate[c] = &_vultimate[c];
           }

           getCoords (&board[i], &camera, vc, vultimate);

           if (i == 0)
               for (c = 0; c < vc; c++)
               {
                   if (vultimate[c]->x < minx) minx = vultimate[c]->x;
                   if (vultimate[c]->x > maxx) maxx = vultimate[c]->x;
                   if (vultimate[c]->y < miny) miny = vultimate[c]->y;
                   if (vultimate[c]->y > maxy) maxy = vultimate[c]->y;
               }
        
           polygon3d_f(bmp, POLYTYPE_PTEX, texture, vc, vultimate);
        }
   }

   if (maxx > bmp->w) distance /= 0.98;
   if (maxy > bmp->h) distance /= 0.98;
   if (minx < 0) distance /= 0.98;
   if (miny < 0) distance /= 0.98;
/*
   set_clip(bmp, 0, 0, bmp->w, bmp->h);
   text_mode(-1);
   char buf[80];
   sprintf(buf, "Field of view: %d (f/F changes)", fov);
   textout(bmp, font, buf, 0, 16, 120);
   sprintf(buf, "Aspect ratio: %.2f (a/A changes)", aspect);
   textout(bmp, font, buf, 0, 24, 120);
   sprintf(buf, "Base Offset: %.2f (b/B changes)", base_offset);
   textout(bmp, font, buf, 0, 32, 120);
   sprintf(buf, "Distance: %.2f (d/D changes)", distance);
   textout(bmp, font, buf, 0, 48, 120);
   sprintf(buf, "Pitch: %d (pgup/pgdn changes)", pitch);
   textout(bmp, font, buf, 0, 64, 120);
   sprintf(buf, "Rotation: %d (r/R changes)", rotation);
   textout(bmp, font, buf, 0, 72, 120);
   sprintf(buf, "Front vector: %.2f, %.2f, %.2f", xfront, yfront, zfront);
   textout(bmp, font, buf, 0, 80, 120);
   sprintf(buf, "Up vector: %.2f, %.2f, %.2f", xup, yup, zup);
   textout(bmp, font, buf, 0, 88, 120);
   sprintf(buf, "Flags, %d", a_info.flags);
   textout(bmp, font, buf, 0, 104, 120);
*/
}



/* deal with user input */
void process_input()
{
   poll_keyboard();

   if (key[KEY_F]) {
      if (key_shifts & KB_SHIFT_FLAG) {
	 if (fov < 96)
	    fov++;
      }
      else {
	 if (fov > 16)
	    fov--;
      }
   }

   if (key[KEY_A]) {
      if (key_shifts & KB_SHIFT_FLAG) {
	 aspect += 0.05;
	 if (aspect > 2)
	    aspect = 2;
      }
      else {
	 aspect -= 0.05;
	 if (aspect < .1)
	    aspect = .1;
      }
   }

   if (key[KEY_B]) {
      if (key_shifts & KB_SHIFT_FLAG)
	 base_offset += 2;
      else
	 base_offset -= 2;
   }

   if (key[KEY_D]) {
      if (key_shifts & KB_SHIFT_FLAG)
	 distance += 5;
      else
	 distance -= 5;
   }

   if (key[KEY_PGUP])
	 pitch -= 2;

   if (key[KEY_PGDN])
	 pitch += 2;

   if (key[KEY_R]) {
      if (key_shifts & KB_SHIFT_FLAG) {
	    rotation += 2;
      }
      else {
	    rotation -= 2;
      }
   }

}

int do_animation ()
{
    if (a_info.flags == 0) return 0;

    bool sign;
    if (a_info.flags & BOARD_TRANSLATE)
    {
        sign = ((board[a_info.board].x - a_info.ex) > 0);

        board[a_info.board].x += a_info.dx;
        if (sign != ((board[a_info.board].x - a_info.ex) > 0))
        {
            a_info.dx = 0;
            board[a_info.board].x = a_info.ex;
        }
        sign = ((board[a_info.board].y - a_info.ey) > 0);
        board[a_info.board].y += a_info.dy;
        if (sign != ((board[a_info.board].y - a_info.ey) > 0))
        {
            a_info.dy = 0;
            board[a_info.board].y = a_info.ey;
        }
        if ((a_info.dx == 0) && (a_info.dy == 0)) a_info.flags -= BOARD_TRANSLATE;
    }

    if (a_info.flags & BOARD_ROTATE)
    {
        sign = ((board[a_info.board].r - a_info.brot_e) > 0);
        board[a_info.board].r += a_info.brot_d;
        if (sign != ((board[a_info.board].r - a_info.brot_e) > 0))
        {
            a_info.flags -= BOARD_ROTATE;
            board[a_info.board].r = a_info.brot_e;
        }
    }
    if (a_info.flags & CAMERA_ROTATE)
    {
        sign = ((rotation - a_info.crot_e) > 0);
        rotation += a_info.crot_d;
        if (sign != ((rotation - a_info.crot_e) > 0))
        {
            a_info.flags -= CAMERA_ROTATE;
            rotation = a_info.crot_e;
        }
    }
    if (a_info.flags & CAMERA_PITCH)
    {
        sign = ((pitch - a_info.pitch_e) > 0);
        pitch += a_info.pitch_d;
        if (sign != ((pitch - a_info.pitch_e) > 0))
        {
            a_info.flags -= CAMERA_PITCH;
            pitch = a_info.pitch_e;
        }
    }
    return a_info.flags;
}

void board_move_to (int b, int x, int y, int step)
{
    a_info.board = b;

    if (board[b].x != x)
    {
        a_info.flags |= BOARD_TRANSLATE;
    
        a_info.ex = x;
        a_info.dx = (x - board[b].x) / step;
        if (a_info.dx == 0)
            a_info.dx = (x - board[b].x) > 0 ? 1 : -1;
    }

    if (board[b].y != y)
    {
        a_info.flags |= BOARD_TRANSLATE;

        a_info.ey = y;
        a_info.dy = (y - board[b].y) / step;
        if (a_info.dy == 0)
            a_info.dy = (y - board[b].y) > 0 ? 1 : -1;
    }
}

void board_rotate_to (int b, int r, int step)
{
    if (board[b].r != r)
    {
        a_info.flags |= BOARD_ROTATE;
    
        a_info.board = b;
    
        a_info.brot_e = r;
        a_info.brot_d = (r - board[b].r) / step;
        if (a_info.brot_d == 0)
            a_info.brot_d = (r - board[b].r) > 0 ? 1 : -1;
    }
}

void camera_pitch_to (int r, int step)
{
    if (r != pitch)
    {
        a_info.flags |= CAMERA_PITCH;
    
        a_info.pitch_e = r;
        a_info.pitch_d = (r - pitch) / step;
        if (a_info.pitch_d == 0)
            a_info.pitch_d = (r - pitch) > 0 ? 1 : -1;
    }
}

void camera_rotate_to (int r, int step)
{
    if (r != rotation)
    {
        a_info.flags |= CAMERA_ROTATE;
    
        a_info.crot_e = r;
        a_info.crot_d = (r - rotation) / step;
        if (a_info.crot_d == 0)
            a_info.crot_d = (r - rotation) > 0 ? 1 : -1;
    }
}

void swap_board_in (int step)
{
    camera_rotate_to (0, step);
    camera_pitch_to (32, step); // angle of 45 degr. above board
}

void swap_board_out (int step)
{
    camera_pitch_to (192, step); // face backside
    camera_rotate_to (0, step);
}

void swap_card_in (int x, int y, int step)
{
    board[2].x = (x < 0) ? -500 : 500;
    board[2].y = (y < 0) ? -500 : 500;
    board[2].r = 128;
    board_rotate_to (2, 0, step);
    board_move_to (2, x, y, step);
}

void swap_card_out (int step)
{
    board_move_to (2,
       (board[2].x < 0) ? -500 : 500,
       (board[2].y < 0) ? -500 : 500, step);
    board_rotate_to (2, 128, step);
}

void make_board (int tx_size)
{
     int i;
     for (i = 0; i < board_max; i++)
     {
          board[i].enabled = FALSE;
     }
     board[0].enabled = TRUE;
     board[0].x = 0;
     board[0].y = 0;
     board[0].tx = 0;
     board[0].ty = 0;
     board[0].w = tx_size;
     board[0].h = tx_size;
     board[0].r = 0;
}

void make_split (int w0, int h0, int x1, int y1, int w1, int h1,
    int w2, int h2, int x2, int y2)
{
     board[0].enabled = TRUE;
     board[0].x = 0;
     board[0].y = 0;
     board[0].tx = 0;
     board[0].ty = 0;
     board[0].w = w0;
     board[0].h = h0;
     board[0].r = 0;
     board[1].enabled = TRUE;
     board[1].x = x1;
     board[1].y = y1;
     board[1].tx = x1;
     board[1].ty = y1;
     board[1].w = w1;
     board[1].h = h1;
     board[1].r = 0;
     board[2].enabled = TRUE;
     board[2].x = x2;
     board[2].y = y2;
     board[2].tx = 512-w2;
     board[2].ty = 512-h2;
     board[2].w = w2;
     board[2].h = h2;
     board[2].r = 0;
}

void make_card (int tx_size, int x, int y)
{
     board[2].enabled = TRUE;
     board[2].x = x;
     board[2].y = y;
     board[2].tx = 512 - tx_size;
     board[2].ty = 512 - tx_size;
     board[2].w = tx_size;
     board[2].h = tx_size;
     board[2].r = 0;
}

void init_3D (int map_size)
{
    a_info.flags = 0;
    xtarget = map_size * 24;
    ytarget = map_size * 24;

}
