#include <assert.h>
#include "main.h"

// needed for "Sleep()"
#ifdef ALLEGRO_WINDOWS
#include <winalleg.h>
#endif

volatile int Main::counter = 0;

void timer_handler()
{
	Main::counter += 10;
}

END_OF_FUNCTION(timer_handler);

void Main::playMusic (const char *id)
{
	if (!(settings.music && settings.soundOn)) return;
	ALSPC_DATA *alspc_data = resources.getSPC (id);
	assert (alspc_data); // fails if requested SPC is not in datafile.	
	if (alspc_player)
	{
		alspc_stop(alspc_player);
		alspc_player = NULL;
	}
	alspc_player = alspc_start (alspc_data, settings.hifi ? 44100 : 22050, 255, 128, settings.stereo, settings.hifi);
}

void Main::playMusic (ALSPC_DATA *alspc_data)
{
	if (!(settings.music && settings.soundOn)) return;	
	assert (alspc_data); 
	if (alspc_player)
	{
		alspc_stop(alspc_player);
		alspc_player = NULL;
	}
	alspc_player = alspc_start (alspc_data, settings.hifi ? 44100 : 22050, 255, 128, settings.stereo, settings.hifi);
}

void Main::playSample (SAMPLE *s)
{
	if (!settings.soundOn) return;
	play_sample (s, 127, 127, 1000, 0);
}

Main::Main()  : menu (this), engine (this), alspc_player (NULL), update_speed(20)
{

		cheatOn = false;
	cheatUnlocked = false;
		
	buffer = NULL;
	last_fps = 0;
	update_counter = 0;
	frame_count = 0;
	frame_counter = 0;
}

int Main::init(int argc, const char *const *argv)
// returns 1 on success, 0 on failure
{
	LOCK_VARIABLE(Main::counter);
	LOCK_FUNCTION(timer_handler);
	if (allegro_init () < 0)
	{
		allegro_message ("init failed");
		return 0;
	}
	
	override_config_file ("xelda.cfg");
		settings.getFromConfig();
		settings.getFromArgs (argc, argv);
	
	if (install_keyboard () < 0)
	{
		allegro_message ("install keyboard failed");
		return 0;
	}
		initKeyboard(); // install custom keyboard handler
		
	if (install_timer() < 0)
	{
		allegro_message ("install timer failed");
		return 0;
	}
	set_volume_per_voice (1);	
	if (settings.soundOn && install_sound(DIGI_AUTODETECT, MIDI_NONE, NULL) < 0)
	{
		// could not get sound to work
		settings.soundOn = false;
		allegro_message ("Could not initialize sound. Sound is turned off.\n%s\n", allegro_error);
	}
	alspc_install();	
	if (install_int (timer_handler, 10) < 0)
	{
		allegro_message ("installation of timer handler failed");
		return 0;
	}
	set_color_depth (16);
	if (settings.windowed)
	{
		if (settings.doublesize)
		{
			if (set_gfx_mode(GFX_AUTODETECT_WINDOWED,640,480,0,0)!=0)
			{
				allegro_message("Unable initialize graphics module\n%s\n", allegro_error);
				return 0;
			}
		}
		else
		{
			if (set_gfx_mode(GFX_AUTODETECT_WINDOWED,320,240,0,0)!=0)
			{
				allegro_message("Unable initialize graphics module\n%s\n", allegro_error);
				return 0;
			}
		}
	}
	else
	{
		if (settings.doublesize)
		{
			if (set_gfx_mode(GFX_AUTODETECT_FULLSCREEN,640,480,0,0)!=0)
			{
				allegro_message("Unable initialize graphics module\n%s\n", allegro_error);
				return 0;
			}
		}
		else
		{
			if (set_gfx_mode(GFX_AUTODETECT_FULLSCREEN,320,240,0,0)!=0)
			{
				allegro_message("Unable initialize graphics module\n%s\n", allegro_error);
				return 0;
			}
		}
	}
	if (settings.doublesize)
	{
		buffer = create_bitmap(SCREEN_W / 2, SCREEN_H / 2);
	}
	else
	{
		buffer = create_bitmap(SCREEN_W, SCREEN_H);
	}
	if (!buffer)
	{
		allegro_message ("Error creating background buffer");
		return 0;
	}
	if (install_mouse() == -1)
	{
		allegro_message ("could not install mouse");
		set_gfx_mode (GFX_TEXT, 0, 0, 0, 0);
		allegro_exit();
		return 0;
	}		
	set_window_title ("Xelda - TINS 2005 - Max & Amarillion");
	teg_install();
	if (!resources.init())
	{
		allegro_message ("Could not load all resources!");
		return 0;
	}
	engine.init();   
	menu.init(); 
	return 1;
}

void Main::setSoundOn(bool value)	
{
	if (value != settings.soundOn)
	{
		settings.soundOn = value;
		if (settings.soundOn && settings.music)
			playMusic ("MMENU");
		else
		{
			if (alspc_player)
			{
				alspc_stop(alspc_player);
				alspc_player = NULL;
			}	
		}		
	}			
}

void Main::setMusicOn(bool value)	
{
	if (value != settings.music)
	{
		settings.music = value;
		if (settings.soundOn && settings.music)
			playMusic ("MMENU");
		else
		{
			if (alspc_player)
			{
				alspc_stop(alspc_player);
				alspc_player = NULL;
			}	
		}		
	}			
}

void Main::setHifiOn(bool value)	
{
	if (value != settings.hifi)
	{
		settings.hifi = value;
		if (settings.soundOn && settings.music)
			playMusic ("MMENU");
	}			
}

void Main::setStereoOn(bool value)	
{
	if (value != settings.stereo)
	{
		settings.stereo = value;
		if (settings.soundOn && settings.music)
			playMusic ("MMENU");
	}			
}

void Main::run()
{
	menu.initStart();
	state = 1;	
	while (state != 2) // state 2 is quit state
	{
		while ((counter - update_counter) > update_speed) // update 25 times/sec
		{
			if (alspc_player) alspc_poll (alspc_player);
			update_counter += update_speed;
			int update_start = counter;			
			update ();
			
			// check to see if update takes too long
			if (counter - update_start > 40)
			{
				// we probably can't catch up anymore.
				// move update_counter ahead, or we have to
				// catch up even more next loop
				update_counter = counter;
				break;
			}
		}
		
		draw();
		if (alspc_player) alspc_poll (alspc_player);
		if (settings.fpsOn)
		{
			textprintf_ex (buffer, font, 0, 0, 
				  WHITE, -1, "fps: %d", last_fps);
		}
		
		if (settings.doublesize)
			stretch_blit (buffer, screen, 0, 0, buffer->w, buffer->h, 0, 0, SCREEN_W, SCREEN_H);
		else
			blit (buffer, screen, 0, 0, 0, 0, buffer->w, buffer->h);
		
		if ((counter - frame_counter) > 1000)
		{
			last_fps = frame_count;
			frame_count = 0;
			frame_counter = counter;
		}
		frame_count++;

#ifdef ALLEGRO_LINUX
		struct timeval tv = {0, 1};
		select (0, NULL, NULL, NULL, &tv);
#elif defined(ALLEGRO_WINDOWS)
		Sleep(1);
#else
		yield_time_slice();
#endif   
	}			
}

void Main::update()
{
	int result;
	switch (state)
	{
		case 1: // run menu
			menu.update();
			result = menu.getState();
			switch (result)
			{
				case 0: state = 2; break; // quit
				case 1:
					 engine.initGame(); 
					 state = 4; 
					 break; // start new game
				case 2: 
					 engine.resume();
					 state = 4; 
					 break; // resume game
				case -1: break; // continue showing menu
				default: assert (false); // shouldn't occur
			}
			break;
		case 2: // quit
			// do nothing
			break;
		case 4: // game is playing, no matter if at intro screen or play screen
			engine.update();
			result = engine.getState();
			switch (result)
			{
				case 0:
				case 1:
				case 2: 
				case 5:
				case 6:
				case 7:
				case 8:
				case 9:
					 break; // do nothing
				case 3: // resume menu
					 menu.initResume();
					 state = 1;
					 break;
				case 4: // start menu
					 menu.initStart();
					 state = 1;
					 break;
				default:
					 assert (false);
			}
			break;
		default:
			assert (false); // shouldn't occur
	}
	if (key[KEY_F10]) state = 2; // emergency bail out
}

void Main::draw()
{
	switch (state)
	{
		case 1:
			menu.draw(buffer);
			break;
		case 2:
			break;
		case 4:
			// draw game
			engine.draw(buffer);
			break;
		default:
			assert (false);
	}
}

Main::~Main()
{
	menu.done();
	if (alspc_player)
	{
		alspc_stop(alspc_player);
		alspc_player = NULL;
	}	
	alspc_uninstall();
	teg_uninstall();
	if (buffer) destroy_bitmap (buffer);	
}

int main(int argc, const char *const *argv)
{
	Main m;
	if (m.init(argc, argv))
	{		
		m.run();
	}
	return 0;
} END_OF_MAIN();
