#include "object.h"
#include "engine.h"
#include "anim.h"
#include "player.h"
#include "monster.h"


TEG_MAP *Object::getMap()
{ 
	return parent->getParent()->getMap(); 
}

// tilestack -> check the tiles in all 3 layers at the same time
int Object::getTileStackFlags(int mx, int my)
{
	int result = 0;
	TEG_MAP *map = getMap();
	if (!map) return 0; // no map found !!!
	
	if (mx < 0 || my < 0 || mx >= map->w || my >= map->h)
	{
		return 0;
	}
	else
	{
		int i1, i2, i3, f1, f2, f3;
		i1 = teg_mapget (map, 0, mx, my);
		i2 = teg_mapget (map, 1, mx, my);
		i3 = teg_mapget (map, 2, mx, my);		
		if (i1 >= 0) f1 = map->tilelist->tiles[i1].flags; else f1 = 0;
		if (i2 >= 0) f2 = map->tilelist->tiles[i2].flags; else f2 = 0;
		if (i3 >= 0) f3 = map->tilelist->tiles[i3].flags; else f3 = 0;
		
		// check for solids
		if (f1 == 1 || f2 == 1 || f3 == 1) result |= TS_SOLID;
		if (f1 == 2 || f2 == 2 || f3 == 2) result |= (TS_SOFT | TS_SOLID);
		if (f1 == 3 || f2 == 3 || f3 == 3) result |= (TS_BRIDGE | TS_SOLID);
	
		return result;
	}
}


void Object::setAnim (Anim *a)
{
	current = a;
	frame = 0;
	animcounter = 0;
	sprdx = a->getFrame(0)->hotx;
	sprdy = a->getFrame(0)->hoty;
	w = a->getFrame(0)->w;
	h = a->getFrame(0)->h;
}

Resources *Object::getResources()
{
	if (parent)
		return parent->getParent()->getResources();
	else
		return NULL;
	
}

Object::Object ()
{
	current = NULL;
	frame = 0;
	animcounter = 0;
	x = 0;
	y = 0;
	w = 8;
	h = 8;
	jump_dy = 0;
	sprdx = 0;
	sprdy = 0;
	visible = false;
	alive = true;
	solid = true;
	parent = NULL;
	hasGravity = false;
}

void Object::setParent (Objects *p)
{
	parent = p;
}

RLE_SPRITE *Object::getSprite ()
{
	if (!current) return NULL;
	return current->getFrameRle (frame);
}

void Object::try_move (fix dx, fix dy)
{
	int dxleft = dx, dyleft= dy;
	int ddx = dx > 0 ? 1 : -1;
	int ddy = dy > 0 ? 1 : -1;	
	int trydx, trydy;
	bool valid = true;
	while ((abs(dxleft) > 0 || abs (dyleft) > 0) && valid)
	{
		if (abs(dxleft) > abs(dyleft))
		{
			trydy = 0;
			if (abs(dxleft) >= 1)
				trydx = ddx;
			else
				trydx = dxleft;
		}
		else
		{
			trydx = 0;
			if (abs(dyleft) >= 1)
				trydy = ddy;
			else
				trydy = dyleft;
		}

		// check with tilemap background, but only if object is solid.
		if (solid)
		{
			
			// check if (x +  |trydx, y + trydy) is valid
			int mx1, my1, mx2, my2;
			int ix, iy;
			TEG_MAP *map = getMap();
			mx1 = ((int)x + trydx) / map->tilelist->tilew;
			my1 = ((int)y + trydy) / map->tilelist->tileh;
			mx2 = ((int)x + trydx + w - 1) / map->tilelist->tilew;
			my2 = ((int)y + trydy + h - 1) / map->tilelist->tileh;
					
			// loop through all map positions we touch with the solid region
			for (ix = mx1; ix <= mx2; ++ix)
			{
				for (iy = my1; iy <= my2; ++iy)
				{
					// see if there is a solid tile at this position
					if (getTileStackFlags (ix, iy) & TS_SOLID)
					{
							valid = false;
					}
				}
			}
		}
		
		if (valid)
		{
			x += trydx;
			dxleft -= trydx;
			y += trydy;
			dyleft -= trydy;
		}		
	}
}
   
void Object::setLocation (fix nx, fix ny)
{
	x = nx;
	y = ny;
}

void Objects::killAll()
{
	list<Object*>::iterator i;
	for (i = objects.begin(); i != objects.end(); ++i)
	{
		delete (*i);
		(*i) = NULL;
	}
	objects.clear();	
}

void Object::update()
{
	assert (parent);
	if (!current) return;
	
	// ask current time
	int counter = parent->getParent()->getCounter();
	
	// if we're lagging behind too much, catch up at once
	if (counter - animcounter > 1000) animcounter = counter;
	
	// if frametime is 0, loop same frame forever
	if (current->getFrameTime (frame) > 0) 
	// otherwise, see if we need to go to the next frame.
	{
		while (counter - animcounter > current->getFrameTime (frame))
		{
			animcounter += current->getFrameTime (frame);
			frame++;
			//allegro_message ("%i, %i", frame, current->getFrameCount());
			if (frame >= current->getFrameCount()) frame = 0;
		}
	}	
		
	// find out what block is supporting player
	int mx = x + 1;
	int my = y + h + 1;
	TEG_MAP *map = getMap();
	supportType = getTileStackFlags (mx / map->tilelist->tilew, my / map->tilelist->tileh);
	mx = x + w - 1;
	my = y + h + 1;
	supportType |= getTileStackFlags (mx / map->tilelist->tilew, my / map->tilelist->tileh);

	if (hasGravity)
	{
		if (solid && (supportType & TS_SOLID))
		{
			if (jump_dy > 0) jump_dy = 0;
		}
		else
		{
			if (jump_dy < 8) jump_dy += 0.5;
		}
		try_move (fix(0), jump_dy);
	}
}


void Objects::add(Object *o)
{
	objects.push_back (o);
	o->setParent (this);
}

class MyObjectRemover
{
   public:
	  bool operator()(Object *o)
	  {
		 if (!o->isAlive())
		 {
			delete o;
			return 1;
		 }
		 return 0;
	  }
};

void Objects::update()
{
	list<Object*>::iterator i;
	for (i = objects.begin(); i != objects.end(); i++)
	{
		if ((*i)->isAlive()) (*i)->update();
	}
	
	// collission detection!	
	list<Object*>::iterator j;
	for (i = objects.begin(); i != objects.end(); i++)
		for (j = objects.begin(); j != i; j++)
	{
		// see if bb interesect
		if ((*i)->isAlive() && (*j)->isAlive())
		{
			int x1 = (*i)->getx();
			int y1 = (*i)->gety();
			int w1 = (*i)->getw();
			int h1 = (*i)->geth();
			int x2 = (*j)->getx();
			int y2 = (*j)->gety();
			int w2 = (*j)->getw();
			int h2 = (*j)->geth();
			if(!((x1 >= x2+w2) || (x2 >= x1+w1) || (y1 >= y2+h2) || (y2 >= y1+h1)))
			{
				(*i)->handleCollission ((*j));
				(*j)->handleCollission ((*i));
			}
		}		
	}
	
	// remove all that are not alive!
	objects.remove_if (MyObjectRemover());
}

void Objects::draw (BITMAP *buffer, int xofst, int yofst)
{
	list<Object*>::iterator i;
	for (i = objects.begin(); i != objects.end(); i++)
	{
		if ((*i)->isVisible() && (*i)->isAlive())
		{
			RLE_SPRITE *rle = (*i)->getSprite();
			if (rle)
			{
				draw_rle_sprite (buffer, (*i)->getSprite(), 
					(*i)->getSprx() - xofst, (*i)->getSpry() -yofst);
			}
			if (parent->isDebug())
			{
				rect (buffer, 
					(*i)->getx() - xofst, 
					(*i)->gety() - yofst,
					(*i)->getx() - xofst + (*i)->getw(), 
					(*i)->gety() - yofst + (*i)->geth(),
					GREEN);
			}			
		}
	}
}

