#include "engine.h"
#include <assert.h>
#include "main.h"
#include "object.h"
#include "player.h"
#include "monster.h"
#include <stdio.h>
#include "cutscene.h"

Resources* Engine::getResources () 
{ return parent->getResources(); }

void Engine::playMusic (const char *id) 
{ parent->playMusic (id); }

void Engine::playSample (SAMPLE *s) 
{ parent->playSample (s); }

void Engine::playMusic (ALSPC_DATA *alspc_data) 
{ parent->playMusic (alspc_data); }

int Engine::getCounter () 
{ return parent->getCounter(); }

bool Engine::isCheatOn ()
{
	return parent->isCheatOn();
}

// fill up target bmp with the same map over and over
void teg_draw_tiled_rle (BITMAP *bmp, const TEG_MAP* map, int layer, int xview, int yview, int frame)
{
	int x, y;
	int tilex, tiley;
	int tilew, tileh;
	tileh = map->tilelist->tileh;
	tilew = map->tilelist->tilew;
	tiley = (yview / tileh) % map->h;
	while (tiley < 0) tiley += map->h;
	for (y = -(yview % tileh); y < bmp->h; y += tileh)
	{
		
		//~ int tempx = ((xview/tilew) + 1);		
		//~ tilex = map->w - (tempx % map->w) - 1;
		tilex = (xview / tilew) % map->w;
		while (tilex < 0) tilex += map->w;
		for (x = - (xview % tilew) ; x < bmp->w; x += tilew)
		{
			RLE_SPRITE *s;
			int i;
			i = teg_mapget(map, layer, tilex, tiley);
			if (i >= 0 && i < map->tilelist->tilenum)
			{
					i += frame * map->tilelist->tilenum;
				s = map->tilelist->tiles[i].rle;
				if (s != NULL)
					draw_rle_sprite (bmp, s, x, y);
			}
			tilex++;
			if (tilex >= map->w) tilex = 0;
		}
		tiley++;
		if (tiley >= map->h) tiley = 0;
	}
}

Engine::Engine (Main *p) : testview(this), objects (this), level (NULL)
{
	parent = p;
	player = NULL;
	debug = false;
}

void Engine::doneLevel()
{
	objects.killAll();
}

void Engine::die()
{
	state = 6;
	outroTimer = 40;
}

void Engine::init()
{
	Player::init(parent->getResources());	
	Monster::init(parent->getResources());	
	ExitObject::init(parent->getResources());
	Bullet::init(parent->getResources());
	
	levelset = parent->getResources()->getLevelset ("xelda");
	srand(time(0));
}

void Engine::initGame ()
{
	// initialize game and player stats.
	// to start a new game.
	state = 9; // show cutscene
	cutscene = parent->getResources()->getCutScene ("intro");
	cutscene->init();
	introTimer = 40;
	currentLevel = 0;
	lives = 3;
	health = 100;
	initLevel();
}

void Engine::initLevel()
{
	parent->playMusic (levelset->levels[currentLevel].music);
	// initialize objects
	//vector <ObjectInfo>::iterator i;
	player = NULL;
	
	// look for objects in layer 3
	
	// make a copy of level, delete old one if necessary
	if (level) teg_destroymap (level);	
	level = teg_create_copy (levelset->levels[currentLevel].map);
	
	// no need to copy background, doesn't change
	background = levelset->levels[currentLevel].background;
	
	int x, y;
	for (x = 0; x < level->w; ++x)
	{
		for (y = 0; y < level->h; ++y)
		{
			int t = teg_mapget (level, 3, x, y);
			if (t >= 0 && t < level->tilelist->tilenum)
			{
				int type = level->tilelist->tiles[t].flags;
				Object *o = NULL;
				int yofst = 0;
				switch (type)
				{
					case 5: // player
						assert (player == NULL); // no more than one player
						player = new Player();
						o = player;
						break;
					case 10: // exit block 
						o = new ExitObject(ExitObject::HASH);
						break;
					case 11: // exit block 
						o = new ExitObject(ExitObject::TEDDY);
						break;
					case 12: // exit block 
						o = new ExitObject(ExitObject::DONUT);
						break;
					case 13:
						o = new ExitObject(ExitObject::PRINCESS);
						yofst = -35;
						break;
					case 7: // enemy 2
						o = new Monster(Monster::BUNNY);
						break;
					case 8: // enemy 1
						o = new Monster(Monster::FOLE);
						break;
					case 9: // enemy 1
						o = new Monster(Monster::SKINNY);
						yofst = -35;
						break;
				}
				if (o)
				{
					objects.add (o);
					o->setLocation (
						fix(x * level->tilelist->tilew), 
						fix(y * level->tilelist->tileh + yofst));
				}
			}
		}
	}
	assert (player != NULL); // there must be a player object now
	camera_x = player->getx() - 160;
	if (camera_x < 0) camera_x = 0;
}

void Engine::resume ()
{
	state = oldState;
	parent->playMusic (levelset->levels[currentLevel].music);
}

// called when exit found
void Engine::nextLevel()
{
	// ignore when already in state==8, to prevent problem with repeated signal
	if (state != 8)
	{
		// go on to next level
		state = 8;
		outroTimer = 40;
	}
}

void Engine::draw (BITMAP *buffer)
{
	int temp;
	switch (state)
	{
		case 1: case 6:		case 8:
			teg_draw_tiled_rle (buffer, background, 0, camera_x / 2, 0, 0);
			teg_draw (buffer, level, 0, camera_x, 0);
			teg_draw (buffer, level, 1, camera_x, 0);
			objects.draw(buffer, camera_x, 0);		
			teg_draw (buffer, level, 2, camera_x, 0);
				   
			textprintf_ex (buffer, font, 1, 1,
				GREY, -1, "LIFE %02i", lives);
			textprintf_ex (buffer, font, 0, 0,
				WHITE, -1, "LIFE %02i", lives);

			// draw a health bar
			health = health < 0 ? 0 : (health > 100 ? 100 : health);
			rectfill (buffer, 218 + health, 1, 318, 6, BLACK);
			rectfill (buffer, 218, 1, 218 + health, 6, RED);
			rect (buffer, 217, 0, 319, 7, WHITE);
			
			temp = (camera_x / 2) % 16;
			if (isDebug()) textprintf_ex (buffer, font, 0, 8,
				GREEN, -1, "camera_x/2 %i, mod 16 %i", camera_x / 2, temp);
			break;
		case 2: 
			clear_to_color (buffer, BLACK);
			textprintf_centre_ex (buffer, font, buffer->w / 2, 
				buffer->h / 2, WHITE, -1, "GAME OVER");
		   
			textprintf_centre_ex (buffer, font, buffer->w / 2, 
				buffer->h * 3 / 4, GREY, -1, "Press ENTER");			
		case 0:
			clear_to_color (buffer, BLACK);
			
			if ((introTimer / 6) & 1)
			{
				textprintf_centre_ex (buffer, font, buffer->w / 2, 
					(buffer->h - text_height (font)) / 2, WHITE, -1, 
					"LEVEL %i", currentLevel+1);
			}
			break;
		case 7:
			testview.draw(buffer);
			break;
		case 9: case 5:
			cutscene->draw(buffer);
			break;
		default:
			assert (false);
	}
}

void Engine::updateCamera()
{
	// update camera
	int x = player->getx();
	int margin = 120;
	if (x < camera_x + margin)
	{
		camera_x -= 2;
		if (camera_x < 0) camera_x = 0;
	}
	if (x > (camera_x + 320 - margin))
	{					
		camera_x += 2;
		if (camera_x > teg_pixelw(level) - 320)
			camera_x = teg_pixelw(level) - 320;
	}
}

void Engine::update ()
{

	int c = 0;
	if (keypressed())
	{	
		c = readkey();
		if (c >> 8 == KEY_ESC)
		{
			oldState = state;
			state = 3;
		}
		if (c >> 8 == KEY_F11)
		{
			debug = !debug;
		}		
		if (c >> 8 == KEY_F12)
		{
			screenshot();
		}	   
		if (c >> 8 == KEY_F9) 
		{
			state = 7; // test view
		}
		
	}
	switch (state)
	{
		case 0: // level intro screen showing
			introTimer--;
			if (introTimer == 0)
			{
				state = 1;
				introTimer = 40;
			}
			break;
		case 6:
			outroTimer--;
			if (outroTimer == 0)
			{
				doneLevel();
				lives--;
				health = 100;
				if (lives == 0)
				{
					state = 5;
				}
				else
				{
					initLevel();
					state = 0;
				}
				break;
			}
			objects.update();
			updateCamera();
			break;
		case 1: // game playing
			// add objects based on position
			objects.update();
			updateCamera();
			break;
		case 2: // "start" before level
			if (c >> 8 == KEY_ENTER) { doneLevel(); state = 4; }
			break;
		case 3: // do nothing, main will switch to resume menu.
			break;
		case 4: // do nothing, game is finished (won or lost)
			break;
		case 5: // game over screen showing
			cutscene->update();
			if (cutscene->finished) state = 4;
			break;
		case 7: // test view screen
			testview.update();
			break;
		case 8:
			outroTimer--;
			if (outroTimer <= 0)
			{
				state = 0;
				doneLevel();
				currentLevel++;
				if (currentLevel >= levelset->levels.size())
				{
					state = 5;
					cutscene = parent->getResources()->getCutScene ("ending");
					cutscene->init();
					parent->setCheatUnlocked (true);
				}
				else
				{
					initLevel();
					switch (currentLevel)
					{
						case 0: cutscene = parent->getResources()->getCutScene ("intro"); break;
						case 1: cutscene = parent->getResources()->getCutScene ("level2"); break;
						case 2: cutscene = parent->getResources()->getCutScene ("level3"); break;
						default : assert (false);
					}
					cutscene->init();
					state = 9;
				}
				break;
			}
			else
			{
				objects.update();
				updateCamera();
			}
			break;			
		case 9:
			cutscene->update();
			if (cutscene->finished) state = 0;
			break;
		default:
			assert (false); // shouldn't occur
	}  
}
