#define ALLEGRO_STATICLINK
#include <allegro.h>
#include <fblend.h>
#include <cstring>
#include <cstdio>

#define CANCEL_GAME 1
#define WIN_GAME 2
#define LOOSE_GAME 3

typedef unsigned char BYTE;

DATAFILE *fontDat;
FONT *myFont;

class Board {
public:    
    enum {
        MAX_W = 12,
        MAX_Z =  6,
        MAX_H =  7,
    };
    enum {
        TILE_DY  = 36,
        TILE_H   = 42,
        TILE_DX  = 51,
        TILE_W   = 60,
    };
    BYTE data[Board::MAX_W * Board::MAX_H * Board::MAX_Z];
    int  countPieces;
    
    inline int calcPos(int x, int y, int z) {
        return x + y * Board::MAX_W + z * Board::MAX_W*Board::MAX_H;
    }
    
    int getAt(int x, int y, int z) {
        //return getAt(x + y * Board::MAX_W + z * Board::MAX_W*Board::MAX_H);
        return getAt(calcPos(x,y,z));
    }
    
    int getAt(int pos) {
        return data[pos];
    }
    
    void setAt(int pos, int value) {
        int old = data[pos];
        
        data[pos] = (BYTE) value;
        if (old == 0) {
            if (value != 0) {
                countPieces++;
            }
        } else {
            if (value == 0) {
                countPieces--;
            }
        }
    }
    
    void setAt(int x, int y, int z, int value) {
        //setAt(x + y * Board::MAX_W + z * Board::MAX_W * Board::MAX_H, value);
        setAt(calcPos(x,y,z), value);
    }
        
    int isPlaceable(int x, int y, int z){
        if (z==0) {
            return TRUE;
        }
        return getAt(x, y, z-1) != 0;
    }
    
    void clear() {
        memset(data, 0, Board::MAX_W * Board::MAX_H * Board::MAX_Z); 
        countPieces = 0;
    }
    
    void save(const char* filename) {
        FILE *f = fopen(filename, "wb");
        for (int a=0; a < Board::MAX_W * Board::MAX_H * Board::MAX_Z; a++) {
            fwrite(&data[a], 1, 1, f);
        }
        fclose(f);
    }
    void load(const char* filename) {
        FILE *f = fopen(filename, "rb");
        countPieces = 0;
        for (int a=0; a < Board::MAX_W * Board::MAX_H * Board::MAX_Z; a++) {
            fread(&data[a], 1, 1, f);
            if (data[a] != 0) {
                countPieces++;
            }
        }
        fclose(f);
    }
};

struct GlobalBitmaps {
    BITMAP *doubleBuffer;
    BITMAP *tile;
    BITMAP *title;
    BITMAP *bg;
    BITMAP *icons;
    
    GlobalBitmaps() :doubleBuffer(NULL), tile(NULL), title(NULL), bg(NULL), icons(NULL) {
    }
    
    void free(){
        if (doubleBuffer != NULL) {
            destroy_bitmap(doubleBuffer);
        }
        if (tile != NULL) {
            destroy_bitmap(tile);
        }
        if (title != NULL) {
            destroy_bitmap(title);
        }
        if (bg != NULL) {
            destroy_bitmap(bg);
        }
        if (icons != NULL) {
            destroy_bitmap(icons);
        }        
    }
};

GlobalBitmaps bmps;
Board curBoard;

int play();

int setGfxMode(int mode) {
    int modes[] = { 16, 32, 15, 24 };
        
    for (int a=0; a < 4; a++) {
        set_color_depth(modes[a]);
        if (set_gfx_mode(mode, 640, 480,0,0) >= 0)  {
            return TRUE;
        }
    }
    return FALSE;
}

void init() {
    allegro_init();    
    install_keyboard();
    install_mouse();
    
    if (!setGfxMode(GFX_AUTODETECT_WINDOWED)) {
        allegro_message("Sorry!");
    }
    bmps.doubleBuffer = create_bitmap(SCREEN_W, SCREEN_H);
    bmps.tile         = load_bitmap("empty.tga", NULL); 
    bmps.bg           = load_bitmap("bg.tga", NULL); 
    bmps.icons        = load_bitmap("geek.tga", NULL); 
    bmps.title        = load_bitmap("title.tga", NULL); 
    
    fontDat = load_datafile("font.dat");
    myFont = (FONT*) fontDat[0].dat;
    
    text_mode(-1);
    
    show_mouse(screen);
}

void editor() {
    clear(bmps.doubleBuffer);    
    curBoard.clear();
    
    char buffer[1024];
    
    
    int total_w = Board::MAX_W * Board::TILE_DX;
    int total_h = Board::MAX_H * Board::TILE_DY;
    int start_x = (SCREEN_W - total_w) / 2;
    int start_y = (SCREEN_H - total_h) / 2;
    
    int x,y,z, a, oz = 0;
    int curX, curY;
    int needsRepaint = 1;
    
    int mode = 0;
    int last_x = -1;
    int last_y = -1;
    
    int saved  = 0;
    int loaded = 0;
    
    z = 0;
    while (!key[KEY_ESC]) {        
        if (needsRepaint) {
            //clear(bmps.doubleBuffer);    
            blit(bmps.bg, bmps.doubleBuffer, 0,0,0,0, bmps.bg->w, bmps.bg->h);
            for (a=0; a <= z; a++) {
                curY = start_y - (Board::TILE_H - Board::TILE_DY)*a;
                for (y=0; y < Board::MAX_H; y++) {
                    curX = start_x - (Board::TILE_W - Board::TILE_DX)*a;
                    for (x=0; x < Board::MAX_W; x++) {                        
                        int piece = curBoard.getAt(x,y,a);
                        if (piece) {
                            draw_sprite(bmps.doubleBuffer, bmps.tile, curX, curY);
                        }
                        if (a==z && curBoard.isPlaceable(x,y,a)) {
                            rect(bmps.doubleBuffer,
                                curX, curY, 
                                curX + Board::TILE_DX, curY + Board::TILE_DY, 
                                makecol(0,128,0));
                            rect(bmps.doubleBuffer,
                                curX+1, curY+1, 
                                curX + Board::TILE_DX-1, curY + Board::TILE_DY-1, 
                                makecol(0,128,0));
                        }            
                        curX+= Board::TILE_DX;
                    }
                    curY+= Board::TILE_DY;
                }
                if (a == z-1) {
                    for (x=0; x < SCREEN_W; x+=2) {
                        vline(bmps.doubleBuffer, x, 0, SCREEN_H, 0);
                    }
                    for (y=0; y < SCREEN_H; y+=2) {
                        hline(bmps.doubleBuffer, y, 0, SCREEN_W, 0);
                    }
                }
            }
            needsRepaint = 0;
            scare_mouse();
            textprintf(bmps.doubleBuffer, font, 5,5, -1, "z: %8i   count: %4i", z, curBoard.countPieces);
            textprintf(bmps.doubleBuffer, font, 5,25, -1, "Keys: 1-7: Choose layer, L: Load Layout, S: Save Layout, P:Play");
            blit(bmps.doubleBuffer, screen, 0,0,0,0, SCREEN_W, SCREEN_H);
            unscare_mouse();
        }
        if (mouse_b & 1) {
            x = mouse_x - start_x + (Board::TILE_W - Board::TILE_DX)*z;
            y = mouse_y - start_y + (Board::TILE_H - Board::TILE_DY)*z;
            
            x /= Board::TILE_DX;
            y /= Board::TILE_DY;      
            
            if (last_x < 0) {            
                if (curBoard.getAt(x,y,z)) {
                    mode = 0;
                } else {
                    mode = 1;
                }                
            }
            if (last_x != x || last_y != y) {
                if (curBoard.getAt(x,y,z) != mode && curBoard.isPlaceable(x,y,z)) {
                    curBoard.setAt(x,y,z, mode);
                    if (mode == 0) {
                        for (a=z+1; a < Board::MAX_Z; a++) {
                            curBoard.setAt(x,y,a, mode);
                        }
                    } 
                    needsRepaint = 1;
                }
                last_x = x;
                last_y = y;
            }
        } else {
            last_x = last_y = -1;
        }
        oz = z;
        if (key[KEY_1]) {
            z = 0;
        } else if (key[KEY_2]) {
            z = 1;
        } else if (key[KEY_3]) {
            z = 2;
        } else if (key[KEY_4]) {
            z = 3;
        } else if (key[KEY_5]) {
            z = 4;
        } else if (key[KEY_6]) {
            z = 5;
        } 
        if (key[KEY_S]) {
            if (!saved) {
                sprintf(buffer, ".");
                if (file_select_ex("Save layout", buffer, "LYT", 1024, 500, 350)) {
                    curBoard.save(buffer);
                }
                saved = 1;
            }
        } else {
            saved = 0;
        }
        if (key[KEY_L]) {
            if (!loaded) {
                sprintf(buffer, ".");
                if (file_select_ex("Load layout", buffer, "LYT", 1024, 500, 350)) {
                    curBoard.load(buffer);
                    needsRepaint = 1;
                }
                loaded= 1;
            }
        } else {
            loaded = 0;
        }
        if (key[KEY_P]) {
            while (key[KEY_P]);
            Board tmp = curBoard;
            play();            
            needsRepaint = 1;
            curBoard = tmp;
        }
        if (z != oz) {
            needsRepaint = 1;
        }
        
    }
    while(key[KEY_ESC]);
}

int pieceSelectable(int x, int y, int z) {
    int piece = curBoard.getAt(x, y, z);
    if (piece != 0) {
        if ((z == Board::MAX_Z-1 || curBoard.getAt(x,y,z+1)==0)) {
            if ((x ==0 || x == Board::MAX_W-1 || curBoard.getAt(x-1, y, z) ==0 || curBoard.getAt(x+1, y, z) == 0)) {
                return 1;
            }
        } 
    } 
    return 0;
}

int findPiece(int x, int y) {
    
    int total_w = Board::MAX_W * Board::TILE_DX;
    int total_h = Board::MAX_H * Board::TILE_DY;
    int start_x = (SCREEN_W - total_w) / 2;
    int start_y = (SCREEN_H - total_h) / 2;
    
    for (int a= Board::MAX_Z-1; a >= 0; a--) {
        int curX = x - start_x + (Board::TILE_W - Board::TILE_DX)*a;
        int curY = y - start_y + (Board::TILE_H - Board::TILE_DY)*a;
        
        curX /= Board::TILE_DX;
        curY /= Board::TILE_DY;      
        
        if (curX >=0 && curX < Board::MAX_W && curY >= 0 && curY < Board::MAX_H && pieceSelectable(curX, curY, a)) {
            return curBoard.calcPos(curX, curY, a);
        }        
    }    
    return -1;
}


void renderBoard(BITMAP *dest, int start_x, int start_y, int topLayer = Board::MAX_Z, int shading = 0, int selected = -1) {
    int a,x,y;
    int curX, curY;
    for (a=0; a < Board::MAX_Z; a++) {            
        //fblend_rect_trans(dest, 0, 0, dest->w, dest->h, makecol(0,0,128), shading * (topLayer-a));
        curY = start_y - (Board::TILE_H - Board::TILE_DY)*a;
        for (y=0; y < Board::MAX_H; y++) {
            curX = start_x - (Board::TILE_W - Board::TILE_DX)*a;
            for (x=0; x < Board::MAX_W; x++) {                        
                int pos   = curBoard.calcPos(x,y,a);
                int piece = curBoard.getAt(pos);
                if (piece) {
                    draw_sprite(bmps.doubleBuffer, bmps.tile , curX   , curY);
                    masked_blit(bmps.icons, bmps.doubleBuffer, (piece-1) * 27, 0, curX+10, curY+4, 27, 27);
                    if (pos == selected) {
                        rect(bmps.doubleBuffer, curX, curY, curX+Board::TILE_DX, curY+Board::TILE_DY, makecol(0,200,0));
                        rect(bmps.doubleBuffer, curX+1, curY+1, curX+Board::TILE_DX-2, curY+Board::TILE_DY-2, makecol(0,150,0));
                    }
                }
                curX+= Board::TILE_DX;
            }
            curY+= Board::TILE_DY;
        }                
    }
}

int isGameLost() {
    static int flat[Board::MAX_W * Board::MAX_H];
    int count = 0;
    for (int z=0; z < Board::MAX_Z; z++) {
        for (int y=0; y < Board::MAX_H; y++) {
            for (int x=0; x < Board::MAX_W; x++) {
                if (pieceSelectable(x,y,z)) {
                    int piece = curBoard.getAt(x,y,z);                    
                    for (int a=0; a < count; a++) {
                        if (flat[a] == piece) {
                            return 0;
                        }
                    }
                    flat[count] = piece;
                    count++;
                }
            }
        }
    }
    return 1;
}


int play() {
    int   count    = curBoard.countPieces;
    int   selected = -1;
    int   topLayer =  0;
    
    if (count == 0) {
        return CANCEL_GAME;
    }
    
    int   cntTiles = bmps.icons->w / 27;
    BYTE *tiles    = new BYTE[count];
    
    
    
    for (int a=0; a < count/2; a++) {
        tiles[a*2  ] = a % cntTiles;
        tiles[a*2+1] = a % cntTiles;
    }

    // shuffle    
    for (int a=0; a < 10000; a++) {
        int q = rand() % count;
        int w = rand() % count;
        
        int t = tiles[q];
        tiles[q] = tiles[w];
        tiles[w] = t;
    } 

    int pos = 0;
    for (int a=0; a < Board::MAX_W * Board::MAX_H * Board::MAX_Z; a++) {

        if (curBoard.getAt(a) != 0 && pos < count) {
            curBoard.setAt(a, tiles[pos] + 1);            
            topLayer = a / (Board::MAX_W*Board::MAX_H);
            pos++;
        }
    }
    int shading = 20;
    for (int a=Board::MAX_Z; a > topLayer; a--) {
        shading += 50;
    }
    
    
    int needsRepaint = 1;
    int needsCheck   = 1;
    int total_w = Board::MAX_W * Board::TILE_DX;
    int total_h = Board::MAX_H * Board::TILE_DY;
    int start_x = (SCREEN_W - total_w) / 2;
    int start_y = (SCREEN_H - total_h) / 2;
    int mouse_down = 0;
    

    int gameOver = 0;
    while (!gameOver) {        
        if (needsRepaint) {
            blit(bmps.bg, bmps.doubleBuffer, 0,0,0,0, bmps.bg->w, bmps.bg->h);
            renderBoard(bmps.doubleBuffer, start_x, start_y, topLayer, shading, selected);
            needsRepaint = 0;
            scare_mouse();
            textprintf(bmps.doubleBuffer,myFont, 5,5, -1, "Pieces remaining: %4i", curBoard.countPieces);
            blit(bmps.doubleBuffer, screen, 0,0,0,0, SCREEN_W, SCREEN_H);
            unscare_mouse();
            needsCheck = 1;
        } else if (needsCheck) {
            needsCheck = 0;
            if (isGameLost()) {
                gameOver = LOOSE_GAME;
            }
            if (curBoard.countPieces == 0) {
                gameOver = WIN_GAME;
            }
        }
        
        
        if (mouse_b) {
            if (!mouse_down) {
                mouse_down = 1;
                int piece = findPiece(mouse_x, mouse_y);
                //allegro_message("%i, %i", piece, selected);
                if (piece >= 0) {
                    if (selected == -1) {
                        selected = piece;
                        needsRepaint = 1;
                    } else {
                        if (piece == selected) {
                            selected = -1;
                            needsRepaint = 1;
                        } else if (curBoard.getAt(selected) == curBoard.getAt(piece)) {
                            curBoard.setAt(selected, 0);
                            curBoard.setAt(piece, 0);
                            selected = -1;
                            needsRepaint = 1;
                        }
                    }
                }
            }
        } else {
            mouse_down = 0;
        }
        if (key[KEY_ESC]) {
            gameOver = CANCEL_GAME;
        } else if (key[KEY_F1]) {
            gameOver = WIN_GAME;
        }
        
    }
    while(key[KEY_ESC]);
    delete [] tiles;
    
    return gameOver;
}

int title() {
    int sel = -1;
    int selected = -1;
    int oldSel = -2;
    char *menu[] = {"Start", "Exit"};
    
    
    while (selected < 0) {        
        blit(bmps.title, bmps.doubleBuffer, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
        int y = 180;        
        sel = (mouse_y - y) / (int) (text_height(myFont)*1.5);
        for (int a=0; a < 2; a++) {
            textprintf_centre(bmps.doubleBuffer, myFont, SCREEN_W/2, y, sel == a ? makecol(20, 240, 170): makecol(64,124,94), menu[a]);
            y+=(int)(text_height(myFont)*1.5);
        }
        if (oldSel != sel) {
            oldSel = sel;
            scare_mouse();
            blit(bmps.doubleBuffer, screen, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
            unscare_mouse();
        }
        if (mouse_b && sel >= 0 && sel < 2) {
            selected = sel;
        }
        
        if (key[KEY_ESC]) {
            selected = 1;
        }
    }
    return selected;
}

int main(int argc, char** argv) {
    char buffer[20];
    
    srand(time(NULL));
    if (argv != NULL) {
        int foo = argc;
        foo++;
    }
    init();
    
    if (argc <= 1) {
        while (title() == 0) {
            while (mouse_b);
                
            sprintf(buffer, "%i.lyt", rand()%3);
            curBoard.load(buffer);
            if (play() != CANCEL_GAME) {
                fblend_rect_trans(bmps.doubleBuffer, 0, 0, SCREEN_W, SCREEN_H, makecol(0,0,128), 128);
                textprintf_centre(bmps.doubleBuffer, myFont, SCREEN_W/2, SCREEN_H/2, makecol(250,255,250), "Game Over");
                scare_mouse();
                
                blit(bmps.doubleBuffer, screen, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
                rest(1500);
                unscare_mouse();
            }        
        }
    } else {
        editor();
    }
    
    /*
    
    blit(bmps.doubleBuffer, screen, 0,0,0,0, SCREEN_W, SCREEN_H);
    while (!keypressed()) {
    }
    */
        
    bmps.free();
    return 0;
} END_OF_MAIN();