#include "LaserRobots.h"
#include <stdio.h>
#include <ctype.h>
#include <string.h>

extern "C" {
double evaluate(char *equation, int *error, double (*variable)(char *name));
}

/* static data */
static PACKFILE *lb_file;
static unsigned char *lb_data;
static int lb_data_length;
static int lb_unget_char;
static int lb_line;

static LB_ROBOT *the_lb = NULL;

	
/* lb_error:
 *  Formats an error message.
 */
static void lb_error(char *error, char *msg, ...) {

	char buf[256];

	va_list ap;
	va_start(ap, msg);
	vsprintf(buf, msg, ap);
	va_end(ap);

	sprintf(error, "%s at line %d", buf, lb_line);
}



/* do_getc:
 *  Reads the next byte of the input stream.
 */
static int do_getc() {
	if (lb_unget_char) {
		int c = lb_unget_char;
		lb_unget_char = 0;
		return c;
	}

	if (lb_file)
		return pack_getc(lb_file);

	if ((lb_data) && (lb_data_length > 0)) {
		lb_data_length--;
		return *(lb_data++);
	}

	return EOF;
}



/* lb_getc:
 *  Reads the next byte of the input stream, applying some bodges.
 */
static int lb_getc() {
	int c = do_getc();

	if ((c >= 'A') && (c <= 'Z'))
		c = c - 'A' + 'a';

	if (c == '#') {
		while ((c != '\n') && (c != EOF))
	 c = do_getc();
	}

	if (c == '\n')
		lb_line++;

	return c;
}



/* lb_ungetc:
 *  Pushes an unwanted character back into the input stream.
 */
static void lb_ungetc(int c) {

	lb_unget_char = c;

	if (c == '\n')
		lb_line--;
}



/* lb_eof:
 *  Have we reached the end of the file?
 */
static int lb_eof() {

	if (lb_file)
		return pack_feof(lb_file);
	else
		return (lb_data_length > 0);
}



/* iswordchar:
 *  Returns true if this character is valid as part of an identifier.
 */
static int iswordchar(int c) {

	return (((c >= 'A') && (c <= 'Z')) ||
		((c >= 'a') && (c <= 'z')) ||
		(c == '_'));
}



/* get_word:
 *  Reads a word from the input stream.
 */
static void get_word(char *buf) {

	int c, i;

	for (;;) {
		c = lb_getc();

		if (c == EOF) {
			buf[0] = 0;
			return;
		}

		if (iswordchar(c)) {
			i = 0;

			for (;;) {
				buf[i++] = c;

				c = lb_getc();

				if (!iswordchar(c)) {
					lb_ungetc(c);
					buf[i] = 0;
					return;
				}
			}
		}

		if (!isspace(c)) {
			buf[0] = c;
			buf[1] = 0;
			return;
		}
	}
}



/* get_formula:
 *  Reads a formula from the input stream, finishing when the specified
 *  terminator character is encountered.
 */
static void get_formula(char *buf, int terminator, char *error) {

	int inspace = FALSE;
	int braces = 0;
	int i = 0;
	int c;

	for (;;) {
		c = lb_getc();

		if ((c == terminator) && (braces <= 0)) {
			buf[i] = 0;
			return;
		}

		if (c == EOF) {
			lb_error(error, "Unexpected EOF");
			return;
		}

		if (isspace(c)) {
			if ((i > 0) && (!inspace)) {
				buf[i++] = ' ';
				inspace = TRUE;
			}
		}
		else {
			buf[i++] = c;
			
			if (terminator == ')') {
				
				if (c == '(')
					braces++;
				else if (c == ')')
					braces--;
			}
		}

		if (i >= 1023) {
			lb_error(error, "Missing '%c'", terminator);
			return;
		}
	}
}



/* get_brace:
 *  Reads a '{' from the input stream, giving an error if it is not found.
 */
static void get_brace(char *error) {

	char buf[256];

	get_word(buf);

	if (strcmp(buf, "{") != 0)
		lb_error(error, "Missing '{'");
}



/* check_ascii_word:
 *  Checks that a string is a valid variable name.
 */
static void check_ascii_word(char *buf, char *error) {
	
	int i;

	for (i=0; buf[i]; i++) {
		if (!iswordchar(buf[i]))
			break;
	}

	if (i <= 0)
		lb_error(error, "Missing identifier");
	else if (buf[i])
		lb_error(error, "Invalid character ('%c') in identifier", buf[i]);
}



/* get_ascii_word:
 *  Reads a word and checks that it is a valid variable name.
 */
static void get_ascii_word(char *buf, char *error) {

	get_word(buf);
	check_ascii_word(buf, error);
}



/* load_lb_cmd:
 *  Reads a list of commands.
 */
static LB_COMMAND *load_lb_cmd(LB_ROBOT *lb, int level, char *error) {
	LB_COMMAND *cmd = NULL;
	LB_COMMAND *tail = NULL;
	//LB_TYPE *type;
	char buf[1024];
	char buf2[256];


	/* helper macro for inserting new commands into the list */
	#define ADD_COMMAND(_type_)                 \
  {                                           \
    if (tail) {                               \
     tail->next = new LB_COMMAND();           \
     tail = tail->next;                       \
    }                                         \
    else                                      \
     tail = cmd = new LB_COMMAND();           \
                                              \
    tail->type = _type_;                      \
    tail->line = lb_line;                     \
    tail->var = NULL;                         \
    tail->exp = NULL;                         \
    tail->cmd = NULL;                         \
    tail->cmd2 = NULL;                        \
    tail->next = NULL;                        \
    printf("Adding command: %s\n", #_type_);  \
  }


	while ((!lb_eof()) && (!error[0])) {
		get_word(buf);

		if (strcmp(buf, "}") == 0) {
			/* block end marker */
			if (level==0)
				lb_error(error, "Unexpected '}'");
			else
				return cmd;
		}
		else if (strcmp(buf, "if") == 0) {
			/* parse an if statement */
			get_word(buf);
			
			if (strcmp(buf, "(") != 0) {
				lb_error(error, "Missing '('");
			}
			else {
				get_formula(buf, ')', error);
				
				if (!error[0]) {
					get_brace(error);

					if (!error[0]) {
						ADD_COMMAND(LB_COMMAND_IF);
		  
						tail->exp = (char*) malloc(strlen(buf)+1);
						strcpy(tail->exp, buf);

						tail->cmd = load_lb_cmd(lb, level+1, error);
					}
				}
			}
		}
		else if (strcmp(buf, "else") == 0) {
			/* parse an else statement */
			if ((!tail) || (tail->type != LB_COMMAND_IF) || (tail->cmd2)) {
				lb_error(error, "Invalid context for 'else'");
			}
			else {
				get_brace(error);

				if (!error[0])
					tail->cmd2 = load_lb_cmd(lb, level+1, error);
			}
		}
		else if (strcmp(buf, "while") == 0) {
			/* parse a while statement */
			get_word(buf);

			if (strcmp(buf, "(") != 0) {
				lb_error(error, "Missing '('");
			}
			else {
				get_formula(buf, ')', error);
						 
				if (!error[0]) {
					get_brace(error);

					if (!error[0]) {
						ADD_COMMAND(LB_COMMAND_WHILE);
						
						tail->exp = (char*) malloc(strlen(buf)+1);
						strcpy(tail->exp, buf);
						
						tail->cmd = load_lb_cmd(lb, level+1, error);
					}
				}
			}
		}
		else if (strcmp(buf, "srand") == 0) {
			/* parse a srand statement */
			ADD_COMMAND(LB_COMMAND_SRAND);

			if (!error[0]) {
				tail->var = (char*) malloc(strlen(buf)+1);
				strcpy(tail->var, buf);

				get_formula(buf, ';', error);

				if (!error[0]) {
					tail->exp = (char*) malloc(strlen(buf)+1);
					strcpy(tail->exp, buf);
				}
			}
		}
		else if (strcmp(buf, "go") == 0) {
			/* parse a go statement */
			ADD_COMMAND(LB_COMMAND_GO);

			if (!error[0]) {
				tail->var = (char*) malloc(strlen(buf)+1);
				strcpy(tail->var, buf);

				get_formula(buf, ';', error);

				if (!error[0]) {
					tail->exp = (char*) malloc(strlen(buf)+1);
					strcpy(tail->exp, buf);
				}
			}
		}
		else if (strcmp(buf, "back") == 0) {
			/* parse a back statement */
			ADD_COMMAND(LB_COMMAND_BACK);

			if (!error[0]) {
				tail->var = (char*) malloc(strlen(buf)+1);
				strcpy(tail->var, buf);
				
				get_formula(buf, ';', error);

				if (!error[0]) {
					tail->exp = (char*) malloc(strlen(buf)+1);
					strcpy(tail->exp, buf);
				}
			}
		}
		else if (strcmp(buf, "left") == 0) {
			/* parse a srand statement */
			ADD_COMMAND(LB_COMMAND_LEFT);

			if (!error[0]) {
				tail->var = (char*) malloc(strlen(buf)+1);
				strcpy(tail->var, buf);
				
				get_formula(buf, ';', error);

				if (!error[0]) {
					tail->exp = (char*) malloc(strlen(buf)+1);
					strcpy(tail->exp, buf);
				}
			}
		}
		else if (strcmp(buf, "right") == 0) {
			/* parse a srand statement */
			ADD_COMMAND(LB_COMMAND_RIGHT);

			if (!error[0]) {
				tail->var = (char*) malloc(strlen(buf)+1);
				strcpy(tail->var, buf);
				
				get_formula(buf, ';', error);

				if (!error[0]) {
					tail->exp = (char*) malloc(strlen(buf)+1);
					strcpy(tail->exp, buf);
				}
			}
		}
		else if (strcmp(buf, "fire") == 0) {
			/* parse a srand statement */
			ADD_COMMAND(LB_COMMAND_FIRE);

			if (!error[0]) {
				tail->var = (char*) malloc(strlen(buf)+1);
				strcpy(tail->var, buf);

				get_formula(buf, ';', error);

				if (!error[0]) {
					tail->exp = (char*) malloc(strlen(buf)+1);
					strcpy(tail->exp, buf);
				}
			}
		}
		else if (buf[0]) {
			/* this must be a variable assignment */
			check_ascii_word(buf, error);

			if (!error[0]) {
				get_word(buf2);

				if (strcmp(buf2, "=") == 0) {
					//printf("SET GEN: buf: %s\n",buf);
					ADD_COMMAND(LB_COMMAND_SET);
				}
				else if (strcmp(buf2, ":") == 0) {
					if (level == 0) {
						get_word(buf2);
						
						if (strcmp(buf2, "=") != 0) {
							lb_error(error, "Missing '='");
						}
						else {
							ADD_COMMAND(LB_COMMAND_INIT);
						}
					}
					else {
						lb_error(
						  error, 
						  "':=' initialization is not at top level "
						  "of robot declaration"
						);
					}
				}
				else {
					lb_error(error, "Fucked up syntax ");
				}
				
				if (!error[0]) {
					
					tail->var = (char*) malloc(strlen(buf)+1);
					strcpy(tail->var, buf);

					get_formula(buf, ';', error);

					if (!error[0]) {
						tail->exp = (char*) malloc(strlen(buf)+1);
						strcpy(tail->exp, buf);
					}
				}
			}
		}
	}

	if ((!error[0]) && (lb_eof()) && (level != 0))
		lb_error(error, "Unexpected EOF");

	if (error[0]) {
		delete cmd;
		return NULL;
	}

	return cmd;
}



/* do_load:
 *  Worker function for loading LB scripts.
 */
static LB_ROBOT *do_load(char *error) {

	LB_ROBOT *lb;
	LB_COMMAND *cmd;

	lb_line = 1;

	lb_unget_char = 0;

	error[0] = 0;

	lb = new LB_ROBOT();

	lb->var = NULL;
	/*
	lb->var = new LB_VARIABLE();
	lb->var->name = "#dummy";
	lb->var->next = NULL;
	lb->var->val = 0;
	*/


	cmd = load_lb_cmd(lb, 0, error);

	if (error[0]) {
		delete lb;
		return NULL;
	}

	process_lb_cmd(NULL, lb, cmd, TRUE, TRUE, error);

	// delete cmd;

	lb->cmd = cmd;

	if (error[0]) {
		delete lb;
		return NULL;
	}

	return lb;
}



/* load_lb:
 *  Reads an LB script from a disk file.
 */
LB_ROBOT *load_lb(char *filename, char *error) {

	LB_ROBOT *lb;

	lb_file = pack_fopen(filename, F_READ);

	if (!lb_file) {
		strcpy(error, "File not found");
		return NULL;
	}

	lb = do_load(error);

	pack_fclose(lb_file);
	lb_file = NULL;

	return lb;
}



/* use_lb:
 *  Reads an LB script from a block of memory.
 */
LB_ROBOT *use_lb(void *data, int length, char *error) {

	LB_ROBOT *lb;

	lb_data = (unsigned char*) data;
	lb_data_length = length;

	lb = do_load(error);

	lb_data = NULL;
	lb_data_length = 0;

	return lb;
}



/* get_lb_variable:
 *  Accesses a robot variable.
 */
double get_lb_variable(LB_ROBOT *lb, char *name) {

	LB_VARIABLE *var = lb->var;
	while (var) {
		if (strcmp(var->name, name) == 0) {
			//printf("Lookup var name %s value is %f\n",name,var->val);
			return var->val;
		}

		var = var->next;
	}

	return 0;
}



/* set_lb_variable:
 *  Sets a robot variable.
 */
void set_lb_variable(LB_ROBOT *lb, char *name, double value) {

	LB_VARIABLE *var, *pvar;
	//printf("Set var %s value to %f\n",name,value);
	
	var = pvar = lb->var;
	
	while (var) {
		if (strcmp(var->name, name) == 0) {
			var->val = value;
			return;
		}
		
		var = var->next;
	}
	
	var = new LB_VARIABLE();

	lb->var = var;
	
	var->name = (char*) malloc(strlen(name)+1);
	strcpy(var->name, name);
	var->val = value;
	var->next = pvar;
}



/* variable_getter:
 *  Callback for reading variables out of the robot structure.
 */
static double variable_getter(char *name) {

	if (the_lb)
		return get_lb_variable(the_lb, name);
	
	return 0;
}



/* evaluate_expression:
 *  Wrapper for the expression evaluator module.
 */
static double evaluate_expression(char *exp, int line, char *error) {

   double ret;
   int err;

   ret = evaluate(exp, &err, variable_getter);

   if (err) {
      sprintf(error, "Error in expression at line %d", line);
      return 0;
   }

   return ret;
}



/* process_lb_cmd:
 *  Interprets a list of commands, applying them to the specified robot.
 */
int process_lb_cmd(LASER_BALL *ball, LB_ROBOT *lb, LB_COMMAND *cmd, int init, int set, char *error) {
  
	double ret;

	the_lb = lb;
	
	while ((cmd) && (!error[0])) {

		//printf("Processing LB command: %s: set mode is %d\n", cmd==NULL? "<NULL>":cmd->exp, set);
		
		switch (cmd->type) {
			
		case LB_COMMAND_INIT:
			/* set variables to their initial values */
			if (init) {
	       ret = evaluate_expression(cmd->exp, cmd->line, error);
				 
				 if (!error[0])
					 set_lb_variable(lb, cmd->var, ret);
			}
			break;
			
		case LB_COMMAND_SET:
			/* modify variables */
			if (set) {
	       ret = evaluate_expression(cmd->exp, cmd->line, error);
				 //printf("SET cmd: var %s, value %f (error %s)\n",cmd->var,ret,error);
				 
				 if (!error[0])
					 set_lb_variable(lb, cmd->var, ret);
			}
			break;
			
		case LB_COMMAND_IF:
			/* process conditionals */
			if (!init) {
	       ret = evaluate_expression(cmd->exp, cmd->line, error);

				 if ((!error[0]) && (ret != 0)) {
	          if (cmd->cmd)
							if (process_lb_cmd(ball, lb, cmd->cmd, init, set, error) != 0)
								return TRUE;
				 }
				 else {
	          if (cmd->cmd2)
							if (process_lb_cmd(ball, lb, cmd->cmd2, init, set, error) != 0)
								return TRUE;
				 }
			}
			break;

			
		case LB_COMMAND_WHILE:
			/* process loops */
			if (!init) {
	      while ((ret = evaluate_expression(cmd->exp, cmd->line, error))) {
					 /*
					 printf("while: var: %s, exp: %s\n",cmd->var,cmd->exp);
					 printf("do: var: %s, exp: %s\n",cmd->cmd->var,cmd->cmd->exp);
					 printf("Commands in while:\n");
					 {LB_COMMAND *c=cmd->cmd;
					 while (c) {printf("type %d\n",c->type);c=c->next;}
					 }
					 printf("vars are now (before while arg):\n");
					  TODO {LB_VARIABLE *v;
					 v=part->var;
					 while (v) {
					 printf("varname: %s, value %e\n",v->name,v->val);
					 v=v->next;
					 }}*/
					if (cmd->cmd)
						if (process_lb_cmd(ball, lb, cmd->cmd, init, set+789, error) != 0)
							return TRUE;
					printf("vars are now (after while arg):\n");
					/* TODO {LB_VARIABLE *v;
					v=part->var;
					while (v) {
					printf("varname: %s, value %e\n",v->name,v->val);
					v=v->next;
					}}
					*/
					printf("\n\n\n");
				}
			}
			break;

		case LB_COMMAND_SRAND:
			/* initialize the random number generator */
			srand((unsigned)evaluate_expression(cmd->exp, cmd->line, error));
			break;

		case LB_COMMAND_GO:
			/* move forwards */
			if (!init)
				ball->go_straight();
			break;
		
		case LB_COMMAND_BACK:
			/* move backwards */
			if (!init)
				ball->go_back();
			break;

		case LB_COMMAND_LEFT:
			/* rotate left */
			if (!init)
				ball->rotate_left();
			break;

		case LB_COMMAND_RIGHT:
			/* rotate right */
			if (!init)
				ball->rotate_right();
			break;

		case LB_COMMAND_FIRE:
			/* fire laser */
			if (!init)
				ball->fire();
			break;
		}
		
		cmd = cmd->next;
	}
	
	return FALSE;
}
