// ============================================
// Laser Balls [version 1.0]
// 
// by MARCO CHIESI
// e-mail: chiesi@libero.it
// web site: http://chiesi.cjb.net
//
// A game created for Allegro Speedhack 2002
//
// ============================================

#include <allegro.h>
#include <string.h>
#include <stdio.h>
#include <math.h>
#include "datafile.h"
#include "LaserBall.h"
#include "LaserRobots.h"

/* global variables defines */
#define BOTTOM_SPACE 42
#define MAX_MENU_ITEMS 10
#define FADE_SPEED 4

/* keys config defines */
#define CONTROL_KEY_FIRST  KEY_A
#define CONTROL_KEY_LAST   KEY_ALTGR

/* game result defines */
#define NO_FINISH -1
#define NO_WINNER -2

/* mode defines */
#define MODE_MENU 0
#define MODE_GAME 1

/* menu class */
class LASER_MENU {
public:
	int num;
	int selected;
	char *line[MAX_MENU_ITEMS];
};

/* global settings variables */
bool sound_install = true, show_fps = false, windowed = false;
int res_width = 800;
int res_height = 600;
int update_mode = 5;

/* other global variables */
int x_min, x_max, y_min, y_max;
LASER_BALL laser_ball[MAX_BALLS];
BITMAP *bmp_ball[MAX_BALLS], *bmp_laser[MAX_BALLS], *bmp_big[MAX_BALLS];
BITMAP *bmp_heart[MAX_BALLS], *bmp_gun[MAX_BALLS];
BITMAP *bmp_texture, *bmp_back, *bmp_title, *bmp_winner;
PALETTE pal;
DATAFILE* data;
MIDI *music;
FONT *font_laser;
char *animation_type_str;
bool first_frame = true;
LASER_MENU *menu_main, *menu_options, *menu_players, *menu_controls, *menu_config;
LASER_MENU *current_menu;
int game_mode, game_winner;
int bottom_bar_color;
bool sound_enabled = true, music_enabled = true, music_playing = false;
LB_ROBOT *lb;

/* scroll message */
#define SPLITTER  "                                "
char *scroll_message = 
"Welcome to Laser Balls"
SPLITTER
"A game written for the Allegro Speedhack 2002"
SPLITTER
"by Marco Chiesi (Web site: http://chiesi.cjb.net)";
int keys_controls[4][5] = {
	{KEY_UP, KEY_DOWN, KEY_LEFT, KEY_RIGHT, KEY_ENTER},
	{KEY_W, KEY_S, KEY_A, KEY_D, KEY_LCONTROL},
	{KEY_Y, KEY_H, KEY_G, KEY_J, KEY_TAB},
	{KEY_O, KEY_L, KEY_K, KEY_COLON, KEY_LSHIFT}
};
char *keys_desc[] = {"UP", "DOWN", "LEFT", "RIGHT", "FIRE"};
int players_controls[4] = {0, 1, 2, 3};
int n_players_types = 6;
char* players_types[MAX_PLAYERS];
char buf_error[256] = "";

/* function prototypes */
void draw_objects(BITMAP* bmp);
void draw_menu(BITMAP* bmp);
void play_sound(int s);
char *get_control_name(int control);
void load_options();
void save_options();

// ---------------------------------------
// Begin of code taken from Allegro Demo
// ---------------------------------------
/* different ways to update the screen */
#define DOUBLE_BUFFER 		 1
#define PAGE_FLIP 				 2
#define RETRACE_FLIP			 3
#define TRIPLE_BUFFER 		 4
#define DIRTY_RECTANGLE 	 5

int animation_type = 0;
BITMAP *s;
BITMAP *page1, *page2, *page3;
int current_page = 0;
int use_retrace_proc = FALSE;
volatile int frame_count, fps;
volatile int game_time;

/* timer callback for controlling the game speed */
void game_timer(void) {
	 game_time++;
}
END_OF_FUNCTION(game_timer);

/* timer callback for measuring the frames per second */
void fps_proc(void) {
	 fps = frame_count;
	 frame_count = 0;
}
END_OF_FUNCTION(fps_proc);

/* dirty rectangle list */
typedef struct DIRTY_RECT {
   int x, y, w, h;
} DIRTY_RECT;

typedef struct DIRTY_LIST {
   int count;
   DIRTY_RECT rect[2000 /* TODO */];
} DIRTY_LIST;

DIRTY_LIST dirty, old_dirty;

/* adds a new screen area to the dirty rectangle list */
void add_to_list(DIRTY_LIST *list, int x, int y, int w, int h) {
	 list->rect[list->count].x = x;
	 list->rect[list->count].y = y;
	 list->rect[list->count].w = w;
	 list->rect[list->count].h = h;
	 list->count++; 
}

/* qsort() callback for sorting the dirty rectangle list */
int rect_cmp(const void *p1, const void *p2) {
	 DIRTY_RECT *r1 = (DIRTY_RECT *)p1;
	 DIRTY_RECT *r2 = (DIRTY_RECT *)p2;
	 return (r1->y - r2->y);
}

/* main screen update function */
void draw_screen(void) {
	int c;
	//int i, j;
	//int x, y;
	BITMAP *bmp;
	//RLE_SPRITE *spr;
	//char *animation_type_str;

	if (animation_type == DOUBLE_BUFFER) {
		/* for double buffering, draw onto the memory bitmap. The first step 
		 * is to clear it...
		 */
		animation_type_str = "double buffered";
		bmp = s;
		clear_bitmap(bmp);
	}
	else if ((animation_type == PAGE_FLIP) || (animation_type == RETRACE_FLIP)) {
		/* for page flipping we draw onto one of the sub-bitmaps which
		 * describe different parts of the large virtual screen.
		 */ 
		if (animation_type == PAGE_FLIP) 
			animation_type_str = "page flipping";
		else
			animation_type_str = "synced flipping";
		if (current_page == 0) {
			bmp = page2;
			current_page = 1;
		}
		else {
			bmp = page1;
			current_page = 0;
		}
		clear_bitmap(bmp);
	}
	else if (animation_type == TRIPLE_BUFFER) {
		/* triple buffering works kind of like page flipping, but with three
		 * pages (obvious, really :-) The benefit of this is that we can be
		 * drawing onto page a, while displaying page b and waiting for the
		 * retrace that will flip across to page c, hence there is no need
		 * to waste time sitting around waiting for retraces.
		 */ 
		animation_type_str = "triple buffered";

		if (current_page == 0) {
			bmp = page2; 
			current_page = 1;
		}
		else if (current_page == 1) {
			bmp = page3; 
			current_page = 2; 
		}
		else {
			bmp = page1; 
			current_page = 0;
		}
		clear_bitmap(bmp);
	}
	else {
		/* for dirty rectangle animation we draw onto the memory bitmap, but 
		 * we can use information saved during the last draw operation to
		 * only clear the areas that have something on them.
		 */
		animation_type_str = "dirty rectangles";
		bmp = s;
	
		for (c=0; c<dirty.count; c++) {
			if ((dirty.rect[c].w == 1) && (dirty.rect[c].h == 1)) {
				putpixel(bmp, dirty.rect[c].x, dirty.rect[c].y, 0);
			}
			else {
				rectfill(bmp, dirty.rect[c].x, dirty.rect[c].y, 
				dirty.rect[c].x + dirty.rect[c].w, 
				dirty.rect[c].y+dirty.rect[c].h, 0);
			}
		}
		old_dirty = dirty;
		dirty.count = 0;
	}
	acquire_bitmap(bmp);

// ---------------------------------------
// End of code taken from Allegro Demo
// ---------------------------------------

	/* draw the background */
	blit(bmp_back, bmp, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
	if (animation_type == DIRTY_RECTANGLE && first_frame) {
		add_to_list(&dirty, 0, 0, bmp->w, bmp->h);
		first_frame = false;
	}

	/* draw the fps information */
	if (show_fps) {
		char score_buf[80];
		sprintf(score_buf, "(%s, %ld fps)", animation_type_str, (long)fps);
		textout(bmp, font, score_buf, 0, 0, makecol(255, 255, 255));
		if (animation_type == DIRTY_RECTANGLE)
			add_to_list(&dirty, 0, 0, text_length(font, score_buf), 8);
	}

	/* show game stuff */
	switch (game_mode) {
		case MODE_GAME:
			draw_objects(bmp);
			break;
		case MODE_MENU:
			draw_menu(bmp);
			break;
	}

// ---------------------------------------
// Begin of code taken from Allegro Demo
// ---------------------------------------

	release_bitmap(bmp);

	if (animation_type == DOUBLE_BUFFER) {
		/* when double buffering, just copy the memory bitmap to the screen */
		blit(s, screen, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
	}
	else if ((animation_type == PAGE_FLIP) || (animation_type == RETRACE_FLIP)) {
			/* for page flipping we scroll to display the image */ 
		show_video_bitmap(bmp);
	}
	else if (animation_type == TRIPLE_BUFFER) {
		/* make sure the last flip request has actually happened */
		do {
		} while (poll_scroll());

		/* post a request to display the page we just drew */
		request_video_bitmap(bmp);
	}
	else {
		/* for dirty rectangle animation, only copy the areas that changed */
		for (c=0; c<dirty.count; c++)
			add_to_list(&old_dirty, dirty.rect[c].x, dirty.rect[c].y, dirty.rect[c].w, dirty.rect[c].h);

		/* sorting the objects really cuts down on bank switching */
		if (!gfx_driver->linear)
			qsort(old_dirty.rect, old_dirty.count, sizeof(DIRTY_RECT), rect_cmp);

		acquire_screen();

		for (c=0; c<old_dirty.count; c++) {
			if ((old_dirty.rect[c].w == 1) && (old_dirty.rect[c].h == 1)) {
				putpixel(screen, old_dirty.rect[c].x, old_dirty.rect[c].y, 
				getpixel(bmp, old_dirty.rect[c].x, old_dirty.rect[c].y));
			}
			else {
				
				blit(bmp, screen, old_dirty.rect[c].x, old_dirty.rect[c].y, 
				old_dirty.rect[c].x, old_dirty.rect[c].y, 
				old_dirty.rect[c].w, old_dirty.rect[c].h);
			}
		}

		release_screen();
	}
}

/* timer callback for controlling the speed of the scrolling text */
volatile int scroll_count;

void scroll_counter(void) {
	scroll_count++;
}

END_OF_FUNCTION(scroll_counter);

// ---------------------------------------
// End of code taken from Allegro Demo
// ---------------------------------------

/* check menu input */
int check_menu(LASER_MENU* menu) {
	int r = -1;
	if (keypressed()) {
		int c = readkey() >> 8;
		switch (c) {
			case KEY_UP:
				menu->selected = (menu->selected + menu->num - 1) % menu->num;
				play_sound(SAMPLE_FLIP);
				break;
			case KEY_DOWN:
				menu->selected = (menu->selected + menu->num + 1) % menu->num;
				play_sound(SAMPLE_FLIP);
				break;
			case KEY_LEFT:
				r = menu->selected + menu->num;
				play_sound(SAMPLE_ENTER);
				break;
			case KEY_RIGHT:
				r = menu->selected + 2 * menu->num;
				play_sound(SAMPLE_ENTER);
				break;
			case KEY_ENTER:
			case KEY_ENTER_PAD:		
			case KEY_SPACE:
				r = menu->selected;
				play_sound(SAMPLE_ENTER);
				break;
			case KEY_ESC:
				r = menu->num - 1;
				break;
		}
	}
	return r;
}

/* draw menu to the screen */
void draw_menu(BITMAP* bmp) {
	int i, length, height;

	/* draw title */
	blit(bmp_title, bmp, 0, 0, SCREEN_W / 2 - bmp_title->w / 2, 30, bmp_title->w, bmp_title->h);
	if (animation_type == DIRTY_RECTANGLE) {
		add_to_list(&dirty, SCREEN_W / 2 - bmp_title->w / 2, 30, bmp_title->w, bmp_title->h);
	}

	/* draw current menu */
	LASER_MENU *menu = current_menu;
	height = text_height(font_laser);
	int dist = height * 4 / 3;
	int space = SCREEN_H - 90 - bmp_title->h - BOTTOM_SPACE;
	int menu_height = menu->num * height + (menu->num - 1) * (dist - height);
	int start = 60 + bmp_title->h + space / 2 - menu_height / 2;
	for (i=0; i<menu->num; i++) {
		/* draw text */
		int color = (i != menu->selected)? makecol(0xFF, 0xFF, 0xFF) : makecol(0xFF, 0xFF, 0x00);
		length = text_length(font_laser, menu->line[i]);
		int x = SCREEN_W / 2 - length / 2;
		int y = start + dist * i;
		textout(bmp, font_laser, menu->line[i], x, y, color);
		if (animation_type == DIRTY_RECTANGLE) {
			add_to_list(&dirty, x, y, length, height);
		}
		/* draw ball if selected */
		if (i == menu->selected) {
			draw_sprite(bmp, bmp_ball[0], x - bmp_ball[0]->w - 20, y + height / 2 - bmp_ball[0]->h / 2);
			if (animation_type == DIRTY_RECTANGLE) {
				add_to_list(&dirty, x - bmp_ball[0]->w - 20, y + height / 2 - bmp_ball[0]->h / 2, bmp_ball[0]->w, bmp_ball[0]->h);
			}
		}
	}

	/* draw scroller */
	rectfill(bmp, 0, SCREEN_H - BOTTOM_SPACE, SCREEN_W - 1, SCREEN_H - 1, bottom_bar_color);
	length = text_length(font_laser, scroll_message);
	height = text_height(font_laser);
	if (scroll_count > length)
		scroll_count = - SCREEN_W;
	textout(bmp,
		font_laser,
		scroll_message,
		-scroll_count,
		SCREEN_H - BOTTOM_SPACE / 2 - height / 2, 
		makecol(255, 255, 255));
	if (animation_type == DIRTY_RECTANGLE) {
		add_to_list(&dirty, 0, SCREEN_H - BOTTOM_SPACE, SCREEN_W, BOTTOM_SPACE);
	}

}

/* draw object games to the screen */
void draw_objects(BITMAP* bmp) {
	int i;
	
	/* draw the lasers */
	if (game_winner == NO_FINISH) {
		for (i=0; i<MAX_BALLS; i++) {
			if (laser_ball[i].laser > 0) {
				int length = laser_ball[i].laser;
				int x1 = (int)(laser_ball[i].x);
				int y1 = (int)(laser_ball[i].y);
				int x2 = (int)(x1 + length * fixtof(fcos(laser_ball[i].fix_angle())));
				int y2 = (int)(y1 + length * fixtof(fsin(laser_ball[i].fix_angle())));
				//line(bmp, x1, y1, x2, y2, getpixel(bmp_ball[i], bmp_ball[i]->w / 2, bmp_ball[i]->h / 2));
				//rotate_sprite(bmp, bmp_laser[i], (x1 + x2) / 2 - bmp_laser[i]->w / 2, (y1 + y2) / 2 - bmp_laser[i]->h / 2, laser_ball[i].fix_angle());
				BITMAP *bmp_las = create_sub_bitmap(bmp_laser[i], 0, 0,length, 3);
				rotate_sprite(bmp, bmp_las, (x1 + x2) / 2 - bmp_las->w / 2, (y1 + y2) / 2 - bmp_las->h / 2, laser_ball[i].fix_angle());
				destroy_bitmap(bmp_las);
				if (animation_type == DIRTY_RECTANGLE) {
					add_to_list(&dirty, (x1<x2?x1:x2) - 3, (y1<y2?y1:y2) - 3, (x1<x2?x2-x1:x1-x2) + 6, (y1<y2?y2-y1:y1-y2) + 6);
					//add_to_list(&dirty, (x1<x2?x1:x2) - 3, (y1<y2?y1:y2) - 3, MAX_LASER + 6, MAX_LASER + 6);
				}
			}
		}
	}
	/* draw the balls */
	if (game_winner == NO_FINISH) {
		for (i=0; i<MAX_BALLS; i++) {
			if (laser_ball[i].is_active()) {
				rotate_sprite(bmp, bmp_ball[i], (int)(laser_ball[i].x - bmp_ball[i]->w / 2), (int)(laser_ball[i].y - bmp_ball[i]->h / 2), laser_ball[i].fix_angle());
			}
			if (laser_ball[i].is_active()) {
				if (animation_type == DIRTY_RECTANGLE) {
					add_to_list(&dirty, (int)(laser_ball[i].x - bmp_ball[i]->w / 2), (int)(laser_ball[i].y - bmp_ball[i]->h / 2), bmp_ball[i]->w, bmp_ball[i]->h);
				}
			}
		}
	}

	/* draw life and energy bars */
	rectfill(bmp, 0, SCREEN_H - BOTTOM_SPACE, SCREEN_W - 1, SCREEN_H - 1, bottom_bar_color);
	for (i=0; i<MAX_BALLS; i++) {
		if (laser_ball[i].is_active()) {
			/* draw icons */
			draw_sprite(bmp, bmp_heart[i], 10 + SCREEN_W / 4 * i, SCREEN_H - BOTTOM_SPACE - 1 + 5);
			draw_sprite(bmp, bmp_gun[i], 11 + SCREEN_W / 4 * i, SCREEN_H - BOTTOM_SPACE - 1 + 25);
			/* draw life bar */
			if (laser_ball[i].life > 0) {
				rectfill(bmp,
				40 + SCREEN_W / 4 * i,
				SCREEN_H - BOTTOM_SPACE - 1 + 6,
				40 + SCREEN_W / 4 * i + laser_ball[i].life * (SCREEN_W / 4 - 50) / MAX_LIFE,
				SCREEN_H - BOTTOM_SPACE - 1 + 18,
				getpixel(bmp_heart[i], 10, 8));
			}
			/* draw energy bar */
			if (laser_ball[i].energy > 0) {
				rectfill(bmp,
				40 + SCREEN_W / 4 * i,
				SCREEN_H - BOTTOM_SPACE - 1 + 24,
				40 + SCREEN_W / 4 * i + laser_ball[i].energy * (SCREEN_W / 4 - 50) / MAX_ENERGY,
				SCREEN_H - BOTTOM_SPACE - 1 + 36,
				getpixel(bmp_heart[i], 10, 8));
			}
		}
	}
	if (animation_type == DIRTY_RECTANGLE) {
		add_to_list(&dirty, 0, SCREEN_H - BOTTOM_SPACE, SCREEN_W, BOTTOM_SPACE);
	}

	/* draw the winner */
	if (game_winner >= 0) {
		blit(bmp_winner, bmp, 0, 0, SCREEN_W / 2 - bmp_winner->w / 2, 30, bmp_winner->w, bmp_winner->h);
		if (animation_type == DIRTY_RECTANGLE) {
			add_to_list(&dirty, SCREEN_W / 2 - bmp_winner->w / 2, 30, bmp_winner->w, bmp_winner->h);
		}
		masked_blit(bmp_big[game_winner], bmp, 0, 0, SCREEN_W / 2 - bmp_big[game_winner]->w / 2, SCREEN_H / 2 - bmp_big[game_winner]->h / 2, bmp_big[game_winner]->w, bmp_big[game_winner]->h);
		if (animation_type == DIRTY_RECTANGLE) {
			add_to_list(&dirty, SCREEN_W / 2 - bmp_big[game_winner]->w / 2, SCREEN_H / 2 - bmp_big[game_winner]->h / 2, bmp_big[game_winner]->w, bmp_big[game_winner]->h);
		}
	}
}

/* the main game update function */
void update_objects() {
	int i, j;
	/* check input */
	for (i=0; i<MAX_BALLS; i++) {
		if(laser_ball[i].is_active()) {
			if(laser_ball[i].is_human()) {
				if(key[laser_ball[i].key[KEY_ROTATE_LEFT]]) 
					laser_ball[i].rotate_left();
				if(key[laser_ball[i].key[KEY_ROTATE_RIGHT]]) 
					laser_ball[i].rotate_right();
				if(key[laser_ball[i].key[KEY_GO_UP]]) 
					laser_ball[i].go_straight();
				if(key[laser_ball[i].key[KEY_GO_DOWN]]) 
					laser_ball[i].go_back();
				if(key[laser_ball[i].key[KEY_FIRE]])
					laser_ball[i].fire();
			}
			else {
				process_lb_cmd(&laser_ball[i], lb, lb->cmd, 0, 1, buf_error);
			}
		}
		int laser = laser_ball[i].laser;
		laser_ball[i].apply_commands();
		if (laser <= 0 && laser_ball[i].laser>0)
			play_sound(SAMPLE_LASER);
	}
	/* chek collisions */
	for (i=0; i<MAX_BALLS; i++) {
		if (laser_ball[i].is_active()) {
			for (j=i; j<MAX_BALLS; j++) {
				if (laser_ball[j].is_active()) {
					VECTOR rel = laser_ball[j].v;
					laser_ball[j].v = laser_ball[j].v.minus(rel);
					laser_ball[i].v = laser_ball[i].v.minus(rel);
					if(laser_ball[i].check_collision(laser_ball[j])) {
						laser_ball[i].collide(laser_ball[j]);
						play_sound(SAMPLE_BALL);
					}
					laser_ball[j].v = laser_ball[j].v.plus(rel);
					laser_ball[i].v = laser_ball[i].v.plus(rel);
				}
			}
		}
	}
	/* check laser hits */
	for (i=0; i<MAX_BALLS; i++) {
		if (laser_ball[i].is_active()) {
			int near_ball = -1;
			float near_dist = 10 * SCREEN_W;
			for (j=0; j<MAX_BALLS; j++) {
				if (laser_ball[j].is_active()) {
					if (i != j && laser_ball[i].check_laser(laser_ball[j])) {
						float dist = laser_ball[i].distance(laser_ball[j]);
						if (near_dist > dist) {
							near_ball = j;
							near_dist = dist;
						}
					}
				}
			}
			if (near_ball != -1) {
				laser_ball[near_ball].add_hit();
				if (laser_ball[i].laser > near_dist)
					laser_ball[i].laser = (int) near_dist;
				play_sound(SAMPLE_BURN);
			}
		}
	}
	/* update status */
	for (i=0; i<MAX_BALLS; i++) {
		bool active = laser_ball[i].is_active();
		if (active) {
			laser_ball[i].update();
		}
		if (!laser_ball[i].is_active() && active) {
			play_sound(SAMPLE_DIE);
		}
	}
}

/* set video mode */
void set_video(int w, int h, int d, bool win) {
	int depth = desktop_color_depth();
	int seq [] = {d, depth, 8, 15, 16, 24, 32};
	bool ok = false;
	for (int i=0; i<2; i++) {
		int mode = i == 0? 
			(win? GFX_AUTODETECT_WINDOWED : GFX_AUTODETECT_FULLSCREEN):
			(win? GFX_AUTODETECT_FULLSCREEN : GFX_AUTODETECT_WINDOWED);
		for (int j=0; j<7; j++) {
				set_color_depth(seq[i]);
				ok = (set_gfx_mode(mode, w, h, 0, 0) >= 0);
				if (ok) break;
		}
		if (ok) break;
	}
	if(!ok) {
			set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
			allegro_message("Unable to set graphics mode");
			allegro_exit();
	}
}

/* initialize the system */
void initialize_system(int w, int h, int d, bool win) {
	allegro_init();
	install_keyboard(); 
	install_mouse();
	install_timer();
	if (sound_install)
		install_sound(DIGI_AUTODETECT, MIDI_AUTODETECT, NULL);
	set_video(w, h, d, win);
	text_mode(-1);
	srand((unsigned) time(NULL));
	set_window_title("Laser Balls");
	set_display_switch_mode(SWITCH_BACKGROUND);
	set_volume(255, 200);
	set_mouse_speed(2, 2);
}

/* allegro remove routines */
void finalize_system() {
	set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
	if (sound_install)
		remove_sound();
	remove_timer();
	remove_mouse();
	remove_keyboard();
	allegro_exit();
}

/* initializes some game engine data */
void initialize_game() {
// ---------------------------------------
// Begin of code taken from Allegro Demo
// ---------------------------------------
	char buf[256];

	switch (animation_type) {

		case PAGE_FLIP:
			/* set up page flipping bitmaps */
			page1 = create_video_bitmap(SCREEN_W, SCREEN_H);
			page2 = create_video_bitmap(SCREEN_W, SCREEN_H);
			if ((!page1) || (!page2)) {
				set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
				allegro_message("Not enough video memory for page flipping\n");
				exit(1);
			}
			break;

		case RETRACE_FLIP: 
			/* set up retrace-synced page flipping bitmaps */
			if (!timer_can_simulate_retrace()) {
				set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
				allegro_message("Retrace syncing is not possible in this environment\n");
				exit(1);
			}
			timer_simulate_retrace(TRUE);
			page1 = create_video_bitmap(SCREEN_W, SCREEN_H);
			page2 = create_video_bitmap(SCREEN_W, SCREEN_H);
			if ((!page1) || (!page2)) {
				set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
				allegro_message("Not enough video memory for page flipping\n");
				exit(1);
			}
			break;

		case TRIPLE_BUFFER: 
			/* set up triple buffered bitmaps */
			if (!(gfx_capabilities & GFX_CAN_TRIPLE_BUFFER))
				enable_triple_buffer();
			if (!(gfx_capabilities & GFX_CAN_TRIPLE_BUFFER)) {
				strcpy(buf, gfx_driver->name);
				set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
				#ifdef ALLEGRO_DOS
				allegro_message("The %s driver does not support triple buffering\n\nTry using mode-X in clean DOS mode (not under Windows)\n", buf);
				#else
				allegro_message("The %s driver does not support triple buffering\n", buf);
				#endif
				exit(1);
			}
			page1 = create_video_bitmap(SCREEN_W, SCREEN_H);
			page2 = create_video_bitmap(SCREEN_W, SCREEN_H);
			page3 = create_video_bitmap(SCREEN_W, SCREEN_H);
			if ((!page1) || (!page2) || (!page3)) {
				set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
				allegro_message("Not enough video memory for triple buffering\n");
				exit(1);
			}
			break;
	}

	use_retrace_proc = timer_is_using_retrace();

	LOCK_VARIABLE(game_time);
	LOCK_FUNCTION(game_timer);
	LOCK_VARIABLE(scroll_count);
	LOCK_FUNCTION(scroll_counter);
	LOCK_VARIABLE(frame_count);
	LOCK_VARIABLE(fps);
	LOCK_FUNCTION(fps_proc);

	s = create_bitmap(SCREEN_W, SCREEN_H);
// ---------------------------------------
// End of code taken from Allegro Demo
// ---------------------------------------
}

/* destroys game engine data */
void finalize_game() {
   destroy_bitmap(s);
   if ((animation_type == PAGE_FLIP) || (animation_type == RETRACE_FLIP)) {
      destroy_bitmap(page1);
      destroy_bitmap(page2);
   }
   else if (animation_type == TRIPLE_BUFFER) {
      destroy_bitmap(page1);
      destroy_bitmap(page2);
      destroy_bitmap(page3);
   }
}

/* creates game objects data */
void initialize_data() {
	int i;
  char buf[256], path[256];
	/* loading datafile */
	//data = load_datafile("LaserBalls.dat");
  get_executable_name(path, sizeof(path));
  data = load_datafile(replace_filename(buf, path, "LaserBalls.dat", sizeof(buf)));
  if (!data) {
    allegro_exit();
    allegro_message("Error loading data file `LaserBalls.dat'...\n");
    exit(1);
  }
	/* set config file */
  get_executable_name(path, sizeof(path));
  set_config_file(replace_filename(buf, path, "LaserBalls.cfg", sizeof(buf)));
	/* loading palette*/ 
	for (i=0; i<256; i++) 
		pal[i] = ((RGB * ) data[PAL].dat)[i];
	set_palette(pal);
	int color1[] = {
		makecol(0xFF, 0x00, 0x00),
		makecol(0x00, 0x00, 0xFF),
		makecol(0x00, 0xFF, 0x00),
		makecol(0xFF, 0xFF, 0x00)
	};
	int color2[] = {
		makecol(0xAA, 0x00, 0x00),
		makecol(0x00, 0x00, 0x88),
		makecol(0x00, 0x88, 0x00),
		makecol(0xAA, 0xAA, 0x00)
	};
	bottom_bar_color = makecol(0, 0, 0);
	for (i=0; i<MAX_BALLS; i++) {
		laser_ball[i].random_position();
		/* creating balls bitmaps */
		bmp_ball[i] = (BITMAP*) data[BALL_1 + i].dat;
		/* creating big balls bitmaps */
		bmp_big[i] = (BITMAP*) data[BIG_BALL_1 + i].dat;
		/* creating hearts and guns bitmaps */
		bmp_gun[i] = (BITMAP*) data[GUN_1 + i].dat;
		bmp_heart[i] = (BITMAP*) data[HEART_1 + i].dat;
		/* creating laser bitmaps */
		bmp_laser[i] = create_bitmap(MAX_LASER, 3);
		line(bmp_laser[i], 0, 0, bmp_laser[i]->w - 1, 0, color2[i]);
		line(bmp_laser[i], 0, 1, bmp_laser[i]->w - 1, 1, color1[i]);
		line(bmp_laser[i], 0, 2, bmp_laser[i]->w - 1, 2, color2[i]);
	}
	/* creating backgrounds */
	bmp_title = (BITMAP*) data[TITLE].dat;
	bmp_winner = (BITMAP*) data[WINNER].dat;
	bmp_texture = (BITMAP*) data[TEXTURE].dat;
	bmp_back = create_bitmap(SCREEN_W, SCREEN_H);
	drawing_mode(DRAW_MODE_COPY_PATTERN, bmp_texture, 0, 0);
	rectfill(bmp_back, 0, 0, SCREEN_W - 1, SCREEN_H - BOTTOM_SPACE - 1, 0);
	drawing_mode(DRAW_MODE_SOLID, NULL, 0, 0);
	/* creating music */
	music = (MIDI*) data[MUSIC].dat;
	/* creating font */
	font_laser = (FONT*) data[FONT_LASER].dat;
}

/* destroys game objects data */
void finalize_data() {
	destroy_bitmap(bmp_back);
	unload_datafile(data);
}

/* create the players menu */
void build_players_menu(char m[MAX_BALLS][40]) {
	for (int i=0; i<MAX_BALLS; i++) {
		sprintf(m[i], "PLAYER %d: %s", i + 1, players_types[players_controls[i]]);
		menu_players->line[i] = m[i];
	}
}

/* create the keys menu */
void build_keys_menu(char m[5][40], int c) {
	for (int i=0; i<5; i++) {
		sprintf(m[i], "%s: %s", keys_desc[i], get_control_name(keys_controls[c][i]));
		menu_config->line[i] = m[i];
	}
}

/* create the menus */
void initialize_menu() {
	players_types[0] = "KEYS 1";
	players_types[1] = "KEYS 2";
	players_types[2] = "KEYS 3";
	players_types[3] = "KEYS 4";
	players_types[4] = "NONE";
	players_types[5] = "CPU";
	n_players_types = 6;
	menu_main = new LASER_MENU();
	menu_main->num = 4;
	menu_main->selected = 0;
	menu_main->line[0] = "START";
	menu_main->line[1] = "PLAYERS";
	menu_main->line[2] = "OPTIONS";
	menu_main->line[3] = "EXIT";
	menu_options = new LASER_MENU();
	menu_options->num = 4;
	menu_options->selected = 0;
	menu_options->line[0] = (char*)(sound_enabled? "SOUND ON" : "SOUND OFF");
	menu_options->line[1] = (char*)(music_enabled? "MUSIC ON" : "MUSIC OFF");
	menu_options->line[2] = "CONTROLS";
	menu_options->line[3] = "BACK";
	menu_players = new LASER_MENU();
	menu_players->num = 5;
	menu_players->selected = 0;
	menu_players->line[0] = "PLAYER 1";
	menu_players->line[1] = "PLAYER 2";
	menu_players->line[2] = "PLAYER 3";
	menu_players->line[3] = "PLAYER 4";
	menu_players->line[4] = "BACK";
	menu_controls = new LASER_MENU();
	menu_controls->num = 5;
	menu_controls->selected = 0;
	menu_controls->line[0] = "KEYS 1";
	menu_controls->line[1] = "KEYS 2";
	menu_controls->line[2] = "KEYS 3";
	menu_controls->line[3] = "KEYS 4";
	menu_controls->line[4] = "BACK";
	menu_config = new LASER_MENU();
	menu_config->num = 6;
	menu_config->selected = 0;
	menu_config->line[0] = "UP";
	menu_config->line[1] = "DOWN";
	menu_config->line[2] = "LEFT";
	menu_config->line[3] = "RIGHT";
	menu_config->line[4] = "FIRE";
	menu_config->line[5] = "BACK";
}

/* destroy menus */
void finalize_menu() {
	delete menu_main;
	delete menu_options;
	delete menu_players;
	delete menu_controls;
	delete menu_config;
}

/* shows the title screen */
bool play_menu() {
	bool esc = false, play = false;
	int current_keys = 0;
	char buf_player[MAX_BALLS][40];
	char buf_keys[5][40];

	/* set up the interrupt routines... */
	install_int(fps_proc, 1000);
	install_int(scroll_counter, 10);
	if (use_retrace_proc)
		retrace_proc = game_timer;
	else
		install_int(game_timer, 10);
	game_time = 0;
	scroll_count = - SCREEN_W;

	/* show the scree */
	current_menu = menu_main;
	first_frame = true;
	draw_screen();
	fade_in(pal, FADE_SPEED);

	/* reset keyboard */
	while (keypressed()) {
		readkey();
	}

	/* title screen loop */
	while(!esc && !play) {
		poll_keyboard();
		while (game_time > 0) {
			int c = check_menu(current_menu);
			/* main menu */
			if (current_menu == menu_main) {
				switch (c) {
					case 0:
						play = true;
						break;
					case 1:
						current_menu = menu_players;
						current_menu->selected = 0;
						build_players_menu(buf_player);
						break;
					case 2:
						current_menu = menu_options;
						current_menu->selected = 0;
						break;
					case 3:
						esc = true;
						break;
				}
			}
			/* players menu */
			else if (current_menu == menu_players) {
				if (c >= 0 && c < menu_players->num - 1 || c >= 2 * menu_players->num && c < 3 * menu_players->num - 1) {
					int i = c % menu_players->num;
					players_controls[i] = (players_controls[i] + 1) % n_players_types;
					build_players_menu(buf_player);
				}
				if (c >= menu_players->num && c < 2 * menu_players->num - 1) {
					int i = c % menu_players->num;
					players_controls[i] = (players_controls[i] + n_players_types - 1) % n_players_types;
					build_players_menu(buf_player);
				}
				if (c == menu_players->num - 1) {
					current_menu = menu_main;
					save_options();
				}
			}
			/* options menu */
			else if (current_menu == menu_options) {
				switch (c) {
					case 0:
						sound_enabled = !sound_enabled;
						menu_options->line[0] = (char*)(sound_enabled? "SOUND ON" : "SOUND OFF");
						break;
					case 1:
						music_enabled = !music_enabled;
						menu_options->line[1] = (char*)(music_enabled? "MUSIC ON" : "MUSIC OFF");
						if (music_enabled && !music_playing && sound_install) {
							play_midi(music, 1);
							music_playing = true;
						}
						else if (!music_enabled && music_playing && sound_install) {
							//stop_midi();
							play_midi(NULL, FALSE);
							music_playing = false;
						}
						break;
					case 2:
						current_menu = menu_controls;
						current_menu->selected = 0;
						break;
				}
				if (c == menu_options->num - 1) {
					current_menu = menu_main;
					save_options();
				}
			}
			/* controls menu */
			else if(current_menu == menu_controls) {
				if (c < menu_controls->num - 1 && c >= 0) {
					current_keys = c;
					current_menu = menu_config;
					current_menu->selected = 0;
					build_keys_menu(buf_keys, current_keys);
				}
				if (c == menu_controls->num - 1) {
					current_menu = menu_options;
				}
			}
			/* controls config menu */
			else if(current_menu == menu_config) {
				if (c >= 0 && c < menu_config->num - 1) {
					int new_control = -1, i;
					/*clear_keybuf();
					for (i=0; i<KEY_MAX; i++) {
						key[i] = 0;
					}*/
					clear_keybuf();
					do {
						poll_keyboard();
					}
					while (key[KEY_ENTER] || key[KEY_ENTER_PAD] || key[KEY_SPACE]);
					while (new_control < 0) {
						poll_keyboard();
						for (i=CONTROL_KEY_FIRST; i<=CONTROL_KEY_LAST; i++) {
							if (key[i]) {
								new_control = i;
								break;
							}
						}
					}
					clear_keybuf();
					do {
						poll_keyboard();
					}
					while (key[new_control]);
					keys_controls[current_keys][c] = new_control;
					build_keys_menu(buf_keys, current_keys);
				}
				if (c == menu_config->num - 1) {
					current_menu = menu_controls;
				}
			}
			game_time--;
		}
		frame_count++;
		draw_screen();
	}

	/* clean up */
	remove_int(fps_proc);
	remove_int(scroll_counter);
	if (use_retrace_proc)
		retrace_proc = NULL;
	else
		remove_int(game_timer);

	while (current_page != 0) {
		draw_screen();
	}
	show_video_bitmap(screen);
	if ((animation_type == PAGE_FLIP) || (animation_type == RETRACE_FLIP) || (animation_type == TRIPLE_BUFFER)) {
		blit(screen, s, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
	}
	fade_out(FADE_SPEED);
	fps = 0;

	if (esc) return false;
	else return true;
}

/* checks if the game is finished */
int check_winner() {
	int count = 0, winner = -1;
	for (int i=0; i<MAX_BALLS; i++) {
		if (laser_ball[i].is_active()) {
			count++;
			winner = i;
		}
	}
	return (count == 1)? winner : (count == 0) ? NO_WINNER : NO_FINISH;
}

/* sets up and plays the game */
void play_game() {
	bool end = false, sound = false;

	/* set up the interrupt routines... */
	install_int(fps_proc, 1000);
	if (use_retrace_proc)
		retrace_proc = game_timer;
	else
		install_int(game_timer, 10);

	/* start the game */
	game_time = 0;
	game_winner = NO_FINISH;
	first_frame = true;
	draw_screen();
	fade_in(pal, FADE_SPEED);
	play_sound(SAMPLE_START);

	/* main game loop */
	while (!end) {
		poll_keyboard();
		while (game_time > 0) {
			if (game_winner == NO_FINISH) {
				update_objects();
			}
			game_time--;
		}
		draw_screen();
		frame_count++;
		if (game_winner == NO_FINISH) {
			game_winner = check_winner();
		}
		if (game_winner >= 0 || game_winner == NO_WINNER) {
			if (!sound && game_winner >= 0) {
				play_sound(SAMPLE_END);
			}
			if (!sound) {
				clear_keybuf();
				do {
					poll_keyboard();
				}	while (key[KEY_ESC] || key[KEY_ENTER] || key[KEY_SPACE] || key[KEY_ENTER_PAD]);
			}
			if (key[KEY_ENTER] || key[KEY_ENTER_PAD] || key[KEY_SPACE]) {
				end = true;
			}
			sound = true;
		}
		if (key[KEY_ESC])
			end = true;
	}

	/* cleanup */
	remove_int(fps_proc);
	if (use_retrace_proc)
		retrace_proc = NULL;
	else
		remove_int(game_timer);
	do {
		poll_keyboard();
	}	while (key[KEY_ESC] || key[KEY_ENTER] || key[KEY_SPACE] || key[KEY_ENTER_PAD]);
	while (current_page != 0) {
		draw_screen();
	}
	show_video_bitmap(screen);
	if ((animation_type == PAGE_FLIP) || (animation_type == RETRACE_FLIP) || (animation_type == TRIPLE_BUFFER)) {
		blit(screen, s, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
	}
	fade_out(FADE_SPEED);
	fps = 0;
}

/* returns the name of a key */
char *get_control_name(int control)	{
	static char *key_name[] =
	{
		"NULL",
		"KEY A",
		"KEY B",
		"KEY C",
		"KEY D",
		"KEY E",
		"KEY F",
		"KEY G",
		"KEY H",
		"KEY I",
		"KEY J",
		"KEY K",
		"KEY L",
		"KEY M",
		"KEY N",
		"KEY O",
		"KEY P",
		"KEY Q",
		"KEY R",
		"KEY S",
		"KEY T",
		"KEY U",
		"KEY V",
		"KEY W",
		"KEY X",
		"KEY Y",
		"KEY Z",
		"KEY 0",
		"KEY 1",
		"KEY 2",
		"KEY 3",
		"KEY 4",
		"KEY 5",
		"KEY 6",
		"KEY 7",
		"KEY 8",
		"KEY 9",
		"KEY 0 PAD",
		"KEY 1 PAD",
		"KEY 2 PAD",
		"KEY 3 PAD",
		"KEY 4 PAD",
		"KEY 5 PAD",
		"KEY 6 PAD",
		"KEY 7 PAD",
		"KEY 8 PAD",
		"KEY 9 PAD",
		"KEY F1",
		"KEY F2",
		"KEY F3",
		"KEY F4",
		"KEY F5",
		"KEY F6",
		"KEY F7",
		"KEY F8",
		"KEY F9",
		"KEY F10",
		"KEY F11",
		"KEY F12",
		"KEY ESC",
		"KEY TILDE",
		"KEY MINUS",
		"KEY EQUALS",
		"KEY BACKSPACE",
		"KEY TAB",
		"KEY OPENBRACE",
		"KEY CLOSEBRACE",
		"KEY ENTER",
		"KEY COLON",
		"KEY QUOTE",
		"KEY BACKSLASH",
		"KEY BACKSLASH2",
		"KEY COMMA",
		"KEY STOP",
		"KEY SLASH",
		"KEY SPACE",
		"KEY INSERT",
		"KEY DEL",
		"KEY HOME",
		"KEY END",
		"KEY PGUP",
		"KEY PGDN",
		"KEY LEFT",
		"KEY RIGHT",
		"KEY UP",
		"KEY DOWN",
		"KEY SLASH PAD",
		"KEY ASTERISK",
		"KEY MINUS PAD",
		"KEY PLUS PAD",
		"KEY DEL PAD",
		"KEY ENTER PAD",
		"KEY PRTSCR",
		"KEY PAUSE",
		"KEY ABNT_C1",
		"KEY YEN",
		"KEY KANA",
		"KEY CONVERT",
		"KEY NOCONVERT",
		"KEY AT",
		"KEY CIRCUMFLEX",
		"KEY COLON2",
		"KEY KANJI",
	/*"KEY MODIFIERS",*/
		"KEY LSHIFT",
		"KEY RSHIFT",
		"KEY LCONTROL",
		"KEY RCONTROL",
		"KEY ALT",
		"KEY ALTGR",
	/* NO!
		"KEY LWIN",
		"KEY RWIN",
		"KEY MENU",
		"KEY SCRLOCK",
		"KEY NUMLOCK",
		"KEY CAPSLOCK",
	*/
	};
	return key_name[control];
}

/* reset players status */
void reset_players() {
	int i, j;
	bool overlap;
	for (i=0; i<MAX_BALLS; i++) {
		laser_ball[i].restart();
		/* check for overlap */
		do {
			laser_ball[i].random_position();
			overlap = false;
			for (j=0; j<MAX_BALLS; j++) {
				if (i != j) {
					if (laser_ball[i].distance(laser_ball[j]) <= laser_ball[i].radius + laser_ball[j].radius) {
						overlap = true;
						break;
					}
				}
			}
		}
		while (overlap);
		/* enable / disable players */
		if (players_controls[i] < 0 || players_controls[i] == 4) {
			laser_ball[i].active = false;
		}
		if (players_controls[i] > 4) {
			laser_ball[i].human = false;
		}
		else {
			laser_ball[i].human =true;
		}
		/* set keys */
		if (players_controls[i] >= 0 && players_controls[i] < MAX_BALLS) {
			laser_ball[i].set_keys(
				keys_controls[players_controls[i]][KEY_GO_UP],
				keys_controls[players_controls[i]][KEY_GO_DOWN],
				keys_controls[players_controls[i]][KEY_ROTATE_LEFT],
				keys_controls[players_controls[i]][KEY_ROTATE_RIGHT],
				keys_controls[players_controls[i]][KEY_FIRE]
			);
		}
	}
}

/* play a sample */
void play_sound(int s) {
	SAMPLE *sam = (SAMPLE*) data[s].dat;
	if (sam && sound_enabled && sound_install)
		play_sample(sam, 255, 128, 1000, 0);
}

/* load options */
void load_options() {
	char buf [20];
	sound_enabled = get_config_int("Audio", "Sound", 1)? true : false;
	music_enabled = get_config_int("Audio", "Music", 1)? true : false;
	for (int i=0; i<MAX_BALLS; i++) {
		sprintf(buf, "Player%d", i + 1);
		for (int j=0; j<5; j++) {
			keys_controls[i][j] = get_config_int(buf, keys_desc[j], keys_controls[i][j]);
		}
		players_controls[i] = get_config_int(buf, "TYPE", i);
	}
}

/* save options */
void save_options() {
	char buf [20];
	set_config_int("Audio", "Sound", sound_enabled? 1 : 0);
	set_config_int("Audio", "Music", music_enabled? 1 : 0);
	for (int i=0; i<MAX_BALLS; i++) {
		sprintf(buf, "Player%d", i + 1);
		for (int j=0; j<5; j++) {
			set_config_int(buf, keys_desc[j], keys_controls[i][j]);
		}
		set_config_int(buf, "TYPE", players_controls[i]);
	}
	flush_config_file();
}

/* print command line options */
void print_usage(char *name, int status) {
	if (!status) {
		fprintf(stdout, "\
Laser Balls, Copyright (C) 2002, by Marco Chiesi\n\
\n\
Usage:\n\
  %s [OPTIONS]\n\
\n\
Options:\n\
  -r, --resolution WIDTH HEIGHT   use a special resolution (in 8 bpp)\n\
  -u, --update METHOD             select screen update method\n\
  -f, --fps                       show fps informations\n\
  -w, --window                    try to start the program in a window\n\
  -s, --nosound                   do not install sounds\n\
  -h, --help                      display this help and exit\n\
      --version                   output version information and exit\n\
\n\
Availale update methods are:\n\
DOUBLE_BUFFER, PAGE_FLIP, RETRACE_FLIP, TRIPLE_BUFFER, DIRTY_RECTANGLE\n\
The default is DIRTY_RECTANGLE\n\
\n\
Report bugs to chiesi@libero.it.\n\
", name);
	}
	else {
		fprintf(stderr, "Try `%s --help' for more information.\n", name);
	}
	exit(status);
}


/* process command line options */
void process_arguments(int argc, char *argv[]) {
	int i, j, k, w = 0, h = 0, z;
	char modes[6][20] = {"", "DOUBLE_BUFFER", "PAGE_FLIP", "RETRACE_FLIP", "TRIPLE_BUFFER", "DIRTY_RECTANGLE"};
	for (i=1; i<argc; i++) {
		if (argv[i][0] == '-') {
			if (argv[i][1] == '-') {
				if (stricmp(argv[i], "--resolution") == 0) {
					if (i+2 >= argc) {
						fprintf(stderr, "%s: `%s' need two parameters\n", argv[0], argv[i]);
						print_usage(argv[0], 1);
					}
					else {
						w = strtol(argv[++i], (char **)NULL, 10);
						h = strtol(argv[++i], (char **)NULL, 10);
					}
				}
				else if (stricmp(argv[i], "--nosound") == 0) {
					sound_install = false;
				}
				else if (stricmp(argv[i], "--help") == 0) {
					print_usage(argv[0], 0);
				}
				else if (stricmp(argv[i], "--version") == 0) {
					fprintf(stdout, "Laser Balls 1.0\n");
					exit(0);
				}
				else if (stricmp(argv[i], "--update") == 0) {
					if (i+1 >= argc) {
						fprintf(stderr, "%s: `%s' need a parameters\n", argv[0], argv[i]);
						print_usage(argv[0], 1);
					}
					z = ++i;
					for (k=DOUBLE_BUFFER; k <= DIRTY_RECTANGLE; k++) {
						if (stricmp(argv[z], modes[k]) == 0) {
							update_mode = k;
							break;
						}
					}
				}
				else if (stricmp(argv[i], "--fps") == 0) {
					show_fps = true;
				}
				else if (stricmp(argv[i], "--window") == 0) {
					windowed = true;
				}
				else {
					fprintf(stderr, "%s: invalid option `%s'\n", argv[0], argv[i]);
					print_usage(argv[0], 1);
				}
			}
			else {
				for (j=1; (j > 0) && (argv[i][j]); j++) {
					switch (argv[i][j]) {
						case 'r':
							if (i+2 >= argc) {
								fprintf(stderr, "%s: `%c' need two parameters\n", argv[0], argv[i][j]);
								print_usage(argv[0], 1);
							}
							else {
								w = strtol(argv[++i], (char **)NULL, 10);
								h = strtol(argv[++i], (char **)NULL, 10);
								j = -1;
							}
							break;
						case 's':
							sound_install = false;
							break;
						case 'u':
							if (i+1 >= argc) {
								fprintf(stderr, "%s: `%c' need a parameters\n", argv[0], argv[i][j]);
								print_usage(argv[0], 1);
							}
							z = ++i;
							for (k=DOUBLE_BUFFER; k <= DIRTY_RECTANGLE; k++) {
								if (stricmp(argv[z], modes[k]) == 0) {
									update_mode = k;
									break;
								}
							}
							j = -1;
							break;
						case 'f':
							show_fps = true;
							break;
						case 'w':
							windowed = true;
							break;
						case 'h':
						case '?':
							print_usage(argv[0], 0);
							break;
						default:
							fprintf(stderr, "%s: invalid option -- %c\n", argv[0], argv[i][j]);
							print_usage(argv[0], 1);
							break;
					}
				}
			}
		}
		else {
			fprintf(stderr, "%s: invalid argument `%s'\n", argv[0], argv[i]);
			print_usage(argv[0], 1);
		}
	}
	if ((w == 640 && h == 480) || (w == 800 && h == 600) || (w == 1024 && h == 768)) {
		res_width = w;
		res_height = h;
	}
	animation_type = update_mode;
}

/* main function */
int main(int argc, char *argv[]) {
	animation_type = DIRTY_RECTANGLE;
	process_arguments(argc, argv);
	initialize_system(res_width, res_height, 8, windowed);
	x_min = 20;
	x_max = SCREEN_W - 20;
	y_min = 20;
	y_max = SCREEN_H - 20 - BOTTOM_SPACE;
	initialize_game();
	initialize_data();
	initialize_menu();
	load_options();
	if (music_enabled && sound_install) {
		play_midi(music ,1);
		music_playing = true;
	}
	game_mode = MODE_MENU;
	
	lb = load_lb("cpu.lb", buf_error);
	printf(buf_error);
	
	while (play_menu()) {
		reset_players();
		game_mode = MODE_GAME;
		play_game(); 
		game_mode = MODE_MENU;
	}
	

	finalize_menu();
	finalize_data();
	finalize_game();
	finalize_system();
	return 0;
}

END_OF_MAIN();
