#include "LaserBall.h"

/* laser ball class constructor */
LASER_BALL::LASER_BALL() : human(true), active(true), laser(0), x(100), y(100), angle(0), radius(20),
 life(MAX_LIFE), v(0,0), a(0,0), hits(0), energy(MAX_ENERGY), reloading(false), 
		command_go(0), command_rotate(0), command_fire(0) {
	key[KEY_GO_UP] = KEY_UP;
	key[KEY_GO_DOWN] = KEY_DOWN;
	key[KEY_ROTATE_LEFT] = KEY_LEFT;
	key[KEY_ROTATE_RIGHT] = KEY_RIGHT;
	key[KEY_FIRE] = KEY_SPACE;
}

/* restart laser ball variables */
void LASER_BALL::restart() {
	active = true;
	laser = 0;
	life = MAX_LIFE;
	energy = MAX_ENERGY;
	v = VECTOR(0, 0);
	a = VECTOR(0, 0);
	hits = 0;
	reloading = false;
	command_go = COMMAND_NONE;
	command_rotate = COMMAND_NONE;
	command_fire = COMMAND_NONE;
}

/* rotate left command */
void LASER_BALL::rotate_left() {
	if (command_rotate == COMMAND_NONE) 
		command_rotate = COMMAND_ROTATE_LEFT;
}

/* rotate right command */
void LASER_BALL::rotate_right() {
	if (command_rotate == COMMAND_NONE) 
		command_rotate = COMMAND_ROTATE_RIGHT;
}

/* fire command */
void LASER_BALL::fire() {
	command_fire = COMMAND_FIRE;
}

/* go back command */
void LASER_BALL::go_back() {
	if (command_go == COMMAND_NONE) 
		command_go = COMMAND_GO_BACK;
}

/* go straight command */
void LASER_BALL::go_straight() {
	if (command_go == COMMAND_NONE) 
		command_go = COMMAND_GO_STRAIGHT;
}

/* angle position in fixed format */
fixed LASER_BALL::fix_angle() {
	return ftofix(angle * 256.0 / 360.0);
}

/* angle posiotion in radians */
float LASER_BALL::rad_angle() {
	return angle * LASER_PI / 180.0;
}

/* adjust angle from 0 to 359 */
void LASER_BALL::adjust_angle() {
	while (angle < 0)
		angle += 360;
	while (angle >= 360)
		angle -= 360;
}

/* apply commands to laser balls */
void LASER_BALL::apply_commands() {
	float dir;
	switch(command_rotate) {
		case COMMAND_ROTATE_LEFT:
			if (angle > 0) angle -= VAR_ANGLE;
			else angle = 359;
			break;
		case COMMAND_ROTATE_RIGHT:
			if (angle < 359) angle += VAR_ANGLE;
			else angle = 0;
			break;
	}
	switch (command_go) {
		case COMMAND_GO_BACK:
			dir = ((float)angle) / 180.0 * LASER_PI;
			a = VECTOR(- cos(dir), - sin(dir));
			a = a.times(VAR_SPEED);
			break;
		case COMMAND_GO_STRAIGHT:
			dir = ((float)angle) / 180.0 * LASER_PI;
			a = VECTOR(cos(dir), sin(dir));
			a = a.times(VAR_SPEED);
			break;
		case COMMAND_NONE:
			if (v.magnitude() > 0) {
				float dir = v.angle();
				a = VECTOR(- cos(dir), - sin(dir));
				a = a.times(VAR_SPEED / 2);
			}
			else {
				a = VECTOR(0, 0);
			}
	}
	if (command_fire == COMMAND_FIRE && !reloading) {
		laser += VAR_LASER;
		energy -= VAR_ENERGY;
	}
	else {
		laser -= VAR_LASER;
	}
	if (laser < 0) laser =0;
	if (laser > MAX_LASER) laser = MAX_LASER;
	command_go = COMMAND_NONE;
	command_rotate = COMMAND_NONE;
	command_fire = COMMAND_NONE;
}

/* update ball status */
void LASER_BALL::update() {
	v = v.plus(a);
	if (v.magnitude() > MAX_SPEED) {
		v = v.times(MAX_SPEED / v.magnitude());
	}
	if (v.magnitude() < 3 * VAR_SPEED / 4) {
		v = VECTOR(0, 0);
	}
	x += v.x;
	y += v.y;
	if (x < x_min) {
		x = x_min;
		v.x = - v.x;
	}
	if (y < y_min) {
		y = y_min;
		v.y = - v.y;
	}
	if (x > x_max) {
		x = x_max;
		v.x = - v.x;
	}
	if (y > y_max) {
		y = y_max;
		v.y = - v.y;
	}
	if (hits > 0) {
		if (life > 0)
			life -= hits * VAR_LIFE;
		if (life <= 0) 
			active = false;
		hits = 0;
	}
	if (energy <= 0) {
		reloading = true;
	}
	if (energy >= MAX_ENERGY) {
		reloading = false;
		energy = MAX_ENERGY;
	}
	if (reloading) {
		energy += VAR_ENERGY;
	}
}

/* position the ball */
void LASER_BALL::position(int x, int y) {
	this->x = x;
	this->y = y;
}

/* random position the ball */
void LASER_BALL::random_position() {
	position(x_min + rand() % (x_max-x_min), y_min + rand() % (y_max-y_min));
	angle = rand() % 360;
}

/* tells if the ball is active */
bool LASER_BALL::is_active() {
	return active;
}

/* tells if the ball is human */
bool LASER_BALL::is_human() {
	return human;
}

/* set keys controls */
void LASER_BALL::set_keys(int up, int down, int left, int right, int fire) {
	key[KEY_GO_UP] = up;
	key[KEY_GO_DOWN] = down;
	key[KEY_ROTATE_LEFT] = left;
	key[KEY_ROTATE_RIGHT] = right;
	key[KEY_FIRE] = fire;
}

/* distance between balls */
float LASER_BALL::distance(LASER_BALL& b) {
	return sqrt((x - b.x) * (x - b.x) + (y - b.y) * (y - b.y));
}

/* check for collision between balls */
bool LASER_BALL::check_collision(LASER_BALL& b) {
	/* Algorithm taken from GAMASUTRA */
	/* http://www.gamasutra.com/features/20020118/vandenhuevel_pfv.htm */
	VECTOR movevec = v;
	// Early Escape test: if the length of the movevec is less
	// than distance between the centers of these circles minus 
	// their radii, there's no way they can hit. 
	double dist = distance(b);
	double sumRadii = (radius + b.radius);
	dist -= sumRadii;
	if(movevec.magnitude() < dist){
		return false;
	}
	// Normalize the movevec
	VECTOR normal = movevec.copy();
	normal.normalize();
	// Find C, the vector from the center of the moving 
	// circle A to the center of B
	VECTOR centers (b.x - x, b.y - y);
	// D = N . C = ||C|| * cos(angle between N and C)
	double d = normal.dot(centers);
	// Another early escape: Make sure that A is moving 
	// towards B! If the dot product between the movevec and 
	// B.center - A.center is less that or equal to 0, 
	// A isn't isn't moving towards B
	if(d <= 0){
		return false;
	} 
	// Find the length of the vector C
	double c = centers.magnitude();
	double f = (c * c) - (d * d);
	// Escape test: if the closest that A will get to B 
	// is more than the sum of their radii, there's no 
	// way they are going collide
	double sumRadiiSquared = sumRadii * sumRadii;
	if(f >= sumRadiiSquared){
		return false;
	}
	// We now have F and sumRadii, two sides of a right triangle. 
	// Use these to find the third side, sqrt(T)
	double t = sumRadiiSquared - f;
	// If there is no such right triangle with sides length of 
	// sumRadii and sqrt(f), T will probably be less than 0. 
	// Better to check now than perform a square root of a 
	// negative number. 
	if(t < 0){
		return false;
	}
	// Therefore the distance the circle has to travel along 
	// movevec is D - sqrt(T)
	double distance = d - sqrt(t);
	// Get the magnitude of the movement vector
	double mag = movevec.magnitude();
	// Finally, make sure that the distance A has to move 
	// to touch B is not greater than the magnitude of the 
	// movement vector. 
	if(mag < distance){
		return false;
	}
	// Set the length of the movevec so that the circles will just 
	// touch
	//movevec.normalize();
	movevec = movevec.times(distance / movevec.magnitude());
	return true; 
}

/* compute collision between balls */
void LASER_BALL::collide(LASER_BALL& b) {
	// First, find the normalized vector n from the center of 
	// circle1 to the center of circle2
	VECTOR n (x -b.x, y - b.y);
	n.normalize(); 
	// Find the length of the component of each of the movement
	// vectors along n. 
	// a1 = v1 . n
	// a2 = v2 . n
	float a1 = v.dot(n);
	float a2 = b.v.dot(n);
	// Using the optimized version, 
	// optimizedP =  2(a1 - a2)
	//              -----------
	//                m1 + m2
	float optimized_p = a1 - a2;
	// Calculate v1', the new movement vector of circle1
	// v1' = v1 - optimizedP * m2 * n
	v = v.minus(n.times(optimized_p));
	// Calculate v1', the new movement vector of circle1
	// v2' = v2 + optimizedP * m1 * n
	b.v = b.v.plus(n.times(optimized_p));
}

/* check for laser collision */
bool LASER_BALL::check_laser(LASER_BALL& b) {
	if (laser <= 0) return false;
	/* calculate distance bnetween balls */
	if (distance(b) > laser + b.radius) return false;
	/* calculate distance between ball and laser line */
	float x1 = x;
	float y1 = y;
	float x2 = x + laser * fixtof(fcos(fix_angle()));
	float y2 = y + laser * fixtof(fsin(fix_angle()));
	float line_a = y1 - y2;
	float line_b = x2 - x1;
	float line_c = x1 * (y2 - y1) - y1 * (x2 - x1);
	float dist_line = fabs(line_a * b.x + line_b * b.y + line_c) / sqrt(line_a * line_a + line_b * line_b);
	/* check if the ball is in the right direction */
	VECTOR v_laser(fixtof(fcos(fix_angle())), fixtof(fsin(fix_angle())));
	VECTOR v_dist(b.x - x, b.y - y);
	float prod = v_laser.dot(v_dist);
	/* return result */
	return dist_line < b.radius && prod > 0;
}

/* adds a hit */
void LASER_BALL::add_hit() {
	hits++;
}
