#include <sys/farptr.h>
#include <go32.h>
#include <math.h>
#include <conio.h>
#include <time.h>
#include <string.h>
#include <libamp.h>
#include "common.h"

// to initialize at program startup
void init_tetris (void)
{
    srandom (time(NULL));
    set_uformat (U_ASCII);
    text_mode (-1);
    enable_m15 = TRUE;
    enable_lasttrk_loop = TRUE;
    
    lock_stuff();          // lock memory
    allegro_init();
    install_timer();       // for use with keyboard, soundcard, and interrupts
    switch_numlock_off();  // switch off numlock
    install_keyboard();    // install allegro keyboard interrupt handler
    //key_led_flag = FALSE;

    //set_color_conversion(0xF000);
    set_color_conversion(COLORCONV_NONE);
    set_color_depth (selected_color_depth);

    clrscr();
    show_blue_bar();
    gotoxy (1, 4);
    printf ("Initializing...");
    gotoxy (1, 7);
    reduce_str();
    detect_necessary_files();
    load_config_cfg();
    init_soundcard();
    init_jgmod();
    init_libamp();
    check_music_lst();
    load_tetris_datafile();
    create_buffer();
    load_chart_cfg();
    print_digi_driver();
    set_graphics_mode();
    install_int_ex (timer_interrupt, BPS_TO_TIMER (60)); //install interrupt 60Hz
    //install_int_ex (libamp_interrupt, BPS_TO_TIMER (70)); //install interrupt 100Hz
}

// initialize soundcard
void init_soundcard (void)
{
    int temp;
    int no_sound_card = FALSE;

    printf ("Initializing Sound Card...");

    reserve_voices (no_channels, -1);
    temp = install_sound(DIGI_AUTODETECT, MIDI_NONE, NULL);

    if (temp < 0)
        {
        reserve_voices (-1, -1);    // this should succeed
        temp = install_sound(DIGI_NONE, MIDI_NONE, NULL);
        no_sound_card = TRUE;

        if (temp < 0)
            {
            printf ("\n\n\nError : Unable to initialize sound card\n");
            exit (1);
            }
        }

    if (no_sound_card == FALSE)
        printf ("\t\t OK\n");
    else
        printf ("\t\t Not Detected\n");
}

// show a blue bar at startup
void show_blue_bar (void)
{
    int index;
    char temp[30];

    color (15, 1);
    for (index=0; index<80; index++)
        cprintf (" ");

    sprintf (temp, "JTris %s", tetris_version);    
    print_center (1, temp);
}

// change forground and background in textmode
void color (int forground, int background)
{
    textcolor (forground);  //set background and forground color
    textbackground (background);
}

// print a string in the center of a specified row in text mode
void print_center (int row, char *strng)
{
    int lenght;

    lenght = strlen (strng);
    gotoxy (40 - (lenght/2), row);
    cprintf ("%s", strng);      // print a string in center of the screen
}

// initialize libamp library
void init_libamp (void)
{
    printf ("Installing LIBAMP...");

    if (!install_amp())
        {
        printf ("\n\n\nError : Unable to install libamp\n");
        exit (1);
        }

    printf ("\t\t\t OK\n");
}

// initialize jgmod library
void init_jgmod (void)
{
    printf ("Installing JGMOD...");

    if (install_mod(no_channels - no_sfx_channel) < 0)
        {
        printf ("\n\n\nError : Unable to allocate %d voices\n", no_channels-6);
        exit (1);
        }

    printf ("\t\t\t OK\n");
}

// set to 640 x 480 16bits graphics mode
void set_graphics_mode (void)
{
    int temp;

    temp = set_gfx_mode (GFX_AUTODETECT, 640, 480, 0, 0);

    if (temp < 0)
        {
        set_color_depth (8);
        set_gfx_mode (GFX_VGA, 320, 200, 0,0);
        set_gfx_mode (GFX_TEXT, 0, 0, 0, 0);
        printf ("\nError : Unable to set to 640 x 480 %d bits graphics mode\n", selected_color_depth);
        exit (1);
        }
}

// load tetris.dat in to memory
void load_tetris_datafile (void)
{
    printf ("Loading Datafile...");

    datafile = load_datafile ("tetris.dat");

    if (datafile == NULL)
        {
        printf ("\n\n\nError : Unable to load tetris.dat\n");
        exit (1);
        }

    printf ("\t\t\t OK\n");
}

// initialize variables at program startup
void init_var (void)
{
    int degree, counter;

    // the shape of the block is stored in binary mode
    // long red stick
    block[0].pos[0] = 3840;
    block[0].pos[1] = 8738;
    block[0].pos[2] = 240;   
    block[0].pos[3] = 17476;
    block[0].bitmap = BK_RED;

    // T shape block
    block[1].pos[0] = 624;
    block[1].pos[1] = 562;
    block[1].pos[2] = 114;   
    block[1].pos[3] = 610;
    block[1].bitmap = BK_LIGHTBLUE;
    
    //Square block
    block[2].pos[0] = 1632;
    block[2].pos[1] = 1632;
    block[2].pos[2] = 1632;   
    block[2].pos[3] = 1632;
    block[2].bitmap = BK_PURPLE;

    // L shape block
    block[3].pos[0] = 368;
    block[3].pos[1] = 547;
    block[3].pos[2] = 116;   
    block[3].pos[3] = 1570;
    block[3].bitmap = BK_GREEN;

    //Inverse L shape block
    block[4].pos[0] = 1136;
    block[4].pos[1] = 802;
    block[4].pos[2] = 113;   
    block[4].pos[3] = 550;
    block[4].bitmap = BK_BLUE;

    // h shape block
    block[5].pos[0] = 864;
    block[5].pos[1] = 17952;
    block[5].pos[2] = 13824;   
    block[5].pos[3] = 561;
    block[5].bitmap = BK_YELLOW;

    //Inverse h shape block
    block[6].pos[0] = 1584;
    block[6].pos[1] = 9792;
    block[6].pos[2] = 25344;
    block[6].pos[3] = 4896;
    block[6].bitmap = BK_PINK;

    //speed of brick falling down for each level
    level_speed[0] = 6;
    level_speed[1] = 15;
    level_speed[2] = 23;
    level_speed[3] = 30;
    level_speed[4] = 36;
    level_speed[5] = 41;
    level_speed[6] = 44;
    level_speed[7] = 48;
    level_speed[8] = 51;
    level_speed[9] = 53;
    level_speed[10] = 55;
    level_speed[11] = 56;
    level_speed[12] = 57;
    level_speed[13] = 58;
    level_speed[14] = 59;

    // prepare sin and cos lookup table for faster calculation for starfield
    for (degree=0; degree<361; degree++)
        {
        sin_table [degree] = sin ((float)degree * 3.142/180);
        cos_table [degree] = cos ((float)degree* 3.142/180);
        }

    // initialize starting position of the stars
    for (counter=0; counter <no_of_stars; counter++)
        {
        stars[counter].x[0]          = random() % 325;
        stars[counter].angle         = random() % 360;
        stars[counter].x_velocity    = ( (random() % 999) /1000 ) + 1 ;
        stars[counter].original_color = 1; 
        }

    // accelerate those stars    
    for (counter=0; counter<400; counter++)
        accelerate_stars();
}

// allocate screen buffer
void create_buffer(void)
{
    int width, height;

    printf ("Creating Buffer...");

    width  = block_width  * game_window_width;
    height = block_height * game_window_height;

    temp_buffer = create_bitmap (screen_width, screen_height);
    player[0].game_window_buffer = create_bitmap (width, height);
    player[1].game_window_buffer = create_bitmap (width, height);

    width  = block_width  * game_info_width;
    height = block_height * game_info_height;

    player[0].game_info_buffer = create_bitmap (width, height);
    player[1].game_info_buffer = create_bitmap (width, height);

    if ( (player[0].game_window_buffer == NULL) ||
         (player[1].game_window_buffer == NULL) ||
         (player[0].game_info_buffer == NULL)   ||
         (player[1].game_info_buffer == NULL)   ||
         (temp_buffer == NULL)  )
        {
        printf ("\n\n\nError :Not enough memory to allocate buffer\n\n");
        exit (1);
        }

    printf ("\t\t\t OK\n");
    
    //now clear those buffer
    clear (player[0].game_window_buffer);
    clear (player[1].game_window_buffer);
    clear (player[0].game_info_buffer);
    clear (player[1].game_info_buffer);
}


void libamp_interrupt (...)
{
    poll_amp();
}END_OF_FUNCTION (libamp_interrupt)

// increase all timer variables. Used to keep track of time
void timer_interrupt (...)
{
    timer[0].left++;
    timer[0].right++;
    timer[0].down++;
    timer[0].cw++;
    timer[0].ccw++;
    timer[0].move_down++;

    timer[1].left++;
    timer[1].right++;
    timer[1].down++;
    timer[1].cw++;
    timer[1].ccw++;
    timer[1].move_down++;
}END_OF_FUNCTION (timer_interrupt)

// this is used to lock variables and functions in to memory so that it
// won't ge swapped to disk. Very important for variables that are used
// by interrupt. Otherwise, it could crash.
void lock_stuff(void)
{
    LOCK_FUNCTION (timer_interrupt);
    LOCK_FUNCTION (libamp_interrupt);
}

// set all the timer to zero
void init_player_timer (void)
{
    int index;

    for (index=0; index<max_player; index++)
        {
        timer[index].left = 0;
        timer[index].right = 0;
        timer[index].down = 0;
        timer[index].cw = 0;
        timer[index].ccw = 0;
        timer[index].move_down = 0;
        }
}

// to initialize player variables when actual game (not program) starting up
void init_player (int no_player, int level, int height)
{
    int player_no;
    int x, y;
    int temp;

    for (player_no=0; player_no < max_player; player_no++)
        clear_old_game_space (player_no);

    if (resume_game == FALSE)
        {
        for (player_no=0; player_no < max_player; player_no++)
            {
            player[player_no].status = inactive;
            player[player_no].score = 0;
            player[player_no].level = level;
            player[player_no].lines_done = 0;
            player[player_no].lines_left = 10;
            player[player_no].curr_block = random() % max_block;
            player[player_no].next_block = random() % max_block;
            player[player_no].block_x = 3;
            player[player_no].block_y = -2;
            player[player_no].rotation = 0;
            clear_game_space (player_no);
            }

        // to set the height of the game space
        for (y = game_window_height-height; y < game_window_height; y++)
            {
            temp = 0;
            for (x = 0; x < game_window_width; x++)
                {
                if (temp >= game_window_width-1)
                    continue;

                if ((random() % 4))
                    {
                    player[0].game_space[x][y] =  random() % max_block;
                    player[1].game_space[x][y] = player[0].game_space[x][y];
                    temp++;
                    }
                }
            }
        }

    if (no_player == 1)
        {
        player[0].game_window_xpos = 225;
        player[0].game_window_ypos = 31;

        player[0].game_info_xpos = 55;
        player[0].game_info_ypos = 31;

        player[0].key_left = player_keys[0].p[kleft];
        player[0].key_down = player_keys[0].p[kdown];
        player[0].key_right = player_keys[0].p[kright];
        player[0].key_rotate_ccw = player_keys[0].p[kccw];
        player[0].key_rotate_cw = player_keys[0].p[kcw];
        if (resume_game == FALSE)
            player[0].status = active;
        }
    else
        {
        // setup player 1 first
        player[0].game_window_xpos = 36;
        player[0].game_window_ypos = 31;

        player[0].game_info_xpos = 263;
        player[0].game_info_ypos = 31;

        player[0].key_left = player_keys[1].p[kleft];
        player[0].key_down = player_keys[1].p[kdown];
        player[0].key_right = player_keys[1].p[kright];
        player[0].key_rotate_ccw = player_keys[1].p[kccw];
        player[0].key_rotate_cw = player_keys[1].p[kcw];


        // now setup player 2
        player[1].game_window_xpos = 413;
        player[1].game_window_ypos = 31;

        player[1].game_info_xpos = 263;
        player[1].game_info_ypos = 259;

        player[1].key_left = player_keys[1].p[5+kleft];
        player[1].key_down = player_keys[1].p[5+kdown];
        player[1].key_right = player_keys[1].p[5+kright];
        player[1].key_rotate_ccw = player_keys[1].p[5+kccw];
        player[1].key_rotate_cw = player_keys[1].p[5+kcw];
        if (resume_game == FALSE)
            {
            player[0].status = active;
            player[1].status = active;
            }
        }
}

// check that all music files exists in music.lst
// otherwise exit.
void check_music_lst(void)
{
    int index;
    char dest[501];

    printf ("Checking Music.lst...");

    no_music = find_no_of_music ("music.lst");
    if (no_music < 0)
        {
        printf ("\n\n\nError : Unable to open music.lst\n\n");        
        exit (1);
        }
    else if (no_music == 0)
        {
        printf ("\n\n\nError : music.lst must contain at least 1 music\n\n");        
        exit (1);
        }

    for (index=0; index<no_music; index++)
        {
        if (get_music_no ("music.lst", dest, index) < 0)
            {
            printf ("\n\n\nError : Unable to open music.lst\n\n");
            exit (1);
            }

        if (exists (dest) == 0)
           {
           printf ("\n\n\nError : Unable to find \"%s\"\n\n", dest);
           exit (1);
           }
        }

    curr_music = random() % no_music;
    printf ("%3d music(s)\t OK\n", no_music);
}

// load configurations in to memory
void load_config_cfg (void)
{
    FILE *file;
    int index;

    printf ("Loading Config.cfg...");

    file = fopen ("config.cfg", "rb");
    if (file == NULL)
        {
        printf ("\n\n\nError : Unable to open Config.cfg\n\n");
        exit (1);
        }

    for (index=0; index < 5; index++)
        {
        player_keys[0].p[index] = fgetc(file);
        if ( (player_keys[0].p[index] > 128) || (player_keys[0].p[index] < 0) )
            player_keys[0].p[index] = 0;
        }

    for (index=0; index < 10; index++)
        {
        player_keys[1].p[index] = fgetc(file);
        if ( (player_keys[1].p[index] > 128) || (player_keys[1].p[index] < 0) )
            player_keys[1].p[index] = 0;
        }

    change_background = (signed char) fgetc(file);
    music_sequential = (signed char) fgetc(file);
    alpha_level = fgetc(file);
    no_channels = fgetc(file);
    sfx_volume = fgetc(file);
    music_volume = fgetc(file);

    for (index=0; index < max_player ; index++)
        {
        player[index].x_movement_speed = fgetc (file);
        if ( (player[index].x_movement_speed < 1) || (player[index].x_movement_speed > 20) )
            player[index].x_movement_speed = 5;

    
        player[index].y_movement_speed = fgetc (file);
        if ( (player[index].y_movement_speed < 0) || (player[index].y_movement_speed > 20) )
            player[index].y_movement_speed = 2;

        player[index].rotate_movement_speed = fgetc (file);
        if ( (player[index].rotate_movement_speed < 1) || (player[index].rotate_movement_speed > 20) )
            player[index].rotate_movement_speed = 10;
        }

    // check if the values are OK.
    if ( (change_background != TRUE) && (change_background != FALSE) )
        change_background = TRUE;

    if ( (music_sequential != TRUE) && (music_sequential != FALSE) )
        music_sequential = TRUE;

    if ( (no_channels != 32) && (no_channels != 64) &&
        (no_channels != 16) && (no_channels != 8) )
        no_channels = 32;

    alpha_level = ((int)(alpha_level /17)) * 17 ;
    sfx_volume = ((int)(sfx_volume /17)) * 17 ;
    music_volume = ((int)(music_volume /17)) * 17 ; 


    set_mod_volume (music_volume);
    set_trans_blender (0, 0, 0, alpha_level);   // set transparency level
    
    fclose (file);
    printf ("\t\t\t OK\n");
}

// this is to switch the numlock off
void switch_numlock_off (void)
{
    uchar status;

    status = _farpeekb (_dos_ds, 0x417);
    status = status & 223;
    _farpokeb (_dos_ds, 0x417, status);
}

// find the necessary files to run. Otherwise exit.
void detect_necessary_files(void)
{
    int index;
    char temp[30];
    char *files[] = {
    "music.lst",
    "starwars.mod",
    "letsdanc.s3m",
    "tetris.dat",
    "pictures.dat",
    "config.cfg",
    "chart.cfg",
    ""
    };

    if (exists ("pictures.dat") != 0)
        {
        if (exists ("config.cfg") == 0)
            {
            FILE *f;

            f = fopen ("config.cfg", "wb");
            if (f != NULL)
                {
                temp[0]  = 'R';
                temp[1]  = 'S';
                temp[2]  = 'U';
                temp[3]  = '';
                temp[4]  = '';
                temp[5]  = '';
                temp[6]  = '';
                temp[7]  = '';
                temp[8]  = '';
                temp[9]  = '';
                temp[10] = 'R';
                temp[11] = 'S';
                temp[12] = 'U';
                temp[13] = 'D';
                temp[14] = 'E';
                temp[15] = 255;
                temp[16] = 255;
                temp[17] = 'w';
                temp[18] = '@';
                temp[19] = 255;
                temp[20] = 'w';
                temp[21] = 5;
                temp[22] = 2;
                temp[23] = 10;
                temp[24] = 5;
                temp[25] = 2;
                temp[26] = 10;
                temp[27] = 0;

                fwrite (temp,1, 27, f);
                fclose (f);
                }
            }

        if (exists ("chart.cfg") == 0)
            save_chart_cfg();
        }


    for (index=0; files[index][0]; index++)
        {
        if (exists (files[index]) == 0)
            {
            printf ("Error : Unable to find %s\n", files[index]);
            exit(1);
            }
        }
}

// print the digital sound card driver Allegro library is using
// NOTE : digi_driver is an Allegro structure.
void print_digi_driver(void)
{
    printf ("\n\nUsing %s\n", digi_driver->desc);
    printf ("\nPress any key to continue");
    readkey();
    fade_out (4);
}

//load top chart from file
int load_chart (void)
{
    FILE *f;
    int index;
    int state = TRUE;
    FAME_INFO temp_fame1[10], temp_fame2[10];

    f = jfopen ("chart.cfg", "rb");

    // now check the IDs
    index = jfgetc (f);
    if (index != 23)
        state = FALSE;

    index = jfgetc (f);
    if (index != 34)
        state = FALSE;

    index = jfgetc (f);
    if (index != 255)
        state = FALSE;

    index = jfgetc (f);
    if (index != 2)
        state = FALSE;

    // check if the ID is correct
    if (state == FALSE)
        {
        jfclose (f);
        return 1;
        }


    for (index=0; index<10; index++)
        {
        temp_fame1[index].lines = jigetl (f);
        temp_fame1[index].score = jigetl (f);
        jfread (temp_fame1[index].name, f);

        temp_fame2[index].lines = jigetl (f);
        temp_fame2[index].score = jigetl (f);
        jfread (temp_fame2[index].name, f);


        if (temp_fame1[index].lines != temp_fame2[index].lines)
            {
            jfclose (f);
            return 2;
            }

        if (temp_fame1[index].score != temp_fame2[index].score)
            {
            jfclose (f);
            return 2;
            }

        if (strcmp (temp_fame1[index].name, temp_fame2[index].name) != 0 )
            {
            jfclose (f);
            return 2;
            }
        }

    for (index=0; index<10; index++)
        {
        fame_info[index].score = temp_fame1[index].score;
        fame_info[index].lines = temp_fame1[index].lines;
        sprintf (fame_info[index].name, "%s", temp_fame1[index].name);
        }

    jfclose (f);
    return 0;
}

void load_chart_cfg(void)
{
    int temp;

    printf ("Loading Chart.cfg...");
    temp = load_chart();
    if (temp == 0)
        printf ("\t\t\t OK\n");
    else if (temp == 1)
        {
        printf ("\t\t\t File corrupted");
        printf ("\n\n\nRestoring Chart.cfg to original\n");
        save_chart_cfg();
        }
    else
        {
        printf ("\t\t\t File being tempered");
        printf ("\n\n\nRestoring Chart.cfg to original. Winners don't cheat\n");
        save_chart_cfg();
        }
}


void reduce_str (void)
{
    int index;
    int temp;
    int len;

    for (index=0; index<10; index++)
        {
        len = strlen(fame_info[index].name);

        for (temp=0; temp<len; temp++)
            fame_info[index].name[temp]--;
        }
}

