//Itana is copyright 2000 by Jason Winnebeck.  You may freely distribute this
//game in its original archive.  If us wish to use code from Itana feel free
//to do so.  If you do a mention in the credits or a mail to
//gillius@webzone.net would be apprciated.

//This file contains the network routines used in the Ship class.

#include "Itana.h"
#include "Ship.h"
#include "PacketConsts.h"
#include "EntityConsts.h"
#include "ShipStats.h"
#include "Packet.h"
#include "Network.h"
#include "Rect.h"
#include "Entity.h"
#include "Game.h"
#include "Bullet.h"
#include "Base.h"
#include "itanadat.h"
#include "Sound.h"

#include "Console.h"

const double BOUNCE_FACTOR  = 125.0; //for collisions
const int    MAX_BOUNCE_VOL = 255;   //vol at aburn speed
const int    MIN_BOUNCE_VOL = 32;    //vol at 0
const int    BOUNCE_VOL_RNG = MAX_BOUNCE_VOL - MIN_BOUNCE_VOL;

void Ship::doCollision(Entity* other) {
  if ((other->getOwner() == this) || dead)
    return;
  if (other->isBullet()) {
    if (control == CTRL_SERVER) {
      DamagePacket* packet = new DamagePacket();
      packet->playerId = network->whoIsThis(this);
      packet->other = network->whoIsThis(other->getOwner());
      packet->frame = Game::getFrame();
      packet->life = ((Bullet*)other)->getDamage();
      if (other->getId() == ID_PARA)
        packet->type = PARADAMAGE;
      else
        packet->type = NORMALDAMAGE;
      network->sendPacket(packet, true);
    }
  } else if (other->isPlanet()) {
    if (control == CTRL_LOCAL) {
      DeathPacket* packet = new DeathPacket();
      packet->type = PLANETDEATH;
      network->sendPacket(packet, true); //Network will handle scores/notification/die
      if (!network->isConnected())
        die(); //Die in single player mode, else let network handle death.
    }
  } else {//Do a "solid object" collision, more speed = more damage
    if (!(other->isRoid() && id == ID_ZYLEX_SHIP)) {
      //Zylex ship 
      if (!other->isShip() || !((Ship*)other)->isDead()) {
        if (control == CTRL_LOCAL) {
          //Do bumping sound, volume based on bounce
          int bounceVolume = int(BOUNCE_VOL_RNG * vect.getMag() / STATS[type].maxaburnspeed + MIN_BOUNCE_VOL);
          Sound::playSound(SFX_DAT_BUMP_WAV, bounceVolume);

          //Do damage calculations
          armor -= (int)(BOUNCE_FACTOR * vect.getMag());
          if ( armor < 0 ) {
            DeathPacket* packet = new DeathPacket();
            packet->type = BANGEDDEATH;
            network->sendPacket(packet, true); //Network will handle scores/notification/die
            if (!network->isConnected())
              die(); //Die in single player mode, else let network handle death.
          }
        }
        MovingEntity::doCollision(other);
      }
    }
  }
}

Packet* Ship::makePacket() {
  //Makes a packet of an approprate type
  ShipPacket* ret = new ShipPacket();
  ret->playerId = network->whoIsThis(this);
  ret->frame = Game::getFrame();
  ret->actions = actions;
  //al_trace("Sending a packet with actions:%i == %i\n", ret->actions, actions);
  ret->vect = vect;
  ret->loc = Point(loc.x, loc.y);
  ret->theta = theta;
  ret->turret = turret;
  return ret;
}

void Ship::usePacket(Packet* packet) {
  switch (packet->getType()) {
  case SHIP:
    {
      ShipPacket* data = (ShipPacket*)packet;
      loc.x = data->loc.x;
      loc.y = data->loc.y;
      vect = data->vect;
      theta = data->theta;
      turret = data->turret;
      actions = data->actions;
      int currFrame = Game::getFrame();
      for(int c=data->frame; c<currFrame; c++)
        update();
    }
    break;
  case DAMAGE:
    {
      //Only client-side should go here.
      if (!dead) {
        DamagePacket* data = (DamagePacket*)packet;
        switch(data->type) {
        case NORMALDAMAGE:
          shield -= (int)(data->life);
          if (shield < 0) {
            armor += shield;
            shield = 0;
          }
          if (armor < 0) {
            DeathPacket* death = new DeathPacket();
            death->type = SHOTDEATH;
            death->other = data->other;
            network->sendPacket(death, true); //Network will handle scores/notification/die
          }
          break;
        case PARADAMAGE:
          paralized = data->life; //Might want to consider adjusting time of actual hit
          actions |= PARALIZED;
          break;
        }
      }
    }
    break;
  case DEATH:
    die(); //and therefore dead = true
    break;
  case RESPAWN:
    init(Rect(loc), Vector(vect));
    //Reset ship to default values, but don't move.
    //we should get a position packet soon, but this will cause a small jump
    break;
  case DYNAMICENTITY:
    //This is a bullet packet.  Technically not useful info, but we can pull
    //position data from this packet.
    DynamicEntityPacket* data = (DynamicEntityPacket*)packet;
    loc.x = data->loc.x + loc.w/2.0;
    loc.y = data->loc.y + loc.h/2.0;
  }
}

void Ship::netInput() {
  //al_trace("Server: frame: %i  actions: %i\n", Game::getFrame(), actions);
  if (!dead && !paralized) {
    if ( actions & TURNLEFT )
      turn(false, actions & SLOWTURN);
    else if ( actions & TURNRIGHT )
      turn(true, actions & SLOWTURN);
    if ( actions & ACCELERATE )
      accel(false);
    else if ( actions & AFTERBURN )
      accel(true);
    if (control != CTRL_SERVER) {
      if ( actions & FIREUP )
        fire(UP);
      if ( actions & FIRELEFT )
        fire(LEFT);
      if ( actions & FIRERIGHT )
        fire(RIGHT);
      if ( actions & FIREDOWN )
        fire(DOWN);
      if ( actions & FIRESPECIAL )
        special();
    }
    if ( actions & DOCKING )
      dock();
    else
      myBase->undock();
  }
}