//Itana is copyright 2000 by Jason Winnebeck.  You may freely distribute this
//game in its original archive.  If us wish to use code from Itana feel free
//to do so.  If you do a mention in the credits or a mail to
//gillius@webzone.net would be appreciated.

#include "Itana.h"
#include "Server.h"
#include "ItanaPort.h"
#include "ItanaServer.h"
#include "Asteroid.h"
#include "Game.h"
#include "Parser.h"
#include "ServerPrivate.h"

char netbuf[NETBUF_SIZE]; //a large but static buf
char netbuf2[NETBUF_SIZE];//another large buf

Server::Server() {
  netbuf[0] = '\0';
  netbuf[1] = '\0';
  netbuf2[0] = '\0';
  netbuf2[1] = '\0';
}

Server::~Server() {
}

PlayerId Server::whoIsThis(void* other) {
  int c=0;
  int l=0;
  l = players.Length();
  LLPos save = players.Save();
  players.SetToStart();
  for (c=0; c<l; c++) {
    NetPlayer* next = players._GetNext();
    if ((other == next->base) || (other == next->ship))
      return next->data.playerId;
    players._Advance();
  }
  players.Load(save);
  
  l = connecting.Length();
  save = connecting.Save();
  connecting.SetToStart();
  for (c=0; c<l; c++) {
    NetPlayer* next = connecting._GetNext();
    if ((other == next->base) || (other == next->ship))
      return next->data.playerId;
    connecting._Advance();
  }
  connecting.Load(save);
  
  return 0;
}

const char* Server::getName(PlayerId id) {
  //Returns the player name associated with this PlayerId
  int l = players.Length();
  LLPos save = players.Save();
  players.SetToStart();
  const char* ret = NULL;
  for (int c=0; (c<l) && (ret == NULL); c++) {
    NetPlayer* next = players._GetNext();
    if (next->data.playerId == id)
      ret = next->data.getName();
    players._Advance();
  }
  players.Load(save);
  
  return ret;
}

const char* Server::getOurName() const {
  return "Server";
}

const NetPeer** Server::getPeers() {
  //For Console -- returns an NEWLY ALLOCATED array of NetPeer*
  //pointing to objects to display which won't be deleted.
  //The last pointer is NULL to signify the end of the array.
  int l = players.Length();
  const NetPeer** ret = new const NetPeer*[l + 1]; //peers + NULL

  int c = 0;
  LLPos save = players.Save();
  players.SetToStart();
  for (; c < l; c++) {
    ret[c] = players._GetNext();
    players._Advance();
  }
  players.Load(save);

  ret[l] = NULL;
  return ret;
}

void Server::sendPacket(Packet* packet, bool reliable) {
  //Sends a packet to all players excluding the one that sent it.
  //Network handles memory allocation of packet, and will delete it.
  if (packet->getType() == DAMAGE)
    //SPECIAL CASE for DamagePacket
    findPlayer(packet->playerId)->TCPqueue.add(packet);
  else {
    //Right now only DamagePackets should be sent from client code
    cout << "Illegal packet send type " << packet->getType() << '.' << endl;
    delete packet;
  }
}

bool Server::doServer(int port) {
  //starts server and exits when server terminates.  If there is an error
  //lastError is filled and the calling function can print an error message.
  sprintf(netbuf, ":%i", port);
  
  if (displayConnectInfo(netbuf)) {
    sprintf(lastError, "Error creating listen port (perhaps port is in use?)");
    return true;
  }
  
  listen = net_openconn(netDriver, netbuf);
  if (listen == NULL) {
    sprintf(lastError, "Error creating listen port");
    return true;
  }
  
  if (net_listen(listen)) {
    sprintf(lastError, "Error while attempting to listen");
    return true;
  }
  
  generate();
  
  bool exit = Game::initForServer();
  bool active = true; //Is the server active or sleeping?
  clearKeybuf();
  while (!exit) {
    exit |= checkInput();
    exit |= checkForNewPlayers();
    if (connecting.Length() != 0 || players.Length() != 0) {
      exit |= pollConnectedPlayers();
      exit |= pollConnectingPlayers();
      Game::doServerUpdate();
      active = true;
      //FUTURE NOTE: I don't think the Planet is in the server's game copy.
      //Obviously this might affect things.
    } else {
      if (active) {
        cout << "No clients connected -- now sleeping..." << endl;
        active = false;
      }
      yield(1000); //sleep for awhile if we have NO clients      
      startTimer();//Don't update while waiting for clients
    }
  }
  Game::closeForServer();
  
  shutDown();
  
  return false;
}

bool Server::update() {
  //returns true only on an unrecoverable error which would require
  //termination of the network connection
  return false;
}

NetPlayer* Server::findPlayer(PlayerId id) {
  int l = players.Length();
  LLPos old = players.Save();
  players.SetToStart();
  NetPlayer* ret = NULL;
  for (int c=0; c<l && !ret; c++) {
    if (players._GetNext()->data.playerId == id)
      ret = players._GetNext();
    else
      players._Advance();
  }
  players.Load(old);
  //WARNING: connecting is not searched
  return ret;
}

int Server::getNextId() {
  //find lowest number not taken
  int ret = 0;
  int l = players.Length();
  LLPos old = players.Save();
  players.SetToStart();
  int c;
  for (c=0; c<l; c++) {
    if (players._GetNext()->data.playerId == ret)
      ret++;
    players._Advance();
  }
  players.Load(old);
  l = connecting.Length();
  old = connecting.Save();
  connecting.SetToStart();
  for (c=0; c<l; c++) {
    if (connecting._GetNext()->data.playerId == ret)
      ret++;
    connecting._Advance();
  }
  connecting.Load(old);
  return ret;
}

bool Server::pollConnectedPlayers() {
  int l = players.Length();
  players.SetToStart();
  for (int c=0; c<l; c++) {
    bool kill = false; //should we kill the current working player
    NetPlayer* next = players._GetNext();
    //Send information to player
    //Check to see if we need to send a ping packet
    if ((Game::getFrame() - next->lastPing) * 10 > PINGFREQ) {
      PingPacket* packet = new PingPacket();
      packet->frame = Game::getFrame();
      packet->playerId = next->data.playerId;
      next->UDPqueue.add(packet);
      next->lastPing = Game::getFrame();
    }
    next->sendTCPPacket();
    next->sendUDPPacket();
    //Check for dropout
    if ((Game::getFrame() - next->lastTime) * 10 > TIMEOUT) {
      cout << next->data.getName() << " dropped." << endl;
      PlayerPartPacket packet;
      packet.frame = Game::getFrame();
      packet.playerId = next->data.playerId;
      packet.status = false;
      addToAllTCP(packet, next, players);
      addToAllTCP(packet, next, connecting);
      kill = true;
    }
    //Get information from player
    int recv = next->getNextPacket(netbuf, NETBUF_SIZE);
    while (recv && !kill) {
      if (recv < 0)
        cout << "Recieve error for " << next->data.getName() << '@' << net_getpeer(next->TCP) << endl;
      else if (recv > 0) {
        dataIn += recv;
        int currPos = 0;
        while (Packet::getNextType(netbuf, currPos) != ENDOFPACKET) {
          Packet* nextPacket = Packet::parsePacket(netbuf, currPos);
          if (handlePacket(nextPacket, next)) {
            cout << next->data.getName() << " has left the game" << endl;
            kill = true;
          }
          delete nextPacket;
        }
      }
      recv = next->getNextPacket(netbuf, NETBUF_SIZE);
    }
    if (kill)
      players._DeleteThis();
    else
      players._Advance();
  }
  return false;
}

bool Server::checkInput() {
  //returns true if we should quit
  return (getChar() != 0);
}

void Server::addToAllTCP(const Packet& packet, NetPlayer* exclude, LinkList<NetPlayer>& theList) {
  int l = theList.Length();
  LLPos save = theList.Save();
  theList.SetToStart();
  for (int c=0; c<l; c++) {
    NetPlayer* next = theList._GetNext();
    if (next != exclude) {
      next->TCPqueue.add(packet.makeClone());
    }
    theList._Advance();
  }
  theList.Load(save);
}

void Server::addToAllUDP(const Packet& packet, NetPlayer* exclude, LinkList<NetPlayer>& theList) {
  int l = theList.Length();
  LLPos save = theList.Save();
  theList.SetToStart();
  for (int c=0; c<l; c++) {
    NetPlayer* next = theList._GetNext();
    if (next != exclude)
      next->UDPqueue.add(packet.makeClone());
    theList._Advance();
  }
  theList.Load(save);
}

void Server::shutDown() {
  //cleans everything up
  net_closeconn(listen);
  int l = players.Length();
  players.SetToStart();
  for (int c=0; c<l; c++) {
    players._GetNext()->shutDown();
    players._Advance();
  }
}

void Server::generate() {
  //generates a new server game
  //Gotta have da 'roids!
  for (int c=0; c < 150; c++) {
    Asteroid *roid = new Asteroid();
    roid->init(getrndd(4500.0, 2250.0), getrndd(2*PI));
    Game::moveEntity(roid);
  }  
}

bool Server::displayConnectInfo(const char* charPort) {
  //Display IP data if any and port, true on error
  NET_CHANNEL* temp = net_openchannel(netDriver, charPort);
  if (temp == NULL)
    return true;
  
  cout << "The server is listening on port " << charPort << endl;
  cout << "The IP address can be found by running winipcfg (Win9x) or ipconfig" << endl;
  cout << "  (WinNT).  This is of course if you are using an internet driver." << endl;
  cout << "Server's best guess at IP is: " << net_getlocaladdress(temp) << endl;
  
  net_closechannel(temp);
  return false;
}