//Itana is copyright 2000 by Jason Winnebeck.  You may freely distribute this
//game in its original archive.  If us wish to use code from Itana feel free
//to do so.  If you do a mention in the credits or a mail to
//gillius@webzone.net would be apprciated.

#include "Itana.h"
#include "Ship.h"
#include "ShipStats.h"
#include "Prefs.h"
#include "Vector.h"
#include "MovingEffect.h"
#include "Game.h"
#include "PacketConsts.h"
#include "Base.h"
#include "Network.h"
#include "Interface.h"
#include "ParaBullet.h"
#include "Video.h"
#include "Sound.h"

const double DOCK_DIST         = 256.0; //max docking distance

void Ship::init(const Rect& rect, const Vector& vect2) {
  loc = rect;
  loc.w = STATS[type].width;
  loc.h = STATS[type].height;
  initScrLoc();
  vect = vect2;
  theta = vect2.getTheta();
  turret = 0.0;
  energy = STATS[type].maxenergy;
  shield = STATS[type].maxshield;
  armor = STATS[type].maxarmor;
  score = normalReload = specialReload = 0;
  obsolete = dead = false;
  paralized = 0;
  actions = 0;
}

void Ship::updateStats(Interface* statusbar) {
  if (!dead) {
    statusbar->setValues((double)energy/STATS[type].maxenergy,
                         (double)shield/STATS[type].maxshield,
                         (double)armor/STATS[type].maxarmor,
                         score);
  } else {
    statusbar->setValues(0, 0, 0, score);
  }
}

void Ship::respawn() {
  //respawn this ship near the base
  Rect tempLoc = myBase->getLoc();
  Vector tempVect;
  tempVect.initPolar(getrndd(2*PI), DOCK_DIST);
  tempVect.addTo(tempLoc);
  tempVect.initPolar(getrndd(2*PI), 0.0);
  init(tempLoc, tempVect);

  if (control != CTRL_SERVER) {
    //Send notification
    RespawnPacket* packet = new RespawnPacket();
    network->sendPacket(packet, true);
  }
}

void Ship::input() {
  switch (control) {
  case CTRL_LOCAL:
    localInput();
    break;
  case CTRL_AI:
    AIInput();
    break;
  default:
    netInput();
    break;
  }
}

void Ship::localInput() {
  actions = 0;
  if (dead) {
    if (key[prefs->c_fire])
      respawn();
    return; //return if the local player is dead
  } else if (!paralized) {
    if (key[prefs->c_left])
      turn(false, key[prefs->c_slow]);
    else if (key[prefs->c_right])
      turn(true, key[prefs->c_slow]);
    if (key[prefs->c_fast])
      accel(true);
    else if (key[prefs->c_up])
      accel(false);    
    if (key[prefs->c_etos])
      eToS();
    else if (key[prefs->c_stoe])
      sToE();
    if (key[prefs->c_fire] || key[prefs->c_fireup])
      fire(UP);
    if (key[prefs->c_fireleft])
      fire(LEFT);
    if (key[prefs->c_fireright])
      fire(RIGHT);
    if (key[prefs->c_firedown])
      fire(DOWN);
    if (key[prefs->c_special])
      special();
    if (key[prefs->c_dock])
      dock();
    else
      myBase->undock();

    //Do afterburner sound actions -- done here since accel code will also
    //execute for network players and we don't want their actions affecting
    //local sound.  Placing them here makes sure we don't make sounds when
    //we don't have the energy to make them ;).
    if (actions & ACCELERATE)
      Sound::engineOn(false);
    else if (actions & AFTERBURN)
      Sound::engineOn(true);
    else
      Sound::engineOff();
  }
}

void Ship::AIInput() {
}

bool Ship::update() {
  if (!dead) {
    MovingEntity::update();
    //"afterburner friction"
    if (vect.getMag() > STATS[type].maxspeed)
      vect.add(STATS[type].aburnfric);
    //Regernerate the ship
    if (energy == STATS[type].maxenergy) {
      //if maxed out on energy, regen shields
      shield += STATS[type].regenrate;
      if (shield > STATS[type].maxshield)
        shield = STATS[type].maxshield;
    } else {
      //else, regen energy
      energy += STATS[type].regenrate;
      if (energy > STATS[type].maxenergy)
        energy = STATS[type].maxenergy;
    }

    if (normalReload)
      normalReload--;
    if (specialReload)
      specialReload--;
    if (control == CTRL_LOCAL && paralized) {
      actions |= PARALIZED;
      paralized--;
    }
  }

  return obsolete;
}

void Ship::draw() {
  if ( control >= CTRL_NET && !(actions & CLOAKED))
    Video::drawMapText(loc, makecol(192,192,192), network->getName(control-CTRL_NET));
  if (actions & PARALIZED) //so this works for netplayers too
    ParaBullet::draw(loc.getCenter()); //draw paralized effect
}

void Ship::getRepaired(int energy2, int shield2, int armor2) {
  energy += energy2;
  shield += shield2;
  armor  += armor2;

  if (energy > STATS[type].maxenergy)
    energy = STATS[type].maxenergy;
  if (shield > STATS[type].maxshield)
    shield = STATS[type].maxshield;
  if (armor > STATS[type].maxarmor)
    armor = STATS[type].maxarmor;
}

void Ship::turn(bool cw, int slow) {
  //true if ship is turning clockwise/slowly
  if (!slow) {
    if (cw) {
      theta += STATS[type].turnrate;
      actions |= TURNRIGHT;
    } else {
      theta -= STATS[type].turnrate;
      actions |= TURNLEFT;
    }
  } else {
    actions |= SLOWTURN;
    if (cw) {
      theta += STATS[type].turnrate / 5.0;
      actions |= TURNRIGHT;
    } else {
      theta -= STATS[type].turnrate / 5.0;
      actions |= TURNLEFT;
    }
  }
}

void Ship::accel(bool fast) {
  Vector thrust;
  if (fast) {
    if (energy > STATS[type].aburnuse) {
      thrust.initPolar(theta, STATS[type].aburnaccel);
      vect.add(thrust);
      vect.clip(STATS[type].maxaburnspeed);
      energy -= STATS[type].aburnuse;
      createEngineParticle(true);
      createEngineParticle(true);
      createEngineParticle(true);
      actions |= AFTERBURN;
    }
  } else {
    //only if we have energy and we are not over maxspeed (so the clip
    //won't jerk us short, and the "abrun fric" does its job
    if ((energy > STATS[type].engineuse) && (vect.getMag() <= STATS[type].maxspeed)) {
      thrust.initPolar(theta, STATS[type].accel);
      vect.add(thrust);
      vect.clip(STATS[type].maxspeed);
      energy -= STATS[type].engineuse;
      createEngineParticle(false);
      actions |= ACCELERATE;
    }
  }
}

const int TRANS_RATE = 15;

void Ship::eToS() {
  //energy to shields
  if ((energy > TRANS_RATE) && (shield <= STATS[type].maxshield-TRANS_RATE)) {
    energy -= TRANS_RATE;
    shield += TRANS_RATE;
  }
}

void Ship::sToE() {
  //shields to energy
  if ((shield > TRANS_RATE) && (energy <= STATS[type].maxenergy-TRANS_RATE)) {
    shield -= TRANS_RATE;
    energy += TRANS_RATE;
  }
}

void Ship::dock() {
  if (loc.distTo(myBase->getLoc()) < DOCK_DIST) {
    actions |= DOCKING;
    myBase->dock();
  } else
    myBase->undock();
}

void Ship::fire(Bullet* bull, double recoilTheta, double recoilMag) {
  bull->setOwner(this);
  Game::moveEntity(bull);

  //Do recoil
  if (recoilMag) {
    Vector v;
    v.initPolar(recoilTheta, recoilMag);
    vect.add(v);
  }
  
  //Send bullet if we are the authoriative shooter
  if (control == CTRL_LOCAL)
    network->sendPacket(bull->makePacket(), true);
}

//Cosmetics

void Ship::paralize(int time) {
  paralized = time;
  actions |= PARALIZED;
}

void Ship::die() {
  Point center = loc.getCenter();
  Explosion* expl = new Explosion(center, vect);
  expl->makeSound();
  Game::moveEffect(expl);
  
  dead = true;
}

void Ship::createEngineParticle(bool aburn) {
  Vector partVect;
  if (aburn) {
    partVect.initPolar(theta + PI + getrndd(toRad(50), toRad(25)),
                     STATS[type].aburnaccel/2 + getrndd(STATS[type].aburnaccel));
  } else {
    partVect.initPolar(theta + PI + getrndd(toRad(50), toRad(25)),
                     STATS[type].accel/2 + getrndd(STATS[type].accel));
  }
  //partVect.add(vect);
  throwEngineParticle(partVect);
}