//Itana is copyright 2000 by Jason Winnebeck.  You may freely distribute this
//game in its original archive.  If us wish to use code from Itana feel free
//to do so.  If you do a mention in the credits or a mail to
//gillius@webzone.net would be appreciated.

#include "Itana.h"
#include "Packet.h"
#include "Parser.h"
#include "Game.h"

Packet::Packet() : type(NONE) {
}

Packet::Packet(PacketType type2) : frame(Game::getFrame()), type(type2) {
}

Packet* Packet::makeClone() const {
  return new Packet(*this);
}

int Packet::createPacket(char* buf, int startPos) const {
  //creates the packet and puts it in buf starting from startPos.  Buf is
  //allocated by caller and must be large enough to accept packet.  Returns
  //index of character immediately after appended data.
  startPos = Parser::makeDirect(buf, startPos, type);
  startPos = Parser::makeDirect(buf, startPos, playerId);
  startPos = Parser::makeDirect(buf, startPos, frame);
  return startPos;
}

int Packet::recvPacket(const char* buf, int startPos) {
  //receives and parses packet, filling fields, returns index of character
  //immediately after data.
  startPos = Parser::parseDirect(buf, startPos, type);
  startPos = Parser::parseDirect(buf, startPos, playerId);
  startPos = Parser::parseDirect(buf, startPos, frame);
  return startPos;
}

int Packet::getNextType(const char* buf, int startPos) {
  //discovers the next packet in the buf
  PacketType tempType;
  Parser::parseDirect(buf, startPos, tempType);
  return (int)tempType;
}

Packet* Packet::parsePacket(const char* buf, int& startPos) {
  Packet* ret = NULL;
  switch(getNextType(buf, startPos)) {
    case NONE:
      ret = new Packet();
      startPos = ret->recvPacket(buf, startPos);
      break;
    case SHIP:
      ret = new ShipPacket();
      startPos = ret->recvPacket(buf, startPos);
      break;
    case DYNAMICENTITY:
      ret = new DynamicEntityPacket();
      startPos = ret->recvPacket(buf, startPos);
      break;
    case PLAYER:
      ret = new PlayerPacket();
      startPos = ret->recvPacket(buf, startPos);
      break;
    case PLAYERPART:
      ret = new PlayerPartPacket();
      startPos = ret->recvPacket(buf, startPos);
      break;
    case PLAYERMESSAGE:
      ret = new PlayerMessagePacket();
      startPos = ret->recvPacket(buf, startPos);
      break;
    case ORBITINGENTITY:
      ret = new OrbitingEntityPacket();
      startPos = ret->recvPacket(buf, startPos);
      break;
    case DAMAGE:
      ret = new DamagePacket();
      startPos = ret->recvPacket(buf, startPos);
      break;
    case FRAME:
      ret = new FramePacket();
      startPos = ret->recvPacket(buf, startPos);
      break;
    case DEATH:
      ret = new DeathPacket();
      startPos = ret->recvPacket(buf, startPos);
      break;
    case RESPAWN:
      ret = new RespawnPacket();
      startPos = ret->recvPacket(buf, startPos);
      break;
    case PING:
      ret = new PingPacket();
      startPos = ret->recvPacket(buf, startPos);
      break;
    case STATSPACKET:
      ret = new StatsPacket();
      startPos = ret->recvPacket(buf, startPos);
      break;
  }
  return ret;
}

Packet* FramePacket::makeClone() const {
  return new FramePacket(*this);
}

Packet* ShipPacket::makeClone() const {
  return new ShipPacket(*this);
}

int ShipPacket::createPacket(char* buf, int startPos) const {
  startPos = Packet::createPacket(buf, startPos);
  startPos = Parser::makeDirect(buf, startPos, actions);
  startPos = Parser::makeVect(buf, startPos, vect);
  startPos = Parser::makePoint(buf, startPos, loc);
  startPos = Parser::makeDirect(buf, startPos, theta);
  startPos = Parser::makeDirect(buf, startPos, turret);
  return startPos;
}

int ShipPacket::recvPacket(const char* buf, int startPos) {
  startPos = Packet::recvPacket(buf, startPos);
  startPos = Parser::parseDirect(buf, startPos, actions);
  startPos = Parser::parseVect(buf, startPos, vect);
  startPos = Parser::parsePoint(buf, startPos, loc);
  startPos = Parser::parseDirect(buf, startPos, theta);
  startPos = Parser::parseDirect(buf, startPos, turret);
  return startPos;
}

Packet* DynamicEntityPacket::makeClone() const {
  return new DynamicEntityPacket(*this);
}

int DynamicEntityPacket::createPacket(char* buf, int startPos) const {
  startPos = Packet::createPacket(buf, startPos);
  //startPos = Parser::makeDirect(buf, startPos, owner);
  startPos = Parser::makeDirect(buf, startPos, objectId);
  startPos = Parser::makeVect(buf, startPos, vect);
  startPos = Parser::makePoint(buf, startPos, loc);
  return startPos;
}

int DynamicEntityPacket::recvPacket(const char* buf, int startPos) {
  startPos = Packet::recvPacket(buf, startPos);
  //startPos = Parser::parseDirect(buf, startPos, owner);
  startPos = Parser::parseDirect(buf, startPos, objectId);
  startPos = Parser::parseVect(buf, startPos, vect);
  startPos = Parser::parsePoint(buf, startPos, loc);
  return startPos;
}

PlayerMessagePacket::PlayerMessagePacket() : Packet(PLAYERMESSAGE) {
  frame = 0;
  message = new char[128];
  message[0] = '\0';
}

PlayerMessagePacket::PlayerMessagePacket(const PlayerMessagePacket &toCopy) : 
Packet(PLAYERMESSAGE) {
  playerId = toCopy.playerId;
  frame = toCopy.frame;
  message = new char[128];
  message = strcpy(message, toCopy.message);
}


PlayerMessagePacket& PlayerMessagePacket::operator = (const PlayerMessagePacket& rhs) {
  playerId = rhs.playerId;
  frame = rhs.frame;
  message = strcpy(message,rhs.message);
  return *this;
}

PlayerMessagePacket::~PlayerMessagePacket() {
  delete[] message;
}

Packet* PlayerMessagePacket::makeClone() const {
  return new PlayerMessagePacket(*this);
}

const char* PlayerMessagePacket::getMessage() {
  return message;
}

void PlayerMessagePacket::setMessage(const char* newMessage) {
  message = strcpy(message, newMessage);
}

int PlayerMessagePacket::createPacket(char* buf, int startPos) const {
  startPos = Packet::createPacket(buf, startPos);
  startPos = Parser::makeString(buf, startPos, message);
  return startPos;
}

int PlayerMessagePacket::recvPacket(const char* buf, int startPos) {
  startPos = Packet::recvPacket(buf, startPos);
  startPos = Parser::parseString(buf, startPos, message);
  return startPos;
}

PlayerPacket::PlayerPacket() : Packet(PLAYER) {
  name = strdup("");
}

PlayerPacket::PlayerPacket(const PlayerPacket &toCopy) : Packet(PLAYER), 
shipType(toCopy.shipType), rate(toCopy.rate), kills(toCopy.kills), 
deaths(toCopy.deaths) {
  playerId = toCopy.playerId;
  frame = toCopy.frame;
  name = strdup(toCopy.name);
}


PlayerPacket& PlayerPacket::operator = (const PlayerPacket& rhs) {
  playerId = rhs.playerId;
  frame = rhs.frame;
  shipType = rhs.shipType;
  rate = rhs.rate;
  kills = rhs.kills;
  deaths = rhs.deaths;
  delete[] name;
  name = strdup(rhs.name);
  return *this;
}

PlayerPacket::~PlayerPacket() {
  delete[] name;
}

Packet* PlayerPacket::makeClone() const {
  return new PlayerPacket(*this);
}

const char* PlayerPacket::getName() const {
  return name;
}

void PlayerPacket::setName(const char* newName) {
  delete[] name;
  name = strdup(newName);
}

int PlayerPacket::createPacket(char* buf, int startPos) const {
  startPos = Packet::createPacket(buf, startPos);
  startPos = Parser::makeDirect(buf, startPos, shipType);
  startPos = Parser::makeDirect(buf, startPos, rate);
  startPos = Parser::makeDirect(buf, startPos, kills);
  startPos = Parser::makeDirect(buf, startPos, deaths);
  startPos = Parser::makeString(buf, startPos, name);
  return startPos;
}

int PlayerPacket::recvPacket(const char* buf, int startPos) {
  startPos = Packet::recvPacket(buf, startPos);
  startPos = Parser::parseDirect(buf, startPos, shipType);
  startPos = Parser::parseDirect(buf, startPos, rate);
  startPos = Parser::parseDirect(buf, startPos, kills);
  startPos = Parser::parseDirect(buf, startPos, deaths);
  delete[] name;
  name = new char[128]; //temp buf
  startPos = Parser::parseString(buf, startPos, name);
  char* newName = strdup(name); //Truncate memory
  delete[] name;
  name = newName;
  return startPos;
}

Packet* PlayerPartPacket::makeClone() const {
  return new PlayerPartPacket(*this);
}

PlayerPartPacket::PlayerPartPacket() : Packet (PLAYERPART) {
  status = false;
}

PlayerPartPacket::~PlayerPartPacket() {
}

int PlayerPartPacket::createPacket(char* buf, int startPos) const {
  startPos = Packet::createPacket(buf, startPos);
  startPos = Parser::makeDirect(buf, startPos, status);
  return startPos;
}

int PlayerPartPacket::recvPacket(const char* buf, int startPos) {
  startPos = Packet::recvPacket(buf, startPos);
  startPos = Parser::parseDirect(buf, startPos, status);
  return startPos;
}

Packet* OrbitingEntityPacket::makeClone() const {
  return new OrbitingEntityPacket(*this);
}

int OrbitingEntityPacket::createPacket(char* buf, int startPos) const {
  startPos = Packet::createPacket(buf, startPos);
  startPos = Parser::makeDirect(buf, startPos, objectId);
  startPos = Parser::makeDirect(buf, startPos, theta);
  startPos = Parser::makeDirect(buf, startPos, dist);
  startPos = Parser::makeDirect(buf, startPos, life);
  return startPos;
}

int OrbitingEntityPacket::recvPacket(const char* buf, int startPos) {
  startPos = Packet::recvPacket(buf, startPos);
  startPos = Parser::parseDirect(buf, startPos, objectId);
  startPos = Parser::parseDirect(buf, startPos, theta);
  startPos = Parser::parseDirect(buf, startPos, dist);
  startPos = Parser::parseDirect(buf, startPos, life);
  return startPos;
}

Packet* DamagePacket::makeClone() const {
  return new DamagePacket(*this);
}

int DamagePacket::createPacket(char* buf, int startPos) const {
  startPos = Packet::createPacket(buf, startPos);
  startPos = Parser::makeDirect(buf, startPos, other);
  startPos = Parser::makeDirect(buf, startPos, type);
  startPos = Parser::makeDirect(buf, startPos, life);
  return startPos;
}

int DamagePacket::recvPacket(const char* buf, int startPos) {
  startPos = Packet::recvPacket(buf, startPos);
  startPos = Parser::parseDirect(buf, startPos, other);
  startPos = Parser::parseDirect(buf, startPos, type);
  startPos = Parser::parseDirect(buf, startPos, life);
  return startPos;
}

int DeathPacket::createPacket(char* buf, int startPos) const {
  startPos = Packet::createPacket(buf, startPos);
  startPos = Parser::makeDirect(buf, startPos, other);
  startPos = Parser::makeDirect(buf, startPos, type);
  return startPos;
}

int DeathPacket::recvPacket(const char* buf, int startPos) {
  startPos = Packet::recvPacket(buf, startPos);
  startPos = Parser::parseDirect(buf, startPos, other);
  startPos = Parser::parseDirect(buf, startPos, type);
  return startPos;
}

Packet* DeathPacket::makeClone() const {
  return new DeathPacket(*this);
}

Packet* RespawnPacket::makeClone() const {
  return new RespawnPacket(*this);
}

Packet* StatsPacket::makeClone() const {
  return new StatsPacket(*this);
}

int StatsPacket::createPacket(char* buf, int startPos) const {
  startPos = Packet::createPacket(buf, startPos);
  startPos = Parser::makeDirect(buf, startPos, latency);
  return startPos;
}

int StatsPacket::recvPacket(const char* buf, int startPos) {
  startPos = Packet::recvPacket(buf, startPos);
  startPos = Parser::parseDirect(buf, startPos, latency);
  return startPos;
}

Packet* PingPacket::makeClone() const {
  return new PingPacket(*this);
}

Packet* GreetPacket::makeClone() const {
  return new GreetPacket(*this);
}

int GreetPacket::createPacket(char* buf, int startPos) const {
  startPos = Packet::createPacket(buf, startPos);
  startPos = Parser::makeDirect(buf, startPos, version);
  return startPos;
}

int GreetPacket::recvPacket(const char* buf, int startPos) {
  startPos = Packet::recvPacket(buf, startPos);
  startPos = Parser::parseDirect(buf, startPos, version);
  return startPos;
}

Packet* ServerQuitPacket::makeClone() const {
  return new ServerQuitPacket(*this);
}