//Itana is copyright 2000 by Jason Winnebeck.  You may freely distribute this
//game in its original archive.  If us wish to use code from Itana feel free
//to do so.  If you do a mention in the credits or a mail to
//gillius@webzone.net would be apprciated.

#include "Itana.h"
#include "Game.h"
#include "Prefs.h"
#include "Video.h"
#include "Planet.h"
#include "Asteroid.h"
#include "ItanaPort.h"
#include "Network.h"
#include "Console.h"
#include "Base.h"
#include "Interface.h"
#include "Radar.h"
#include "Star.h"
#include "Ship.h"

//Includes for Game::Generate
#include "ZylexShip.h"
#include "AlsettiShip.h"
#include "HumanShip.h"
#include "OrnonShip.h"

template <class LL>
static void LLInput(LL &list);    //To call input to Entities in a LinkList
template <class LL>
static void LLUpdate(LL &list);   //To update a LinkList
template <class LL>
static void LLDraw(LL &list);     //To draw objects in a LinkList

volatile int frame;
volatile bool fpsdisplay;

#ifdef ALLEGRO_GCC
void fpsproc(...) {
#else
void fpsproc() {
#endif
  frame++;
} END_OF_FUNCTION(fpsproc);

#ifdef ALLEGRO_GCC
void fpsdisp(...) {
#else
void fpsdisp() {
#endif
  fpsdisplay = true;
} END_OF_FUNCTION(fpsdisp);

int Game::currFrame;//current known frame
Interface* const Game::statusBar = new Interface();
Radar* const Game::radar = new Radar();
Stars* const Game::stars = new Stars();
Ship* Game::player = NULL; //the local player is included in the linklists below!
LinkList<MovingEntity> Game::mEnts;   //collision is checked between mEntities
LinkList<Entity> Game::oEnts;         //oEntities only collide with mEntities
LinkList<Effect> Game::effects;

void Game::construct() {
}

void Game::destruct() {
  delete statusBar;
  delete radar;
  delete stars;
}

void Game::init() {
  oEnts.Clear();
  mEnts.Clear();

  //Big 'ol planet...  Everything revolves around this
  oEnts._MoveData(new Planet());
}

void Game::initVideo() {
  Video::initVideo();
  radar->init();
  stars->init();
  Video::setupGameMouse();
}

void Game::setPlayer(Ship* player2, Base* base) {
  player = player2;
  player->bind(CTRL_LOCAL);
  player->bindBase(base);
  base->bindShip(player);
}

void Game::generate(int race) {
  //Player ship and its base
  switch (race) {
  case ID_ZYLEX_SHIP:
    player = new ZylexShip();
    break;
  case ID_ALSETTI_SHIP:
    player = new AlsettiShip();
    break;
  case ID_HUMAN_SHIP:
    player = new HumanShip();
    break;
  case ID_ORNON_SHIP:
  default:
    player = new OrnonShip();
    race = ID_ORNON_SHIP;
    break;
  }
  player->bind(CTRL_LOCAL);
  Base* base = new Base(getrndd(4500.0, 2250.0), getrndd(2*PI), race - MIN_ID_SHIP + MIN_ID_BASE);
  player->bindBase(base);
  player->respawn();
  base->bindShip(player);
  moveEntity(player);
  moveEntity(base);

  //Gotta have da 'roids!
  for (int c=0; c < 150; c++) {
    Asteroid *roid = new Asteroid();
    roid->init(getrndd(4500.0, 2250.0), getrndd(2*PI));
    moveEntity(roid);
  }
}

void Game::playGame() {
  //Set up variables
  stars->init();

  Video::dirtyAll(); //next update should be a FULL update
  LOCK_FUNCTION(fpsproc);
  LOCK_VARIABLE(frame);
  LOCK_FUNCTION(fpsdisp);
  LOCK_VARIABLE(fpsdisplay);
  install_int(fpsproc, 10);
  install_int(fpsdisp, 1000);
  gameLoop();
  remove_int(fpsproc);
  remove_int(fpsdisp);
}

void Game::setFrame(int currFrame2) {
  currFrame = currFrame2;
}

void Game::runToFrame(int currFrame2) {
  while (currFrame <= currFrame2) {
    LLUpdate(oEnts);
    LLUpdate(mEnts);
    LLUpdate(effects);
    doCollisions();
    currFrame++;
  }
}

void Game::gameLoop() {
  bool drawn = true; //has the screen been updated?
  int lastfps = 0, currfps = 0;
  while (!key[prefs->c_quit]) {
    while (frame > 0) {
      doInput();
      doUpdate();
      frame--;
      currFrame++;
      drawn = false;
    }
    if (!drawn) {
      doDraw();
      text_mode(0);
      acquire_screen();
      textprintf(screen, font, 0, scry-text_height(font), makecol(255, 255, 255), "%i/100fps  ", lastfps);
      release_screen();
      text_mode(1);
      if (key[prefs->c_shot]) { //take a screenshot
        BITMAP* visible = create_sub_bitmap(screen, 0, 0, screen->w, screen->h);
        save_bitmap("scrnshot.bmp", visible, NULL);
        destroy_bitmap(visible);
      }
      currfps++;
      drawn = true;
    }
    if (fpsdisplay) {
      lastfps = currfps;
      currfps = 0;
      fpsdisplay = false;
    }
  }
}

void Game::doInput() {
  console->input();
  if (!console->isOpen()) {
    LLInput(mEnts);
    radar->input();
  }
}

bool Game::initForServer() {
  initTimer(10);
  startTimer();
  return false;
}

void Game::doServerUpdate() {
  //a special version of doUpdate for the server
  while (getTimerCount() > 0) {
    LLInput(mEnts); //For server-side prediction
    LLUpdate(oEnts);
    LLUpdate(mEnts);
    LLUpdate(effects);
    doCollisions();
    decrementTimer();
    currFrame++;
  }
}

void Game::closeForServer() {
}

void Game::doUpdate() {
  //Future optimization notes:  network->update() sends faster if at end
  //of doUpdate or recieves faster if at beginning, by +/- 10ms either
  //way.  But right now placing both send/recv parts at end will cause a
  //player to be obsolete for one extra frame which now causes Ship::draw
  //to fail when it tries to get a name, so in future we might want to 
  //separate network into send and receive update portions.
  network->update();
  LLUpdate(oEnts);
  LLUpdate(mEnts);
  LLUpdate(effects);
  doCollisions();
  console->update();
}

void Game::doDraw() {
  Video::setView(player->getLoc().getCenter(), stars);
  radar->setCenter(player->getLoc().getCenter());
  radar->update();

  player->updateStats(statusBar);
  stars->draw();
  LLDraw(oEnts);
  LLDraw(effects);
  LLDraw(mEnts);
  console->draw();
  Video::update(statusBar, radar);
}

void Game::doCollisions() {
  //Do the collision checks -- compare every mEnt to every other mEnt
  mEnts.SetToStart();
  int l = mEnts.Length();
  int c, d;
  for (c=0; c<l-1; c++) {
    MovingEntity* curr = mEnts._GetNext();
    mEnts._Advance();
    LLPos save = mEnts.Save(); //save position in list
    for (d=c+1; d<l; d++) {
      if (mEnts._GetNext()->isCollision(curr)) {
        mEnts._GetNext()->doCollision(curr);
        curr->doCollision(mEnts._GetNext());
      }
      mEnts._Advance();
    }
    mEnts.Load(save);           //restore position
  }

  //Collision checks between mEnts and oEnts
  mEnts.SetToStart();
  for (c=0; c<l; c++) {
    MovingEntity* curr = mEnts._GetNext();
    oEnts.SetToStart();
    int olen = oEnts.Length();
    for (d=0; d<olen; d++) {
      if (oEnts._GetNext()->isCollision(curr)) {
        oEnts._GetNext()->doCollision(curr); 
          //It may be possible to entirely remove this statement if OrbitingEntities do nothing on collision
        curr->doCollision(oEnts._GetNext());
      }
      oEnts._Advance();
    }
    mEnts._Advance();
  }
}

template <class LL>
static void LLInput(LL &list) {   //To call input to Entities in a LinkList
  list.SetToStart();
  int len = list.Length();
  for (int c=0; c<len; c++) {
    list._GetNext()->input();
    list._Advance();
  }
}

template <class LL>
static void LLUpdate(LL &list) {
  list.SetToStart();
  int len = list.Length();
  for (int c=0; c<len; c++)
    if (list._GetNext()->update())
      list._DeleteThis();
    else
      list._Advance();
}

template <class LL>
static void LLDraw(LL &list) {
  list.SetToStart();
  int len = list.Length();
  for (int c=0; c<len; c++) {
    list._GetNext()->draw();
    list._Advance();
  }
}