//Itana is copyright 2000 by Jason Winnebeck.  You may freely distribute this
//game in its original archive.  If us wish to use code from Itana feel free
//to do so.  If you do a mention in the credits or a mail to
//gillius@webzone.net would be appreciated.

#include "Itana.h"
#include "Client.h"
#include "Packet.h"
#include "Ship.h"
#include "NetPeer.h"
#include "Asteroid.h"
#include "Base.h"
#include "Ship.h"
#include "AlsettiShip.h"
#include "HumanShip.h"
#include "ZylexShip.h"
#include "OrnonShip.h"
#include "ItanaPort.h"
#include "Console.h"
#include "Game.h"
#include "itanadat.h"
#include "Sound.h"
#include "ParaBullet.h"

void Client::doBindings(NetPeer* peer) {
  al_trace("Doing bindings for player %i.\n", peer->data.playerId);
  if (peer->ship && peer->base) {
    peer->base->bindShip(peer->ship);
    peer->ship->bindBase(peer->base);
    Game::moveEntity(peer->ship);
    Game::moveEntity(peer->base);
    if (peer == us) {
      Game::setPlayer(peer->ship, peer->base);
      peer->ship->bind(CTRL_LOCAL);
    }
  }
  console->addLine("%s has entered the game.", peer->data.getName());
}

void Client::handlePacket(Packet* next) {
  if (next->getType() != FRAME)
    if (next->frame > Game::getFrame())
      Game::runToFrame(next->frame);

  switch(next->getType()) {
    case SHIP:
      handleShipPacket((ShipPacket*)next);
      break;
    case DYNAMICENTITY:
      handleDynamicEntityPacket((DynamicEntityPacket*)next);
      break;
    case PLAYER:
      handlePlayerPacket((PlayerPacket*)next);
      break;
    case PLAYERPART:
      handlePlayerPartPacket((PlayerPartPacket*)next);
      break;
    case PLAYERMESSAGE:
      Sound::playSound(SFX_DAT_INMSG_WAV);
      console->addLine("%s: %s", getName(next->playerId), ((PlayerMessagePacket*)next)->getMessage());
      break;
    case ORBITINGENTITY:
      handleOrbitingEntityPacket((OrbitingEntityPacket*)next);
      break;
    case FRAME:
      Game::setFrame(next->frame);
      break;
    case DAMAGE:
    case RESPAWN:
      {
        NetPeer* peer = findPlayer(next->playerId);
        if (peer) //ignore this packet if the player doesn't exist.
          peer->ship->usePacket(next);
      }
      break;
    case DEATH:
      handleDeathPacket((DeathPacket*)next);
      break;
    case STATSPACKET:
      findPlayer(next->playerId)->latency = ((StatsPacket*)next)->latency;
      break;
    case PING:
      sendPacket(next->makeClone(), false);
      break;
    case SERVERQUIT:
      //terminate game session here
      break;
    default:
      al_trace("Unknown packet recieved.\n");
  }
  delete next;
}

void Client::handleDynamicEntityPacket(DynamicEntityPacket* packet) {
  if (packet->objectId == ID_PARA) {
    ParaBullet* bull = new ParaBullet();
    bull->usePacket(packet);
    bull->setOwner(findPlayer(packet->playerId)->ship);
    Game::moveEntity(bull);
  } else
    al_trace("Illegal DynamicEntityPacket");
}

void Client::handleShipPacket(ShipPacket* packet) {
  NetPeer* peer = findPlayer(packet->playerId);
  if (peer) { //ignore this packet if the player doesn't exist.
    if (!peer->ship) {
      al_trace("Creating ship for player %i\n", packet->playerId);
      switch (peer->data.shipType) {
      case ID_ALSETTI_SHIP:
        peer->ship = new AlsettiShip();
        break;
      case ID_ORNON_SHIP:
        peer->ship = new OrnonShip();
        break;
      case ID_HUMAN_SHIP:
        peer->ship = new HumanShip();
        break;
      case ID_ZYLEX_SHIP:
        peer->ship = new ZylexShip();
        break;
      }
      peer->ship->init(Rect(), Vector());
      peer->ship->bind(CTRL_NET + peer->data.playerId);
      if (peer->base)
        doBindings(peer);
    }
    peer->ship->usePacket(packet);
  } else
    al_trace("Recieved an illegal ship packet.\n");
}

void Client::handlePlayerPacket(PlayerPacket* packet) {
  al_trace("Recieved a PlayerPacket for player %i\n", packet->playerId);
  if (packet->playerId == us->data.playerId) {
    al_trace("Creating our own player data.  Our ID is %i\n", packet->playerId);
    us->data = *packet;
    al_trace("Player %i(us) == %s\n", us->data.playerId, us->data.getName());
  } else {
    NetPeer* peer = findPlayer(packet->playerId);
    if (!peer) { //ignore packet if this player exists.  In the future
      //we might want to replace players for stability issues, in case for
      //some reason we didn't get a PlayerPartPacket.
      al_trace("Creating a new player: %i.\n", packet->playerId);
      peer = new NetPeer();
      peer->data = *packet;
      peers._MoveData(peer);
      al_trace("Player %i == %s\n", peer->data.playerId, peer->data.getName());
    }
  }
}

void Client::handlePlayerPartPacket(PlayerPartPacket* packet) {
  al_trace("Recieved a PlayerPartPacket for player %i\n", packet->playerId);
  NetPeer* peer = findPlayer(packet->playerId);
  if (peer) {
    //only can delete existing players
    if (packet->status)
      console->addLine("%s has left the game.", peer->data.getName());
    else
      console->addLine("%s dropped.", peer->data.getName());
    peer->ship->makeObsolete();
    peer->base->makeObsolete();
    deletePlayer(peer);
  }
}

void Client::handleOrbitingEntityPacket(OrbitingEntityPacket* packet) {
  if (packet->objectId == ID_ROID) {
    Asteroid* newObj = new Asteroid();
    newObj->usePacket(packet);
    Game::moveEntity(newObj);
  } else if ((packet->objectId >= MIN_ID_BASE) && (packet->objectId <= MAX_ID_BASE)) {
    al_trace("Received a base packet for player %i\n", packet->playerId);
    NetPeer* peer = findPlayer(packet->playerId);
    if (peer) { //ignore this packet if the player doesn't exist.
      if (!peer->base) {
        al_trace("Creating a new base for player %i\n", packet->playerId);
        switch (peer->data.shipType) {
        case ID_ALSETTI_SHIP:
          peer->base = new Base(0.0, 0.0, ID_ALSETTI_BASE);
          break;
        case ID_ORNON_SHIP:
          peer->base = new Base(0.0, 0.0, ID_ORNON_BASE);
          break;
        case ID_HUMAN_SHIP:
          peer->base = new Base(0.0, 0.0, ID_HUMAN_BASE);
          break;
        case ID_ZYLEX_SHIP:
          peer->base = new Base(0.0, 0.0, ID_ZYLEX_BASE);
          break;
        }
        if (peer->ship)
          doBindings(peer);
      }
      peer->base->usePacket(packet);
    }
  }
}

void Client::handleDeathPacket(DeathPacket* packet) {
  NetPeer* peer = findPlayer(packet->playerId);
  if (peer) {//ignore this packet if the player doesn't exist.
    peer->ship->usePacket(packet);
    peer->data.deaths++;
    switch( packet->type ) {
    case PLANETDEATH:
      console->addLine("%s couldn't manage to avoid an entire planet.",
        peer->data.getName());
      break;
    case BANGEDDEATH:
      console->addLine("%s got banged up.",
        peer->data.getName());
      break;
    case SHOTDEATH:
      {
        NetPeer* otherPeer = findPlayer(packet->other);
        otherPeer->data.kills++;
        console->addLine("%s was blown to bits by %s.",
          peer->data.getName(), otherPeer->data.getName());
      }
      break;
    }
  }
}