//Itana is copyright 2000 by Jason Winnebeck.  You may freely distribute this
//game in its original archive.  If us wish to use code from Itana feel free
//to do so.  If you do a mention in the credits or a mail to
//gillius@webzone.net would be appreciated.

#include "Itana.h"
#include "Client.h"
#include "Packet.h"
#include "Game.h"
#include "Parser.h"
#include "Base.h"
#include "Ship.h"
#include "ItanaPort.h"
#include "Prefs.h"
#include "NetPeer.h"

const int NETBUF_SIZE = 1024;
static char netbuf[NETBUF_SIZE]; //a large but static buf
static char netbuf2[NETBUF_SIZE];//another large buf

Client::Client() {
  netbuf[0] = '\0';
  netbuf[1] = '\0';
  netbuf2[0] = '\0';
  netbuf2[1] = '\0';
  us = NULL;
  serverTCP = NULL;
  serverUDP = NULL;
  connected = false;
  tempBuf = new char[NETBUF_SIZE];
  tempBufLen = 0;
}

Client::~Client() {
  delete us;
}

void Client::clientInit() {
  serverTCP = NULL;
  serverUDP = NULL;
  us = new NetPeer();
  peers.Clear();
}

NetPeer* Client::findPlayer(PlayerId id) {
  if (us->data.playerId == id)
    return us;

  int l = peers.Length();
  LLPos save = peers.Save();
  peers.SetToStart();
  NetPeer* ret = NULL;
  for (int c=0; c<l && !ret; c++) {
    if (peers._GetNext()->data.playerId == id)
      ret = peers._GetNext();
    else
      peers._Advance();
  }
  peers.Load(save);

  return ret;
}

void Client::deletePlayer(NetPeer* peer) {
  //This function cannot be called while traversing a list
  int l = peers.Length();
  peers.SetToStart();
  for (int c=0; c<l; c++) {
    if (peers._GetNext() == peer) {
      peers._DeleteThis();
      return;
    } else
      peers._Advance();
  }
}

PlayerId Client::whoIsThis(void* other) {
  if (!connected)
    return 0; //return dummy value in single player
  int l = peers.Length();
  LLPos save = peers.Save();
  peers.SetToStart();
  int ret = -1;
  if (us->base == other || us->ship == other)
    ret = us->data.playerId;
  for (int c=0; (c<l) && (ret == -1); c++) {
    NetPeer* next = peers._GetNext();
    if (next->base == other || next->ship == other)
      ret = next->data.playerId;
    peers._Advance();
  }
  peers.Load(save);
  return ret;
}

const char* Client::getName(PlayerId id) {
  if (!connected)
    return "Player"; //Return default name in single player
  //Returns the player name associated with this PlayerId
  int l = peers.Length();
  LLPos save = peers.Save();
  peers.SetToStart();
  const char* ret = NULL;
  if (us->data.playerId == id)
    ret = us->data.getName();
  for (int c=0; (c<l) && (ret == NULL); c++) {
    NetPeer* next = peers._GetNext();
    if (next->data.playerId == id)
      ret = next->data.getName();
    peers._Advance();
  }
  peers.Load(save);

  return ret;
}

const char* Client::getOurName() const {
  if (connected)
    return us->data.getName();
  else
    return "Player";
}

const NetPeer** Client::getPeers(){
  //For Console -- returns an NEWLY ALLOCATED array of NetPeer*
  //pointing to objects to display which won't be deleted.
  //The last pointer is NULL to si\gnify the end of the array.
  int l = peers.Length();
  const NetPeer** ret = new const NetPeer*[l + 2]; //peers + us + NULL
  if (!connected) {
    //display nothing since deaths/kills aren't tracked in single
    ret[0] = NULL;
    return ret;
  }

  ret[0] = us;
  int c = 0;
  LLPos save = peers.Save();
  peers.SetToStart();
  for (; c < l; c++) {
    ret[c+1] = peers._GetNext();
    peers._Advance();
  }
  peers.Load(save);

  ret[l+1] = NULL;
  return ret;
}

void Client::sendPacket(Packet* packet, bool reliable) {
  if (connected) {
    switch (packet->getType()) {
    case DEATH:
      //Handle score calcs if this is a death packet
      packet->playerId = us->data.playerId;
      packet->frame = Game::getFrame();
      handleDeathPacket((DeathPacket*)packet);
      us->ship->usePacket(packet); //Kills us off
      break;
    case PING:
      //We want to send the identical packet back
      break;
    default:
      packet->playerId = us->data.playerId;
      packet->frame = Game::getFrame();
    }
    if (reliable)
      TCPqueue.add(packet);
    else
      UDPqueue.add(packet);
  }
}

bool Client::connect(char* target, const char* name, int shipType) {
  clientInit();
  //connect to the server and perform all nessacary operations
  //return true if error
  serverTCP = net_openconn(netDriver, NULL);
  if (serverTCP == NULL) {
    sprintf(lastError, "Could not create a channel for this driver.");
    return true;
  }

  if (net_connect_wait_time(serverTCP, target, 5)) {
    sprintf(lastError, "Server timed-out or refused to connect.");
    net_closeconn(serverTCP);
    return true;
  }

  //get port
  while (net_receive_rdm(serverTCP, netbuf, NETBUF_SIZE) <= 0) {
    yield();
  }
  strcpy(netbuf2, target);
  int idx = 0;
  for (; netbuf2[idx] != ':'; idx++) {}
  idx++;
  strcpy(&netbuf2[idx], netbuf); //put port after address
  serverUDP = net_openchannel(netDriver, "");
  if (serverUDP == NULL) {
    sprintf(lastError, "Could not create a channel to %s.", netbuf2);
    net_closeconn(serverTCP);
    return true;
  }
  net_assigntarget(serverUDP, netbuf2);

  //get Id/Greeting
  while (net_receive_rdm(serverTCP, netbuf, NETBUF_SIZE) <= 0) {
    yield();
  }
  GreetPacket versionCheck;
  versionCheck.recvPacket(netbuf, 0);
  if (versionCheck.version != PROTOCOLVERSION) {
    //sprintf(lastError, "Server has a different network protocol version than you. Us: %i Server: %i", int(PROTOCOLVERSION), int(versionCheck.version));
    sprintf(lastError, "Protocol version error.");
    //Send back a greeting packet so the server knows too
    versionCheck.version = PROTOCOLVERSION;
    idx = versionCheck.createPacket(netbuf, 0);
    netbuf[idx++] = ENDOFPACKET;
    net_send_rdm(serverTCP, netbuf, idx); //We don't poll because for some reason we have serious problems polling here.
    net_closeconn(serverTCP);
    net_closechannel(serverUDP);
    Game::init(); //clear out any game data we might have received
    return true;
  }

  //Create our PlayerPacket
  us->data.playerId = versionCheck.playerId;
  us->data.setName(name);
  us->data.rate = inRate;
  us->data.shipType = shipType;
  us->data.frame = 0;
  us->data.kills = us->data.deaths = 0;

  //Send player data and greeting
  idx = 0;
  idx = versionCheck.createPacket(netbuf, idx);
  idx = us->data.createPacket(netbuf, idx);
  netbuf[idx++] = ENDOFPACKET;
  if (net_send_rdm(serverTCP, netbuf, idx) == -1)
    return true;

  //Get game data
  bool exit = false;
  while (!exit) {
    int check = net_receive_rdm(serverTCP, netbuf, NETBUF_SIZE);
    if (check == -1)
      al_trace("net_receive_rdm error!!!");
    else if (check != 0) {
      int currPos = 0;
      int type = Packet::getNextType(netbuf, currPos);
      while (type != ENDOFPACKET) {
        if (type == NONE) { //This is the marker meaning we are through
          exit = true;
          delete Packet::parsePacket(netbuf, currPos); //Skip marker packet
        } else if (type != ENDOFPACKET)
          handlePacket(Packet::parsePacket(netbuf, currPos));

        type = Packet::getNextType(netbuf, currPos);
      }
      dataIn += check;
    } else
      yield();
    if (key[prefs->c_quit]) {
      sprintf(lastError, "User canceled network connection.");
      net_closeconn(serverTCP);
      net_closechannel(serverUDP);
      Game::init(); //clear game data received so far
      return true;
    }
  }

  //Send server UDP contact info
  netbuf[0] = ENDOFPACKET;
  net_send(serverUDP, netbuf, 1); //Send end of packet in case server gets > 1
  net_send(serverUDP, netbuf, 1); //Sending 3 increases chances this gets there.
  net_send(serverUDP, netbuf, 1); 

  //We've finished connecting
  al_trace("Successful connection.\n");
  connected = true;
  
  return false;
}

void Client::disconnect() {
  if (connected) {
    while (key[prefs->c_quit]) {} //wait for ESC to be released

    PlayerPartPacket quitNotice;
    quitNotice.status = true;
    quitNotice.playerId = us->data.playerId;
    quitNotice.frame = Game::getFrame();
    int idx = quitNotice.createPacket(netbuf, 0);
    netbuf[idx++] = ENDOFPACKET;
    //send quit notice
    while (net_send_rdm(serverTCP, netbuf, idx) != -1 && !key[prefs->c_quit]) {}
    int in=0, out=1;
    while (out && !key[prefs->c_quit]) { //poll until the server gets it
      net_conn_stats(serverTCP, &in, &out);
      yield();
    }
    net_closeconn(serverTCP);
    net_closechannel(serverUDP);
    connected = false;
    Game::init(); //Clear out the Game data
  }
}

bool Client::isConnected() {
  return connected;
}

bool Client::update() {
  if (connected) {
    //Receive all packets pending
    int recv = getNextPacket(netbuf, NETBUF_SIZE);
    while (recv > 0) {
      dataIn += recv;
      int currPos = 0;
      while (Packet::getNextType(netbuf, currPos) != ENDOFPACKET) {
        Packet* nextPacket = Packet::parsePacket(netbuf, currPos);
        handlePacket(nextPacket);
      }
      recv = getNextPacket(netbuf, NETBUF_SIZE);
    }
    if (recv == -1)
      return true; //There was an error -- bail out like the wussy we are.

    //This is only temporary code and should be replaced with the rate
    //management code.  Sets network updates to once every 3 frames.
    //A 28.8 modem can handle this, and 33 fps updaterate might
    //be good enough.
    const int UPDATE_RATE = 3;
    static int sendDelay = 0;
    bool positionSent = false;

    if (sendDelay <= 0) {
      //Send pending packets and position packet.
      if (!TCPqueue.isEmpty()) {
        //Send a position packet
        TCPqueue.add(us->ship->makePacket());
        dataOut += sendTCPPacket();
        positionSent = true;
      }
      if (!positionSent)
        UDPqueue.add(us->ship->makePacket());
      dataOut += sendUDPPacket();
      sendDelay = UPDATE_RATE;
    } else
      sendDelay--;
  }

  //returns true only on an unrecoverable error which would require
  //termination of the network connection
  return false;
}

int Client::getNextPacket(char* buf, int bufsize) {
  int recv = 0;
  recv = net_receive_rdm(serverTCP, buf, bufsize);
  if (recv == 0)
    recv = net_receive(serverUDP, netbuf, NETBUF_SIZE, NULL);
  return recv;
}

int Client::sendTCPPacket() {
  int dataOut = 0;
  if (tempBufLen) {
    if (net_send_rdm(serverTCP, tempBuf, tempBufLen) == 0) {
      dataOut = tempBufLen;
      tempBufLen = 0;
    }
  } else if (!TCPqueue.isEmpty()) {
    while (!TCPqueue.isEmpty() && tempBufLen < PACKETMAX) {
      //try to send as many game packets as possible in this network packet.
      Packet* nextPacket = TCPqueue.pop();
      tempBufLen = nextPacket->createPacket(tempBuf, tempBufLen);
      delete nextPacket;
    }
    tempBuf[tempBufLen++] = ENDOFPACKET;
    if (net_send_rdm(serverTCP, tempBuf, tempBufLen) == 0) {
      dataOut = tempBufLen;
      tempBufLen = 0;
    } //else tempBufLen != 0 and will be saved for later resend
  }
  return dataOut;
}

int Client::sendUDPPacket() {
  int dataOut = 0;  
  if (!UDPqueue.isEmpty() && tempBufLen == 0) {
    //check tempBufLen as to not overflood the client when RDMs are waiting
    while (!UDPqueue.isEmpty() && tempBufLen < PACKETMAX) {
      //try to send as many game packets as possible in thie network packet.
      Packet* nextPacket = UDPqueue.pop();
      tempBufLen = nextPacket->createPacket(tempBuf, tempBufLen);
      delete nextPacket;
    }
    
    tempBuf[tempBufLen++] = ENDOFPACKET;
    net_send(serverUDP, tempBuf, tempBufLen);
    dataOut = tempBufLen;
    tempBufLen = 0;
  }
  return dataOut;
}
