//Itana is copyright 2000 by Jason Winnebeck.  You may freely distribute this
//game in its original archive.  If us wish to use code from Itana feel free
//to do so.  If you do a mention in the credits or a mail to
//gillius@webzone.net would be apprciated.

#include "Itana.h"
#include "itanadat.h"
#include "Video.h"
#include "IntRect.h"
#include "IntPoint.h"
#include "Interface.h"
#include "Radar.h"
#include "Rect.h"
#include "Star.h"
#include "Prefs.h"
#include "MovingEffect.h"
#include "ParaBullet.h"

//Declaring static variables
int Video::depth;         //The current color depth
BITMAP* Video::buffer;    //the local memory buffer
DATAFILE* Video::sprites; //graphics data
Interface* Video::bar;    //the interface
int Video::viewx, Video::viewy;  //viewpoints
LinkList<Pixel> Video::oldPixels;  //Special one-pixel rect cases. This is beneficial for our
LinkList<Pixel> Video::newPixels;  //many calculations on rects when we have several hundred stars
IntRect* Video::noClearRects;        //Rects that don't clear
int Video::numNoClearRects;
IntRect* Video::oldRects;            //an "old list" of clears
int Video::numOldRects;
IntRect* Video::newRects;            //For dirty rectangling
int Video::numNewRects;              //number of rects currently stored
LinkList<IntRect> Video::optRects;   //finalized(optimized) rectanges to copy to screen
bool Video::blitAll;                 //if true then the whole screen is dirty
IntRect Video::mouseRect;            //the rectangle the mouse occupies
bool Video::mouse;                   //is the mouse on the screen?

//the main statusbar area above is 48 pixels tall
//this is the viewport on the PHYSICAL screen not the map!
const int BAR_HEIGHT = 48;
const IntRect viewport(0, BAR_HEIGHT, scrx, scry-BAR_HEIGHT);
const int MAX_RECTS = 128;  //ABITRARY VALUE FOR NOW!!!
const int MAX_NOCLEARRECTS = 16; //There should NEVER be more no-clear rects than this number

void display_callback() {
  Video::dirtyAll();
}

void Video::construct() {
  noClearRects = new IntRect[MAX_NOCLEARRECTS];
  newRects = new IntRect[MAX_RECTS];
  oldRects = new IntRect[MAX_RECTS];
  depth = 8;
  buffer = NULL;
  sprites = NULL;
  mouse = false;
}

void Video::installCallbacks() {
  //Set both display switching modes for most compatability.  Doesn't check
  //for errors because if the callbacks don't work there's nothing we can do
  //anyways.
  set_display_switch_callback(SWITCH_OUT, display_callback);
  set_display_switch_callback(SWITCH_IN, display_callback);
  set_display_switch_mode(SWITCH_BACKGROUND);
}

void Video::destruct() {
  destroy_bitmap(buffer);
  delete[] newRects;
  delete[] oldRects;
  delete[] noClearRects;
}

void Video::init(DATAFILE* datafileRoot) {
  if (buffer)
    destroy_bitmap(buffer);
  buffer = create_bitmap(scrx, scry);
  clear(buffer);

  sprites = (DATAFILE*)datafileRoot[GRFX_DAT].dat;
  viewx = viewy = 0;
  numNewRects = numOldRects = numNoClearRects = 0;
  blitAll = true;
  mouse = false;

  //Initalize other classes using video with animations
  Explosion::initVideo(datafileRoot);
  ParaBullet::initVideo(datafileRoot);
}

void Video::initVideo() {
  //This clipping is always true unless otherwise set, and if it is changed,
  //the function should return the clipping to this clip.
  set_clip(screen, viewport.x, viewport.y, viewport.getRight(), viewport.getBottom());
  set_clip(buffer, viewport.x, viewport.y, viewport.getRight(), viewport.getBottom());
  text_mode(-1);
  setDefaultBlender();
}

void Video::setupGameMouse() {
  show_mouse(NULL);
  set_mouse_sprite_focus(7, 7);
  set_mouse_range(viewport.x, viewport.y, viewport.getRight(), viewport.getBottom());
  position_mouse(viewport.getCenter().x, viewport.getCenter().y);
  mouseRect.w = mouseRect.h = 16;
}

void Video::setupGUIMouse() {
  if (prefs->fastmouse)
    set_mouse_speed(1, 1);
  else
    set_mouse_speed(2, 2);
  set_mouse_sprite_focus(0, 0);
  set_mouse_range(0, 0, SCREEN_W-1, SCREEN_H-1);
  /* //Old code used when converting from datafile to current
  destroy_bitmap(guiMouse); //kill old one and convert to current gfx mode
  BITMAP* dataMouse = (BITMAP*)(((DATAFILE*)root[GRFX_DAT].dat)[GRFX_DAT_GUIMOUSE_BMP].dat);
  BITMAP* guiMouse = create_bitmap(dataMouse->w, dataMouse->h);
  blit(dataMouse, guiMouse, 0, 0, 0, 0, dataMouse->w, dataMouse->h);
  */
  set_mouse_sprite((BITMAP*)sprites[GRFX_DAT_GUIMOUSE_BMP].dat);
  //position_mouse((SCREEN_W-1)/2, (SCREEN_H-1)/2);
  show_mouse(screen);
}

int Video::getDepth() {
  return depth;
}

Rect Video::getView() {
  return Rect(viewx, viewy, viewport.w, viewport.h);
}

bool Video::tryDepth(int depth) {
  //Helper function for setVideoMode
  set_color_depth(depth);
  Video::depth = depth;
#ifdef _DEBUG
  return (set_gfx_mode(GFX_DIRECTX_WIN, scrx, scry, 0, 0) < 0);
#else
  return (set_gfx_mode(GFX_AUTODETECT, scrx, scry, 0, 0) < 0);
#endif
}

bool Video::setVideoMode() {
  //Attempt to set the proper depth
  if (tryDepth(16))
    if (tryDepth(15))
      if (tryDepth(32))
        if (tryDepth(24))
          return true;
  installCallbacks();
  return false;
}

void Video::setAddBlender() {
  //sets transparency to additive mode.  You MUST return to default mode
  //when you are done drawing
  set_add_blender(0, 0, 0, 255);
}

void Video::setDefaultBlender() {
  //sets blending to default mode.
  set_trans_blender(0, 0, 0, 32);
}

void Video::drawRLESprite(int idx, IntRect rect) {
  rect.x -= viewx;
  rect.y -= viewy;
  if (rect.isCollision(viewport)) {
    draw_rle_sprite(buffer, (RLE_SPRITE*)sprites[idx].dat, rect.x, rect.y);
    addRect(rect);
  }
}

void Video::drawSprite(int idx, const IntRect& realrect, const IntRect& vidrect, fixed theta) {
  drawSprite((BITMAP*)sprites[idx].dat, realrect, vidrect, theta);
}

void Video::drawSprite(BITMAP* bmp, IntRect realrect, const IntRect& vidrect, fixed theta) {
    //realrect contains the map x, y, width, and height for the object.
    //vidrect contains offset x and y and big width and height for a rect
    //which will hold all possible area the rotated sprite can contain
  realrect.x -= viewx;
  realrect.y -= viewy;
  if (realrect.isCollision(viewport)) {
    //only draw if the sprite is on the screen
    rotate_sprite(buffer, bmp, realrect.x, realrect.y, theta);
    realrect.x += vidrect.x; //add offsets for dirty rect
    realrect.y += vidrect.y;
    realrect.w = vidrect.w;  //copy new widths and heights
    realrect.h = vidrect.h;
    addRect(realrect);
  }
}

void Video::drawTransSprite(int idx, IntRect realrect, const IntRect& vidrect, fixed theta) {
    //realrect contains the map x, y, width, and height for the object.
    //vidrect contains offset x and y and big width and height for a rect
    //which will hold all possible area the rotated sprite can contain
  realrect.x -= viewx;
  realrect.y -= viewy;
  if (realrect.isCollision(viewport)) {
    //only draw if the sprite is on the screen
    BITMAP *temptrans = create_bitmap(realrect.w * 2, realrect.h * 2);
    clear_to_color(temptrans, makecol(255, 0, 255));
    rotate_sprite(temptrans, (BITMAP*)sprites[idx].dat, realrect.w/2, realrect.h/2, theta);
    draw_trans_sprite(buffer, temptrans, realrect.x - realrect.w/2, realrect.y - realrect.h/2);
    destroy_bitmap(temptrans);

    realrect.x += vidrect.x; //add offsets for dirty rect
    realrect.y += vidrect.y;
    realrect.w = vidrect.w;  //copy new widths and heights
    realrect.h = vidrect.h;
    addRect(realrect);
  }
}

void Video::drawTransSprite(BITMAP* bmp, IntRect rect) {
  rect.x -= viewx;
  rect.y -= viewy;
  if (rect.isCollision(viewport)) {
    draw_trans_sprite(buffer, bmp, rect.x, rect.y);
    addRect(rect);
  }
}

void Video::drawDot(IntPoint point, int color) {
  point.x -= viewx;
  point.y -= viewy;
  if (viewport.isCollision(point)) {
    putpixel(buffer, point.x, point.y, color);
    newPixels._MoveData(new Pixel(point.x, point.y, color));
  }
}

void Video::drawTransDot(IntPoint point, int color) {
  point.x -= viewx;
  point.y -= viewy;
  if (viewport.isCollision(point)) {
    drawing_mode(DRAW_MODE_TRANS, NULL, 0, 0);
    putpixel(buffer, point.x, point.y, color);
    drawing_mode(DRAW_MODE_SOLID, NULL, 0, 0);
    newPixels._MoveData(new Pixel(point.x, point.y, color));
  }
}

void Video::drawStar(int x, int y, int color) {
  putpixel(buffer, x, y, color);
  newPixels._MoveData(new Pixel(x, y, color));
}

void Video::drawMapText(IntRect rect, int color, const char* format, ...) {
  //create the string
  char buf[512];
  va_list ap;
  va_start(ap, format);
  vsprintf(buf, format, ap);
  va_end(ap);

  //create video rectangle and draw on screen
  rect.x -= viewx;
  rect.y -= viewy;
  rect.w = text_length(font, buf);
  rect.h = text_height(font);
  text_mode(-1);
  textout(buffer, font, buf, rect.x, rect.y, color);
  addRect(rect);
}

void Video::drawText(IntRect rect, int color, const char* format, ...) {
  //prints text at an absolute position on the screen.
  //create the string
  char buf[512];
  va_list ap;
  va_start(ap, format);
  vsprintf(buf, format, ap);
  va_end(ap);

  //create video rectangle and draw on screen
  rect.w = text_length(font, buf);
  rect.h = text_height(font);
  text_mode(-1);
  textout(buffer, font, buf, rect.x, rect.y, color);
  addRect(rect);
}

void Video::drawTransCircle(IntPoint point, int radius, int color) {
  point.x -= viewx;
  point.y -= viewy;
  if (viewport.isCollision(point)) {
    drawing_mode(DRAW_MODE_TRANS, NULL, 0, 0);
    circlefill(buffer, point.x, point.y, radius, color);
    drawing_mode(DRAW_MODE_SOLID, NULL, 0, 0);
    IntRect scrRect(point.x - radius, point.y - radius, point.x + radius, point.y + radius);
    addRect(scrRect);
  }
}

void Video::addNoClearRect(const IntRect& rect) {
  ASSERT(numNoClearRects < MAX_NOCLEARRECTS);
  noClearRects[numNoClearRects] = rect;
  numNoClearRects++;
}

void Video::showMouse(bool shown) {
  //true if mouse is on, false if off
  mouse = shown;
}

void Video::addRect(const IntRect& rect) {
  if (numNewRects < MAX_RECTS) {
    newRects[numNewRects] = rect;
    numNewRects++;
  } else
    //after MAX_RECTS it's better just to blit the whole thing
    blitAll = true;
}

void Video::setView(const IntPoint& center, Stars* stars) {
  int newviewx = center.x - viewport.w/2 - viewport.x;
  int newviewy = center.y - viewport.h/2 - viewport.y;
  stars->update(viewx - newviewx, viewy - newviewy);
  viewx = newviewx;
  viewy = newviewy;
}

void Video::update(Interface* statusbar, Radar* radar) {
  //Draws the mouse cursor if shown and updates screen
  drawEdges();
  if (blitAll) {
    acquire_screen();
    if (prefs->vsync)
      vsync();
    updateAll(statusbar, radar);
    release_screen();
  } else {
    //draw interface and radar
    set_clip(buffer, 0, 0, buffer->w-1, buffer->h-1);
    statusbar->draw(buffer, sprites);
    radar->draw(buffer);
    set_clip(buffer, viewport.x, viewport.y, viewport.getRight(), viewport.getBottom());
    if (mouse)
      drawMouse();
    optimizeRects();

    acquire_screen();
    if (prefs->vsync)
      vsync();
    clearPixels();
    updatePixels();
    updateNoClearRects();
    updateOptRects();
    release_screen();
    doClears();
  }

  //reset new and opt rects, swap old and new
  numOldRects = numNewRects;
  numNewRects = numNoClearRects = 0;
  optRects.Clear();
  IntRect* temp = oldRects;
  oldRects = newRects;
  newRects = temp;
  oldPixels.Swap(newPixels);
  newPixels.Clear();
} 

void Video::snowball(const IntRect& rect) {
  //Add a rect to optRect using snowball algorithm
  IntRect* working = new IntRect(rect); //make a copy
  optRects._MoveData(working);          //add to list
  optRects.SetToStart();
  optRects._Advance();                  //proceed to first element to compare
  while (!optRects.IsOverEnd()) {       //proceed until the end
    if (working->isCollision(*optRects._GetNext())) {
      working->combineWith(*optRects._GetNext());//combine
      optRects._DeleteThis();           //delete and advance
    } else {
      optRects._Advance();              //advance
    }
  }
}

void Video::optimizeRects() {
  int c;
  for (c=0; c<numOldRects; c++)
    snowball(oldRects[c]);
  for (c=0; c<numNewRects; c++)
    snowball(newRects[c]);
}

void Video::doClears() {
  int c;
  for (c=0; c<numNewRects; c++) {
    //clear used areas from buffer
    rectfill(buffer, newRects[c].x, newRects[c].y,
      newRects[c].x+newRects[c].w-1, newRects[c].y+newRects[c].h-1, 0);
  }
  newPixels.SetToStart();
  int l = newPixels.Length();
  for (c=0; c<l; c++) {
    Pixel* pixel = newPixels._GetNext();
    putpixel(buffer, pixel->x, pixel->y, 0);
    newPixels._Advance();
  }
}

void Video::clearPixels() {
  oldPixels.SetToStart();
  int l = oldPixels.Length();
  for (int c=0; c<l; c++) {
    Pixel* pixel = oldPixels._GetNext();
    putpixel(screen, pixel->x, pixel->y, 0);
    oldPixels._Advance();
  }
}

void Video::updateAll(Interface* statusbar, Radar* radar) {
  if (mouse)
    drawMouse();
  set_clip(buffer, 0, 0, buffer->w-1, buffer->h-1);
  statusbar->dirtyAll();
  statusbar->draw(buffer, sprites);
  radar->draw(buffer);
  set_clip(screen, 0, 0, screen->w-1, screen->h-1);
  blit(buffer, screen, 0, 0, 0, 0, scrx, scry);
  //now clear the whole volatile part of the buffer
  rectfill(buffer, viewport.x, viewport.y, viewport.getRight(), viewport.getBottom(), 0);
  set_clip(screen, viewport.x, viewport.y, viewport.getRight(), viewport.getBottom());
  set_clip(buffer, viewport.x, viewport.y, viewport.getRight(), viewport.getBottom());
  blitAll = false;
}

void Video::updateNoClearRects() {
  //update the no clear rects
  set_clip(screen, 0, 0, screen->w-1, screen->h-1);
  for (int c=0; c<numNoClearRects; c++) {
    blit(buffer, screen, noClearRects[c].x, noClearRects[c].y,
      noClearRects[c].x, noClearRects[c].y,
      noClearRects[c].w, noClearRects[c].h);
  }
  set_clip(screen, viewport.x, viewport.y, viewport.getRight(), viewport.getBottom());
}

void Video::updatePixels() {
  //draw the new pixels to the screen
  newPixels.SetToStart();
  int l = newPixels.Length();
  for (int c=0; c<l; c++) {
    Pixel* pixel = newPixels._GetNext();
    putpixel(screen, pixel->x, pixel->y, pixel->color);
    newPixels._Advance();
  }
}

void Video::updateOptRects() {
  optRects.SetToStart();
  int l = optRects.Length();
  //blit changed areas to the screen
  for (int c=0; c<l; c++) {
    IntRect* curr = optRects._GetNext();
    blit(buffer, screen, curr->x, curr->y, curr->x, curr->y, curr->w, curr->h);
    optRects._Advance();
  }
}

void Video::drawMouse() {
  if (mouse_needs_poll())
    poll_mouse();
  draw_rle_sprite(buffer, (RLE_SPRITE*)sprites[GRFX_DAT_MOUSEPTR_BMP].dat, mouse_x-7, mouse_y-7);
  mouseRect.x = mouse_x-7;
  mouseRect.y = mouse_y-7;
  addRect(mouseRect);
}

#define MS ((int)(MAP_SIZE))

void Video::drawEdges() {
  if (viewx <= -MS) {
    vline(buffer, -MS-viewx, -MS-viewy, MS-viewy, makecol(192, 192, 192));
    addRect(IntRect(-MS-viewx, -MS-viewy, 1, MS*2));
  } else if (viewx+viewport.w-1 >= MS) {
    vline(buffer, MS-viewx, -MS-viewy, MS-viewy, makecol(192, 192, 192));
    addRect(IntRect(MS-viewx, -MS-viewy, 1, MS*2));
  }
  if (viewy <= -MS) {
    hline(buffer, -MS-viewx, -MS-viewy, MS-viewx, makecol(192, 192, 192));
    addRect(IntRect(-MS-viewx, -MS-viewy, MS*2, 1));
  } else if (viewy+viewport.h-1 >= MS) {
    hline(buffer, -MS-viewx, MS-viewy, MS-viewx, makecol(192, 192, 192));
    addRect(IntRect(-MS-viewx, MS-viewy, MS*2, 1));
  }
}