/*
 * IDLE : Incomplete Draft of a Lisa Emulator 
 * 
 * File :  mmu.c 
 *         memory handling and i/o entry point
 * 
 * This file is distributed under the GPL, version 2 or at your
 * option any later version.  See doc/license.txt for details.
 *
 */
static char     sccsid[] = "$Id: mmu.c,v 1.11 2007/01/15 21:16:49 gilles_fetis Exp $";
#include "../config.h"

#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <memory.h>

#ifdef _CAST_
#include "../cpu/68000.h"
#include "../cpu/op68k.h"
#else
#include "../cpu/m68k.h"
// for Bus error
#include "../cpu/m68kcpu.h"
#endif

#include "mmu.h"

#ifdef DEBUG
#include <assert.h>
#endif                          /* DEBUG */

// the ROM and RAM spaces
uint8           *ramBase, *romBase, *videoRomBase;

// the MMU description
static int mmuSTART;
static int mmuCONTEXT=0;

// the Lisa video register
static int videoADDR=0;

// video irq enable
static int mmuVIDINTR=0;
static int mmuSTATUS=0xFFFF;

// we use a char[2] to avoid big/little endian issue
// the emulator stick to real mmu structure
// performance is not the goal (yet)
typedef struct mmuSegment {
        uint8 SOR[2];
        uint8 SLR[2];
        } mmuSegment;
        
#define MMU_CONTEXT_NUM 4
#define MMU_SEGMENT_BY_CONTEXT 128

mmuSegment mmu[MMU_CONTEXT_NUM][MMU_SEGMENT_BY_CONTEXT];

#define MEM_IS_ROM 0
#define MEM_IS_RAM_RW 1
#define MEM_IS_RAM_RO 2
#define MEM_IS_IO 3
#define MEM_IS_BUS_ERROR 4
#define MEM_IS_SOR 5
#define MEM_IS_SLR 6
#define MEM_IS_SN 7
#define MEM_IS_UNMAPPED 8



/*
 * memory access jump tables
 */
void            (*mem_set_b) (uint32 address, uint8 value);
void            (*mem_set_w) (uint32 address, uint16 value);
void            (*mem_set_l) (uint32 address, uint32 value);
uint8           (*mem_get_b) (uint32 address);
uint16          (*mem_get_w) (uint32 address);
uint32          (*mem_get_l) (uint32 address);


uint32 clock68k;


void refresh_mmu_box(void)
{
     char buf[256];
     int context,seg;
     mmu_box[0]='\0';
     sprintf(buf,"MMU START=%d CTX=%d\n",mmuSTART,mmuCONTEXT);
     strcat(mmu_box,buf);  
     for (int context=0;context<4;context++)
         for (int seg=0;seg<128;seg++)
         {
             sprintf(buf,"[%d][%02x] or%03x l%02x t%01x\n",
                                     context,
                                     seg,
                                     ((mmu[context][seg].SOR[0])<<8|mmu[context][seg].SOR[1])&0xFFF,
                                     (mmu[context][seg].SLR[1]),
                                     (mmu[context][seg].SLR[0]&0x0F));
             strcat(mmu_box,buf);  
         }                           
 }


// IO Space handling
// define a 64Kb adress space for IO
#define IO_SPACE_MAX 0x10000
void (* ioTable[IO_SPACE_MAX]) (int *ioval,int rwMode,int size);
void (* ioTableAdr[IO_SPACE_MAX]) (uint32 adr,int *ioval,int rwMode,int size);


// IO access entry point
void accessIo(uint32 address,int *ioval,int rwMode,int size)
{
     IDLE_INIT_FUNC("accessIo()");
     if (ioTable[address&0xFFFF]!=NULL)
        // jump to registered function (if any)
        ioTable[address&0xFFFF](ioval,rwMode,size);
     else
     {
         if (ioTableAdr[address&0xFFFF]!=NULL)
         {
                 ioTableAdr[address&0xFFFF](address,ioval,rwMode,size);
         }
         else
         {
             switch (address&0xF000) {
                    case 0x0000 :
                    case 0x1000 :
                         IDLE_WARN("Slot #1 low decode io=%06x pc=%06x",address,get_pc);
                         if (rwMode==IO_READ)
                            *ioval=0;
                         return;
                    case 0x2000 :
                    case 0x3000 :
                         IDLE_WARN("Slot #1 high decode io=%06x pc=%06x",address,get_pc);
                         if (rwMode==IO_READ)
                            *ioval=0;
                         return;
                    case 0x4000 :
                    case 0x5000 :
                         IDLE_WARN("Slot #2 low decode io=%06x pc=%06x",address,get_pc);
                         if (rwMode==IO_READ)
                            *ioval=0;
                         return;
                    case 0x6000 :
                    case 0x7000 :
                         IDLE_WARN("Slot #2 high decode io=%06x pc=%06x",address,get_pc);
                         if (rwMode==IO_READ)
                            *ioval=0;
                         return;
                    case 0x8000 :
                    case 0x9000 :
                         IDLE_WARN("Slot #3 low decode io=%06x pc=%06x",address,get_pc);
                         if (rwMode==IO_READ)
                            *ioval=0;
                         return;
                    case 0xA000 :
                    case 0xB000 :
                         IDLE_WARN("Slot #3 high decode io=%06x pc=%06x",address,get_pc);
                         if (rwMode==IO_READ)
                            *ioval=0;
                         return;
                         
                    default:
                            IDLE_WARN("unexpected io=%06x pc=%06x",address,get_pc);
                            if (rwMode==IO_READ)
                            if (size==IO_BYTE)
                               *ioval=0xff;
                            else
                                *ioval=0xffff;
             }
          }
     }
}

void initIo(void) 
{
     int i;
     for (i=0;i<IO_SPACE_MAX;i++)
         ioTable[i]=NULL;
}

// register an IO function
void registerIoFunc(uint32 address,void (*func)(int*,int,int))
{
     ioTable[address]=func;
 }

void registerIoFuncAdr(uint32 address,void (*func)(uint32,int*,int,int))
{
     ioTableAdr[address]=func;
 }

// the IO funcs attached to CPU Card
void setSTART(int* val,int mode,int size)
{
     IDLE_INIT_FUNC("setSTART()");
     mmuSTART=1; 
     IDLE_DEBUG("Start Mode pc=%06x",get_pc);                 
}
void resetSTART(int* val,int mode,int size)
{
     IDLE_INIT_FUNC("resetSTART()");
     mmuSTART=0; 
     IDLE_DEBUG("Back mapped mode get_pc=%06x",get_pc); 
}

void setSEG1(int* val,int mode,int size)
{
     mmuCONTEXT|=1;
}
void resetSEG1(int* val,int mode,int size)
{
     mmuCONTEXT&=2;
}
void setSEG2(int* val,int mode,int size)
{
     mmuCONTEXT|=2;
}
void resetSEG2(int* val,int mode,int size)
{
     mmuCONTEXT&=1;
}

void setVIDEOADDR(int* val,int mode,int size)
{
     IDLE_INIT_FUNC("setVIDEOADDR()");
     if (size==IO_WORD)
        IDLE_WARN("is WORD");
     if (mode==IO_WRITE)
     {
         videoADDR=(*val)&0x3F;
     }
     else
     {
         *val=videoADDR;
     }
}
int isVBL=0;

void setVIDINTR(int* val,int mode,int size)
{
     mmuVIDINTR=1;
    }

void resetVIDINTR(int* val,int mode,int size)
{
     mmuVIDINTR=0;
     mmuSTATUS|=4;
  }

void endVBL(void)
{
     mmuSTATUS|=4;
     isVBL=0;
}

int beginVBL(void)
{
     isVBL=1;
     if (mmuVIDINTR)
          mmuSTATUS&=0xFFFB;
     return mmuVIDINTR;
}

void getSTATUS(int* val,int mode,int size)
{
     IDLE_INIT_FUNC("getSTATUS");
     if (mode==IO_WRITE)
     {
        IDLE_WARN("is written");
        return;
     }
     else
     if (size==IO_WORD)
     {
         *val=mmuSTATUS&0xFFFF;
     }
     else
     // read high byte
     *val=(mmuSTATUS&0xFF00)>>8;
}

void getSTATUSodd(int* val,int mode,int size)
{
     if (mode==IO_WRITE)
     {
        return;
     }
     else
     if (size==IO_BYTE)
       *val=(mmuSTATUS&0xFF);
}

uint8* getVideoMemory(void)
{
       return (&ramBase[videoADDR<<15]);
       }

void initMMUIo(void)
{
     int i;
     registerIoFunc(0xE010,&setSTART);
     registerIoFunc(0xE012,&resetSTART);
     registerIoFunc(0xE00A,&setSEG1);
     registerIoFunc(0xE008,&resetSEG1);
     registerIoFunc(0xE00E,&setSEG2);
     registerIoFunc(0xE00C,&resetSEG2);
     registerIoFunc(0xE01A,&setVIDINTR);
     registerIoFunc(0xE018,&resetVIDINTR);
     for (i=0;i<0x100;i+=2) {
         registerIoFunc(0xE800+i,&setVIDEOADDR);
         registerIoFunc(0xF800+i,&getSTATUS);
         registerIoFunc(0xF801+i,&getSTATUSodd);   
     }
}

/*
* Lisa mmu entry points
*/

uint8 unmapped[4]={0xff,0xff,0xff,0xff};

int lisaMMU(uint32 address,uint8** physaddress)
{
    IDLE_INIT_FUNC("lisaMMU()");
       IDLE_DEBUG("Accessed address %06X",address);
       // if START mode and bit 14 not set   
       if ((mmuSTART) && ((address & 0x4000)==0x0000))
       {
            // mask bits 15 and 3
            switch (address & 0x8008) {
                   // bit 15 not set => ROM
                   case 0x0000:
                   case 0x0008:
                        *physaddress=&romBase[address&0x3FFF];
                        IDLE_DEBUG("IS ROM");
                        return MEM_IS_ROM;
                   // bit 15 set and bit 3 set => SOR
                   case 0x8008:
                        *physaddress=(uint8*)&mmu[mmuCONTEXT][(address&0xFE0000)>>17].SOR; 
                        IDLE_DEBUG("IS SOR");
                        return MEM_IS_SOR;
                   // bit 15 set and bit 3 not set => SLR
                   case 0x8000:
                        *physaddress=(uint8*)&mmu[mmuCONTEXT][(address&0xFE0000)>>17].SLR; 
                        IDLE_DEBUG("IS SLR");
                        return MEM_IS_SLR;
                   default:
                           return MEM_IS_BUS_ERROR;
                   } // of switch
            } // of if (mmuSTART...
       // normal MMU mapped operation
       {
          uint32 seg= (address&0xFE0000)>>17;
          uint32 pageAdr=(address&0x01FE00);
          uint32 page=pageAdr>>9;
          uint32 offset=(address&0x1FF);
          int context;
          uint32 sorg;
          uint32 slim;
          
          // if supervisor mode, stick to 0 context (this enable interrupt to go to OS context)
          if (GetS())
// don't know if the following is correct, boot is not far enough
// to check...
//          if ( (GetS())  || ((mmuSTART) && ((address & 0x4000)==0x4000)))
             context=0;
             else context=mmuCONTEXT;
          sorg=(uint32)(((((mmu[context][seg].SOR[0])<<8)|
                                   mmu[context][seg].SOR[1])
                                   &0xFFF)<<9);
          slim=(uint32)mmu[context][seg].SLR[1];
          switch (mmu[context][seg].SLR[0]&0xF) {
                 // RAM RO stack
                 case 0x4 :
                      // segment limit check (stack)
                      if (((page+slim+1)&0x100)==0x000)
                      {
                         IDLE_WARN("IS BUS ERROR RAM RO");
                         return MEM_IS_BUS_ERROR;
                      }
                      *physaddress=&ramBase[((sorg+pageAdr)&0x1FFE00)|offset];
                      IDLE_DEBUG("IS RAM STACK RO");
                      return MEM_IS_RAM_RO;
                      break;
                 // RAM RO
                 case 0x5 :
                      // segment limit check
                      if (((page+slim)&0x100)==0x100)
                      {
                         IDLE_WARN("IS LIMIT ERROR RAM RO");
                         return MEM_IS_BUS_ERROR;
                      }
                      *physaddress=&ramBase[((sorg+pageAdr)&0x1FFE00)|offset];
                      IDLE_DEBUG("IS RAM RO");
                      return MEM_IS_RAM_RO;
                      break;
                 // RAM RW stack
                 case 0x6 :
                      // segment limit check (stack)
                      if (((page+slim+1)&0x100)==0x000)
                      {
                         IDLE_WARN("IS LIMIT ERROR RAM STK adr=%06x",address);
                         IDLE_WARN("page=%x slim=%x res=",page,slim,page+slim+1);
                         return MEM_IS_BUS_ERROR;
                         }
                      *physaddress=&ramBase[((sorg+pageAdr)&0x1FFE00)|offset];
                         IDLE_DEBUG("IS RAM STACK RW");
                         return MEM_IS_RAM_RW;
                      break;
                 // RAM RW
                 case 0x7 :
                      // segment limit check
                      if (((page+slim)&0x100)==0x100)
                      {
                         IDLE_WARN("IS LIMIT ERROR RAM");
                         return MEM_IS_BUS_ERROR;
                      }
                      *physaddress=&ramBase[((sorg+pageAdr)&0x1FFE00)|offset];
                     IDLE_DEBUG("IS RAM RW");
                         return MEM_IS_RAM_RW;
                      break;
                 // IO space
                 case 0x8 :
                 case 0x9 : 
  
                     IDLE_DEBUG("IS IO");
                      return MEM_IS_IO;
                      break;
                 case 0xC :
                     IDLE_WARN("IS UNMAPPED %06x seg=%x get_pc=%06x",address,seg,get_pc);
                     return MEM_IS_UNMAPPED;
//                     unmapped[0]=0xFF;
//                     unmapped[1]=0xFF;
//                     unmapped[2]=0xFF;
//                     unmapped[3]=0xFF;
//                     *physaddress=&unmapped[0];
//                     return MEM_IS_RAM_RW;
                 // ROM
                 case 0xF :
                      // serial num 
                      if (pageAdr==0x8000)
                      {
                         // simplified version of Raphael Nabet's unfinished MESS lisa driver
                         static uint32 addr=0x00;
                         static int count=0;
                         if (addr==56) addr=0;
                         if (count==(56*2)) count=0;
                         if (count>=56) 
                               *physaddress=&videoRomBase[addr|0x80];
                         else
                               *physaddress=&videoRomBase[addr];
                         addr=(addr+1)&0x7F;
                         count++;
                         return MEM_IS_SN;
                         }
                      else
                      if (pageAdr>=0x4000)
                      {
                        *physaddress=&romBase[(pageAdr+offset)&0x3FFF];
                        IDLE_WARN("mapping at %06x",pageAdr);                                         
                      }     
                      else
                        *physaddress=&romBase[(pageAdr+offset)&0x3FFF];
                     IDLE_DEBUG("IS ROM2");
                        return MEM_IS_ROM;
                 default:
                     IDLE_WARN("IS ERROR %d",mmu[context][seg].SLR[0]&0xF);
                       return MEM_IS_BUS_ERROR;
          } // of switch
       } // of normal MMU mode
       return MEM_IS_BUS_ERROR;
}


uint8 LisaGetB(uint32 address)
{
      int ret;
      uint8* physAddress;
      int ioVal;
      IDLE_INIT_FUNC("LisaGetB()");
      clock68k++;
      ret=lisaMMU(address,&physAddress);
      switch (ret) {
             case MEM_IS_ROM :
             case MEM_IS_RAM_RW :
             case MEM_IS_RAM_RO :
                    return physAddress[0];
             case MEM_IS_SOR :
             case MEM_IS_SLR :
                    if ((address&0x1)==0)
                    {
                    IDLE_DEBUG("MMU %s read %02x address=%06d" ,((ret==MEM_IS_SOR)?"SOR":"SLR"),physAddress[0],address);
                    return physAddress[0];
                    }
                    else
                    {
                    IDLE_DEBUG("MMU %s read %02x address=%06d" ,((ret==MEM_IS_SOR)?"SOR":"SLR"),physAddress[1],address);
                    return physAddress[1];
                    }
             case MEM_IS_IO :
                    accessIo(address,&ioVal,IO_READ,IO_BYTE);
                    return (uint8)ioVal;
             case MEM_IS_SN:
                  // fixme : the read here is strange...
                  IDLE_WARN("Should not read byte address=%06x",address);
                  return (0);
             case MEM_IS_BUS_ERROR :
                     ExceptionGroup0(BUSERR, address, 1);
                     break;
             default :
                     ExceptionGroup0(BUSERR, address, 1);
             }
}
uint16 LisaGetW(uint32 address)
{
      int ret;
      uint8* physAddress;
      int ioVal;
      IDLE_INIT_FUNC("LisaGetW()");
      clock68k++;
      if ((address&0x1)==0x1) 
      {
// fixme
         return 0;
      }
      ret=lisaMMU(address,&physAddress);
      switch (ret) {
             case MEM_IS_SOR :
             case MEM_IS_SLR :
                    IDLE_DEBUG("MMU %s read %02x address=%06x" ,((ret==MEM_IS_SOR)?"SOR":"SLR"),(physAddress[0]<<8)|physAddress[1],address);
             case MEM_IS_ROM :
             case MEM_IS_RAM_RW :
             case MEM_IS_RAM_RO :
                    return (physAddress[0]<<8)|physAddress[1];
             case MEM_IS_SN:
                  return ((physAddress[0]&0x80)<<8);
             case MEM_IS_IO :
                    accessIo(address,&ioVal,IO_READ,IO_WORD);
                    return (uint16)ioVal;
             case MEM_IS_BUS_ERROR :
                     ExceptionGroup0(BUSERR, address, 1);
                     break;
             default :
                     ExceptionGroup0(BUSERR, address, 1);
             }
}
uint16 LisaGetWDebug(uint32 address)
{
      int ret;
      uint8* physAddress;
      int ioVal;
      if ((address&0x1)==0x1) 
      {
// fixme
         return 0;
      }
      ret=lisaMMU(address,&physAddress);
      switch (ret) {
             case MEM_IS_ROM :
             case MEM_IS_RAM_RW :
             case MEM_IS_RAM_RO :
             case MEM_IS_SOR :
             case MEM_IS_SLR :
                    return (physAddress[0]<<8)|physAddress[1];
             case MEM_IS_IO :
                    return (uint16)0xFFFF;
             case MEM_IS_BUS_ERROR :
                    return (uint16)0xFFFF;
             default :
                    return (uint16)0xFFFF;
             }
}
void LisaSetB(uint32 address,uint8 value)
{
      int ret;
      uint8* physAddress;
      int ioVal;
      IDLE_INIT_FUNC("LisaSetB()");
      clock68k++;
      ret=lisaMMU(address,&physAddress);
      switch (ret) {
             case MEM_IS_ROM :
             case MEM_IS_RAM_RO :
                     ExceptionGroup0(BUSERR, address, 1);
                     break;
             case MEM_IS_RAM_RW :
                    physAddress[0]=value;
                    return;
             case MEM_IS_SOR :
             case MEM_IS_SLR :
                    IDLE_DEBUG("MMU BYTE %s write %02x address=%06x" ,((ret==MEM_IS_SOR)?"SOR":"SLR"),value,address);
                    if ((address&0x01)==0)
                    {
                       physAddress[0]=value;
                       return;
                    }
                    else
                    {
                       physAddress[1]=value;
                       return;
                    }
             case MEM_IS_IO :
                    ioVal=value;
                    accessIo(address,&ioVal,IO_WRITE,IO_BYTE);
                    return;
             case MEM_IS_BUS_ERROR :
                     ExceptionGroup0(BUSERR, address, 0);
                     break;
             default :
                     ExceptionGroup0(BUSERR, address, 0);
             }
}
void LisaSetW(uint32 address,uint16 value)
{
      int ret;
      uint8* physAddress;
      int ioVal;
      IDLE_INIT_FUNC("LisaSetW()");
      clock68k++;

      if ((address&0x1)==0x1) 
      {
// fixme
         return; 
      }
      ret=lisaMMU(address,&physAddress);
      switch (ret) {
             case MEM_IS_ROM :
             case MEM_IS_RAM_RO :
                     ExceptionGroup0(BUSERR, address, 1);
                     break;
             case MEM_IS_SOR :
             case MEM_IS_SLR :
                    IDLE_DEBUG("MMU %s write %04x address=%06x" ,((ret==MEM_IS_SOR)?"SOR":"SLR"),(physAddress[0]<<8)|physAddress[1],address);
                    physAddress[0]=((value&0xFF00)>>8);
                    physAddress[1]=value&0xFF;
                    break;
             case MEM_IS_RAM_RW :
                    physAddress[0]=((value&0xFF00)>>8);
                    physAddress[1]=value&0xFF;
                    break;
             case MEM_IS_IO :
                    ioVal=value;
                    accessIo(address,&ioVal,IO_WRITE,IO_WORD);
                    return;
             case MEM_IS_BUS_ERROR :
                     ExceptionGroup0(BUSERR, address, 0);
                     break;
             default :
                     ExceptionGroup0(BUSERR, address, 0);
             }
}


void    LisaSetL (uint32 address, uint32 value)
{
    ON_WRITE(address, value);
    LisaSetW(address,(value&0xFFFF0000)>>16);
    LisaSetW(address+2,value&0xFFFF);
}

uint32  LisaGetL(uint32 address)
{
    uint32 ret;
    ret=(LisaGetW(address)<<16);
    ret|=LisaGetW(address+2);
    return ret;
}

/*
 * MemTableSet - Add memory bank handler to memory regions
 *
 * not really usefull for IDLE but to keep castaway source 
 * compatibility for now.
 */
void    MemTableSet(void (*setbyte)(uint32, uint8),
                void (*setword)(uint32, uint16),
                void (*setlong)(uint32, uint32 ),
                uint8  (*getbyte)(uint32),
                uint16 (*getword)(uint32),
                uint32  (*getlong)(uint32) )
{
        mem_set_b = setbyte;
        mem_set_w = setword;
        mem_set_l = setlong;
        mem_get_b = getbyte;
        mem_get_w = getword;
        mem_get_l = getlong;
}



void            MemInit(void)
{
    FILE *rom;
    int ret;
    uint8 romtmp[512*32];
    int i;
    /* Set default memory access handlers */
    MemTableSet(
        LisaSetB, LisaSetW, LisaSetL,
        LisaGetB, LisaGetW, LisaGetL);

        //  
        ramBase=(uint8*)malloc(512*4096); 
        romBase=(uint8*)malloc(512*32);
        videoRomBase=(uint8*)malloc(256);
        
        rom=fopen("bios/booth.hi","rb");
        if (rom==NULL) exit(1);
        ret=fread(&romtmp[0],512*16,1,rom);
        fclose(rom);
        
        rom=fopen("bios/booth.lo","rb");
        if (rom==NULL) exit(1);
        ret=fread(&romtmp[512*16],512*16,1,rom);
        fclose(rom);
        
        rom=fopen("bios/vidstate.rom","rb");
        if (rom==NULL) exit(1);
        ret=fread(&videoRomBase[0],256,1,rom);
        fclose(rom);

        // reorder rom
        for (i=0;i<512*16;i++)
        {
         romBase[i*2]=romtmp[i];
         romBase[i*2+1]=romtmp[i+512*16];
        }
        mmuSTART=1;
        initIo();
        initMMUIo();

        // voids checksum        
        romBase[0x01A8]=0x4E;
        romBase[0x01A9]=0x71;
        romBase[0x01AA]=0x4E;
        romBase[0x01AB]=0x71;

        // fixme ?
        // void parity test
        // no parity check for now 
        for (i=0x0DA4;i<0x0DC0;i+=2)
        {
                romBase[i]=0x4E;
                romBase[i+1]=0x71;
        }

        // fixme skip basic mem test
        // (my libretto ct100 is too slow with tests...)
        romBase[0x0E6A]=0x4E;
        romBase[0x0E6B]=0x75;

        
        // fixme jump after rs232
        romBase[0x1000]=0x60;
        romBase[0x1001]=0x00;
        romBase[0x1002]=0x01;
        romBase[0x1003]=0x0A;
        
        
        // fixme jump after disk (11C0)
        // romBase[0x110C]=0x60;
        // romBase[0x110D]=0x00;
        // romBase[0x110E]=0x00;
        // romBase[0x110F]=0xB2;
        // 0x1D94 -> jump (14*NOP)
/*        for (i=0x1D94;i<0x1DB0;i+=2)
        {
                romBase[i]=0x4E;
                romBase[i+1]=0x71;
        } */
        
}

void mmuReset(void)
{
     mmuSTART=1;
}

int             QueryIRQ (int intlev)
{
                /*
      switch (intlev) {
      case 1 : return AUTOINT1;
      case 2 : return AUTOINT2;
      case 3 : return AUTOINT3;
      case 4 : return AUTOINT4;
      case 5 : return AUTOINT5;
      case 6 : return AUTOINT6;
      case 7 : return AUTOINT7;
      default : return BUSERR;
      }
      */
}
unsigned int  m68k_read_memory_8(unsigned int address) {
         return LisaGetB(address);
}
unsigned int  m68k_read_memory_16(unsigned int address){
         return LisaGetW(address);
}
unsigned int  m68k_read_memory_32(unsigned int address){
         return LisaGetL(address);
}

void m68k_write_memory_8(unsigned int address, unsigned int value){
         LisaSetB(address,(uint8) value);
}
void m68k_write_memory_16(unsigned int address, unsigned int value){
         LisaSetW(address,(uint16) value);
}
void m68k_write_memory_32(unsigned int address, unsigned int value){
         LisaSetL(address,(uint32) value);
}
