/*
 * 05/10/2006 : debugger 
 * 07/10/2006 : first code executed => ROM checksum error
 * 08/10/2006 : _ checksum patched => MMU test pass but no RAM found
 *              _ added breakpoint
 * 09/10/2006 : _ added MMU debug
 *              _ corrected mmu & Long Write
 *              _ added lock to 0 context when in supervisor mode (for mmu use)
 *              _ now crash on first RTS. 0xFE06E0 (stack translation error?)
 *              _ => corrected HWReset (bad ssp)
 *              _ fail for VIA test 
 *              _ FE08A2 
 *              _ ROM was not broken, MemInit() was...
 *              _ corrected screen
 *              _ Now go graphical mode => IO Board error code 50
 *                (normal, VIA are not emulated yet).
 * 11/10/2006 : _ VIA partially emulated 
 *              => VIA test ok but crashes in castaway code (move to SR)
                => try SR_IMPLEMENTATION 2 instead of 0
                not better address of crash is FE09B2
 * 12/10/2006   _ workaround the error (the long_jump vector was not set in CPUStep)
                => will need to verify if long_jump is really needed.
                _ now io board error 52 (io board COPS error) => OK
 * 13/10/2006   _ added less primitive VIA support
                _ COPS test now ok
                _ error 42 (video error) due to lack of video interrupt
                _ FE0B96
                _ video interrupt now OK
                _ serial number getting error => should support videorom
   28/10/2006   _ fe1c96 : boot sector seems ok : error 47
 */
#include <allegro.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef _CAST_
#include "cpu/68000.h"
#else
#include "cpu/m68k.h"
#endif

#include "mmu/mmu.h"
#include "io/via.h"
#include "disk/disk_image.h"
#include "disk/disk_ctrl.h"

#include "debug/mem_explorer.h"

extern int    disass(char *buf, uint16 *inst_stream);
extern char *GetStatus(void);

int ask_bkpt=0;

void init();
void deinit();

static  int insert_dc42_image(void);
static  int insert_dart_image(void);

DIALOG fast_popup[] =
{
   /* (dialog proc)     (x)   (y)   (w)   (h)  (fg) (bg) (key) (flags)     (d1) (d2)    (dp)                   (dp2) (dp3) */
   { d_shadow_box_proc,  20,  10,  280,  180,    0,   0,    0,    D_DISABLED, 0,   0,    (void*)NULL },
   { d_button_proc,     30,   30,  260,   16,    0,   0,    0,    D_EXIT,     0,   0,    (void*)"Disk image DC 42" },
   { d_button_proc,     30,   50,  260,   16,    0,   0,  'l',    D_EXIT,     0,   0,    (void*)"Disk image DART" },
   { d_button_proc,     30,   74,  260,   16,    0,   0,  'r',    D_EXIT,     0,   0,    (void*)"Call Debugger" },
   { d_button_proc,    164,  170,  126,   16,    0,   0,    0,    D_EXIT,     0,   0,    (void*)"Return" },
   { d_yield_proc,       0,    0,    0,    0,    0,   0,    0,      0,        0,   0,    NULL,                   NULL, NULL  },
   { NULL,               0,    0,    0,    0,    0,   0,    0,      0,        0,   0,    NULL,                   NULL, NULL  }
};

void fastPopup(void) {
    int ret;

   /* set up colors */
   gui_fg_color = makecol(0, 0, 0);
   gui_mg_color = makecol(128, 128, 128);
   gui_bg_color = makecol(200, 200, 200);
   set_dialog_color (fast_popup, gui_fg_color, gui_bg_color);


    clear_keybuf();
    ret=popup_dialog(fast_popup,-1);

    if (ret==1) {
                insert_dc42_image();
    }
    if (ret==2) {
                insert_dart_image();
    }
    if (ret==3) {
                ask_bkpt=1;
    }
    
}

int quit(void)
{
   if (alert("Really Quit?", NULL, NULL, "&Yes", "&No", 'y', 'n') == 1)
      return D_CLOSE;
   else
      return D_REDRAW;
}

int about(void)
{
   alert("IDLE (incomplete Draft of a Lisa Emulator)\nversion alpha 0.3 - 08/11/2006"  
   , NULL, NULL, "&OK", "&No", 'y', 'n');
      return D_REDRAW;
}



char log_box_init[0x10000];

// switch to this area while disassembling

char log_box_init2[60*20*4+1];
char *log_box=log_box_init;

char mmu_box_init[0x10000];
char *mmu_box=mmu_box_init;

char disass_box[60*20*4+1];


void disass_debugger(void) {
#ifdef _CAST_                  
                  uint32 l_pc=pc;
#else
                  uint32 l_pc=m68k_get_reg(NULL,M68K_REG_PC);
#endif
                  uint16 l_inststream[256];
                  int i,j;
                  int l;
                  int inst_size;
                  char l_inst[256];
                  char addr[256];
                  // clears the text
                  disass_box[0]='\0';
                  no_trace();
                  for (l=0;l<20;l++)
                  {
                      log_box[0]='\0';
                      // fills inst stream
                      for (i=0;i<32;i++)
                          l_inststream[i]=LisaGetWDebug(l_pc+i*2);
                      // 
                      sprintf(addr,"%06X ",l_pc);
                      strcat(disass_box,addr);
                      // disass one instruction
                      inst_size=disass(l_inst,l_inststream);
                      for (j=0;j<6;j++)
                      {
                          if (j<inst_size)
                             sprintf(addr,"%04X",l_inststream[j]);
                          else
                              strcpy(addr,"    ");
                          strcat (disass_box,addr);
                      }
                      // adds to box
                      strcat(disass_box,l_inst);
                      // plus a carriage return
                      strcat(disass_box,"\n");
                      l_pc+=(2*inst_size);
                  }
                  restore_trace();
     }


char registers_box[30*20*4+1];

void regs_debugger(void)
{
     char buf[256];
     registers_box[0]='\0';
#ifdef _CAST_
     sprintf(buf,"PC=x%08X SR=%s\n",pc,GetStatus());
     strcat(registers_box,buf);
     sprintf(buf,"D0=x%08X A0=x%08X\n",reg[0],reg[8]);
     strcat(registers_box,buf);
     sprintf(buf,"D1=x%08X A1=x%08X\n",reg[1],reg[9]);
     strcat(registers_box,buf);
     sprintf(buf,"D2=x%08X A2=x%08X\n",reg[2],reg[10]);
     strcat(registers_box,buf);
     sprintf(buf,"D3=x%08X A3=x%08X\n",reg[3],reg[11]);
     strcat(registers_box,buf);
     sprintf(buf,"D4=x%08X A4=x%08X\n",reg[4],reg[12]);
     strcat(registers_box,buf);
     sprintf(buf,"D5=x%08X A5=x%08X\n",reg[5],reg[13]);
     strcat(registers_box,buf);
     sprintf(buf,"D6=x%08X A6=x%08X\n",reg[6],reg[14]);
     strcat(registers_box,buf);
     sprintf(buf,"D7=x%08X A7=x%08X\n",reg[7],reg[15]);
     strcat(registers_box,buf);
#else
     sprintf(buf,"PC=x%08X SR=%s\n",get_pc,GetStatus());
     strcat(registers_box,buf);
     sprintf(buf,"D0=x%08X A0=x%08X\n",m68k_get_reg(NULL,M68K_REG_D0),m68k_get_reg(NULL,M68K_REG_A0));
     strcat(registers_box,buf);
     sprintf(buf,"D1=x%08X A1=x%08X\n",m68k_get_reg(NULL,M68K_REG_D1),m68k_get_reg(NULL,M68K_REG_A1));
     strcat(registers_box,buf);
     sprintf(buf,"D2=x%08X A2=x%08X\n",m68k_get_reg(NULL,M68K_REG_D2),m68k_get_reg(NULL,M68K_REG_A2));
     strcat(registers_box,buf);
     sprintf(buf,"D3=x%08X A3=x%08X\n",m68k_get_reg(NULL,M68K_REG_D3),m68k_get_reg(NULL,M68K_REG_A3));
     strcat(registers_box,buf);
     sprintf(buf,"D4=x%08X A4=x%08X\n",m68k_get_reg(NULL,M68K_REG_D4),m68k_get_reg(NULL,M68K_REG_A4));
     strcat(registers_box,buf);
     sprintf(buf,"D5=x%08X A5=x%08X\n",m68k_get_reg(NULL,M68K_REG_D5),m68k_get_reg(NULL,M68K_REG_A5));
     strcat(registers_box,buf);
     sprintf(buf,"D6=x%08X A6=x%08X\n",m68k_get_reg(NULL,M68K_REG_D6),m68k_get_reg(NULL,M68K_REG_A6));
     strcat(registers_box,buf);
     sprintf(buf,"D7=x%08X A7=x%08X\n",m68k_get_reg(NULL,M68K_REG_D7),m68k_get_reg(NULL,M68K_REG_A7));
     strcat(registers_box,buf);
#endif
 }

char breakpoint_str[6*4+1];

// one cpu instruction
int step(void)
{
    log_box[0]='\0';
#ifdef _CAST_
    CPUStep();
#else
     m68k_execute(1);
#endif
    disass_debugger();
    regs_debugger();
    refresh_mmu_box();
    get_log_str(log_box,sizeof(log_box_init));
    return D_REDRAW;
}

// loop CPU instruction until PC is greater (use this for loop)
int stepover(void)
{
#ifdef _CAST_                  
                  uint32 savedPC=pc;
#else
                  uint32 savedPC=m68k_get_reg(NULL,M68K_REG_PC);
#endif

    do {
       log_box[0]='\0';
#ifdef _CAST_
    CPUStep();
#else
     m68k_execute(1);
#endif
       
#ifdef _CAST_                  
    } while (((pc<=savedPC) || (pc>savedPC+8)) && !key[KEY_SPACE]);
 #else
    } while (((m68k_get_reg(NULL,M68K_REG_PC)<=savedPC) || (m68k_get_reg(NULL,M68K_REG_PC)>savedPC+8)) && !key[KEY_SPACE]);
#endif
   while (key[KEY_SPACE]);
    disass_debugger();
    regs_debugger();
    refresh_mmu_box();
    get_log_str(log_box,sizeof(log_box_init));
    return D_REDRAW;
}

volatile int last_scancode;
      
void keypress_handler(int scancode)
{
     last_scancode=scancode;
} END_OF_FUNCTION(keypress_handler)


void lisa_keyb(void) {
     keyboard_lowlevel_callback = keypress_handler;
}

void allegro_keyb(void) {
      keyboard_lowlevel_callback = NULL;
      clear_keybuf();      
}

volatile int loop_exit=0;

int pc2lisa_scancode(int code)
{
    int mode=(code&0x80)^0x80;
    int lkey;
    switch (code&0x7F) {
           case KEY_LCONTROL : lkey=0x7F ;break;
           case KEY_ALT : lkey=0x7F ;break;
           case KEY_RCONTROL : lkey=0x68 ;break;
           case KEY_ENTER : lkey=0x48 ;break;
           case KEY_BACKSPACE : lkey=0x45 ;break;


           case KEY_1 : lkey=0x74 ;break;
           case KEY_2 : lkey=0x71 ;break;
           case KEY_3 : lkey=0x72 ;break;
           case KEY_4 : lkey=0x73 ;break;
           case KEY_5 : lkey=0x64 ;break;
           case KEY_6 : lkey=0x61 ;break;
           case KEY_7 : lkey=0x62 ;break;
           case KEY_8 : lkey=0x63 ;break;
           case KEY_9 : lkey=0x50 ;break;
           case KEY_0 : lkey=0x51 ;break;
           case KEY_MINUS : lkey=0x40 ;break;
           case KEY_EQUALS : lkey=0x41 ;break;
           
           case KEY_Q : lkey=0x75 ;break;
           case KEY_W : lkey=0x77 ;break;
           case KEY_E : lkey=0x60 ;break;
           case KEY_R : lkey=0x65 ;break;
           case KEY_T : lkey=0x66 ;break;
           case KEY_Y : lkey=0x67 ;break;
           case KEY_U : lkey=0x52 ;break;
           case KEY_I : lkey=0x53 ;break;
           case KEY_O : lkey=0x49 ;break;
           case KEY_P : lkey=0x44 ;break;
           
           case KEY_A : lkey=0x70 ;break;
           case KEY_S : lkey=0x76 ;break;
           case KEY_D : lkey=0x7b ;break;
           case KEY_F : lkey=0x69 ;break;
           case KEY_G : lkey=0x6a ;break;
           case KEY_H : lkey=0x6b ;break;
           case KEY_J : lkey=0x54 ;break;
           case KEY_K : lkey=0x55 ;break;
           case KEY_L : lkey=0x59 ;break;

           case KEY_Z : lkey=0x79 ;break;
           case KEY_X : lkey=0x7a ;break;
           case KEY_C : lkey=0x6d ;break;
           case KEY_V : lkey=0x6c ;break;
           case KEY_B : lkey=0x6e ;break;
           case KEY_N : lkey=0x6f ;break;
           case KEY_M : lkey=0x58 ;break;

           case KEY_F9 : loop_exit=1;
           // under allegro this will cause program end
           case KEY_ESC : loop_exit=1;
           
           default : lkey=0x5c; // SPACE by default
    }
    return (lkey|mode);
}

int getKeyEvent() {
    static int prev_code=0;
    int ret;
    int code=last_scancode; // pseudo atomic read
    if (prev_code!=code)
    {
       ret=via_keyb_event_key(pc2lisa_scancode(code));
       prev_code=code;
       return ret;
    }
    return 0;
}

int getMouseButton() {
    static int prev=0;
    int button=mouse_b&1;
    int ret;
    if (prev!=button)
    {
       if (button)
              ret=via_keyb_event_key(0x86);
       else
              ret=via_keyb_event_key(0x06);
       prev=button;
       return ret;
    }
    return 0;
}

int redraw_lisa(void)
{
    static int first=1;
    static int live=0;
    static BITMAP *lisa_screen=NULL;
    int y,x,xx;
    uint8* video=getVideoMemory();
    int mask;
    int black=makecol(0, 0, 0);
    int white=makecol(255, 255, 255);
    int red=makecol(255, 0, 0);
    if (first)
    {
         lisa_screen=create_bitmap(720,364);
         first=0;
    }
              
    acquire_bitmap(lisa_screen);
    for (y=0;y<364;y++)
        for (x=0;x<90;x++)
        {
            mask=0x80;
            for (xx=0;xx<8;xx++)
            {
                if ((video[x+y*90]&mask)==mask)
                   putpixel(lisa_screen,x*8+xx,y,black);
                else
                   putpixel(lisa_screen,x*8+xx,y,white);
                mask=mask>>1;
            }
        }
        // add a "galactica" like live indicator
        putpixel(lisa_screen,live,363,red);
        live=(live+1)%720;
//        for (y=0;y<36;y++)
//        blit(lisa_screen,screen,0,y*10,0,y*10,720,10);
        blit(lisa_screen,screen,0,0,0,0,720,364);
        release_bitmap(lisa_screen);
}

int redraw_macxl(void)
{
    static int first=1;
    static int live=0;
    static BITMAP *lisa_screen=NULL;
    int y,x,xx;
    uint8* video=getVideoMemory();
    int mask;
    int black=makecol(0, 0, 0);
    int white=makecol(255, 255, 255);
    int red=makecol(255, 0, 0);
    if (first)
    {
         lisa_screen=create_bitmap(608,431);
         first=0;
    }
              
//    acquire_bitmap(lisa_screen);
    for (y=0;y<431;y++)
        for (x=0;x<76;x++)
        {
            mask=0x80;
            for (xx=0;xx<8;xx++)
            {
                if ((video[x+y*76]&mask)==mask)
                   putpixel(lisa_screen,x*8+xx,y,black);
                else
                   putpixel(lisa_screen,x*8+xx,y,white);
                mask=mask>>1;
            }
        }
        // add a "galactica" like live indicator
        putpixel(lisa_screen,live,431,red);
        live=(live+1)%608;
        blit(lisa_screen,screen,0,0,0,0,608,431);
}


void lisa_loop(uint32 targetPC)
{
     static int vblnum=0;
     int skip=3;
     int line;
     loop_exit=0;
     scare_mouse();
     clock68k=0;
     do {
              via_hdd_tick();
              floppy_tick();
              getKeyEvent();
              for (line=0;line<360;line++)
              {
               set_trace_time((vblnum<<16)|line);
               via_keyb_tick();
               // execute more or less 55 mem cycles
               while (clock68k<55) 
               {

//                  log_box[0]='\0';

#ifdef _CAST_
    CPUStep();
#else
     if ((m68k_execute(1)==1)||(ask_bkpt)||(get_pc==targetPC)) goto end;
#endif
#ifdef _CAST_
                  if ((pc==targetPC) ||(ask_bkpt)) goto end;
#endif
              }
              clock68k-=55;
              }
             if (beginVBL()) m68k_set_irq(M68K_IRQ_1);
             {
             int l_mouse_x,l_mouse_y;
             // many thanx to allegro coders for this relative mouse routine ;-)
             get_mouse_mickeys(&l_mouse_x,&l_mouse_y);
             if ((l_mouse_x!=0) || (l_mouse_y!=0)) {
                if (l_mouse_x<-128) l_mouse_x=-128;
                if (l_mouse_x>127) l_mouse_x=127;
                if (l_mouse_y<-128) l_mouse_y=-128;
                if (l_mouse_y>127) l_mouse_y=127;
                
                via_keyb_update_mouse_pos((int8)l_mouse_x,(int8)l_mouse_y);
                }
                else
                    getMouseButton();
                // check for popup
                if ((mouse_b & 2)==2) {
                   unscare_mouse();
                   fastPopup();
                   scare_mouse();
                }
             }
              for (line=360;line<374;line++)
              {
               set_trace_time((vblnum<<16)|line);
               via_keyb_tick();
               // execute more or less 55 mem cycles
               while (clock68k<55) 
               {

//                  log_box[0]='\0';

#ifdef _CAST_
    CPUStep();
#else
     if ((m68k_execute(1)==1)||(ask_bkpt)||(get_pc==targetPC)) goto end;
#endif
#ifdef _CAST_
                  if ((pc==targetPC)||(ask_bkpt)) goto end;
#endif
              }
              clock68k-=55;
              }
             endVBL();
          
             // to speed up on my libretto...    
              skip=(skip+1)%4;
              if (skip==0)
                  redraw_lisa();
              vblnum++;
          } while (!loop_exit);
      // wait for space release
      while (key[KEY_F9]);
end:
      unscare_mouse();
      ask_bkpt=0;
}

int tobreak(void)
{
    uint32 targetPC;
    char str_alert[256];
    sscanf(breakpoint_str,"%x",&targetPC);
    sprintf(str_alert,"Go to %06x ?",targetPC);
   if (alert(str_alert, NULL, NULL, "&Yes", "&No", 'y', 'n') == 1)
   {
      int end=0;
      lisa_keyb();
      lisa_loop(targetPC);
      allegro_keyb();
      disass_debugger();
      regs_debugger();
      refresh_mmu_box();
    }
    get_log_str(log_box,sizeof(log_box_init));
    return D_REDRAW;
}

int tobreaknoconfirm(void)
{
    uint32 targetPC;
    char str_alert[256];
    sscanf(breakpoint_str,"%x",&targetPC);
   {
      int end=0;
      lisa_keyb();
      lisa_loop(targetPC);
      allegro_keyb();
      disass_debugger();
      regs_debugger();
      refresh_mmu_box();
    }
    get_log_str(log_box,sizeof(log_box_init));
    return D_REDRAW;
}




int preview(void)
{
    redraw_lisa();
    while (!key[KEY_SPACE]);
    return D_REDRAW;
}

int lisa_reset(void)
{
    mmuReset();
#ifdef _CAST_
    HWReset();
#else
    m68k_init();
    m68k_pulse_reset();
#endif
      disass_debugger();
      regs_debugger();
    get_log_str(log_box,sizeof(log_box_init));
    return D_REDRAW;
}




int insert_dc42_image(void)
{
    char path[6*1024];
    path[0]='\0';
    file_select_ex("Insert a Disc copy 4.2 image",
                           path,
                           NULL,
                           1024,
                           480,
                           384);
    dc42_insert(path,0);
        return D_REDRAW;
}

int insert_dart_image(void)
{
    char path[6*1024];
    path[0]='\0';
    file_select_ex("Insert a Disc copy 4.2 image",
                           path,
                           NULL,
                           1024,
                           480,
                           384);
    dart_insert(path,0);
        return D_REDRAW;
}

int call_memory_explorer_mmu(void)
{
    memory_explorer_mmu();
    return D_REDRAW;
}

int call_memory_explorer_raw(void)
{
    return D_REDRAW;
}

/* the first menu in the menubar */
MENU commands[] =
{
   
   { "Step \tF7", step,  NULL,      0,  NULL  },
   { "Step over\tF8", stepover,  NULL,      0,  NULL  },
   { "Go to breakpoint", tobreak,  NULL,      0,  NULL  },
   { "screen preview", preview,  NULL,      0,  NULL  },
   { "&Quit debugger\tESC",        quit,  NULL,      0,  NULL  },
   { NULL,                   NULL,  NULL,      0,  NULL  }
};

/* the help menu */
MENU helpmenu[] =
{
   
   { "&About \tF1",      about,  NULL,      0,  NULL  },
   { NULL,               NULL,  NULL,      0,  NULL  }
};

/* the disk menu */
MENU diskmenu[] =
{
   
   { "Insert DC42 image",      insert_dc42_image,  NULL,      0,  NULL  },
   { "Insert DART image (not complete)",      insert_dart_image,  NULL,      0,  NULL  },
   { NULL,               NULL,  NULL,      0,  NULL  }
};

/* the memory menu */
MENU memory_menu[] =
{
   
   { "MMU memory explorer",      call_memory_explorer_mmu,  NULL,      0,  NULL  },
   { "Raw memory explorer",      call_memory_explorer_raw,  NULL,      0,  NULL  },
   { NULL,               NULL,  NULL,      0,  NULL  }
};

/* the main menu-bar */
MENU the_menu[] =
{
   
   { "&Commands",  NULL,   commands,          0,      NULL  },
   { "Disk images",  NULL,   diskmenu,          0,      NULL  },
   { "Memory",  NULL,   memory_menu,          0,      NULL  },
   { "&Help",   NULL,   helpmenu,       0,      NULL  },
   { NULL,      NULL,   NULL,           0,      NULL  }
};

DIALOG the_dialog[] =
{
   /* (dialog proc)     (x)   (y)   (w)   (h) (fg)(bg) (key) (flags)     (d1) (d2)    (dp)                   (dp2) (dp3) */
   { d_clear_proc,        0,   0,    0,    0,   0,  0,    0,      0,       0,   0,    NULL,                   NULL, NULL  },
   { d_menu_proc,         0,   0,    0,    0,   0,  0,    0,      0,       0,   0,    the_menu,               NULL, NULL  },
   { d_text_proc,         0,  20,    0,    0,   0,  0,    0,      0,       0,   0,    (void *)"Lisa emulator integrated debugger",          NULL, NULL  },
   // a 60x20 char box for disass
   { d_textbox_proc,    0, 30,  480,   160,   0,  0,    0,      0,       0,   0,    (void *)disass_box,       NULL, NULL  },
   // a 60x20 char box for loging events
   { d_textbox_proc,    0, 200,  480,   152,   0,  0,    0,      0,       0,   0,    (void *)log_box,       NULL, NULL  },
   // a 30x20 char box for registers
   { d_textbox_proc,    488, 30,  232,   160,   0,  0,    0,      0,       0,   0,    (void *)registers_box,       NULL, NULL  },
   // the breakpoint
   { d_edit_proc,    488, 206,  232,   8,   0,  0,    0,      0,       6,   0,    (void *)breakpoint_str,       NULL, NULL  },
   // the mmu Entry
   { d_textbox_proc,    488, 222,  232,   130,   0,  0,    0,      0,       0,   0,    (void *)mmu_box,       NULL, NULL  },
   { d_keyboard_proc,     0,   0,    0,    0,   0,  0,    0,      0,  KEY_F7,   0,    (void *)step,          NULL, NULL  },
   { d_keyboard_proc,     0,   0,    0,    0,   0,  0,    0,      0,  KEY_F8,   0,    (void *)stepover,          NULL, NULL  },
   { d_keyboard_proc,     0,   0,    0,    0,   0,  0,    0,      0,  KEY_B,   0,    (void *)tobreaknoconfirm,          NULL, NULL  },
   { d_keyboard_proc,     0,   0,    0,    0,   0,  0,    0,      0,  KEY_R,   0,    (void *)lisa_reset,          NULL, NULL  },
   { d_yield_proc,        0,   0,    0,    0,   0,  0,    0,      0,       0,   0,    NULL,                   NULL, NULL  },
   { NULL,                0,   0,    0,    0,   0,  0,    0,      0,       0,   0,    NULL,                   NULL, NULL  }
};

void debugger(void)
{
     int i;
   /* set up colors */
   gui_fg_color = makecol(0, 0, 0);
   gui_mg_color = makecol(128, 128, 128);
   gui_bg_color = makecol(200, 200, 200);
   set_dialog_color (the_dialog, gui_fg_color, gui_bg_color);

   /* white color for d_clear_proc and the d_?text_procs */
   the_dialog[0].bg = makecol(255, 255, 255);
   for (i = 4; the_dialog[i].proc; i++)
      if (the_dialog[i].proc == d_text_proc ||
          the_dialog[i].proc == d_ctext_proc ||
          the_dialog[i].proc == d_rtext_proc)
         the_dialog[i].bg = the_dialog[0].bg;
   /* shift the dialog 2 pixels away from the border */
   position_dialog (the_dialog, 2, 2);

   breakpoint_str[0]='\0';
   mmu_box[0]='\0';
   disass_debugger();
   regs_debugger();
   /* do the dialog */
   do_dialog(the_dialog, -1);
}

int main() {
    char buf[256];
    int line;
    IDLE_INIT_FUNC("main()");
	init();
	IDLE_DEBUG("Init mem");
    MemInit();
	IDLE_WARN("Init via1");
    init_via_keyb_io();
    init_via_hdd_io();
    init_floppy_io(); 

    get_log_str(log_box,sizeof(log_box_init));

#ifdef _CAST_
    HWReset();
#else
    m68k_init();
    m68k_set_cpu_type(M68K_CPU_TYPE_68000);
    m68k_pulse_reset();
#endif

// go live at boot (bkpt 0 never reached (in theory...))
    lisa_keyb();
    lisa_loop(0);
    allegro_keyb();

    debugger();

	
    allegro_keyb();
	deinit();
	return 0;
}
END_OF_MAIN()

void init() {
	int depth, res;
	allegro_init();
	depth = desktop_color_depth();
	if (depth == 0) depth = 16;
	set_color_depth(depth);
	res = set_gfx_mode(GFX_DIRECTX_WIN, 720, 364, 0, 0);
//	res = set_gfx_mode(GFX_DIRECTX, 800, 600, 0, 0);
	if (res != 0) {
		allegro_message(allegro_error);
		exit(-1);
	}

	install_timer();
	install_keyboard();
	install_mouse();
    LOCK_FUNCTION(keypress_handler);
    LOCK_VARIABLE(last_scancode);
}

void deinit() {
	clear_keybuf();
	/* add other deinitializations here */
}
