#include <stdlib.h>
#include "../config.h"

#include "../cpu/m68k.h"

// to register io functions
#include "../mmu/mmu.h"

#include "via.h"

// pin assignement is as follow
// 
// PA0 - PA7 : COP421
//
// PB0 : reset keyboard (spec83) or record sound (spec81)
// PB0 is reset keyboard => enable to force low for a period
// PB1-3 : speaker volume
// PB4 : floppy dir interrupt status
// PB5 : ??? parallel port
// PB6 : COPS421 handshake (D3)
// PB7 : ??? parallel port
//
// CA1 : COPS421 S0
// CA2 : COPS421 S1
//
// T2 and SR used for sound (CVSD)
// CB1 : out clock to sound (CVSD)

// for pc debug
#ifdef _CAST_
#include "../cpu/68000.h"
#include "../cpu/op68k.h"
#else
#include "../cpu/m68k.h"
#endif


// the mapping address of this VIA
#define VIABASE 0xDD81
// the increment value
#define VIA_INC 2

// these are the outputs
uint8 via_keyb_orb;
uint8 via_keyb_ora;
// the inputs
// don't know why but this must be init to 0 (for now)
uint8 via_keyb_irb=0x00;
uint8 via_keyb_ira;

uint8 via_keyb_latch_a=0;

uint8 via_keyb_ddrb;
uint8 via_keyb_ddra;

uint8 via_keyb_t1c_l;
uint8 via_keyb_t1c_h;

uint8 via_keyb_t1l_l;
uint8 via_keyb_t1l_h;

uint8 via_keyb_t2c_l;
uint8 via_keyb_t2c_h;

uint8 via_keyb_sr;
uint8 via_keyb_acr;
uint8 via_keyb_pcr;
uint8 via_keyb_ifr;
uint8 via_keyb_ier;


// COP state machine
#define COP_IDLE 0
#define COP_CMD_READY 1
#define COP_CMD_BUSY 2


#define COP_CMD_INIT 3
#define COP_CMD_INIT2 4

#define COP_READ_CLK 10
#define COP_READ_CLK2 11

#define COP_MOUSE 20
#define COP_MOUSE2 21


char key_id[2]={0x80,0x3F};
char clk_data[7]={0x80,0xE7,0x11,0x11,0x11,0x11,0x11};
uint8 mouse_data[3]={0x00,0x00,0x00};

int mouse_enable=0;
// COP state
int cop_state=COP_IDLE;
int cop_clk_rec=0;
int cop_id=0;
int cop_busy=0;

#define COP_DELAY1 100

// check VIA status update
void via_keyb_tick(void)
{
    IDLE_INIT_FUNC("via_keyb_tick()");
 // fixme : should handle timers
 
 // test T1
 // T1 in timed interrupt
 if ((via_keyb_acr&0xC0) == 0x40)
    if (via_keyb_ier&0x40) {
       static int c=1;
       c=(c+1)%1000;
       if (c==0) {
           IDLE_TRACE("random T1 interrupt");
           via_keyb_ifr|=0xC0;
           m68k_set_irq(M68K_IRQ_2);
       }
 }
 
 // fixme : should resend blocked key events
 
 if (cop_state==COP_CMD_INIT)
 {
    via_keyb_latch_a=1;
    // simulate handshake for direct mode
    via_keyb_irb|=0x40;
    // reply key id
    via_keyb_ira=key_id[cop_id];
    IDLE_DEBUG("KEYB INIT: COP reply %02x",key_id[cop_id]);
    if (via_keyb_ier&0x02)
    {
           via_keyb_ifr|=0x82;
           m68k_set_irq(M68K_IRQ_2);
     }
     cop_id++;
     cop_busy=0;
     if (cop_id<2)
         cop_state=COP_CMD_INIT2;
     else
         cop_state=COP_CMD_BUSY;
    return;
 }
 
 if (cop_state==COP_CMD_INIT2)
 {
   cop_busy++;
   if (cop_busy>COP_DELAY1)
   {
      cop_state=COP_CMD_INIT;
      via_keyb_irb&=0xBF;
      cop_busy=0;
      return;
      }
 }

 if (cop_state==COP_READ_CLK)
 {
    // reply clock data
    via_keyb_irb|=0x40;
    via_keyb_latch_a=1;
    via_keyb_ira=clk_data[cop_clk_rec];
    IDLE_DEBUG("KEYB CLK: COP reply x%02x",clk_data[cop_clk_rec]);
    if (via_keyb_ier&0x02)
    {
           via_keyb_ifr|=0x82;
           m68k_set_irq(M68K_IRQ_2);
     }
     cop_clk_rec++;
     cop_busy=0;
    if (cop_clk_rec<7)
         cop_state=COP_READ_CLK2;
     else
     {
         cop_state=COP_CMD_BUSY;
     }
     return;
 }

if (cop_state==COP_READ_CLK2)
{
   cop_busy++;
   if (cop_busy>COP_DELAY1)
   {
                      cop_state==COP_READ_CLK;
                      via_keyb_irb&=0xBF;
                      return;
   }
}

 if (cop_state==COP_CMD_READY)
 {
    via_keyb_irb|=0x40;
    cop_busy=0;
    cop_state=COP_CMD_BUSY;
    return;
 }

 if (cop_state==COP_CMD_BUSY)
 {
   cop_busy++;
   if (cop_busy>COP_DELAY1)
   {
       cop_state=COP_CMD_READY;
      via_keyb_irb&=0xBF;
      cop_busy=0;
      }
    return;
 }

 if (cop_state==COP_MOUSE)
 {
    via_keyb_latch_a=1;
    // simulate handshake for direct mode
    via_keyb_irb|=0x40;
    // reply key id
    via_keyb_ira=mouse_data[cop_id];
    IDLE_DEBUG("KEYB INIT: COP reply %02x",mouse_data[cop_id]);
    if (via_keyb_ier&0x02)
    {
           via_keyb_ifr|=0x82;
           m68k_set_irq(M68K_IRQ_2);
     }
     cop_id++;
      cop_busy=0;
     if (cop_id<3)
         cop_state=COP_MOUSE2;
     else
         cop_state=COP_CMD_BUSY;
    return;
 }
 
 if (cop_state==COP_MOUSE2)
 {
   cop_busy++;
   if (cop_busy>COP_DELAY1)
   {
      cop_state=COP_MOUSE;
      via_keyb_irb&=0xBF;
      cop_busy=0;
      return;
      }
 }

 }

// code is already in lisa's scancode
int via_keyb_event_key(uint8 code)
{
    IDLE_INIT_FUNC("via_keyb_event_key()");
    IDLE_DEBUG("keyb event x%02x",code);
     // was the previously event read
     if (via_keyb_latch_a)
     {
      IDLE_WARN("event lost");
      // no => should stack (the COP does it)
      return 0;
     }
     via_keyb_latch_a=1;
     via_keyb_ira=code;
     
     // do we generate an IRQ ?
     // fixme (is it CA1 or CA2)
     if (via_keyb_ier&0x02)
     {
           IDLE_DEBUG("IRQ req");
           via_keyb_ifr|=0x82;
           m68k_set_irq(M68K_IRQ_2);
           return 1;                
     }
     return 0;
}
 

void via_keyb_update_mouse_pos(uint8 x, uint8 y)
{   
    if (!mouse_enable) return;
     mouse_data[1]=x;
     mouse_data[2]=y;
     IDLE_INIT_FUNC("via_keyb_update_mouse_pos()");
     IDLE_DEBUG("called at pc=x%06x",get_pc);
     cop_id=0;
     cop_state=COP_MOUSE;
}

void via_keyb_update_mouse_button(uint8 value)
{
     // nothing here... mouse button is a key in lisa protocol
}

// cop can send reset codes
// 0x80 xx
// where xx:
// FF keyb fail
// FD keyb unplugged
// FC clock timer interrupt
// fb soft power pressed
// Ey clock data follows
//          80 Ey qq qh hm ms st 
// 00 - df key id (01)


// normal code is
// drrr nnnn : where
//      d : 0=up 1=down
//   rrr=row nnnn=column


// commands sent to cop
// 0000 0000 turn io port on
// 0000 0001 turn io off
// 0000 0010 read clock
// 0001 nnnn write nnnn to clock
// 0010 spmm set clock mode
// 0011 nnnn write nnnn to low indicator
// 0100 nnnn write nnnn to high indicator
// 0101 nnnn set high nibble of NMI
// 0110 nnnn set low nibble of NMI
// 0111 nnnn mouse update interval
// 1xxx xxxx Null command 
void send_COP(uint8 code)
{
     IDLE_INIT_FUNC("send_COP()");
     IDLE_DEBUG("pc=x%06x",get_pc);
     switch (code & 0xF0) {
            case 0x00 :
                 {
                      switch (code) {
                            // turn port on
                            case 0x00 :
                                 IDLE_DEBUG("COP: Sent 00");
                                 cop_state=COP_CMD_INIT;
                                 cop_id=0;
                                 return;
                            // turn port off
                            case 0x01 :
                                 IDLE_DEBUG("COP: Sent 01");
                                 cop_state=COP_IDLE;
                                 return;
                            // read clock
                            case 0x02 :
                                        IDLE_DEBUG("COP: Sent 02");
                                        cop_state=COP_READ_CLK;
                                        cop_clk_rec=0;
                                 return;  
                      default:
                              return;
                      }
                  }
            // mouse delay
            case 0x70:
                 mouse_enable=code&0x0F;
                 return;
            // write nnn to clock     
            case 0x10 :
            // set clock mode
            case 0x20 :
            // write low indicator
            case 0x30 :
            // write high indicator
            case 0x40 :
            // set nmi high
            case 0x50 :
            // set nmi low
            case 0x60 :
            default:
                    IDLE_WARN("COP: Sent x%02x",code);
                    return;
            }     
        
     
}

void via_keyb_access_orb(int* val,int mode,int size)
{
     uint8 prev=via_keyb_orb;
     IDLE_INIT_FUNC("via_keyb_access_orb");
     if (mode==IO_WRITE)
     {
        via_keyb_orb=*val&0xFF;
        // check if a reset keyboard ended
        if (((prev&0x01)==0) && ((via_keyb_orb&0x01)==1)) {
           IDLE_TRACE("Keyboard has been reset");
           cop_state=COP_CMD_INIT;
           cop_id=0;
        }
     }
     else
     {
         // fixme ?
         // always read irb no matter ddrb
         *val=via_keyb_irb;
     }
}

void via_keyb_access_ora(int* val,int mode,int size)
{
     IDLE_INIT_FUNC("via_keyb_access_ora()");
     if (mode==IO_WRITE)
     {
        // this sends a command to COP421
        via_keyb_ora=*val&0xFF;
        // fixme (should use ddra)
        if ((via_keyb_ddra&0xFF)!=00)
           send_COP((*val&via_keyb_ddra)&0xFF);
     }
     else
     {
         // select ora or ira         
         *val=(int)((via_keyb_ora&via_keyb_ddra) |
               (via_keyb_ira&(~via_keyb_ddra))) & 0xFF;

          if (via_keyb_latch_a)
          {
             IDLE_DEBUG("keyb read x%02x pc=x%06x",*val,get_pc);
          }

     }



     // clears the latch
     via_keyb_latch_a=0;
     // clears ifr flag
     if (via_keyb_ier&0x3)
     {
         via_keyb_ifr&=0xFC;
         // reset global IRQ flag if needed
         if ((via_keyb_ifr&0x7F)==0)
            via_keyb_ifr=0;
      }
      
      if (cop_state==COP_READ_CLK2)
         cop_state=COP_READ_CLK;
         
      if (cop_state==COP_CMD_INIT2)
         cop_state=COP_CMD_INIT;

}

void via_keyb_access_ddrb(int* val,int mode,int size)
{
     if (mode==IO_WRITE)
     {
        via_keyb_ddrb=*val&0xFF;
     }
     else
     {
         *val=via_keyb_ddrb;
     }
}

void via_keyb_access_ddra(int* val,int mode,int size)
{
     if (mode==IO_WRITE)
     {
        via_keyb_ddra=*val&0xFF;
        if (via_keyb_ddra!=0)
           send_COP((via_keyb_ora&via_keyb_ddra)&0xFF);
      }
     else
     {
         *val=via_keyb_ddra;
     }
}

void via_keyb_access_t1c_l(int* val,int mode,int size)
{
     IDLE_INIT_FUNC("via_keyb_access_t1c_l");
     if (mode==IO_WRITE)
     {
        IDLE_TRACE("set %02x pc=%06d",*val,get_pc);
        via_keyb_t1c_l=*val&0xFF;
     }
     else
     {
         *val=via_keyb_t1c_l;
     }
}

void via_keyb_access_t1c_h(int* val,int mode,int size)
{
     IDLE_INIT_FUNC("via_keyb_access_t1c_h");
     if (mode==IO_WRITE)
     {
        IDLE_TRACE("set %02x get_pc=%06d",*val,get_pc);
        via_keyb_t1c_h=*val&0xFF;
     }
     else
     {
         *val=via_keyb_t1c_h;
     }
}

void via_keyb_access_t1l_l(int* val,int mode,int size)
{
     IDLE_INIT_FUNC("via_keyb_access_t1l_l");
     if (mode==IO_WRITE)
     {
        IDLE_DEBUG("set %02x get_pc=%06d",*val,get_pc);
        via_keyb_t1l_l=*val&0xFF;
     }
     else
     {
         *val=via_keyb_t1l_l;
     }
}

void via_keyb_access_t1l_h(int* val,int mode,int size)
{
     IDLE_INIT_FUNC("via_keyb_access_t1l_h");
     if (mode==IO_WRITE)
     {
        IDLE_DEBUG("set %02x get_pc=%06d",*val,get_pc);
        via_keyb_t1l_h=*val&0xFF;
     }
     else
     {
         *val=via_keyb_t1l_h;
     }
}

void via_keyb_access_t2c_l(int* val,int mode,int size)
{
     IDLE_INIT_FUNC("via_keyb_access_t2c_l");
     if (mode==IO_WRITE)
     {
        IDLE_TRACE("set %02x get_pc=%06d",*val,get_pc);
        via_keyb_t2c_l=*val&0xFF;
     }
     else
     {
         *val=via_keyb_t2c_l;
     }
}

void via_keyb_access_t2c_h(int* val,int mode,int size)
{
     IDLE_INIT_FUNC("via_keyb_access_t2c_h");
    if (mode==IO_WRITE) 
     {
        IDLE_TRACE("set %02x get_pc=%06d",*val,get_pc);
        via_keyb_t2c_h=*val&0xFF;
     }
     else
     {
         *val=via_keyb_t2c_h;
     }
}

void via_keyb_access_sr(int* val,int mode,int size)
{
     if (mode==IO_WRITE)
     {
        via_keyb_sr=*val&0xFF;
     }
     else
     {
         *val=via_keyb_sr;
     }
}

void via_keyb_access_acr(int* val,int mode,int size)
{
     if (mode==IO_WRITE)
     {
        via_keyb_acr=*val&0xFF;
     }
     else
     {
         *val=via_keyb_acr;
     }
}

void via_keyb_access_pcr(int* val,int mode,int size)
{
     if (mode==IO_WRITE)
     {
        via_keyb_pcr=*val&0xFF;
     }
     else
     {
         *val=via_keyb_pcr;
     }
}

void via_keyb_access_ifr(int* val,int mode,int size)
{
     if (mode==IO_WRITE)
     {
        via_keyb_ifr=via_keyb_ifr&((!(*val&0x7F))&0xFF);
        if ((via_keyb_ifr&0x7F)!=0x00) via_keyb_ifr|=0x80;
        else via_keyb_ifr&=0x7f;
     }
     else
     {
         *val=via_keyb_ifr;
     }
}

void via_keyb_access_ier(int* val,int mode,int size)
{
     if (mode==IO_WRITE)
     {
        // set ier                
        if (*val&0x80)
            via_keyb_ier|=(*val&0x7F);
        // reset ier
        else
            via_keyb_ier=(via_keyb_ier&(!(*val&0x7F)))&0x7F;
       }
     else
     {
         *val=(via_keyb_ier|0x80);
     }
}

void via_keyb_access_ora_direct(int* val,int mode,int size)
{
     // for now, no difference in direct or handshake mode
     via_keyb_access_ora(val,mode,size);
}

void via_set_FDIR(void)
{
     via_keyb_irb|=0x10;
}

void via_reset_FDIR(void)
{
     via_keyb_irb&=0xEF;
}

void init_via_keyb_io(void)
{
     int i,j;
     for (i=0xDC00;i<0xDE00;i=i+0x20) {
             registerIoFunc(i+1+VIA_ORB*VIA_INC     ,&via_keyb_access_orb);
             registerIoFunc(i+1+VIA_ORA*VIA_INC     ,&via_keyb_access_ora);
             registerIoFunc(i+1+VIA_DDRB*VIA_INC    ,&via_keyb_access_ddrb);
             registerIoFunc(i+1+VIA_DDRA*VIA_INC    ,&via_keyb_access_ddra);
             registerIoFunc(i+1+VIA_T1C_L*VIA_INC   ,&via_keyb_access_t1c_l);
             registerIoFunc(i+1+VIA_T1C_H*VIA_INC   ,&via_keyb_access_t1c_h);
             registerIoFunc(i+1+VIA_T1L_L*VIA_INC   ,&via_keyb_access_t1l_l);
             registerIoFunc(i+1+VIA_T1L_H*VIA_INC   ,&via_keyb_access_t1l_h);
             registerIoFunc(i+1+VIA_T2C_L*VIA_INC   ,&via_keyb_access_t2c_l);
             registerIoFunc(i+1+VIA_T2C_H*VIA_INC   ,&via_keyb_access_t2c_h);
             registerIoFunc(i+1+VIA_SR*VIA_INC      ,&via_keyb_access_sr);
             registerIoFunc(i+1+VIA_ACR*VIA_INC     ,&via_keyb_access_acr);
             registerIoFunc(i+1+VIA_PCR*VIA_INC     ,&via_keyb_access_pcr);
             registerIoFunc(i+1+VIA_IFR*VIA_INC     ,&via_keyb_access_ifr);
             registerIoFunc(i+1+VIA_IER*VIA_INC     ,&via_keyb_access_ier);
             registerIoFunc(i+1+VIA_ORA_DIRECT*VIA_INC,&via_keyb_access_ora_direct);
     }
}
