#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "../config.h"

// to register io functions
#include "../mmu/mmu.h"
#ifdef _CAST_
#include "../cpu/68000.h"
#include "../cpu/op68k.h"
#else
#include "../cpu/m68k.h"
#endif
#include "../io/via.h"


#include "disk_image.h"


/*
real mapping is
DISKMEM .EQU $00FCC001 ;base address of shared memory
CMD .EQU 2 ;offset for command byte
DRV .EQU CMD+2 ;offset for drive #
SIDE .EQU DRV+2 ;side #
SCTR .EQU SIDE+2 ;sector #
TRAK .EQU SCTR+2 ;track #
SPEED .EQU TRAK+2 ;motor speed control
CNFRM .EQU SPEED+2 ;confirm for format cmd
STAT .EQU CNFRM+2 ;error status
INTLV .EQU STAT+2 ;interleave factor CHG022
TYPE .EQU INTLV+2 ;drive type id CHG009
STST .EQU TYPE+2 ;self-test result CHG022
ROMV .EQU $30 ;ROM version #
RTRYCNT .EQU $58 ;retry count
INTSTAT .EQU $5E ;interrupt status
CHKCNT .EQU $BA ;data checksum error count
CHKCNT2 .EQU $C4 ;address checksum error count
DSKBUFF .EQU $3E8 ;start of disk buffer
DSKDATA .EQU DSKBUFF+24 ;first 12 bytes are header
DISKROM .EQU $FCC031 ;absoulte address for disk ROM id

real commands are (from Boot rom listing)

READS .EQU 0 ;read sector w/checksum
WRT .EQU 1
UNCLAMP .EQU 2 ;unclamp diskette
FMT .EQU 3
VFY .EQU 4 ;verify disk
CLAMP .EQU 9 ;clamp disk
OK .EQU $FF ;confirmation for format
SEEK .EQU $83 ;seek cmd
EXRW .EQU $81 ;execute cmd
CLRSTAT .EQU $85 ;clear status cmd
ENBLINT .EQU $86 ;enable intrpt
DSABLINT .EQU $87 ;disable intrpt
SLEEP .EQU $88 ;loop in RAM cmd
DIE .EQU $89 ;loop in ROM cmd


*/

static int drive;
static int side;
static int sector;
static int track;
static int error;

static int command_time;

uint8 floppy_shared_tab[0x400];


#ifdef _UNDER_TEST_
FILE *f_debug=NULL;
int seqnum=0;
int lastpc=0;

void newFile(void)
{
     char fnm[256];
     sprintf(fnm,"FILE%03d.bin",seqnum);
     seqnum++;
     if (f_debug!=NULL) fclose(f_debug);
     f_debug=fopen(fnm,"wb");
}

void pushBlock(char * block,int sz,uint32 pc)
{
     // we assume same file for same pc value command
     // ok for boot
     if ((f_debug==NULL) || (pc!=lastpc)) newFile();
     fwrite(block,1,sz,f_debug);
     lastpc=pc;
}

#endif

#define CMD_NONE -1
#define CMD_UNCLAMP 0
#define CMD_READ 1
#define CMD_WRITE 2
#define CMD_RESET_INT 3
#define CMD_HANDSHAKE 4
#define CMD_DRIVE_ENABLE 5
#define CMD_CALL         6

static int command_type=CMD_NONE;

uint8 command;
uint8 gobyte;

void floppy_command(int* val,int mode,int size)
{
     static int init=0;
     IDLE_INIT_FUNC("floppy_command()");
     // fixme what if byte access?
     if (mode==IO_WRITE)
     {
            IDLE_DEBUG("set command %02x at get_pc=%06x",*val&0xff,get_pc);
            command=*val&0xFF;
     }
     else
     {
         *val=command;
     }
}

void floppy_stst(int* val,int mode,int size)
{
     static int init=0;
     IDLE_INIT_FUNC("floppy_stst()");
     // fixme what if byte access?
     if (mode==IO_WRITE)
     {
        IDLE_TRACE("write floppy stst");
     }
     else
     {
         *val=0x00;
     }
}

// this is the end of command routine (mapped in 0x1518 in IO ROM A8)
void E_1518(void) {
     if (floppy_shared_tab[0x002f]&0x70) floppy_shared_tab[0x002f]|=0x80;
     floppy_shared_tab[0x002E]=floppy_shared_tab[0x002C]&floppy_shared_tab[0x002F];    
     if (floppy_shared_tab[0x002E]!=0) { 
        via_set_FDIR();
        m68k_set_irq(M68K_IRQ_1);
     }
     else via_reset_FDIR();
}

int floppy_tick(void)
{
    IDLE_INIT_FUNC("floppy_tick()");
    // take command
    gobyte=0;
    if (command_time>1) command_time--;
    else
    if (command_time==1)
    {
        TRACE1("sim disk handshake : get_pc=x%06x",get_pc);
                
        command_time=0;
        if ((command_type==CMD_READ) || (command_type==CMD_UNCLAMP))
           {
              via_reset_DIAG();
              switch (command_type) {
                     case CMD_READ :
                          floppy_shared_tab[0x002f]|=0x40;
                          break;
                     case CMD_UNCLAMP :
                          floppy_shared_tab[0x002f]|=0x40;
                          // clears disk in place status
                          floppy_shared_tab[0x0020]=0x00;
                          break;
              }
           E_1518();           
           // macworks hack...
           if (command_type==CMD_CALL)
                floppy_shared_tab[0x0006]=0xFF;          
        }
    }
    else
    if (command_time==0) {
       via_set_DIAG();
       command_type=CMD_NONE;
    }
}



void floppy_gobyte(int* val,int mode,int size)
{
     uint8 *data;
     uint8 *tags;
     int ret;
     int i;
  
     IDLE_INIT_FUNC("floppy_gobyte()");     
     // fixme what if byte access?
     if (mode==IO_WRITE)
     {
         gobyte=(*val)&0xFF;
         IDLE_DEBUG("Got command %02x gobyte %02x",command,gobyte);
         if ((gobyte==0x80)) {
         // handshake
               IDLE_TRACE("Handshake");
               command_type=CMD_HANDSHAKE;
               floppy_shared_tab[0x0008]=0;
               command_time=1;
         }
         if ((gobyte==0x84)) {
         // handshake
         // this is a hack for macworks 2.0 & 3.0 ....
               IDLE_TRACE("Call user prog");
               command_type=CMD_CALL;
               floppy_shared_tab[0x0006]=0;
               command_time=2;
         }
         if (gobyte==0x86) {
         // drive enable
               IDLE_TRACE("drive int enable");
               command_type=CMD_DRIVE_ENABLE;
               floppy_shared_tab[0x0008]=0;
               floppy_shared_tab[0x002C]|=command;
               command_time=1;
         }
         if (gobyte==0x87) {
         // drive int disable
               IDLE_TRACE("drive int disable");
               command_type=CMD_HANDSHAKE;
               floppy_shared_tab[0x0008]=0;
               floppy_shared_tab[0x002C]&=(!command)&0xFF; 
               command_time=1;
         }
         if (gobyte==0x81) {
            switch (command) {
                // read sector
                case 0x00 :
                     command_type=CMD_READ;
                     IDLE_TRACE("Read sect%d track%d",sector,track);
                     command_time=3;
                     ret=read_sector(0,track,sector,side,
                             &data,&tags);
                     IDLE_DEBUG("ret=%d",ret);
                     if (ret==0)
                     {
                             floppy_shared_tab[0x0008]=0;
                             for (i=0x005B;i<0x0063;i++)
                                 floppy_shared_tab[i]=0;
                             
                             memcpy(&floppy_shared_tab[0x1F4+12],data,512);
                             memcpy(&floppy_shared_tab[0x1F4],tags,12);
                             IDLE_TRACE("FILEID %02x%02x",tags[4],tags[5]);
#ifdef _UNDER_TEST_
                             pushBlock((char *)data,512,get_pc);
#endif
                     }
                     break;
                // case 01 Write
                case 0x01 :
                     {
                          uint8 wdata[512];
                          uint8 wtags[12];
                          command_type=CMD_READ;
                          IDLE_TRACE("Write sect%d track%d",sector,track);
                          command_time=3;
                          floppy_shared_tab[0x0008]=0;
                          for (i=0x005B;i<0x0063;i++)
                            floppy_shared_tab[i]=0;
                             
                             memcpy(wdata,&floppy_shared_tab[0x1F4+12],512);
                             memcpy(wtags,&floppy_shared_tab[0x1F4],12);
                             IDLE_TRACE("FILEID %02x%02x",tags[4],tags[5]);

                             ret=write_sector(0,track,sector,side,
                             wdata,wtags);
                             IDLE_DEBUG("ret=%d",ret);

                     }
                     break;
                // unclamp
                case 0x02 :
                     command_type=CMD_UNCLAMP;
                     IDLE_TRACE("Unclamp");
                     command_time=1;
                     floppy_shared_tab[0x0008]=0;
                default :
                     IDLE_TRACE("Got command %02x",command);
                } // of switch
                } // of if gobyte=0x81
                    
                if (gobyte==0x85) {                // reset INT
                     IDLE_TRACE("REset INT");
                     command_type=CMD_RESET_INT;
                     command_time=1;
                     floppy_shared_tab[0x002f]&=(!command)&0xFF;
                     floppy_shared_tab[0x0008]=0;
                }

     } // of mode WRITE
     else
     {
         *val=gobyte;
     }
}

void floppy_drive(int* val,int mode,int size)
{
     IDLE_INIT_FUNC("floppy_drive()");
     if (size==IO_WORD) IDLE_WARN("Is Word!");
     // fixme what if byte access?
     if (mode==IO_WRITE)
     {
         drive=(*val)&0xFF;
     }
     else
     {
         *val=drive;
     }
}

void floppy_side(int* val,int mode,int size)
{
     IDLE_INIT_FUNC("floppy_side()");
     // fixme what if byte access?
     if (mode==IO_WRITE)
     {
         side=(*val)&0xFF;
     }
     else
     {
         *val=side;
     }
}

void floppy_sector(int* val,int mode,int size)
{
     // fixme what if byte access?
     if (mode==IO_WRITE)
     {
         sector=(*val)&0xFF;
     }
     else
     {
         *val=sector;
     }
}

void floppy_track(int* val,int mode,int size)
{
     // fixme what if byte access?
     if (mode==IO_WRITE)
     {
         track=(*val)&0xFF;
     }
     else
     {
         *val=track;
     }
}

void floppy_error(int* val,int mode,int size)
{
     // fixme what if byte access?
     if (mode==IO_WRITE)
     {
         error=(*val)&0xFF;
     }
     else
     {
         *val=error;
     }
}

void floppy_shared(uint32 adr,int* val,int mode,int size)
{
     IDLE_INIT_FUNC("floppy_shared()");
//     uint32 offset=(adr&0x07FF)>>1;
     uint32 offset=(adr&0x0fff)>>1;
     if (size==IO_WORD) IDLE_WARN("Is Word!");
     if (mode==IO_WRITE)
     {
         if (offset<0xB0)
            IDLE_TRACE("write off%x",offset);
if (offset!=0x2f)
         floppy_shared_tab[offset]=(*val)&0xFF;
     }
     else
     {
         if (offset<0xB0)
            IDLE_TRACE("read off%x",offset);
         *val=floppy_shared_tab[offset];
#if 1
//         if (offset==0x20) ask_bkpt=1;// 20 is isk in place
//         if (offset==0x2f) ask_bkpt=1; // 2f is drive int status... 
           
//         if (offset==0x4a) ask_bkpt=1; 
//         if (offset==0x5c) ask_bkpt=1; 
#endif
     }
}

void init_floppy_io(void)
{
     int i;

     registerIoFunc(0xC001 ,&floppy_gobyte);
     
     registerIoFunc(0xC003 ,&floppy_command);
     registerIoFunc(0xC005 ,&floppy_drive);
     registerIoFunc(0xC007 ,&floppy_side);
     registerIoFunc(0xC009 ,&floppy_sector);
     registerIoFunc(0xC00B ,&floppy_track);
     
     for (i=0xC00D;i<0xC801;i+=2)
          registerIoFuncAdr(i ,&floppy_shared);

     for (i=0;i<0x400;i++)     
         floppy_shared_tab[i]=0x00; 

     floppy_shared_tab[0x18]=0xA8; // rom version
     floppy_shared_tab[0x0A]=0x01; // disk type = sony single sided
     floppy_shared_tab[0x46]=0xFF; // 
     floppy_shared_tab[0x0009]=2; // lisa disk
     floppy_shared_tab[0x005C]=10; // lisa disk
     floppy_shared_tab[0x20]=0xff; // disk is in place
     
         

     registerIoFunc(0xC017 ,&floppy_stst);
     
}
