/*

              __________  __    __    ________________  ____ 
             / ____/ __ \/ /   / /   / ____/ ____/ __ \/ __ \
            / /   / / / / /   / /   / __/ / / __/ /_/ / / / /
           / /___/ /_/ / /___/ /___/ /___/ /_/ / _, _/ /_/ / 
           \____/\____/_____/_____/_____/\____/_/ |_|\____/  
          
                 - Collision Detection for Allegro -


The collegro library is a collision detection library for use with allegro.

Currently, collegro supprts bounding boxes, bouding circles, and bit maps.  You
can test all the above against each other.  

You may use this library freely.  Feel free to change anything you wish.  
The only restriction is that if you plan to sell or otherwise profit off
of collegro without it being used in a program you made, let me know first.
You can leave a message on my website in the forums.

Also, feel free to suggest new features and report bugs on my website.

Thank you for using collegro,
- Kaitlyn Handelman

--------------------------------------------------------------------------------

Website: http://www.playingwithyarn.net
*/


#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <allegro.h>
#include "collegro.h"

int clgo_error = CLGO_NO_ERROR;

static CLGO_BIT_MASK *allocate_bitmask_memory(int x_pos, int y_pos, int width, int height);
/* Allocates all the memory space needed for a bit mask. */

static void make_bit_mask(CLGO_BIT_MASK *fill_me, BITMAP *allegro_bitmap, int x_offset, 
                          int y_offset, int width, int height, 
                          int (*func)(int bitmap_x, int bitmap_y, int depth, 
                                      unsigned long int pixel));
/* Fills in all the bitmask data. */

static void get_overlapping_rect(CLGO_BOUNDING_BOX *bb_a, CLGO_BOUNDING_BOX *bb_b, 
                                 int *x, int *y, int *width, 
                                 int *height);
/* Fills the arguments with an overlapping rectangle.  Note: make sure
that the two bounding boxes overlap first. They overlap if there's a collision. */

static int get_bit(CLGO_BIT_MASK *bit_mask, int x, int y);
/* Grabs a bit from a bit mask at x, y.  It will automatically adjust for any
vertical or horizontal flipping. */

CLGO_BOUNDING_BOX *clgo_create_bbox(int x_placement, int y_placement, int width, int height)
{
  CLGO_BOUNDING_BOX *new_box;  /* The new collision box we'll make.  */

  /* Allocate memory for the new box. */
  new_box = (CLGO_BOUNDING_BOX *)malloc(sizeof(CLGO_BOUNDING_BOX));

  /* If there was an error in allocating space, return NULL. */
  if (new_box == NULL)
  {
    clgo_error = CLGO_ALLOCATION_ERROR;
    return NULL;
  }

  /* Now we set our starting data. */
  new_box->x = x_placement;
  new_box->y = y_placement;
  new_box->width = width;
  new_box->height = height;

  return new_box;
}



CLGO_BOUNDING_BOX *clgo_bitmap_create_bbox(int x_placement, int y_placement, 
                                           BITMAP *allegro_bitmap)
{  
  /* Now we just steal the width and height from the bitmap. */
  return clgo_create_bbox(x_placement, y_placement, allegro_bitmap->w, allegro_bitmap->h);
}



CLGO_BOUNDING_CIRCLE *clgo_create_bcircle(int center_x, int center_y, float radius)
{
  CLGO_BOUNDING_CIRCLE *new_circle; /* The new bounding circle to make. */

  /* Allocating the space needed for our circle. */
  new_circle = (CLGO_BOUNDING_CIRCLE *)malloc(sizeof(CLGO_BOUNDING_CIRCLE));

  /* If there was an error in allocating space, return NULL. */
  if (new_circle == NULL)
  {
    clgo_error = CLGO_ALLOCATION_ERROR;
    return NULL;
  }

  /* Now we set our starting data. */
  new_circle->x = center_x;
  new_circle->y = center_y;
  new_circle->radius = radius;
  new_circle->radius_squared = radius * radius;

  return new_circle;
}



CLGO_BIT_MASK *clgo_create_bitmask_ex(int x_placement, int y_placement, int x_offset, int y_offset, int width, int height, BITMAP *allegro_bitmap)
{
  CLGO_BIT_MASK *new_bit_mask; /* The new bitmask to make. */
  int color_depth;

  /* Now, make sure the width and height specified are within range. */
  if (x_offset + width > allegro_bitmap->w || y_offset + height > allegro_bitmap->h)
  {
    clgo_error = CLGO_INVALID_PARAMS;
    return NULL;
  }

  /* Allocate new memory for the bit mask. */
  new_bit_mask = allocate_bitmask_memory(x_placement, y_placement, width, height);

  if (new_bit_mask == NULL)
  {
    clgo_error = CLGO_ALLOCATION_ERROR;
    return NULL;
  }

  /* Now determine the bit depth and check the image for transparent pixels. */
  color_depth = bitmap_color_depth(allegro_bitmap);
  make_bit_mask(new_bit_mask, allegro_bitmap, x_offset, y_offset, width, height, clgo_standard_bitmask_func);

  return new_bit_mask;
}



CLGO_BIT_MASK *clgo_create_bitmask(int x_placement, int y_placement, BITMAP *allegro_bitmap)
{
  /* Just call the ex function, encompassing the entire bitmap. */
  return clgo_create_bitmask_ex(x_placement, y_placement, 0, 0, allegro_bitmap->w, allegro_bitmap->h, allegro_bitmap);
}



CLGO_BIT_MASK *clgo_create_custom_bitmask_ex(int x_placement, int y_placement, int x_offset, int y_offset, int width, int height, BITMAP *allegro_bitmap, int (*func)(int bitmap_x, int bitmap_y, int depth, unsigned long int pixel))
{
  CLGO_BIT_MASK *new_bit_mask; /* The new bitmask to make. */
  int color_depth;

  /* Now, make sure the width and height specified are within range. */
  if (x_offset + width > allegro_bitmap->w || y_offset + height > allegro_bitmap->h)
  {
    clgo_error = CLGO_INVALID_PARAMS;
    return NULL;
  }

  /* Allocate new memory for the bit mask. */
  new_bit_mask = allocate_bitmask_memory(x_placement, y_placement, width, height);

  /* Now determine the bit depth and check the image for transparent pixels. */
  color_depth = bitmap_color_depth(allegro_bitmap);
  make_bit_mask(new_bit_mask, allegro_bitmap, x_offset, y_offset, width, height, func);

  return new_bit_mask;
}




CLGO_BIT_MASK *clgo_create_custom_bitmask(int x_placement, int y_placement, BITMAP *allegro_bitmap, int (*func)(int bitmap_x, int bitmap_y, int depth, unsigned long int pixel))
{  
  /* Call the ex function to emcompass the entire bitmap. */
  return clgo_create_custom_bitmask_ex(x_placement, y_placement, 0, 0, allegro_bitmap->w, allegro_bitmap->h, allegro_bitmap, func);
}




void clgo_update_bbox_ex(CLGO_BOUNDING_BOX *bb, int new_x_placement, int new_y_placement, int new_width, int new_height)
{
  /* Update! */
  bb->x = new_x_placement;
  bb->y = new_y_placement;
  bb->width = new_width;
  bb->height = new_height;
}




void clgo_update_bbox(CLGO_BOUNDING_BOX *bb, int new_x_placement, int new_y_placement)
{
  /* Update! */
  bb->x = new_x_placement;
  bb->y = new_y_placement;
}




void clgo_update_bcircle_ex(CLGO_BOUNDING_CIRCLE *circle, int new_center_x, int new_center_y, float new_radius)
{
  /* Update. */
  circle->x = new_center_x;
  circle->y = new_center_y;
  if (new_radius >= 0.0f)
  {
    circle->radius = new_radius;
    circle->radius_squared = new_radius * new_radius;
  }
}




void clgo_update_bcircle(CLGO_BOUNDING_CIRCLE *circle, int new_center_x, int new_center_y)
{
  /* Update. */
  circle->x = new_center_x;
  circle->y = new_center_y;
}




void clgo_update_bitmask(CLGO_BIT_MASK *bitmask, int new_x_placement, int new_y_placement)
{
  /* Update the bounding box info. */
  bitmask->bb.x = new_x_placement;
  bitmask->bb.y = new_y_placement;
}




void clgo_flip_bitmask(CLGO_BIT_MASK *bitmask, int h_flip, int v_flip)
{
  /* Set the flags for flipping. */
  bitmask->h_flip = h_flip;
  bitmask->v_flip = v_flip;
}




void clgo_destroy_bbox(CLGO_BOUNDING_BOX *destroy_me)
{
  free(destroy_me);
}




void clgo_destroy_bcircle(CLGO_BOUNDING_CIRCLE *destroy_me)
{
  free(destroy_me);
}




void clgo_destroy_bitmask(CLGO_BIT_MASK *destroy_me)
{
  /* First we have to delete the memory inside, first. */
  free(destroy_me->entries);
  free(destroy_me);
}




void clgo_draw_bbox(CLGO_BOUNDING_BOX *bb, BITMAP *destination, int color)
{
  /* Draw a rect in the shape of the bounding box. */
  clgo_inline_draw_bbox(bb, destination, color);
}




void clgo_draw_bcircle(CLGO_BOUNDING_CIRCLE *bcircle, BITMAP *destination, int color)
{
  /* Now just draw a circle using the info from the bounding circle. */
  clgo_inline_draw_bcircle(bcircle, destination, color);
}




void clgo_draw_bitmask_bbox(CLGO_BIT_MASK *bitmask, BITMAP *destination, int color)
{
  /* We store the dimensions of the bit mask as a bounding box anyways. */
  clgo_inline_draw_bitmask_bbox(bitmask, destination, color);
}




void clgo_draw_bitmask(CLGO_BIT_MASK *bitmask, BITMAP *destination, int x, int y, int color_trans, int color_solid)
{
  int bit_x; /* Bit mask x position. */
  int start_bit_x; /* Starting bit mask x position. */
  int bit_y; /* Bit mask y position. */
  int end_bit_x; /* Ending bit map x position. */
  int end_bit_y; /* Ending bit map y position. */
  int add_x; /* Amount to add to bit_x at end of reading pixel. */
  int add_y; /* Amount to add to bit_y after reading a line. */
  int color_depth; /* Color depth of the bitmap. */
  int byte_depth; /* Color depth in bytes. */
  int bmp_x; /* Bitmap x position. */
  int bmp_y; /* Bitmap y position. */
  unsigned int write_address; /* Write address. */
  unsigned char *entries; /* bit mask entries. */
  int width; /* Bit mask width. */
  int height; /* Bit mask height. */

  width = bitmask->bb.width;
  height = bitmask->bb.height;

  /* Determine the end x location if we horizontally flip the bit mask or not. */
  if (bitmask->h_flip)
  {
    end_bit_x = -1;
    add_x = -1;
    bit_x = width -1;
  } else
  {
    end_bit_x = width;
    bit_x = 0;
    add_x = 1;
  }

  start_bit_x = bit_x;

  /* Determine the vertical y position if we vertically flip the bit mask or not. */
  if (bitmask->v_flip)
  {
    end_bit_y = -1;
    add_y = -1;
    bit_y = height - 1;
  } else
  {
    end_bit_y = height;
    add_y = 1;
    bit_y = 0;
  }

  if (destination->w < width || destination->h < height)
  {
    /* Not enough space to blit. */
    clgo_error = CLGO_INVALID_PARAMS;
    return;
  }

  /* Figure out the color depth of our destination. */
  color_depth = bitmap_color_depth(destination);
  if (color_depth == 15)
  {
    byte_depth = 2;
  } else
  {
    byte_depth = color_depth / 8;
  }

  bmp_y = y;

  entries = bitmask->entries + bit_y * width + start_bit_x;

  /* Now go through each bit and place a pixel on the detination bitmap. */
  while (bit_y != end_bit_y)
  {
    write_address = bmp_write_line(destination, bmp_y);
    bit_x = start_bit_x;
    bmp_x = x;
    while (bit_x != end_bit_x)
    {
      switch (color_depth)
      {
      case 8:
        if (*entries)
          bmp_write8(write_address + bmp_x * byte_depth, color_solid);
        else
          bmp_write8(write_address + bmp_x * byte_depth, color_trans);
        break;
      case 15:
        if (*entries)
          bmp_write15(write_address + bmp_x * byte_depth, color_solid);
        else
          bmp_write15(write_address + bmp_x * byte_depth, color_trans);
        break;
      case 16:
        if (*entries)
          bmp_write16(write_address + bmp_x * byte_depth, color_solid);
        else
          bmp_write16(write_address + bmp_x * byte_depth, color_trans);
        break;
      case 24:
        if (*entries)
          bmp_write24(write_address + bmp_x * byte_depth, color_solid);
        else
          bmp_write24(write_address + bmp_x * byte_depth, color_trans);
        break;
      case 32:
        if (*entries)
          bmp_write32(write_address + bmp_x * byte_depth, color_solid);
        else
          bmp_write32(write_address + bmp_x * byte_depth, color_trans);
        break;
      default:
        return;
      }

      entries += add_x;
      bmp_x = bmp_x + 1;
      bit_x = bit_x + add_x;
    }
    bmp_unwrite_line(destination);
    bmp_y = bmp_y + 1;
    bit_y = bit_y + add_y;
    if (add_y != add_x)
    {
      entries += add_y * width * 2;
    }
  }
}




int clgo_collide_bitmaps(BITMAP *bitmap_a, int blit_x_a, int blit_y_a, BITMAP *bitmap_b, int blit_x_b, int blit_y_b)
{
  return clgo_inline_collide_bbox_coords(blit_x_a, blit_y_a, bitmap_a->w, 
                                         bitmap_a->h, blit_x_b, blit_y_b, 
                                         bitmap_b->w, bitmap_b->h);
}




int clgo_collide_bcircle_bitmap(CLGO_BOUNDING_CIRCLE *circle_a, BITMAP *bitmap_b, int blit_x_b, int blit_y_b)
{
  CLGO_BOUNDING_BOX *bb_b; /* Bounding box for bitmap_b. */
  int result; /* Result of the collision test. */

  /* Create our bounding box for our bitmap. */
  bb_b = clgo_bitmap_create_bbox(blit_x_b, blit_y_b, bitmap_b);

  /* Check for an error. */
  if (bb_b == NULL)
  {
    clgo_error = CLGO_ALLOCATION_ERROR;
    return 0; /* Yikes! */
  }

  /* Now do our boudning box test. */
  result = clgo_collide_bbox_bcircle(bb_b, circle_a);

  clgo_destroy_bbox(bb_b);

  return result;
}




int clgo_collide_bbox_coords(int x1, int y1, int width1, int height1, int x2, 
                             int y2, int width2, int height2)
{
  return clgo_inline_collide_bbox_coords(x1, y1, width1, height1, x2, y2, width2, height2);
}




int clgo_collide_point_bbox(int x, int y, CLGO_BOUNDING_BOX *bounding_box)
{
  return clgo_inline_collide_point_bbox(x, y, bounding_box);
} 




int clgo_collide_point_bcircle(float x, float y, CLGO_BOUNDING_CIRCLE *bounding_circle)
{
  return clgo_inline_collide_point_bcircle(x, y, bounding_circle);
}



int clgo_collide_bbox_bbox(CLGO_BOUNDING_BOX *bb_a, CLGO_BOUNDING_BOX *bb_b)
{  
  /* See if the Xes and Ys are both between the other collision box's points. */
  return clgo_inline_collide_bbox_bbox(bb_a, bb_b);
}





int clgo_collide_bbox_bbox_ex(CLGO_BOUNDING_BOX *bb_a, CLGO_BOUNDING_BOX *bb_b, int *overlapping_rect_x, int *overlapping_rect_y,int *overlapping_rect_width, int *overlapping_rect_height)
{
  /* First, make sure the bounding boxes collide. */
  if (!clgo_inline_collide_bbox_bbox(bb_a, bb_b))
  {
    return 0;
  }

  /* Everything look AOK, let's fill in that data. */
  get_overlapping_rect(bb_a, bb_b, overlapping_rect_x, overlapping_rect_y, 
                       overlapping_rect_width, overlapping_rect_height);

  return 1;
}




int clgo_collide_bcircle_bcircle(CLGO_BOUNDING_CIRCLE *circle_a, CLGO_BOUNDING_CIRCLE *circle_b)
{
  /* This is the easiest one.  Find the distance on the x and y axis...
  then the total distance squared will be distance_x^2 + distance_y^2.  If the 
  total distance is less than the sum of the two radii. */

  /* Check to see if the total distance between the center of the two
  circles are less than or equal to their actual distances. */
  return clgo_inline_collide_bcircle_bcircle(circle_a, circle_b);
}





int clgo_collide_bitmask_bitmask(CLGO_BIT_MASK *bitmask_a, CLGO_BIT_MASK *bitmask_b)
{
  CLGO_BOUNDING_BOX *bb_a; /* First bounding box. */
  CLGO_BOUNDING_BOX *bb_b; /* Second bounding box. */
  int bit_a_start_x; /* The starting x bit for bit mask a. */
  int bit_a_start_y; /* The starting y bit for bit mask a. */
  int bit_b_start_x; /* The starting x bit for bit mask b. */
  int bit_b_start_y; /* The starting x bit for bit mask b. */
  int overlapping_box_x; /* Overlapping box left-most x coordinate. */
  int overlapping_box_y; /* Overlapping box upper y coordinate. */
  int overlapping_box_width;  /* The width of the overlapping bounding box. */
  int overlapping_box_height; /* The height of the overlapping bounding box. */
  int line_x; /* Current x position while comparing bitmasks. */
  int line_y; /* Current y position while comparing bitmasks. */
  int bit_a; /* Bit from bitmask a. */
  int bit_b; /* Bit from bitmask b. */

  bb_a = &(bitmask_a->bb);
  bb_b = &(bitmask_b->bb);

  /* Both bounding boxes should overlap first. */
  if (clgo_inline_collide_bbox_bbox(bb_a, bb_b))
  {
    get_overlapping_rect(bb_a, bb_b, &overlapping_box_x, &overlapping_box_y, &overlapping_box_width, &overlapping_box_height);
    bit_a_start_x = overlapping_box_x - bb_a->x;
    bit_a_start_y = overlapping_box_y - bb_a->y;
    bit_b_start_x = overlapping_box_x - bb_b->x;
    bit_b_start_y = overlapping_box_y - bb_b->y; 

    /* Find each bit in the overlapping box and see if there are any
    overlapping 1 bits.  */
    for (line_y = 0; line_y < overlapping_box_height; line_y = line_y + 1)
    {
      for (line_x = 0; line_x < overlapping_box_width; line_x = line_x + 1)
      {
        bit_a = get_bit(bitmask_a, line_x + bit_a_start_x, line_y + bit_a_start_y);
        bit_b = get_bit(bitmask_b, line_x + bit_b_start_x, line_y + bit_b_start_y);

        /* Hit! */
        if (bit_a && bit_b)
        {
          return 1;
        }
      }
    }
  }
  /* Restore our old data. */

  return 0;
}




int clgo_collide_bbox_bcircle(CLGO_BOUNDING_BOX *bb_a, CLGO_BOUNDING_CIRCLE *circle_b)
{
  /* This test is rather simple.  If an outter point extending along the x and y
  axes from the circle's center (think 0, 90, 180, 270 degree angles), are 
  inside the box, it collided.  Otherwise, in order for there to be a collision,
  one of the four box corners will be in the circle.  Check four outter points
  on each object.  If any return true of being inside one another, we return
  true. */

  float bb_center_point_x; /* Center point of the bounding box. */
  float bb_center_point_y; /* Center point of the bounding box. */
  float total_width; /* Width between center of bounding box and circle. */
  float total_height; /* Height between center of bounding box and circle. */
  int outter_point_x; /* Outter x point on the outter edge of the circle. */
  int outter_point_y; /* Outter y point on the outter edge of the circle. */ 
  float box_point_x; /* Box coordinates converted to floating point. */
  float box_point_y; /* Box coordinates converted to floating point. */
  int bb_a_x; /* Bounding box a X coordinate. */
  int bb_a_y; /* Bounding box a Y coordinate. */
  int bb_a_width; /* Bounding box a width. */
  int bb_a_height; /* Bounding box a height. */
  int circle_b_x; /* Circle b center X coordinate. */
  int circle_b_y; /* Circle b center Y coordinate. */
  float circle_b_radius; /* Circle b radius. */

  bb_a_x = bb_a->x;
  bb_a_y = bb_a->y;
  bb_a_width = bb_a->width;
  bb_a_height = bb_a->height;

  circle_b_x = circle_b->x;
  circle_b_y = circle_b->y;
  circle_b_radius = circle_b->radius;

  /* Before we do our tests, let's just do a quick test to see if it's even 
  possible for these two to even collide in the first place. */
  bb_center_point_x = (float)(bb_a_x + bb_a_x + bb_a_width- 1) / 2.0f;
  bb_center_point_y = (float)(bb_a_y + bb_a_y + bb_a_height - 1) / 2.0f;

  /* Find the distance between the center of the circle and bounding box. */
  total_width = circle_b_x - bb_center_point_x;
  if (total_width < 0)
  {
    total_width = total_width * -1;
  }

  /* If the width is too great for any collision, don't go any further. */
  if (total_width > (float)(bb_a_width) / 2.0f + circle_b_radius)
  {
    return 0;
  }

  /* Find the vertical distance between the circle and bounding box. */
  total_height = circle_b_y - bb_center_point_y;
  if (total_height < 0)
  {
    total_height = total_height * -1;
  }

  /* If the height difference is too great, stop now. */
  if (total_height > (float)(bb_a_height) / 2.0f + circle_b_radius)
  {
    return 0;
  }

  /* Point inside circle tests. */

  /* Upper left point. */
  box_point_x = (float)bb_a_x + 0.5f;
  box_point_y = (float)bb_a_y + 0.5f;
  if (clgo_inline_collide_point_bcircle(box_point_x, box_point_y, circle_b))
  {
    return 1;
  }

  /* Upper right point. */
  box_point_x = (float)(bb_a_x + bb_a_width - 1) + 0.5f;
  if (clgo_inline_collide_point_bcircle(box_point_x, box_point_y, circle_b))
  {
    return 1;
  }

  /* Lower Right Point. */
  box_point_y = (float)(bb_a_y + bb_a_height - 1) + 0.5f;
  if (clgo_inline_collide_point_bcircle(box_point_x, box_point_y, circle_b))
  {
    return 1;
  }

  /* Lower Left Point. */
  box_point_x = (float)(bb_a_x) + 0.5f;
  if (clgo_inline_collide_point_bcircle(box_point_x, box_point_y, circle_b))
  {
    return 1;
  }

  /* Point inside bounding box tests. */

  /* Right Point. */
  outter_point_x = circle_b_x + (int)circle_b_radius;
  outter_point_y = circle_b_y;

  if (clgo_inline_collide_point_bbox(outter_point_x, outter_point_y, bb_a))
  {
    return 1;
  }

  /* Left point - y stays the same. */
  outter_point_x = circle_b_x - (int)circle_b_radius;

  if (clgo_inline_collide_point_bbox(outter_point_x, outter_point_y, bb_a))
  {
    return 1;
  }

  /* Upper point. */
  outter_point_x = circle_b_x;
  outter_point_y = circle_b_y  - circle_b_radius;

  if (clgo_inline_collide_point_bbox(outter_point_x, outter_point_y, bb_a))
  {
    return 1;
  }

  /* Lower point - x stays the same.*/
  outter_point_y = circle_b_y + (int)circle_b_radius;

  return(clgo_inline_collide_point_bbox(outter_point_x, outter_point_y, bb_a));
}




int clgo_collide_bbox_bitmask(CLGO_BOUNDING_BOX *bb_a, CLGO_BIT_MASK *bitmask_b)
{
  /* This works very much the same way as bitmask -> bitmask testing works, 
  except now, we only need to test to see if one bit mask has a non-zero 
  inside the overlapping area. So, I copy paste! */

  CLGO_BOUNDING_BOX *bb_b; /* Bounding box for bit mask. */
  int bit_b_start_x; /* The starting x bit for bit mask b. */
  int bit_b_start_y; /* The starting x bit for bit mask b. */
  int overlapping_box_x; /* Overlapping box left-most x coordinate. */
  int overlapping_box_y; /* Overlapping box upper y coordinate. */
  int overlapping_box_width;  /* The width of the overlapping bounding box. */
  int overlapping_box_height; /* The height of the overlapping bounding box. */
  int line_x; /* Current x position while comparing bitmasks. */
  int line_y; /* Current y position while comparing bitmasks. */
  int bit_b; /* Bit from bitmask b. */

  bb_b = &(bitmask_b->bb);

  /* Both bounding boxes should overlap first. */
  if (clgo_inline_collide_bbox_bbox(bb_a, bb_b))
  {
    get_overlapping_rect(bb_a, bb_b, &overlapping_box_x, &overlapping_box_y, &overlapping_box_width, &overlapping_box_height);
    bit_b_start_x = overlapping_box_x - bb_b->x;
    bit_b_start_y = overlapping_box_y - bb_b->y; 

    /* Find each bit in the overlapping box and see if there are any
    overlapping 1 bits.  */
    for (line_y = 0; line_y < overlapping_box_height; line_y = line_y + 1)
    {
      for (line_x = 0; line_x < overlapping_box_width; line_x = line_x + 1)
      {
        bit_b = get_bit(bitmask_b, line_x + bit_b_start_x, line_y + bit_b_start_y);

        /* Hit! */
        if (bit_b)
        {
          return 1;
        }
      }
    }
  }

  return 0;
}




int clgo_collide_bcircle_bitmask(CLGO_BOUNDING_CIRCLE *circle_a, CLGO_BIT_MASK *bitmask_b)
{
  /* This test is easier than it sounds.  Create a bounding box around
  the circle.  Use the code from the previous bit masks to create an 
  overlapping rectangle.  Then test every bit inside that rectangle that
  is non-zero if it's inside the circle.  So once again, copy-paste! */

  CLGO_BOUNDING_BOX *bb_a; /* The bounding box around the circle. */
  CLGO_BOUNDING_BOX *bb_b; /* Bounding box for the bit mask. */
  int bit_b_start_x; /* The starting x bit for bit mask b. */
  int bit_b_start_y; /* The starting x bit for bit mask b. */
  int overlapping_box_x; /* Overlapping box left-most x coordinate. */
  int overlapping_box_y; /* Overlapping box upper y coordinate. */
  int overlapping_box_width;  /* The width of the overlapping bounding box. */
  int overlapping_box_height; /* The height of the overlapping bounding box. */
  int line_x; /* Current x position while comparing bitmasks. */
  int line_y; /* Current y position while comparing bitmasks. */
  int bit_b; /* Bit from bitmask b. */

  /* The bounding rectangle should be a little bigger than the circle. */
  bb_a = clgo_create_bbox((int)(circle_a->x - circle_a->radius - 1.0f), 
                          (int)(circle_a->y - circle_a->radius - 1.0f), 
                          (int)(circle_a->radius * 2.0f + 2.0f), 
                          (int)(circle_a->radius * 2.0f + 2.0f));

  if (bb_a == NULL)
  {
    clgo_error = CLGO_ALLOCATION_ERROR;
    return 0;
  }

  bb_b = &(bitmask_b->bb);

  /* Both bounding boxes should overlap first. */
  if (clgo_inline_collide_bbox_bbox(bb_a, bb_b))
  {
    get_overlapping_rect(bb_a, bb_b, &overlapping_box_x, &overlapping_box_y, &overlapping_box_width, &overlapping_box_height);
    bit_b_start_x = overlapping_box_x - bb_b->x;
    bit_b_start_y = overlapping_box_y - bb_b->y; 

    /* Find each bit in the overlapping box and see if there are any
    overlapping 1 bits.  */
    for (line_y = 0; line_y < overlapping_box_height; line_y = line_y + 1)
    {
      for (line_x = 0; line_x < overlapping_box_width; line_x = line_x + 1)
      {
        bit_b = get_bit(bitmask_b, line_x + bit_b_start_x, line_y + bit_b_start_y);

        /* Hit! */
        if (bit_b)
        {
          /* Test to see if the middle of the pixels are insize the circle. */
          if (clgo_collide_point_bcircle((float)(line_x + overlapping_box_x) + 0.5f, 
                                         (float)(line_y + overlapping_box_y) + 0.5f, 
                                         circle_a))
          {
            clgo_destroy_bbox(bb_a);
            return 1;
          }
        }
      }
    }
  }

  clgo_destroy_bbox(bb_a);
  return 0;
}





CLGO_BOUNDING_BOX *clgo_get_bitmask_bbox(CLGO_BIT_MASK *bitmask)
{
  return clgo_inline_get_bitmask_bbox(bitmask);
}




unsigned char clgo_get_bitmask_bit(CLGO_BIT_MASK *bitmask, int x, int y)
{
  int index; /* Index inside the bit mask. */

  /* Check for invalid x and y values. */
  index = y * bitmask->bb.width + x;
  if (index >= bitmask->size || index < 0)
  {
    clgo_error = CLGO_INVALID_PARAMS;
    return 0;
  }

  /* Grab the bit mask value. */
  return get_bit(bitmask, x, y);
}




int clgo_get_error(void)
{
  int error_copy; /* Copy of the current error message. */

  /* Copy the error message and reset the error message. */
  error_copy = clgo_error;
  clgo_error = CLGO_NO_ERROR;

  /* Return the previous error. */
  return error_copy;
}




CLGO_BIT_MASK *allocate_bitmask_memory(int x_pos, int y_pos, int width, int height)
{
  CLGO_BIT_MASK *new_bit_mask;

  /* Allocating the space needed for our bit mask. */
  new_bit_mask = (CLGO_BIT_MASK *)malloc(sizeof(CLGO_BIT_MASK));

  /* If there was an error in allocating space, return NULL. */
  if (new_bit_mask == NULL)
  {
    clgo_error = CLGO_ALLOCATION_ERROR;
    return NULL;
  }

  /* Allocate space for the bits themselves and set our default values. */
  new_bit_mask->bb.x = x_pos;
  new_bit_mask->bb.y = y_pos;
  new_bit_mask->bb.width = width;
  new_bit_mask->bb.height = height;
  new_bit_mask->h_flip = 0;
  new_bit_mask->v_flip = 0;
  new_bit_mask->size = width * height;
  new_bit_mask->entries = (unsigned char *)calloc(new_bit_mask->size, sizeof(unsigned char));

  /* Make sure the allocation worked OK. */
  if (new_bit_mask->entries == NULL)
  {
    clgo_error = CLGO_ALLOCATION_ERROR;
    free(new_bit_mask);
    return NULL;
  }

  return new_bit_mask;
}



void make_bit_mask(CLGO_BIT_MASK *fill_me, BITMAP *allegro_bitmap, int x_offset, int y_offset, int width, int height, int (*func)(int bitmap_x, int bitmap_y, int depth, unsigned long int pixel))
{
  unsigned char *bmp_row_ptr; /* Pointer to the start of a row in a bitmap. */
  unsigned long int pixel; /* Pixel Data. */
  int depth; /* Color depth of the bmp. */
  int depth_bytes; /* Color depth in bytes. */
  int x; /* Current x location in the bitmap. */
  int y; /* Current y location in the bitmap. */
  int bit_x; /* Current x location in the bit mask. */
  int bit_y; /* Current y location in the bit mask. */
  unsigned char *entries; /* Quick pointer to entries. */

  entries = fill_me->entries;

  depth = bitmap_color_depth(allegro_bitmap);

  /* Figure out how many bytes per pixel */
  if (depth == 15)
  {
    depth_bytes = 2;
  } else
  {
    depth_bytes = depth / 8;
  }

  /* Now read every pixel in every row. */
  bit_y = 0;
  for (y = y_offset; y < y_offset + height; y = y + 1)
  {
    bit_x = 0;
    bmp_row_ptr = (unsigned char *)bmp_read_line(allegro_bitmap, y);
    for (x = x_offset; x < x_offset + width; x = x + 1)
    {
      /* Copying the pixel data into a long int.  We have to make sure that
      only the pixel we want is copied into our data. */
      pixel = 0;

      /* Let's grab our pixel! */
      switch (depth)
      {
      case 8:
        pixel = bmp_read8((long int)bmp_row_ptr);
        break;
      case 15:
        pixel = bmp_read15((long int)bmp_row_ptr);
        break;
      case 16:
        pixel = bmp_read16((long int)bmp_row_ptr);
        break;
      case 24:
        pixel = bmp_read24((long int)bmp_row_ptr);
        break;
      case 32:
        pixel = bmp_read32((long int)bmp_row_ptr);
        break;
      }

      /* Call our functor to see if the pixel is to be masked or not. */
      /* entries[bit_y * width + bit_x] = (func)(x, y, depth, pixel); */
      *entries = (func)(x, y, depth, pixel);
      entries++;
      bmp_row_ptr = bmp_row_ptr + depth_bytes;
      bit_x = bit_x + 1;
    }
    bit_y = bit_y + 1;
  }
}



int clgo_standard_bitmask_func(int bitmap_x, int bitmap_y, int depth, unsigned long int pixel)
{
  /* Test each depth for the pixel mask color. */
  switch (depth)
  {
  case 32:
    return(pixel != MASK_COLOR_32);
  case 24:
    return(pixel != MASK_COLOR_24);
  case 16:
    return(pixel != MASK_COLOR_16);
  case 15:
    return(pixel != MASK_COLOR_15);
  case 8:
    return(pixel != MASK_COLOR_8);
  }
  return 0;
}

void get_overlapping_rect(CLGO_BOUNDING_BOX *bb_a, CLGO_BOUNDING_BOX *bb_b, int *x, int *y, int *width, int *height)
{
  /* The coordinates of the overlapping rect will be defined by the middle
  two coordinates between the highest and lowest coordinates of both
  rectangles projected onto an axis.  In other words, you know how we looked
  for the highest and lowest coordinates in the bounding box collision 
  function?  Now we want to find the two coordiantes that are between
  those two high and low ends. */

  /* For this function, I assume NULL testing has already occured since this
  is an internal function. */

  /* Find the 2nd to left-most x coordinate. */

  int bb_a_x; /* Bounding box a X coordinate. */
  int bb_a_y; /* Bounding box a Y coordinate. */
  int bb_a_width; /* Bounding box a width. */
  int bb_a_height; /* Bounding box a height. */
  int bb_b_x; /* Bounding box b X coordinate. */
  int bb_b_y; /* Bounding box b Y coordinate. */
  int bb_b_width; /* Bounding box b width. */
  int bb_b_height; /* Bounding box b height. */

  bb_a_x = bb_a->x;
  bb_a_y = bb_a->y;
  bb_a_width = bb_a->width;
  bb_a_height = bb_a->height;

  bb_b_x = bb_b->x;
  bb_b_y = bb_b->y;
  bb_b_width = bb_b->width;
  bb_b_height = bb_b->height;

  if (bb_a_x >= bb_b_x)
  {
    *x = bb_a_x;
  } else
  {
    *x = bb_b_x;
  }

  /* Find 2nd to right-most x coordinate. */
  if (bb_a_x + bb_a_width <= bb_b_x + bb_b_width)
  {
    *width = bb_a_x + bb_a_width - *x;
  } else
  {
    *width = bb_b_x + bb_b_width - *x;
  }

  /* Find the 2nd highest y coordinate. */
  if (bb_a_y >= bb_b_y)
  {
    *y = bb_a_y;
  } else
  {
    *y = bb_b_y;
  }

  /* Find 2nd to lowest y coordinate. */
  if (bb_a_y + bb_a_height <= bb_b_y + bb_b_height)
  {
    *height = bb_a_y + bb_a_height - *y;
  } else
  {
    *height = bb_b_y + bb_b_height - *y;
  }
}

int get_bit(CLGO_BIT_MASK *bit_mask, int x, int y)
{
  int bit; /* Resulting bit (1 or 0). */
  int index; /* Array index to check. */
  int h_flip; /* Horizontal flip. */
  int v_flip; /* Vertical flip. */

  h_flip = bit_mask->h_flip;
  v_flip = bit_mask->v_flip;

  if (!v_flip && !h_flip)
  {
    index = y * bit_mask->bb.width + x;
  } else if (!v_flip && h_flip)
  {
    index = y * bit_mask->bb.width + (bit_mask->bb.width - 1) - x;
  } else if (v_flip && !h_flip)
  {
    index = (bit_mask->bb.height - y - 1) * bit_mask->bb.width + x;
  } else if (v_flip && h_flip)
  {
    index = (bit_mask->bb.height - y - 1) * bit_mask->bb.width + (bit_mask->bb.width - 1) - x;
  }

  bit = bit_mask->entries[index];
  return bit; 
}
