/*

              __________  __    __    ________________  ____ 
             / ____/ __ \/ /   / /   / ____/ ____/ __ \/ __ \
            / /   / / / / /   / /   / __/ / / __/ /_/ / / / /
           / /___/ /_/ / /___/ /___/ /___/ /_/ / _, _/ /_/ / 
           \____/\____/_____/_____/_____/\____/_/ |_|\____/  
          
                 - Collision Detection for Allegro -


The collegro library is a collision detection library for use with allegro.

Currently, collegro supprts bounding boxes, bouding circles, and bit maps.  You
can test all the above against each other.  

You may use this library freely.  Feel free to change anything you wish.  
The only restriction is that if you plan to sell or otherwise profit off
of collegro without it being used in a program you made, let me know first.
You can leave a message on my website in the forums.

Also, feel free to suggest new features and report bugs on my website.

Thank you for using collegro,
- Kaitlyn Handelman

--------------------------------------------------------------------------------

Website: http://www.playingwithyarn.net
*/

/*! \file collegro.h
    \brief The collegro header file.
*/


#ifndef COLLEGRO_H
#define COLLEGRO_H

#ifndef ALLEGRO_H
#include <allegro.h>
#endif

#ifdef __cplusplus
extern "C" {
#endif

#define COLLEGRO_VERSION 0.82

/*! \def CLGO_NO_ERROR 
    No error has occured. 
*/
/*! \def CLGO_ALLOCATION_ERROR 
    A memory allocation error has occured. 
*/
/*! \def CLGO_INVALID_PARAMS 
    Invalid parameters were sent to a function, such as a NULL pointer. 
*/

 /** @defgroup error Error Messages
 *  Use the function clgo_get_error() to retrieve an error message. 
 *  @{
 */

#define CLGO_NO_ERROR 0 
#define CLGO_ALLOCATION_ERROR 1 
#define CLGO_INVALID_PARAMS 2 

/** @} */ 

typedef struct _CLGO_BOUNDING_BOX
{
  int x; /*!< \brief The upper left x-coordinate of the box. */ 
  int y; /*!< \brief The upper left y coordinate of the box. */
  int width; /*!< \brief The width of the box. */
  int height;  /*!< \brief The height of the box. */
} CLGO_BOUNDING_BOX;
/*!
 * \struct _CLGO_BOUNDING_BOX
 * A bounding box.  Use the typedef CLGO_BOUNDING_BOX to use.
 * \see CLGO_BOUNDING_BOX
 */

typedef struct _CLGO_BOUNDING_CIRCLE
{
  int x; /*!< \brief The center x coordinate of the bounding circle. */
  int y;  /*!< \brief The center y coordinate of the bounding circle. */
  float radius; /*!< \brief The radius of the circle. */
  float radius_squared; /*!< \brief Pre-computed square of the radius. */
} CLGO_BOUNDING_CIRCLE;
/*!
 * \struct _CLGO_BOUNDING_CIRCLE
 * A bounding circle.  Use the typedef CLGO_BOUNDING_CIRCLE to use.
 * \see CLGO_BOUNDING_CIRCLE
 */

typedef struct _CLGO_BIT_MASK
{
  CLGO_BOUNDING_BOX bb; /*!< \brief A bounding box describing the dimensions and position of the bit mask. */
  int h_flip; /*!< \brief Horizontal flip flag.  Set to 1 to enable horizontal flipping when reading. */
  int v_flip; /*!< \brief Vertical flip flag.  Set to 1 to enable vertical flipping when reading. */
  int size;  /*!< \brief Size of the bitmask in bytes. */
  unsigned char *entries; /*!< \brief All the bit mask values as an array. */
  unsigned int *bit_pack; /* \brief All the bits packed into an int for quick testing. */
  unsigned int *bit_pack_h_flip; /* \brief All the bits horizontally flipped packed into an int for quick testing. */
  unsigned int *bit_pack_v_flip; /* \brief All the bits vertically flipped packed into an int for quick testing. */
  unsigned int *bit_pack_vh_flip;/* \brief All the bits horizontally & vertically flipped packed into an int for quick testing. */
} CLGO_BIT_MASK;
/*!
 * \struct _CLGO_BIT_MASK
 * A bit mask struct.  Use the typedef CLGO_BIT_MASK to use.
 * \see CLGO_BIT_MASK
 */
 
 /** @defgroup data_types Data Types
 *  These are the data types available to collegro users.
 *  @{
 */
 
  /*!
 * \typedef CLGO_BOUNDING_BOX
 * A CLGO_BOUNDING_BOX represents a bounding box which is commonly
 * used in 2D collision detection algorithms.  The box represents the size
 * and x / y location of the bitmap or sprite.  Testing for two overlapping
 * bounding boxes is quick and easy.  However, this method lacks pixel per
 * pixel testing.
 */
 
 /*!
 * \typedef CLGO_BOUNDING_CIRCLE
 * A CLGO_BOUNDING_CIRCLE represents a bounding circle.  A bounding circle
 * is defined by two center coordinates and its radius.  Bounding circles
 * work well with round, circular objects.  Also, they are very good
 * if you want to test to see if any object comes within a certain range
 * of the center coordinates.  If you want to make sure something doesn't
 * come within 5 feet of your desired coordinates, use a bounding circle. 
*/
 
/*!
 * \typedef CLGO_BIT_MASK
 * A CLGO_BIT_MASK represents a bit mask, which can be used when you need pixel
 * perfect collision detection.  Given any bitmap, the pixels can be read
 * and asigned a value of 0 for non-colliding pixels and 1 for pixels that 
 * can collide.  This library, collegro, allows you to not only use bit masks
 * to test for collisions, but also to draw the bit masks using two colors to
 * another allegro BITMAP.  Bit masks are the slowest form of collision 
 * detection in collegro.  Use bounding boxes or circles when appropriate.
*/

/** @} */ 



/** @defgroup creation Creating Collision Objects
 *  These functions allocate memory and create the various collision objects.
 *  @{
 */

/*! Creates a bounding box from user-specific input.
 
  \param x_placement The upper-left x coordinate of the bounding box you want to make. 
  \param y_placement The upper-left y coordinate of the bounding box you want to make.
  \param width The width of the bounding box.
  \param height The height of the bounding box.
  
  \return A newly created CLGO_BOUNDING_BOX object.
 */
CLGO_BOUNDING_BOX *clgo_create_bbox(int x_placement, int y_placement, 
                                  int width, int height);


/*! Creates a bounding box, grabbing the width and height straight from the 
    allegro BITMAP.
    
    \param x_placement The upper-left x coordinate of the bounding box you want to make. 
    \param y_placement The upper-left y coordinate of the bounding box you want to make.
    \param allegro_bitmap A pointer to a valid allegro BITMAP structure.
    
    \return A newly created CLGO_BOUNDING_BOX object with the width and height
            taken from allegro_bitmap.
*/ 
CLGO_BOUNDING_BOX *clgo_bitmap_create_bbox(int x_placement, int y_placement, 
                                         BITMAP *allegro_bitmap);

/*! Creates a new bounding circle based on the parameters.
    
    \param center_x The center x coordinate for the circle.
    \param center_y The center y coordinate for the circle.
    \param radius The radius of the circle.  A radius is the length from
           the center of the circle to the outside edge.

  \return a newly created CLGO_BOUNDING_CIRCLE object centered at 
          (center_x, center_y) with the radius copied from the parameter of the 
          same name. 
*/                                      
CLGO_BOUNDING_CIRCLE *clgo_create_bcircle(int center_x, int center_y, float radius);

/*! Creates a bit mask from a portion of an allegro BITMAP.  This function
    sets a bit in the mask to the "transpaent" setting if the corresponding color
    of the pixel in the allegro_bitmap BITMAP is a standard allegro masked color.
    For example, in 8 bit modes, the bits in the bitmap that correspond to 
    all pixels with the index of 0 will be labelled as transparent.  In truecolor
    modes, the function searches for all pixels with a color of RGB(255, 0, 255)
    to set as transparent.  In 32 bit modes, it searches for RGBA(255, 0, 255, 0).
    
    \param x_placement The upper-left x location of the bit mask for 
                       collision-testing purposes.  This describes the
                        upper-left location of the internal bounding box and
                        does not alter any bits inside the mask.
    \param y_placement The upper-left y location of the bit mask for 
                       collision-testing purposes.  This describes the
                        upper-left location of the internal bounding box and
                        does not alter any bits inside the mask.
    \param x_offset Number of pixels to the right from the start of BITMAP to 
                    begin making the bit mask from.  This can be used to grab
                    a portion of the BITMAP.
    \param y_offset Number of pixels down from the top of the BITMAP to begin
                    making the bit mask from.  This can be used to grab
                    a portion of the BITMAP.
    \param width The width of the bitmap.  Once again, this can be used to 
                 make bit masks that don't extend all the way across the BITMAP.
    \param height The height of the bitmap.  Set to select all or some of
                  the height of the bitmap.
    \param allegro_bitmap The BITMAP to create the bitmap from.  The function
                  will read through every pixel and make a bit mask based on
                  which pixels are defined as transparent by allegro. 
                  
    \return A newly created CLGO_BIT_MASK object that represents all of, or a 
            portion of an allegro BITMAP using the standard allegro color masks
            to determine what pixels are transparent, and what are not.   
*/
CLGO_BIT_MASK *clgo_create_bitmask_ex(int x_placement, int y_placement, 
                                      int x_offset, int y_offset, 
                                      int width, int height, 
                                      BITMAP *allegro_bitmap);

/*! Creates a bit mask that encompasses an entire allegro BITMAP.  Unlike 
    clgo_create_bitmask_ex(), this function assumes you want to make a
    standard bit mask spreading across the entire BITMAP.
    
    \param x_placement The upper-left x location of the bit mask for 
                       collision-testing purposes.  This describes the
                        upper-left location of the internal bounding box and
                        does not alter any bits inside the mask.
    \param y_placement The upper-left y location of the bit mask for 
                       collision-testing purposes.  This describes the
                        upper-left location of the internal bounding box and
                        does not alter any bits inside the mask.
    \param allegro_bitmap The BITMAP to make the bit mask from. 
    
    \return A newly created CLGO_BIT_MASK that represents all transparent
            and non-transparent pixels within allegro_bitmap.
    
    \see clgo_create_bitmask_ex
*/  
CLGO_BIT_MASK *clgo_create_bitmask(int x_placement, int y_placement, 
                                   BITMAP *allegro_bitmap);

/*! Creates a custom bitmask from a portion of or all of an allegro BITMAP. This
    function works almost the same as clgo_create_bitmask_ex(), except that
    the user can supply their own function to determine what is or is not
    a transparent pixel.  
    
    \param x_placement The upper-left x location of the bit mask for 
                       collision-testing purposes.  This describes the
                        upper-left location of the internal bounding box and
                        does not alter any bits inside the mask.
    \param y_placement The upper-left y location of the bit mask for 
                       collision-testing purposes.  This describes the
                        upper-left location of the internal bounding box and
                        does not alter any bits inside the mask.
    \param x_offset Number of pixels to the right from the start of BITMAP to 
                    begin making the bit mask from.  This can be used to grab
                    a portion of the BITMAP.
    \param y_offset Number of pixels down from the top of the BITMAP to begin
                    making the bit mask from.  This can be used to grab
                    a portion of the BITMAP.
    \param width The width of the bitmap.  Once again, this can be used to 
                 make bit masks that don't extend all the way across the BITMAP.
    \param height The height of the bitmap.  Set to select all or some of
                  the height of the bitmap.
    \param allegro_bitmap The BITMAP to create the bitmap from.  The function
                  will read through every pixel and make a bit mask based on
                  which pixels are defined as transparent by allegro. 
    \param func A pointer to a function which will return 0 for a transparent
                pixel or a 1 for a solid pixel.  The function can recieve the 
                internal bitmap x and y coordinates as well as the color depth
                of the image (8, 16, 24, etc...) and the color of the pixel.
                
    \return A newly created CLGO_BIT_MASK object that used a custom bit mask
            function to determine which pixels are transparent and which are
            solid.
            
    \see clgo_create_bitmask_ex
    \see clgo_standard_bitmask_func
*/
CLGO_BIT_MASK *clgo_create_custom_bitmask_ex(int x_placement, int y_placement,
                                             int x_offset, int y_offset, 
                                             int width, 
                                             int height, 
                                             BITMAP *allegro_bitmap, 
                                             int (*func)(int bitmap_x, 
                                             int bitmap_y, 
                                             int depth, 
                                             unsigned long int pixel));

/*!  Creates a custom bit mask using all the pixels from an allegro BITMAP.

     \param x_placement The upper-left x location of the bit mask for 
                        collision-testing purposes.  This describes the
                        upper-left location of the internal bounding box and
                        does not alter any bits inside the mask.
     \param y_placement The upper-left y location of the bit mask for 
                        collision-testing purposes.  This describes the
                        upper-left location of the internal bounding box and
                        does not alter any bits inside the mask.
     \param allegro_bitmap The allegro BITMAP to create the bit mask from.
     \param func A custom function pointer for creating bit masks.  The
            function recieves data about a pixel that has been read
            from the allegro BITMAP (its x and y postion, bit depth, and
            pixel color), and returns a 0 if the pixel should 
            be treated as transparent or a 1 if it should be treated as a
            solid. 
            
     \return A newly created CLGO_BIT_MASK object that contains a custom 
             bit mask for an allegro BITMAP.
             
     \see clgo_standard_bitmask_func        
*/
CLGO_BIT_MASK *clgo_create_custom_bitmask(int x_placement, int y_placement,
                                          BITMAP *allegro_bitmap, 
                                          int (*func)(int bitmap_x, 
                                          int bitmap_y, int depth, 
                                          unsigned long int pixel));


/** @} */ 

/** @defgroup update Updating Collision Objects
 *  These functions allow you to update the data within collegro objects
    without having to destroy and re-allocate memory every time you want to
    change some small thing.
 *  @{
 */

/*!  Updates an already created bounding box with new x, y coordinates and a new 
     width and height.
     
     \param bb The bounding box to update. 
     \param new_x_placement The new upper-left x coordinate of the bounding box 
            you wish to set it to. 
     \param new_y_placement The new upper-left y coordinate of the bounding box
            you wish to set it to.
     \param new_width The width you wish to set the bounding box to. 
     \param new_height The height you wish to set the bounding box to.
*/
void clgo_update_bbox_ex(CLGO_BOUNDING_BOX *bb, int new_x_placement, 
                       int new_y_placement, int new_width, 
                       int new_height);

/*! Updates the upper-left coordinates of an already created bounding box.

    \param bb The bounding box object to update. 
    \param new_x_placement The new upper-left x coordinate you wish to set the
           bounding box to. 
    \param new_y_placement The new upper-left y coordinate you wish to set the
           bounding box to.
*/
void clgo_update_bbox(CLGO_BOUNDING_BOX *bb, int new_x_placement, 
                      int new_y_placement);

/*! Updates a bounding circle that has already been created with new x and
    y center locations as well as a new radius.  Set new_radius to 0.0f or less
    to keep the previous radius value for the bounding circle.
    
    \param circle The bounding circle to update. 
    \param new_center_x The x coordinate for the center point of the circle.
    \param new_center_y The y coordinate for the center point of the circle.
    \param new_radius If new_radius is set to 0.0f or less, the internal value 
                      for the radius of the bounding circle will not 
                      change.  However, if you set new_radius to a value above
                      0, the radius of the bounding circle will become that
                      of new_radius.
*/

void clgo_update_bcircle_ex(CLGO_BOUNDING_CIRCLE *circle, int new_center_x, 
                           int new_center_y, float new_radius);

/*! Updates the location of a bounding circle that has already been created. 
    
    \param circle A pointer to the bounding circle to update.
    \param new_center_x The x coordinate for the center point of the circle.
    \param new_center_y The y coordinate for the center point of the circle.
*/
void clgo_update_bcircle(CLGO_BOUNDING_CIRCLE *circle, int new_center_x, 
                        int new_center_y);

/*! Updates an already created bit mask's location. 

   \param bitmask The bit mask that you wish to update.
   \param new_x_placement The new upper-left x coordinate for the bit mask. 
   \param new_y_placement The new upper-left y coordinate for the bit mask.
*/                        
void clgo_update_bitmask(CLGO_BIT_MASK *bitmask, int new_x_placement, 
                         int new_y_placement);

/*! Sets the internal flags to swap the horizontal and / or vertical 
bit mask locations. This function will not actually write over 
the bit mask data, but rather set flags that allow collegro to read the
bit mask backwards and / or upside down. 

    \param bitmask The bit mask you wish to set the internal flipping flags to. 
    \param h_flip Set to 1 to set the horizontal flipping flag.  Otherwise, set 
                  to 0 to turn off horizontal (left to right) flipping.
    \param v_flip Set to 1 to set the vertical flipping flag.  Otherwise, set 
                  to 0 to turn off vertical (upside down) flipping.
*/
void clgo_flip_bitmask(CLGO_BIT_MASK *bitmask, int h_flip, int v_flip);


/** @} */ 

/** @defgroup destroy Destroying Collision Objects
 *  These functions allow you to destroy and free memory from collision 
 *  objects you have previously allocated before.  Call these functions when
 *  you have finished using a collision object.
 *  @{
 */

/*! Destroys a bounding box.  Call when you are finished with the current 
bounding box and wish to destroy the data. 

    \param destroy_me The bounding box you wish to destroy.
*/
void clgo_destroy_bbox(CLGO_BOUNDING_BOX *destroy_me);

/*! Destroys a bounding circle.  Call when you are finished with the current 
    bounding circle and wish to destroy the data. 

    \param destroy_me The bounding circle you wish to destroy. 
*/
void clgo_destroy_bcircle(CLGO_BOUNDING_CIRCLE *destroy_me);

/*! Destroys a bit mask.  Call when you are finished with the current bit mask 
    and wish to destroy the data. 

    \param destroy_me The bit mask you wish to destroy.
*/
void clgo_destroy_bitmask(CLGO_BIT_MASK *destroy_me);

/** @} */ 

/** @defgroup drawing Drawing Collision Objects
 *  There are times when you may want to draw the various representations of your 
 *  collision objects to an allegro BITMAP.  One such example would be to test 
 *  to make sure your collision ojbects are in the right place and of the right 
 *  size.  Also, you may want to view a bit mask to make sure your it was 
 *  created without error. Another good use for drawing bit maps is create bit 
 *  mapped effects like shadows.    
 *  @{
 */

/*! Draws the bounding box onto an allegro BITMAP.  If you're not getting collisions
    when you should be getting a collision, chances are that your bounding box
    is in the wrong place.  You can use this function to draw a bounding box
    at its current location.
 
    \param bb The bounding box to draw.
    \param destination The BITMAP to draw the box onto.
    \param color The color of the box.  In 8 bit mode, this would be a color index;
                 however, most users will want to use the allegro function, 
                 makecol(int red, int green, int blue), in true color modes.   
*/
void clgo_draw_bbox(CLGO_BOUNDING_BOX *bb, BITMAP *destination, int color);

/*! Inline version of clgo_draw_bbox()
    \param bb The bounding box (CLGO_BOUNDING_BOX *) to draw.
    \param destination The BITMAP to draw the box onto.
    \param color The color of the box.  In 8 bit mode, this would be a color index;
                 however, most users will want to use the allegro function, 
                 makecol(int red, int green, int blue), in true color modes.  
    \see clgo_draw_bbox
*/
#define clgo_inline_draw_bbox(bb,destination,color) rect(destination, \
                              bb->x, bb->y, bb->x + bb->width - 1, bb->y + \
                              bb->width - 1, color)

/*! Draws the bounding circle onto an allegro BITMAP.  If you're not getting collisions
    when you should be getting a collision, chances are that your bounding circle
    is in the wrong place or the wrong size.  You can use this function to draw a bounding 
    circle at its current location.
 
    \param bcircle The bounding circle to draw.
    \param destination The BITMAP to draw the circle onto.
    \param color The color of the circle.  In 8 bit mode, this would be a color index;
                 however, most users will want to use the allegro function, 
                 makecol(int red, int green, int blue), in true color modes.   
*/
void clgo_draw_bcircle(CLGO_BOUNDING_CIRCLE *bcircle, BITMAP *destination, int color);


/*! Inline version of clgo_draw_bcircle()
    \param bcircle The bounding circle (CLGO_BOUNDING_CIRCLE *) to draw. 
    \param destination The BITMAP to draw the circle onto.
    \param color The color (int) of the circle.  In 8 bit mode, this would be a 
                 color index; however, most users will want to use the allegro 
                 function, makecol(int red, int green, int blue), in true color 
                 modes.   
    \see clgo_draw_bcircle
*/
#define clgo_inline_draw_bcircle(bcircle,destination,color) \
        circle(destination, bcircle->x, bcircle->y, (int)bcircle->radius, \
        color)

/*! Draws the bounding box around a bit mask. Bit masks contain an internal 
    bounding box which it uses to quickly check for a collision with another 
    object before going into the slow bit per bit testing code.  The bounding 
    box also defines the x and y position of the bit mask as well as stores the 
    the width and height of the bit mask.  You can display this bounding box on
    the screen to help debug any problems you may be having. 
  
    \param bitmask The bit mask you wish to get the bounding box data to draw 
                   from.
    \param destination The allegro BITMAP you want to draw the bounding box 
                   into.
    \param color The color of the box.  In 8 bit mode, this would be a color index;
                 however, most users will want to use the allegro function, 
                 makecol(int red, int green, int blue), in true color modes. 
    
    \see clgo_draw_bbox
*/
void clgo_draw_bitmask_bbox(CLGO_BIT_MASK *bitmask, BITMAP *destination, int color);

/*! Inline version of clgo_draw_bitmask_bbox()

    \param bitmask The bit mask (CLGO_BIT_MASK *) you wish to get the bounding 
                   box data to draw from.
    \param destination The allegro BITMAP you want to draw the bounding box 
                       into.
    \param color The color (int) of the box.  In 8 bit mode, this would be a 
                 color index; however, most users will want to use the allegro 
                 function,  makecol(int red, int green, int blue), in true 
                 color modes. 
                 
    \see clgo_draw_bitmask_bbox
*/
#define clgo_inline_draw_bitmask_bbox(bitmask, destination, color) \
        rect(destination, bitmask->bb.x, bitmask->bb.y, \
        bitmask->bb.x + bitmask->bb.width - 1, \
        bitmask->bb.y + bitmask->bb.width - 1, color)
             
/*lgo_draw_bbox(&(bitmask->bb) , destination, color) */

/*! Draws a bit mask to a BITMAP.  All bits marked to be transparent will
    be drawn in one color, and all the bits marked to be solid will be drawn
    in another color.  This is not only useful for debugging (you can create
    your own bit-masking code), but also for special effects.  Drawing to a
    BITMAP can be a bit slow.  If you plan to draw the bit mask for special 
    effects, you might want to draw it to a BITMAP sprite, and blit the
    sprite later on.
    
    \param bitmask The bit mask you wish to draw.
    \param destination The allegro BITMAP you want to draw it to.
    \param x The upper-left x coordinate you want to draw the bit mask at in the
             destination BITMAP.
    \param y The upper-left y coordinate you want to draw the bit mask at in the
             destination BITMAP.
    \param color_trans The allegro color you wish to draw transparent bits with.
    \param color_solid The allegro color you wish to draw solid bits with.
*/  
void clgo_draw_bitmask(CLGO_BIT_MASK *bitmask, BITMAP *destination, 
                       int x, int y, int color_trans, int color_solid);
 

/** @} */ 

/** @defgroup quick Quick Collision Detections
 *  These function give you a simple way of doing collision detection without
 *  creating two collision detection objects. Many functions allow you to use 
 *  an allegro BITMAP to act as a bounding box, for example.
 *  @{
 */

/*! Puts collision boxes around the two allegro BITMAPs and checks to see if
    they collide. 

    \param bitmap_a The first allegro BITMAP to check.
    \param blit_x_a The upper-left x coordinate that you display the first 
                    BITMAP at.
    \param blit_y_a The upper-left y coordinate that you display the first 
                    BITMAP at.
    \param bitmap_b The second allegro BITMAP to check.
    \param blit_x_b The upper-left x coordinate that you display the second 
                    BITMAP at.
    \param blit_y_b The upper-left y coordinate that you display the second 
                    BITMAP at.
                    
    \return 1 if the two BITMAPs collide.  0 if they do not collide.
*/
int clgo_collide_bitmaps(BITMAP *bitmap_a, int blit_x_a, int blit_y_a, 
                         BITMAP *bitmap_b, int blit_x_b, int blit_y_b);

/*! Automatically creates a collision box around an allegro BITMAP and tests if 
    it collides with a bounding circle. 

    \param circle_a The bounding circle to check.
    \param bitmap_b The allegro BITMAP to check.
    \param blit_x_b The upper-left x coordinate that you display the 
                    BITMAP at.
    \param blit_y_b The upper-left y coordinate that you display the 
                    BITMAP at.
                    
    \return 1 if the bounding circle and a bounding box around the BITMAP 
            collide.  0 if they do not collide.
*/
int clgo_collide_bcircle_bitmap(CLGO_BOUNDING_CIRCLE *circle_a,
                                BITMAP *bitmap_b, int blit_x_b, int blit_y_b);

/*! Checks to see if the raw values of two bounding boxes collide. 

    \param x1 Upper-left x coordinate of the first box.
    \param y1 Upper-left y coordinate of the first box.
    \param width1 The width of the first box.
    \param height1 The height of the first box.
    \param x2 Upper-left x coordinate of the second box.
    \param y2 Upper-left y coordinate of the second box.
    \param width2 The width of the second box.
    \param height2 The height of the second box.
    
    \return 1 If there's a collision, 0 if they do not collide.
*/
int clgo_collide_bbox_coords(int x1, int y1, int width1, int height1, int x2, 
                             int y2, int width2, int height2);

/*! Inline version of clgo_collide_bbox_coords().

    \param x1 Upper-left x coordinate (int) of the first box.
    \param y1 Upper-left y coordinate (int) of the first box.
    \param width1 The width (int) of the first box.
    \param height1 The height (int) of the first box.
    \param x2 Upper-left x coordinate (int) of the second box.
    \param y2 Upper-left y coordinate (int) of the second box.
    \param width2 The width (int) of the second box.
    \param height2 The height (int) of the second box.
    
    \see clgo_collide_bbox_coords
*/
#define clgo_inline_collide_bbox_coords(x1, y1, width1, height1, x2, y2, width2, height2) \
  (!( (x1 >= x2 + width2) || (x2 >= x1 + width1) || (y1 >= y2 + height2) \
  || (y2 >= y1 + height1) ))

/** @} */ 

/** @defgroup collision Detecting Collisions
 *  The following functions will return 1 if a collision occured, otherwise
 *  they will return 0.
 *  @{
 */

/*! Tests to see if a point indicated by (x, y) is inside a bounding box.

    \param x The x coordinate to check.
    \param y The y coordinate to check.
    \param bounding_box The bounding box to check if the point (x, y) is inside.
    
    \return 1 if point is inside bounding box.  0 if it is outside the bounding
            box. 
*/   
int clgo_collide_point_bbox(int x, int y, CLGO_BOUNDING_BOX *bounding_box); 

/*! Inline version of clgo_collide_point_bbox()

  \param x1 The x (int) coordinate to check.
  \param y1 The y (int) coordinate to check.
  \param bounding_box The bounding box (CLGO_BOUNDING_BOX *) to check if the 
                      point (x, y) is inside.
                      
  \return 1 if point is inside bounding box.  0 if it is outside the bounding
            box. 
*/
#define clgo_inline_collide_point_bbox(x1, y1, bounding_box) \
        (((x1) >= (bounding_box)->x && (x1) < (bounding_box)->x + \
        (bounding_box)->width) && ((y1) >= (bounding_box)->y && (y1) < \
        (bounding_box)->y + (bounding_box)->height)) 

/*! Tests to see if a point indicated by (x, y) is inside the bounding circle. 
    Because bounding circles use floats for their radius, the coordinates are
    floats, unlike clgo_draw_bitmask().
    
    \param x The x coordinate to check.
    \param y The y coordinate to check.
    \param bounding_circle The bounding circle to check if the point (x, y) 
            is inside.
    
    \return 1 if point is inside bounding cirlce.  0 if it is outside the 
              bounding circle. 
*/ 
int clgo_collide_point_bcircle(float x, float y, 
                               CLGO_BOUNDING_CIRCLE *bounding_circle);

/*! Inline version of clgo_collid_point_bcircle().

    \param x1 The x (int) coordinate to check.
    \param y1 The y (int) coordinate to check.
    \param bounding_circle The bounding circle (CLGO_BOUNDING CIRCLE *) to 
                           check if the point (x, y) is inside.
    
    \return 1 if point is inside bounding cirlce.  0 if it is outside the 
              bounding circle. 
*/                             
#define clgo_inline_collide_point_bcircle(x1, y1, bounding_circle) \
((((((bounding_circle)->x - (x1)) * ((bounding_circle)->x - (x1))) + \
(((bounding_circle)->y - (y1)) * ((bounding_circle)->y - (y1)))) <= \
((bounding_circle)->radius_squared)))

/*! Tests to see if a bounding box and another bounding box collide.

    \param bb_a The first bounding box. 
    \param bb_b The second bounding box.
    
    \return 1 if the bounding boxes collide.  0 if they do not collide.
*/ 
int clgo_collide_bbox_bbox(CLGO_BOUNDING_BOX *bb_a, CLGO_BOUNDING_BOX *bb_b);   

/*! Inline version of clgo_collide_bbox_bbox()
    \param bb_a The first bounding box (CLGO_BOUNDING_BOX *). 
    \param bb_b The second bounding box (CLGO_BOUNDING_BOX *).
    
    \return 1 if the bounding boxes collide.  0 if they do not collide.
*/
#define clgo_inline_collide_bbox_bbox(bb_a, bb_b) \
        clgo_inline_collide_bbox_coords(bb_a->x, bb_a->y, bb_a->width, \
                                        bb_a->height, bb_b->x, bb_b->y, \
                                        bb_b->width, bb_b->height)

/*! Tests two bounding boxes for a collision.  If there's a collision, it'll 
    write data about the resulting overlapping rectangle created from the area
    both bounding boxes share in the overlapping_rect* variables. 
    
    \param bb_a The first bounding box. 
    \param bb_b The second bounding box.
    \param overlapping_rect_x A pointer to an integer to write the upper-left x
                              coordinate of the resulting overlapping rectangle 
                              to.
    \param overlapping_rect_y A pointer to an integer to write the upper-left y
                              coordinate of the resulting overlapping rectangle 
                              to.
    \param overlapping_rect_width A pointer to an integer to write the width
                                  of the resulting overlapping rectangle to.
    \param overlapping_rect_height A pointer to an integer to write the height
                                   of the resulting overlapping rectangle to.
*/  
int clgo_collide_bbox_bbox_ex(CLGO_BOUNDING_BOX *bb_a, CLGO_BOUNDING_BOX *bb_b, 
                        int *overlapping_rect_x, int *overlapping_rect_y,
                        int *overlapping_rect_width, 
                        int *overlapping_rect_height);  

/*! Tests to see if two bounding circles collide.
    
    \param circle_a The first circle.
    \param circle_b The second circle.
    
    \return 1 if the bounding circles collide.  0 if they do not collide.
*/
int clgo_collide_bcircle_bcircle(CLGO_BOUNDING_CIRCLE *circle_a, 
                                 CLGO_BOUNDING_CIRCLE *circle_b);

/*! Inline version of clgo_collide_bcircle_bcircle()
    \param circle_a The first circle (CLGO_BOUNDING_CIRCLE *).
    \param circle_b The second circle CLGO_BOUNDING_CIRCLE *).
    
    \return 1 if the bounding circles collide.  0 if they do not collide.
*/                               
#define clgo_inline_collide_bcircle_bcircle(circle_a, circle_b) \
        (((circle_a->radius + circle_b->radius) * (circle_a->radius + \
        circle_b->radius)) >= (float)(((circle_b->x - circle_a->x) * \
        (circle_b->x - circle_a->x)) + ((circle_b->y - circle_a->y) * \
        (circle_b->y - circle_a->y))))
                            
/*! Tests to see if two bit masks collide.  In other words, it checks for 
    overlapping pixels between the two bit masks.

    \param bitmask_a The first bit mask.
    \param bitmask_b The second bit mask.
    
    \return 1 if the bit masks collide.  0 if they do not collide.    
*/
int clgo_collide_bitmask_bitmask(CLGO_BIT_MASK *bitmask_a,
                                 CLGO_BIT_MASK *bitmask_b);
                                 
/*! Tests to see if a bounding box and a bounding circle collide.

    \param bb_a The bounding box to check.
    \param circle_b The bounding circle to check.
    
    \return 1 if the objects collide. 0 if they do not collide.
*/
int clgo_collide_bbox_bcircle(CLGO_BOUNDING_BOX *bb_a, 
                            CLGO_BOUNDING_CIRCLE *circle_b);

/*! Tests to see if a bounding box and a bit mask collide.  In other words, it 
    checks to see if there any solid pixels inside the bounding box.

    \param bb_a The bounding box to check.
    \param bitmask_b The bit mask to check.
    
    \return 1 if there are solid pixels inside the bounding box. 
            0 if there was no solid pixels inside the bounding box.
*/
int clgo_collide_bbox_bitmask(CLGO_BOUNDING_BOX *bb_a, 
                            CLGO_BIT_MASK *bitmask_b);

/*! Tests to see if a bounding circle and a bit mask collide.  In other words, 
    it tests to see if any solid pixels inside the bit mask are inside the
    circle.

    \param circle_a The bounding circle to check.
    \param bitmask_b The bit mask to check.
    
    \return 1 if there are solid pixels inside the circle. 
            0 if there are no solid pixels inside the circle.
*/
int clgo_collide_bcircle_bitmask(CLGO_BOUNDING_CIRCLE *circle_a, 
                                 CLGO_BIT_MASK *bitmask_b);

/** @} */ 



/** @defgroup help Helper Functions
 *  These functions can be helpful to collegro users, even if they don't 
 *  belong to any of the other sub-catagories.
 *  @{
 */

/*! Returns a pointer to a bit mask's internal bounding box. You can alter
    the x and y coordinates of the bounding box to move the bit mask.  
    However, do not alter the width or height values of the bounding box. 

    \param bitmask The bitmask to get the pointer from.
    
    \return A pointer to the bit mask's internal bounding box.     
*/
CLGO_BOUNDING_BOX *clgo_get_bitmask_bbox(CLGO_BIT_MASK *bitmask);

/*! Inline version of clgo_get_bitmask_bbox().

    \param bitmask The bitmask (CLGO_BIT_MASK *) to get the pointer from.
    \return The internal CLGO_BOUNDING BOX pointer of the bit mask. 

    \see clgo_get_bitmask_bbox
*/
#define clgo_inline_get_bitmask_bbox(bitmask) &(bitmask->bb)

/*! Returns the bit mask bit at (x,y) inside the bit mask. In other words, 
    you can check the value of a single pixel inside the bit map. 
    
    \param bitmask The bitmask to grab the value from. 
    \param x The x pixel location within the bit mask to check.  Valid x values
             range from 0 to the bit mask's width - 1.
    \param y The y pixel location within the bit mask to check.  Valid y values
             range from 0 to the bit mask's height - 1.
             
    \return A bit mask bit representing a pixel at x,y.  0 means it's 
            transparent, 1 means it's solid.
*/   
unsigned char clgo_get_bitmask_bit(CLGO_BIT_MASK *bitmask, int x, int y); 

/*! Checks for the latest error message.  After calling this function, the
latest error message is deleted.  If the function returns CLGO_NO_ERROR or 0, 
no error has occured. 

    \return The last error to occur.
    \see error
*/
int clgo_get_error(void);

/*! A standard masking function for making bit masks that tests each color with 
    the allegro predefined transparent pixel values. If you are writing your
    own bit masking function, you may find this useful.
    
    \param bitmap_x The x location within an allegro BITMAP.
    \param bitmap_y The y location within an allegro BITMAP.
    \param depth The bit depth of the allegro BITMAP.
    \param pixel The color of the pixel a (x, y).
    
    \return 1 if the pixel color is not transparent (AKA solid bit).  0 if the 
              pixel color is transparent. 
    
*/
int clgo_standard_bitmask_func(int bitmap_x, int bitmap_y, int depth, 
                               unsigned long int pixel);


/** @} */ 

#ifdef __cplusplus
}
#endif

#endif
