/****************************************************
 * Project:   Glizda Game
 * FILE:      glizda.cc
 * Author:    Andrew Deren
 * Date:      1/2/98
 * Length:    2159
 ****************************************************/

#include <ctype.h>
#include "agui.h"
#include "glizda.h"

#define MAX_GLIZDA 150
#define DIR_UP 1
#define DIR_DOWN 2
#define DIR_LEFT 3
#define DIR_RIGHT 4

#define MAX_MUSICS 15
int musics[MAX_MUSICS];

HISCORE_INFO hi_scores[MAX_HIGHSCORES];
int glizda_timer = 0;
int global_timer = 0;
int time_for_level;
BITMAP *mapa;
BITMAP *score_bitmap;
MAP game_map;
MAP loaded_map;
int game_on;
int score;
int level_finished;
int current_level;
int animation;
// used for playing custom map
int custom_map_speed = 5;
int custom_map_time = 60;

// high scores for this episod currently displayed
char high_scores_view_name[100];
int high_scores_view_place = -1;

int draw_background;    // used when scrolling the map at the beginning of
                        // each level
int redraw_score_box;
int show_fps = FALSE;   // display fps?
// used to show fps
int frames_per_second;

// used for showing time
int start_second;
int current_second;

// used for slow machines
int show_background = TRUE;
int show_fog = TRUE;

// music and sounds vars
int music_volume = 8;
int sound_volume = 8;

int mask[18][18];
dialog *play_dialog;

char cheat_code[11];
int current_cheat_char;

dialog *game_dialog;

void init_mask(void)
{
   int i, j;
   for (i=0; i<18; i++)
     for (j=0; j<18; j++)
        mask[i][j] = 1;
}

void draw_mask(void)
{
   int i,j;
   for (i=0; i<18; i++) {
      for (j=0; j<18; j++) {
         if (mask[i][j])
            draw_compiled_sprite(mapa, (COMPILED_SPRITE*)main_data[MASK].dat, i*24, j*24);
      }
   }
}

void start_music(void)
{
   int i;
   if (!music_on) return;
   set_volume(25*sound_volume, 25*music_volume);
   for (i=0; i<MAX_MUSICS; i++)
      musics[i] = FALSE;
   i = random() % MAX_MUSICS;
   musics[i] = TRUE;
   play_midi((MIDI*)music_data[i].dat, FALSE);
}
void check_music(void)
{
   if (!music_on) return;
   if (midi_pos >= 0) return;
   int temp = random() % MAX_MUSICS;
   int start_temp = temp;
   while (1) {
      if (musics[temp] == FALSE) {
         play_midi((MIDI*)music_data[temp].dat, FALSE);
         musics[temp] = TRUE;
         return;
      }
      temp++;
      if (temp == MAX_MUSICS) temp = 0;
      if (temp == start_temp) {
         start_music();
         return;
      }
   }
}
void stop_music(void)
{
   if (!music_on) return;
   stop_midi();
}

void clear_point_mask(int x, int y)
{
   mask[x][y] = 0;
   // clear at left
   if (x-1 >= 0)
      mask[x-1][y] = 0;
   if (x+1 < 18)
      mask[x+1][y] = 0;
   if (y-1 >= 0)
      mask[x][y-1] = 0;
   if (y+1 < 18)
      mask[x][y+1] = 0;
   //upper left corner
   if (x-1 >= 0 && y-1 >= 0)
      mask[x-1][y-1] = 0;
   //upper right
   if (x+1 < 18 && y-1 >= 0)
      mask[x+1][y-1] = 0;
   //lower left
   if (x-1 >= 0 && y+1 < 18)
      mask[x-1][y+1] = 0;
   //lower right
   if (x+1 >= 0 && y+1 < 18)
      mask[x+1][y+1] = 0;
}

void draw_map(void);
int HighScoresProc(void);

void play_sound(int datafile_index)
{
    if (!sound_on) return;
    play_sample((SAMPLE*)sounds_data[datafile_index].dat, sound_volume*25, 128, 1000, FALSE);
}


class SpecialEffects {
protected:
   int x, y; // location of the effect
   int start_time;       // time the effect started
   int frames;         // how many frames the effects lasts (-1 forever)
   int duration;           // time between each execution of effect
public:
   SpecialEffects(int x1, int y1) { x = x1; y = y1; start_time = global_timer;}
   virtual ~SpecialEffects(void) {}
   // do effect returns TRUE if the effect still active
   // and false if effect is done
   virtual int do_effect(void) {return FALSE;}
};

// effect when the wall is destroyed by the bomb
class Effect1 : public SpecialEffects {
protected:
   int start_frame;
   int limit;
public:
   Effect1(int sf, int x1, int y1, int l=4);
   virtual int do_effect(void);
};

Effect1::Effect1(int sf, int x1, int y1, int l=4)
        :SpecialEffects(x1, y1)
{
   frames = 0;
   duration = 15;
   start_frame = sf;
   limit = l;
}

int Effect1::do_effect(void)
{
   int time = global_timer - start_time;
   // check if time for the effect to change
   if (time > duration) {
      frames++;
      // set beginning of this event
      start_time = global_timer;
   }
   if (frames >= limit) return FALSE;
   draw_sprite(mapa, (BITMAP*)effects_data[start_frame+frames].dat, x*24, y*24);
   return TRUE;
}

// effect to display some integer value
class Effect2 : public SpecialEffects {
protected:
   int value;
public:
   Effect2(int x1, int y1, int v);
   virtual int do_effect(void);
};

Effect2::Effect2(int x1, int y1, int v)
        :SpecialEffects(x1, y1)
{
   frames = y1;
   duration = 10;
   value = v;
}

int Effect2::do_effect(void)
{
   int time = global_timer - start_time;
   // check if time for the effect to change
   if (time > duration) {
      frames--;
      // set beginning of this event
      start_time = global_timer;
   }
   if (frames < 0) return FALSE;

   text_mode(-1);
   textprintf(mapa, font, x*24, frames*24, 15, "%4d", value);
   return TRUE;
}


// effect to display a some text scrolling up to top
class Effect3 : public SpecialEffects {
protected:
   char *text;
public:
   Effect3(int x1, int y1, char *t);
   virtual int do_effect(void);
};

Effect3::Effect3(int x1, int y1, char *t)
        :SpecialEffects(x1, y1)
{
   frames = y1;
   duration = 10;
   text = t;
}

int Effect3::do_effect(void)
{
   int time = global_timer - start_time;
   // check if time for the effect to change
   if (time > duration) {
      frames--;
      // set beginning of this event
      start_time = global_timer;
   }
   if (frames < 0) return FALSE;

   text_mode(-1);
   textprintf(mapa, font, x*24, frames*24, 15, "%s", text);
   return TRUE;
}

// displayed text with a big font centered at x and y
class Effect4 : public SpecialEffects {
protected:
   char text[25];
public:
   Effect4(int x1, int y1, char *t);
   virtual int do_effect(void);
};

Effect4::Effect4(int x1, int y1, char *t)
        :SpecialEffects(x1, y1)
{
   strcpy(text, t);
   duration = 1000;
}

int Effect4::do_effect(void)
{
   text_mode(-1);
   int time = global_timer - start_time;
   // check if time for the effect to change
   if (time > duration) {
      return FALSE;
   }

   if (duration < 0) return FALSE;
   textout_centre(mapa, (FONT*)main_data[TITLE_FONT].dat, text, x*24, y*24, 15);
   return TRUE;
}


// effect used when the level is finished and the time goes down to 0
class Effect5: public SpecialEffects {
protected:
   char text[25];
   int *total_time;
   int time_bonus;
public:
   Effect5(int *t);
   virtual int do_effect(void);
};

Effect5::Effect5(int *t)
        :SpecialEffects(216, 210)
{
   total_time = t;
   duration = 5;
   time_bonus = 0;
}

int Effect5::do_effect(void)
{
   text_mode(-1);
   int time = global_timer - start_time;
   // check if time for the effect to change
   if (time > duration) {
      (*total_time)--;
      play_sound(TIME_DOWN);
      time_bonus += 50;
      start_time = global_timer;
      if (*total_time <= 0) {
         score += time_bonus;
         sprintf(text, "Time: %2d", *total_time);
         textout(mapa, (FONT*)main_data[TITLE_FONT].dat, text, x, y, 15);
         textprintf(mapa, (FONT*)main_data[TITLE_FONT].dat, x-20, y + 25, 15, "Time Bonus: %3d", time_bonus);
         redraw_score_box;
         rest(500);
         return FALSE;
      }
   }
   sprintf(text, "Time: %2d", *total_time);
   textout(mapa, (FONT*)main_data[TITLE_FONT].dat, text, x, y, 15);
   textprintf(mapa, (FONT*)main_data[TITLE_FONT].dat, x-20, y + 25, 15, "Time Bonus: %3d", time_bonus);
   return TRUE;
}


#define MAX_EFFECTS 20
SpecialEffects *effects[MAX_EFFECTS];

void kill_effects(void)
{
   int i;
   for (i=0; i<MAX_EFFECTS; i++) {
      if (effects[i]) {
         delete effects[i];
         effects[i] = NULL;
      }
   }
}

class GLIZDA {
public:
   int length;
   int x[MAX_GLIZDA];
   int y[MAX_GLIZDA];
   bool dead;
   int dead_frame;
   int has_helmet;
   int face;
   int teleport;
   int x_dir, y_dir;
   int has_light;
   GLIZDA(void);
   void reset(int,int);
   int lives;
   int moved;           // used for changing directions so that not allowed to move back
   int time_limit;
   int get_length(void) {return length;}
   void set_x_dir(int dir);
   void set_y_dir(int dir);
   void move(void);
   // those are for moving glizda when various special sprites
   // are encoutered
   void move201(int tx, int ty);
   void move202(int tx, int ty);
   void move203(int tx, int ty);
   void move204(int tx, int ty);
   void move205(int tx, int ty);
   void move206(int tx, int ty);
   void move207(int tx, int ty);
   void move208(int tx, int ty);
   void move209(int tx, int ty);
   void move210(int tx, int ty);
   void move211(int tx, int ty);
   void move212(int tx, int ty);
   void move213(int tx, int ty);
   void move214(int tx, int ty);
   void move215(int tx, int ty);
   void move216(int tx, int ty);
   void move217(int tx, int ty);
   void move218(int tx, int ty);
   void move219(int tx, int ty);
   void move220(int tx, int ty);


   void draw(void);
   void add(void);
   bool IsDead(void) {return dead;}
   void glizda_die(void);
   int speed;
   int offset;
};

GLIZDA glizda;
int sprites_to_eat;

GLIZDA::GLIZDA(void)
{
   reset(0, 0);
}

void GLIZDA::reset(int x_start, int y_start)
{
   length = 4;
   for (int i=0; i<MAX_GLIZDA; i++)
      x[i] = y[i] = 0;
   x[0] = x_start;
   y[0] = y_start;
   x[1] = x[0] - 1;
   x[2] = x[0] - 2;
   x[3] = x[0] - 3;
   y[1] = y[2] = y[3] = y[0];
   clear_point_mask(x[0], y[0]);
   clear_point_mask(x[0]-1, y[0]);
   clear_point_mask(x[0]-2, y[0]);
   clear_point_mask(x[0]-3, y[0]);
   x_dir = 0;
   y_dir = 0;
   dead = FALSE;
   face = 0;
   has_helmet = 0;
   teleport = FALSE;
   animation = 0;
   offset = 0;
   has_light = FALSE;
   moved = FALSE;
   dead_frame = 0;
   time_limit = time_for_level;
   memcpy(&game_map, &loaded_map, sizeof(MAP));
   redraw_score_box = TRUE;
}

void GLIZDA::set_x_dir(int dir)
{
  if (y_dir != 0) {
      x_dir = dir;
      y_dir = 0;
   }
   else {
      if (x_dir == (-dir) && (x_dir != 0))
         return;
      x_dir = dir;
   }
}

void GLIZDA::set_y_dir(int dir)
{
   if (x_dir != 0) {
      y_dir = dir;
      x_dir = 0;
   }
   else {
      if (y_dir == (-dir) && (y_dir != 0))
         return;
      y_dir = dir;
   }
}


void GLIZDA::add(void)
{
   length++;
   if (length > 150) length = 150;
}

int find_on_map(int to_find, int *x, int *y)
{
   int i, j;
   for (i=0; i<18; i++) {
      for (j=0; j<18; j++) {
         if (game_map.values[i][j] == to_find) {
            if (*x == i && *y == j) continue;
            else {
               *x = i;
               *y = j;
               return TRUE;
            }
         }
      }
   }
   return FALSE;
}

void check_component3(int x, int y, int check_value, int set_value)
{
   int found = 0;
   int i;
   int effect;
   if (check_value == 213) effect = EF13;
   else if (check_value == 217) effect = EF17;
   else effect = EF17;
   if (game_map.values[x][y+1] == check_value) {
      found++;
      game_map.values[x][y+1] = set_value;
      for (i=0; i<MAX_EFFECTS; i++) {
         if (effects[i] == NULL) {
            effects[i] = new Effect1(effect, x, y+1);
            break;
         }
      }
   }
   if (game_map.values[x][y-1] == check_value) {
      found++;
      game_map.values[x][y-1] = set_value;
      for (i=0; i<MAX_EFFECTS; i++) {
         if (effects[i] == NULL) {
            effects[i] = new Effect1(effect, x, y-1);
            break;
         }
      }

   }
   if (game_map.values[x+1][y] == check_value) {
      found++;
      game_map.values[x+1][y] = set_value;
      for (i=0; i<MAX_EFFECTS; i++) {
         if (effects[i] == NULL) {
            effects[i] = new Effect1(effect, x+1, y);
            break;
         }
      }

   }
   if (game_map.values[x-1][y] == check_value) {
      found++;
      game_map.values[x-1][y] = set_value;
      for (i=0; i<MAX_EFFECTS; i++) {
         if (effects[i] == NULL) {
            effects[i] = new Effect1(effect, x-1, y);
            break;
         }
      }

   }
   if (found) {
      game_map.values[x][y] = set_value;
      for (i=0; i<MAX_EFFECTS; i++) {
         if (effects[i] == NULL) {
            effects[i] = new Effect1(effect, x, y);
            break;
         }
      }
      play_sound(COLLIDE_BLOCK);
   }
}

void GLIZDA::glizda_die(void)
{
   int i;
   draw_map();
   if (has_helmet) {
      if (x[0] <= 0 || x[0] >= 17 || y[0] <= 0 || y[0] >= 17) {
      }
      else {
         has_helmet--;
         play_sound(EFFECT_5);
         return;
      }
   }
   dead = TRUE;
   dead_frame = 0;
   if (game_on) play_sound(DIE);
   game_on = FALSE;
   do {
      glizda.length--;
      dead_frame++;
      draw_map();
      if (glizda.length <= 3 || dead_frame > 6) {
         play_sound(BOMB);
         draw_sprite(mapa, (BITMAP*)glizda_data[EXPLODE].dat, 24*x[0]-48, 24*y[0]-48);
         blit(mapa, screen, 0, 0, 25, 25, mapa->w, mapa->h);
         rest(1000);
         break;
      }
      rest(100);
   } while (1);
   reset(game_map.glizda_x, game_map.glizda_y);
   current_second = glizda.time_limit;
   lives--;
   kill_effects();
}


// push a block in all directions
void GLIZDA::move201(int tx, int ty)
{
   // checked if block not pushed outside the map
   if (tx < 0 || tx > 17 || ty < 0 || ty > 17)
      glizda_die();
   else if (game_map.values[tx][ty] == 0) {
      int i;
      // check if block not pushed on glizda
      for (i = 1; i<length-1; i++) {
          if (x[i] == tx && y[i] == ty) {
             glizda_die();
             return;
          }
      }
      game_map.values[tx][ty] = 201;
      game_map.values[x[0]][y[0]] = 0;
      play_sound(MOVE_BLOCK);
   }
   else {
      glizda_die();
   }
}

// push a block right and left
void GLIZDA::move202(int tx, int ty)
{
   if (tx < 0 || tx > 17 || ty < 0 || ty > 17)
      glizda_die();
   else if (game_map.values[tx][ty] == 0 && x_dir != 0) {
      if (game_map.values[tx][ty] != 0) // trying to push a block into
         // not empty space
         glizda_die();
       else {
         int i;
         // check if block not pushed on glizda
         for (i = 1; i<length-1; i++) {
            if (x[i] == tx && y[i] == ty) {
               glizda_die();
               return;
            }
         }
         game_map.values[tx][ty] = 202;
         game_map.values[x[0]][y[0]] = 0;
         play_sound(MOVE_BLOCK);
       }
    }
    else {
       glizda_die();
    }
}

// push a block up and down
void GLIZDA::move203(int tx, int ty)
{
   if (tx < 0 || tx > 17 || ty < 0 || ty > 17)
      glizda_die();
   else if (game_map.values[tx][ty] == 0 && y_dir != 0) {
      if (game_map.values[tx][ty] != 0)
         glizda_die();
      else {
         int i;
         // check if block not pushed on glizda
         for (i = 1; i<length-1; i++) {
            if (x[i] == tx && y[i] == ty) {
               glizda_die();
               return;
            }
         }

         game_map.values[tx][ty] = 203;
         game_map.values[x[0]][y[0]] = 0;
         play_sound(MOVE_BLOCK);
      }
   }
   else {
      glizda_die();
   }
}

// decreasse size of glizda
void GLIZDA::move204(int tx, int ty)
{
   if (length > 3) {
      game_map.values[x[0]][y[0]] = 0;
      length--;
      play_sound(GLIZDA_SHORTER);
   }
}

// get helmet
void GLIZDA::move205(int tx, int ty)
{
   has_helmet++;
   game_map.values[x[0]][y[0]] = 0;
   redraw_score_box = TRUE;
   play_sound(EFFECT_5);
}

void GLIZDA::move206(int tx, int ty)
{
   // not used; code is in glizda::move
}

// can walk onto sprite from right
void GLIZDA::move207(int tx, int ty)
{
   if (x_dir != 1) glizda_die();
   else play_sound(MOVEON_1DIR);
}

// can walk onto sprite from left
void GLIZDA::move208(int tx, int ty)
{
   if (x_dir != -1) glizda_die();
   else play_sound(MOVEON_1DIR);
}

// can walk onto sprite when going up
void GLIZDA::move209(int tx, int ty)
{
   if (y_dir != -1) glizda_die();
   else play_sound(MOVEON_1DIR);
}

// can walk onto sprite when going down
void GLIZDA::move210(int tx, int ty)
{
   if (y_dir != 1) glizda_die();
   else play_sound(MOVEON_1DIR);
}

// push a bomb
void GLIZDA::move211(int tx, int ty)
{
   int i;
   if (tx < 0 || tx > 17 || ty < 0 || ty > 17)
      glizda_die();
   // if the square that the bomb is being moved to has nothing
   else if (game_map.values[tx][ty] == 0) {
      game_map.values[tx][ty] = 211; // move bomb there
      game_map.values[x[0]][y[0]] = 0;
      play_sound(MOVE_BLOCK);
   }
   else {
      if (game_map.values[tx][ty] != 219) {
         int i;
         // check if bomb not pushed on glizda
         for (i = 1; i<length-1; i++) {
            if (x[i] == tx && y[i] == ty) {
               glizda_die();
               return;
            }
         }

         if (game_map.values[tx][ty] > 0 && game_map.values[tx][ty] <= 20) {
            game_map.sprites_to_eat--;
         }
         play_sound(BOMB);
         for (i=0; i<MAX_EFFECTS; i++) {
            if (effects[i] == NULL) {
               effects[i] = new Effect1(EF11, tx-1, ty-1, 11);
               break;
            }
         }
         game_map.values[tx][ty] = 0;
      }
      game_map.values[x[0]][y[0]] = 0;
   }
}

// vertically flip 207 and 208 and horizontally 209 and 210
void GLIZDA::move212(int tx, int ty)
{
   int i, j;
   play_sound(TURN);
   for (i = 0; i < 18; i++) {
      for (j = 0; j < 18; j++) {
         if (game_map.values[i][j] == 207)
            game_map.values[i][j] = 208;
         else if (game_map.values[i][j] == 208)
            game_map.values[i][j] = 207;
         else if (game_map.values[i][j] == 209)
            game_map.values[i][j] = 210;
         else if (game_map.values[i][j] == 210)
            game_map.values[i][j] = 209;
      }
   }
}

// if two sprites of this type are close to each other
// delete them and replace with sprite number 1
void GLIZDA::move213(int tx, int ty)
{
   if (tx < 0 || tx > 17 || ty < 0 || ty > 17)
      glizda_die();
   else if (game_map.values[tx][ty] == 0) {
      int i;
      // check if block not pushed on glizda
      for (i = 1; i<length-1; i++) {
          if (x[i] == tx && y[i] == ty) {
             glizda_die();
             return;
          }
      }

      game_map.values[tx][ty] = 213;
      game_map.values[x[0]][y[0]] = 0;
      play_sound(MOVE_BLOCK);
      check_component3(tx, ty, 213, 1);
   }
   else {
      glizda_die();
   }
}

// found light
void GLIZDA::move214(int tx, int ty)
{
   game_map.values[x[0]][y[0]] = 0;
   glizda.has_light = TRUE;
   redraw_score_box = TRUE;
   play_sound(LIGHT_ON);
}

// give a lot of points
void GLIZDA::move215(int tx, int ty)
{
   int i;
   score += 2500;
   redraw_score_box = TRUE;
   game_map.values[x[0]][y[0]] = 0;
   play_sound(ALOT_POINTS);
   for (i=0; i<MAX_EFFECTS; i++) {
      if (effects[i] == NULL) {
         effects[i] = new Effect2(x[0], y[0], 2500);
         break;
      }
   }
}

// more time
void GLIZDA::move216(int tx, int ty)
{
   int i;
   game_map.values[x[0]][y[0]] = 0;
   current_second += 10;
   redraw_score_box = TRUE;
   play_sound(PLUS_TIME);
   for (i=0; i<MAX_EFFECTS; i++) {
      if (effects[i] == NULL) {
         effects[i] = new Effect3(x[0], y[0], "Time: +10s");
         break;
      }
   }
}

// when two blocks of the same type are next to each other
// they disapear, also when they touch 220 both are gone
void GLIZDA::move217(int tx, int ty)
{
   if (tx < 0 || tx > 17 || ty < 0 || ty > 17)
      glizda_die();
   else if (game_map.values[tx][ty] == 0) {
      int i;
      // check if block not pushed on glizda
      for (i = 1; i<length-1; i++) {
          if (x[i] == tx && y[i] == ty) {
             glizda_die();
             return;
          }
      }

      game_map.values[tx][ty] = 217;
      game_map.values[x[0]][y[0]] = 0;
      play_sound(MOVE_BLOCK);
      check_component3(tx, ty, 217, 0);
      check_component3(tx, ty, 220, 0);
   }
   else {
      glizda_die();
   }
}

// save game at current stage
void GLIZDA::move218(int tx, int ty)
{
   int i;
   game_map.values[x[0]][y[0]] = 0;
   memcpy(&loaded_map, &game_map, sizeof(MAP));
   play_sound(SAVE_MAP);
   for (i=0; i<MAX_EFFECTS; i++) {
      if (effects[i] == NULL) {
         effects[i] = new Effect3(x[0], y[0], "Game Saved");
         break;
      }
   }
}

// end of level
void GLIZDA::move219(int tx, int ty)
{
   if (game_map.sprites_to_eat == 0) {
      play_sound(END_LEVEL);
      do {
         glizda.length--;
         if (glizda.length < 1) break;
         draw_map();
         rest(20);
      } while (1);
      level_finished = TRUE;
   }
}

// when glizda walks on it,it dies but this block
//can be eliminated with 217
void GLIZDA::move220(int tx, int ty)
{
   glizda_die();
}

void GLIZDA::move(void)
{
   animation++;
   if (animation > 3) animation = 0;

   moved = FALSE;
   int i, j;
   int old_x_length = x[length-1];
   int old_y_length = y[length-1];
   for (i=length-1; i>0; i--) {
      x[i] = x[i-1];
      y[i] = y[i-1];
   }
   if (game_on) play_sound(MOVE);
   if (game_map.values[x[0]][y[0]] == 206 && !teleport) {
      find_on_map(206, &(x[0]), &(y[0]));
      teleport = TRUE;
      play_sound(TELEPORT);
   }
   else {
      x[0] = x[0] + x_dir;
      y[0] = y[0] + y_dir;
      teleport = FALSE;
   }

   clear_point_mask(x[0], y[0]);

   //check if glizda went outside the map
   if (x[0] < 0 || x[0] > 17 || y[0] < 0 || y[0] > 17) {
      glizda_die();
      return;
   }

   //check if glizda ate itself
   for (i=1; i<length; i++) {
      if (x[0] == x[i] && y[0] == y[i]){
         if (game_on) glizda_die();
         return;
      }
   }
   //alternate graphics for head
   if (face == 1) face = 0;
   else if (face == 0) face = 1;

   i = game_map.values[x[0]][y[0]];
   //check if ate sprite
   if (i > 0 && i <= 20) {
      add();
      x[length-1] = old_x_length;
      y[length-1] = old_y_length;
      game_map.values[x[0]][y[0]] = 0;
      game_map.sprites_to_eat--;
      if (game_map.sprites_to_eat <= 0) {
         int t_x, t_y;
         if (find_on_map(219, &t_x, &t_y)) {
            clear_point_mask(t_x, t_y);
         }
         play_sound(SPRITES_EATEN);
      }
      if (game_on) play_sound(EAT_SPRITE);
      j = 100 + i * 5;
      score += j;
      for (i=0; i<MAX_EFFECTS; i++) {
         if (effects[i] == NULL) {
            effects[i] = new Effect2(x[0], y[0], j);
            break;
         }
      }
      redraw_score_box = TRUE;
   }
   //check if hit the wall
   else if (i > 100 && i <= 120) {
      glizda_die();
      return;
   }
   // next square it's going to move to
   int tx = x[0] + x_dir;
   int ty = y[0] + y_dir;
   switch(i) {
      case 201: move201(tx, ty); return;
      case 202: move202(tx, ty); return;
      case 203: move203(tx, ty); return;
      case 204: move204(tx, ty); return;
      case 205: move205(tx, ty); return;
      case 207: move207(tx, ty); return;
      case 208: move208(tx, ty); return;
      case 209: move209(tx, ty); return;
      case 210: move210(tx, ty); return;
      case 211: move211(tx, ty); return;
      case 212: move212(tx, ty); return;
      case 213: move213(tx, ty); return;
      case 214: move214(tx, ty); return;
      case 215: move215(tx, ty); return;
      case 216: move216(tx, ty); return;
      case 217: move217(tx, ty); return;
      case 218: move218(tx, ty); return;
      case 219: move219(tx, ty); return;
      case 220: move220(tx, ty); return;
   }
}

void GLIZDA::draw(void)
{
   int xt, yt;
   for (int i=1; i<length-1; i++) {
      if (has_helmet)
         draw_trans_sprite(mapa, (BITMAP*)glizda_data[BODY].dat, x[i]*24, y[i]*24);
      else
         draw_sprite(mapa, (BITMAP*)glizda_data[BODY].dat, x[i]*24, y[i]*24);
   }
   int head;
   int tail;
   if (x_dir == 1) {
      if (face) head = RIGHT_OPEN;
      else head = RIGHT_CLOSED;
   }
   else if (x_dir == -1) {
      if (face) head = LEFT_OPEN;
      else head = LEFT_CLOSED;
   }
   else if (y_dir == -1) {
      if (face) head = UP_OPEN;
      else head = UP_CLOSED;
   }
   else if (y_dir == 1) {
      if (face) head = DOWN_OPEN;
      else head = DOWN_CLOSED;
   }
   else head = RIGHT_OPEN;

   // draw head and tail
   if (dead) {
      stretch_sprite(mapa, (BITMAP*)glizda_data[head].dat, x[0]*24-dead_frame*3, y[0]*24-dead_frame*3, 24+dead_frame*6, 24+dead_frame*6);
   }
   else if (has_helmet)
      draw_trans_sprite(mapa, (BITMAP*)glizda_data[head].dat, x[0]*24, y[0]*24);
   else
      draw_sprite(mapa, (BITMAP*)glizda_data[head].dat, x[0]*24, y[0]*24);

   if (length > 1) {
      if (x[length-1] == x[length-2]) tail = TAIL_UP;
      else tail = TAIL_RIGHT;
      if (has_helmet)
         draw_trans_sprite(mapa, (BITMAP*)glizda_data[tail].dat, x[length-1]*24, y[length-1]*24);
      else
         draw_sprite(mapa, (BITMAP*)glizda_data[tail].dat, x[length-1]*24, y[length-1]*24);
   }
}

int draw_texture(void)
{
   drawing_mode(DRAW_MODE_COPY_PATTERN, (BITMAP*)main_data[TEXTURE].dat, 0, 0);
   rectfill(global_bitmap, 0, 0, SCREEN_W, SCREEN_H, 1);
   solid_mode();
   draw_trans_sprite(global_bitmap, (BITMAP*)main_data[BACKGROUND].dat, 0, 0);
   vsync(); vsync();
   blit(global_bitmap, screen, 0, 0, 0, 0, 640, 480);
   draw_rle_sprite(screen, (RLE_SPRITE*)main_data[FRAME3].dat, 0, 0);
   text_mode(-1);
   textout(screen, font, GLIZDA_INFO1, 500, 440, makecol8(255, 255, 255));
   textout(screen, font, GLIZDA_INFO2, 500, 450, makecol8(255, 255, 255));
	draw_sprite(screen, (BITMAP*)main_data[GLIZDA_TITLE].dat, 95, 15);
   return D_O_K;
}


int LoadMap(void)
{
 	FILE *file;       //file
   char ttext[265];
   ttext[0] = '\0';
   //open file dialog
   if (file_select("Open Map", ttext, "map") != 0) {
     	file = fopen(ttext, "rb");      //try to open file
      if (!file) {         //opening failed
        	alert("Error opening file.", NULL, NULL, "OK", NULL, KEY_ENTER, 0);
         return FALSE;
      }
      else {          //read data from file
        	fread(&game_map, sizeof(struct MAP), 1, file);
         fclose(file);              //close file
         memcpy(&loaded_map, &game_map, sizeof(MAP));

         //check if valid file type
         if (strcmp(game_map.IdString, "glizda") != 0) {
            alert("Error", "File of invalid type", NULL, "OK", NULL, KEY_ENTER, 0);
            return FALSE;
         }
         //check if valid version
         else if (strcmp(game_map.version, "v1.0") != 0) {
            alert("Error!!!", " file of invalid version:", game_map.version, "OK", NULL, KEY_ENTER, 0);
            return FALSE;
         }
         else {
            glizda.reset(game_map.glizda_x, game_map.glizda_y);
            text_mode(0);
            textout(screen, font, ttext, 300, 1, 15);
            return TRUE;
         } //end else
      } //end else !file
      return TRUE;
   }  //end if file_select
   else return FALSE;
}

void draw_score_box(void)
{
   text_mode(0);
   clear(score_bitmap);
   textprintf(score_bitmap, font, 10, 0, 32, "Score:     %06d", score);
   textprintf(score_bitmap, font, 10, 20, 15, "Sprites left: %3d", game_map.sprites_to_eat);
   textprintf(score_bitmap, font, 10, 40, 15, "Lives: %3d", glizda.lives);
   textprintf(score_bitmap, font, 10, 60, 15, "Time:  %3d", current_second);
   textprintf(score_bitmap, font, 10, 80, 15, "Level: %3d", current_level);
   if (glizda.has_light)
      stretch_sprite(score_bitmap, (BITMAP*)special_data[14].dat, 10, 100, 16, 16);
   if (glizda.has_helmet) {
      stretch_sprite(score_bitmap, (BITMAP*)special_data[5].dat, 30, 100, 16, 16);
      textprintf(score_bitmap, font, 50, 105, 15, "%d", glizda.has_helmet);
   }
   if (show_fps)
      textprintf(screen, font, 1, 1, 15, "%3d", frames_per_second);
   blit(score_bitmap, screen, 0, 0, 500, 50, 123, 133);
   redraw_score_box = FALSE;
}


void draw_map(void)
{
   int i, j, temp;
   BITMAP *bg = (BITMAP*)main_data[BACKGROUND].dat;
   if (draw_background)
     if (show_background)
        blit(bg, mapa, 25, 25, 0, 0, mapa->w, mapa->h);
     else clear(mapa);
   else
       clear(mapa);
   for (i=0; i<18; i++) {
      for (j=0; j<18; j++) {
         temp = game_map.values[i][j];
         if (temp > 0 && temp <= 20) {
            //draw sprites; sprites values are taken from map.values[i][j] and they
            //correspond to array index of sprites_data file
            draw_sprite(mapa, (BITMAP*)sprites_data[temp].dat, i*24, j*24);
         }
         else if (temp > 100 && temp <= 120) {
            //draw walls (object glizda cannot enter). the values are map.values[i][j]
            // - 100 and graphics are taken from walls_data
            draw_sprite(mapa, (BITMAP*)walls_data[temp-100].dat, i*24, j*24);
         }  //end else if
         // special sprites
         else if (temp > 200 && temp <= 220) {
            draw_sprite(mapa, (BITMAP*)special_data[temp-200+animation*20].dat, i*24, j*24);
         }
      }  //end for j
   }  //end for i

   if (!glizda.has_light)
      if (show_fog)
         draw_mask();
   glizda.draw();
   for (i=0; i<MAX_EFFECTS; i++) {
      if (effects[i]) {
         if (!(effects[i]->do_effect())) {
            delete effects[i];
            effects[i] = NULL;
         }
      }
   }
   if (draw_background)
      blit(mapa, screen, 0, 0, 25, 25, mapa->w, mapa->h);
   if (redraw_score_box)
      draw_score_box();
}

void process_cheat_code(void)
{
   if (current_cheat_char < 0 || current_cheat_char > 10) return;
   if (strcmp(cheat_code, "eatbug") == 0)
      glizda.lives++;
   else if (strcmp(cheat_code, "eatrock") == 0)
      glizda.has_helmet++;
   else if (strcmp(cheat_code, "eatshit") == 0) {
      if (glizda.length > 3)  glizda.length--;
   }
   else if (strcmp(cheat_code, "gotohell") == 0)
      level_finished = TRUE;
   else if (strcmp(cheat_code, "notsofast") == 0)
      glizda.speed = 100;
   else if (strcmp(cheat_code, "upmyass") == 0)
      glizda.speed = 20;
   else if (strcmp(cheat_code, "lightmyass") == 0)
      glizda.has_light = TRUE;
   else if (strcmp(cheat_code, "fps") == 0)
      show_fps = (show_fps == TRUE) ? FALSE : TRUE;
   else if (strcmp(cheat_code, "eatgod") == 0)
      glizda.lives += 10;
   current_cheat_char = 0;
   redraw_score_box = TRUE;
}


int StartGame(void)
{
   int i;
   char ttext[25];
   game_on = FALSE;
   current_cheat_char = 0;
   init_mask();
   glizda.reset(game_map.glizda_x, game_map.glizda_y);
   level_finished = FALSE;

   // start level scrolling
   // global_bitmap holds old level and background if first level
   // mapa holds the new level without the background
   // t_bitmap is the bitmap that is drawn on screen
   play_sound(SCROLLMAP);
   BITMAP *t_bitmap = create_bitmap(24*18, 24*18);
   draw_background = FALSE;
   if (current_level <= 1)
      blit((BITMAP*)main_data[BACKGROUND].dat, global_bitmap, 25, 25, 0, 0, mapa->w, mapa->h);
   // get the new map for the level
   draw_map();
   sprintf(ttext, "Level %d", current_level);
   text_mode(-1);
   for (i=36; i>0; i--) {
       // first draw old level
       blit(global_bitmap, t_bitmap, 0,0, 0, 0, mapa->w, mapa->h);
       // now draw the new level on top of it
       draw_sprite(t_bitmap, mapa, i*12, 0);
       /*
       blit(mapa, global_bitmap, 0, 0, i*12, 0, (36-i)*12, mapa->h);*/
       // put level number
       textout_centre(t_bitmap, (FONT*)main_data[TITLE_FONT].dat, ttext,
                      (36-i)*10 , 9*24, 15);
       // put the t_bitmap on screen
       blit(t_bitmap, screen, 0, 0, 25, 25, mapa->w, mapa->h);
       rest(25);
   }
   draw_background = TRUE;
   destroy_bitmap(t_bitmap);

   for (i=0; i<MAX_EFFECTS; i++)
      effects[i] = NULL;

   start_second = global_timer;
   current_second = glizda.time_limit;
   frames_per_second = 0;
   redraw_score_box = TRUE;
   int num_frames = 0;

   do {
      poll_joystick();
      if (key[KEY_ESC]) {
         break;
      }
      if ((key[KEY_UP] || joy_up) && !glizda.moved) {
         glizda.moved = TRUE;
         glizda.set_y_dir(-1);
         if (!game_on) {
            game_on = TRUE;
            glizda_timer = 0;
         }
      }
      else if ((key[KEY_DOWN] || joy_down) && !glizda.moved) {
         glizda.moved = TRUE;
         glizda.set_y_dir(1);
         if (!game_on) {
            game_on = TRUE;
            glizda_timer = 0;
         }
      }
      else if ((key[KEY_RIGHT] || joy_right) && !glizda.moved) {
          glizda.moved = TRUE;
          glizda.set_x_dir(1);
          if (!game_on) {
             game_on = TRUE;
             glizda_timer = 0;
          }
      }
      else if ((key[KEY_LEFT] || joy_left) && !glizda.moved) {
        glizda.moved = TRUE;
        glizda.set_x_dir(-1);
        if (!game_on) {
           game_on = TRUE;
           glizda_timer = 0;
        }
      }
      else if (key[KEY_F3]) {
         game_on = FALSE;
         memcpy(&game_map, &loaded_map, sizeof(MAP));
      }
      else if (key[KEY_F5])
         stop_midi();
      else if (key[KEY_F10]) {
         //create temp bitmap
         BITMAP *tbmp = create_sub_bitmap(screen, 0, 0, SCREEN_W, SCREEN_H);
         //save the picture with game pallete
         save_pcx("glizda.pcx", tbmp, (RGB*)main_data[GAME_PALLETE].dat);
         destroy_bitmap(tbmp);     //destroy bitmap
         while (key[KEY_F10]) {}
      }
      if (keypressed()) {
         int k = readkey();
         if ((k >> 8) == KEY_ENTER) {
            cheat_code[current_cheat_char] = '\0';
            process_cheat_code();
         }
         else {
           if (isalpha(k & 0xff)) {
              cheat_code[current_cheat_char++] = (k & 0xff);
              cheat_code[current_cheat_char] = '\0';
              if (current_cheat_char > 10) current_cheat_char = 0;
           }
         }
         clear_keybuf();
      }
      if (glizda_timer > glizda.speed) {
         if (game_on) {
            glizda_timer -= glizda.speed;
            glizda.move();
         }
      }
      if (global_timer - start_second > 100) {
         if (glizda.x_dir != 0 || glizda.y_dir != 0) {
            current_second--;
            redraw_score_box = TRUE;
            if (current_second < 10)
               play_sound(TIME_UP);
         }
         start_second = global_timer;
         frames_per_second = num_frames;
         num_frames = 0;
         if (current_second <= 0)
            glizda.glizda_die();
      }
      check_music();
      draw_map();
      num_frames++;
      if (level_finished) {
         game_on = FALSE;
         for (i=0; i<MAX_EFFECTS; i++) {
           if (effects[i] == NULL) {
              effects[i] = new Effect5(&current_second);
              break;
           }
        }
        while (current_second > 0)
           draw_map();
        rest(100);
        draw_map();
        return TRUE;
      }
      if (glizda.lives == 0) {
         game_on = FALSE;
         draw_sprite(screen, (BITMAP*)main_data[GAME_OVER].dat, 150,150);
         while (1) {
            if (key[KEY_SPACE] || mouse_b)
               break;
         }
         return FALSE;
      }
   } while (1);
   return FALSE;
}

void prepare_game_graphics(void)
{
   drawing_mode(DRAW_MODE_COPY_PATTERN, (BITMAP*)main_data[TEXTURE].dat, 0, 0);
   rectfill(global_bitmap, 0, 0, SCREEN_W, SCREEN_H, 1);
   solid_mode();
   draw_trans_sprite(global_bitmap, (BITMAP*)main_data[BACKGROUND].dat, 0, 0);
   vsync(); vsync();
   blit(global_bitmap, screen, 0, 0, 0, 0, 640, 480);
   draw_rle_sprite(screen, (RLE_SPRITE*)main_data[FRAME2].dat, 480, 30);
   draw_rle_sprite(screen, (RLE_SPRITE*)main_data[FRAME1].dat, 4, 4);
   text_mode(-1);
   textout(screen, font, GLIZDA_INFO1, 500, 450, makecol8(255, 255, 255));
   textout(screen, font, GLIZDA_INFO2, 500, 460, makecol8(255, 255, 255));
}

int PlayMapProc(void)
{
   show_mouse(NULL);
   if (!LoadMap()) return FALSE;
   score = 0;
   glizda.lives = 3;
   time_for_level = custom_map_time;
   switch (custom_map_speed) {
      case 1: glizda.speed = 10; break;
      case 2: glizda.speed = 15; break;
      case 3: glizda.speed = 20; break;
      case 4: glizda.speed = 25; break;
      case 5: glizda.speed = 30; break;
      case 6: glizda.speed = 35; break;
      case 7: glizda.speed = 40; break;
      case 8: glizda.speed = 45; break;
      case 9: glizda.speed = 50; break;
      default: glizda.speed = 30; break;
   }
   current_level = 0;
   prepare_game_graphics();
   start_music();
   StartGame();
   stop_music();
   clear_keybuf();
   play_dialog->redraw();
   return FALSE;
}

int play_shareware_dialog()
{
   // Object definitions - DEGUI 1.2 beta
   box_object object_1;
   button_object object_2("OK",0,true);
   panel_ridge_object object_4;
   centerd_text_object object_3("Error");
   centerd_text_object object_5("Sorry, you need to");
   centerd_text_object object_6("register the game");
   centerd_text_object object_7("for the next levels");
   centerd_text_object object_8("to be available");

   // Add Objects to the dialog
   dialog the_dialog;
   the_dialog.add(object_1,150,150,215,130);
   the_dialog.add(object_2,240,250,50,20);
   the_dialog.add(object_4,155,155,205,20);
   the_dialog.add(object_3,250,160,20,8);
   the_dialog.add(object_5,260,185,72,8);
   the_dialog.add(object_6,260,200,68,8);
   the_dialog.add(object_7,260,215,72,8);
   the_dialog.add(object_8,255,230,48,8);
   the_dialog.center();
   the_dialog.popup();
   return FALSE;
}

void PlayEpisod(EPISOD_INFO *episod)
{
   show_mouse(NULL);
   char ttext[25];
   int which = 0, i;
   score = 0;
   glizda.lives = 3;
   if (episod->num_maps < 1) return;
   start_music();
   prepare_game_graphics();
   while (1) {
      memcpy(game_map.values, episod->maps[which].data, (MAP_SIZE*MAP_SIZE)*(sizeof(BYTE)));
      game_map.glizda_x = episod->maps[which].glizda_x;
      game_map.glizda_y = episod->maps[which].glizda_y;
      game_map.sprites_to_eat = episod->maps[which].num_sprites;

      // get a copy of current map
      blit(mapa, global_bitmap, 0, 0, 0, 0, mapa->w, mapa->h);
      memcpy(&loaded_map, &game_map, sizeof(MAP));
      current_level = which+1;
      int sp = episod->maps[which].speed;
      switch (episod->maps[which].speed) {
         case 1: glizda.speed = 10; break;
         case 2: glizda.speed = 15; break;
         case 3: glizda.speed = 20; break;
         case 4: glizda.speed = 25; break;
         case 5: glizda.speed = 30; break;
         case 6: glizda.speed = 45; break;
         case 7: glizda.speed = 50; break;
         case 8: glizda.speed = 55; break;
         case 9: glizda.speed = 60; break;
         default: glizda.speed = 30; break;
      }
      time_for_level = episod->maps[which].time_limit;
      if (!StartGame()) break;
      which++;
      if (shareware && which > 1) {
         play_shareware_dialog();
         break;
      }

      if (which >= episod->num_maps) {
         alert("Congratulations!!!", "You have finished whole episod.",
               NULL, "OK", NULL, KEY_ENTER, KEY_ESC);
         break;
      }
   }
   stop_music();
   clear_keybuf();
   play_dialog->redraw();
}

dialog *got_highscore_dialog;

int exit_gothighscore_dialog(void)
{
   got_highscore_dialog->close();
   return D_O_K;
}

void check_highscores(char *filename)
{
   char file[256];
   char name[11];
   filename = get_filename(filename);
   strcpy(file, filename);
   char *temp = index(file, '.');
   strcpy(temp, ".hsc");
   load_highscores(file);
   if (is_highscore(score)) {
      // Object definitions - DEGUI 1.2 beta
      play_sound(GOT_HIGHSCORE);
      char t_string[100];
      sprintf(t_string, "Your score: %d", score);
      strcpy(name, "glizda");
      shadow_box_object object_1;
      panel_sunken_object object_3;
      centerd_text_object object_2("Congratulations!!!");
      centerd_text_object object_4("You got a High Score.");
      centerd_text_object object_5(t_string);
      centerd_text_object object_6("Enter Your Name:");
      edittext_object edtName(name,10);
      button_object object_8("OK",0,true);
      // Add Objects to the dialog
      dialog the_dialog;
      got_highscore_dialog = &the_dialog;
      the_dialog.add(object_1,165,175,265,165);
      the_dialog.add(object_3,175,180,245,20);
      the_dialog.add(object_2,300,185,72,8);
      the_dialog.add(object_4,295,215,96,8);
      the_dialog.add(object_5,290,235,48,8);
      the_dialog.add(object_6,285,255,64,8);
      the_dialog.add(edtName,190,275,215,15);
      the_dialog.add(object_8,265,305,65,25);
      the_dialog.center();
      the_dialog.popup(&edtName);
      strcpy(name, edtName.get_text());
      update_highscores(name, current_level, score);
      save_highscores(file);
      strcpy(high_scores_view_name, get_filename(file));
      HighScoresProc();
   }
}

int play_episod1_proc(void)
{
   if (!episod1.loaded)
      alert("Episod not found. ", "File: ", EPISOD1_DATA, "OK", NULL, KEY_ENTER, KEY_ESC);
   else {
      PlayEpisod(&episod1);
      check_highscores(EPISOD1_DATA);
   }
   return FALSE;
}

int play_episod2_proc(void)
{
   if (!episod2.loaded)
      alert("Episod not found. ", "File: ", EPISOD2_DATA, "OK", NULL, KEY_ENTER, KEY_ESC);
   else {
      PlayEpisod(&episod2);
      check_highscores(EPISOD2_DATA);
   }
   return FALSE;
}

int play_episod3_proc(void)
{
   if (!episod3.loaded)
      alert("Episod not found. ", "File: ", EPISOD3_DATA, "OK", NULL, KEY_ENTER, KEY_ESC);
   else {
      PlayEpisod(&episod3);
      check_highscores(EPISOD3_DATA);
   }
   return FALSE;
}

int play_episod4_proc(void)
{
   if (!episod4.loaded)
      alert("Episod not found. ", "File: ", EPISOD4_DATA, "OK", NULL, KEY_ENTER, KEY_ESC);
   else {
      PlayEpisod(&episod4);
      check_highscores(EPISOD4_DATA);
   }
   return FALSE;
}

int play_episod5_proc(void)
{
   if (!episod5.loaded)
      alert("Episod not found. ", "File: ", EPISOD5_DATA, "OK", NULL, KEY_ENTER, KEY_ESC);
   else {
      PlayEpisod(&episod5);
      check_highscores(EPISOD5_DATA);
   }
   return FALSE;
}

// play custom episod
int play_custom_episod_proc(void)
{
   char t_file_name[256];
   t_file_name[0] = '\0';
   if (file_select("Change Episod (*.gep)", t_file_name, "gep") != 0) {
      load_episod(&episod_custom, t_file_name);
      PlayEpisod(&episod_custom);
      free(episod_custom.maps);
      check_highscores(t_file_name);
      game_dialog->redraw();
   }
   return FALSE;
}

//////////////////////////////// OPTIONS DIALOG /////////////////////
dialog *options_dialog;
texture_text_object *p_txtSoundVolume;
texture_text_object *p_txtMusicVolume;
texture_button_object *p_cmdSoundOn;
texture_button_object *p_cmdMusicOn;
texture_button_object *p_cmdFog;
texture_button_object *p_cmdBackground;

char *volume_list[] = {" 0", " 1", " 2", " 3", " 4", " 5", " 6", " 7", " 8", " 9", "10"};

int ClearOptionsProc(void)
{
   draw_custom_background("O P T I O N S");
   blit(global_bitmap, screen, 0, 0, 0, 0, 640, 480);
   return D_O_K;
}

int exit_options_dialog_proc(void)
{
   options_dialog->close();
   return FALSE;
}

int sound_up_proc(void)
{
   if (sound_volume < 10) {
      sound_volume++;
      p_txtSoundVolume->change_text(volume_list[sound_volume]);
      p_txtSoundVolume->redraw();
   }
   return FALSE;
}

int sound_down_proc(void)
{
   if (sound_volume > 0) {
      sound_volume--;
      p_txtSoundVolume->change_text(volume_list[sound_volume]);
      p_txtSoundVolume->redraw();
   }
   return FALSE;
}

int music_up_proc(void)
{
   if (music_volume < 10) {
      music_volume++;
      p_txtMusicVolume->set_text(volume_list[music_volume]);
      p_txtMusicVolume->redraw();
   }
   return FALSE;
}

int music_down_proc(void)
{
   if (music_volume > 0) {
      music_volume--;
      p_txtMusicVolume->set_text(volume_list[music_volume]);
      p_txtMusicVolume->redraw();
   }
   return FALSE;
}

int change_sound_proc(void)
{
   if (sound_on) {
      sound_on = FALSE;
      p_cmdSoundOn->set_text("Sound Off");
   }
   else {
      sound_on = TRUE;
      p_cmdSoundOn->set_text("Sound On");
   }
   p_cmdSoundOn->redraw();
   return FALSE;
}

int change_music_proc(void)
{
   if (music_on) {
      music_on = FALSE;
      p_cmdMusicOn->set_text("Music Off");
   }
   else {
      music_on = TRUE;
      p_cmdMusicOn->set_text("Music On");
   }
   p_cmdMusicOn->redraw();
   return FALSE;
}

int change_fog_proc(void)
{
   if (show_fog) {
      show_fog = FALSE;
      p_cmdFog->set_text("Fog Off");
   }
   else {
      show_fog = TRUE;
      p_cmdFog->set_text("Fog On");
   }
   p_cmdFog->redraw();
   return FALSE;
}

int change_background_proc(void)
{
   if (show_background) {
      show_background = FALSE;
      p_cmdBackground->set_text("Background Off");
   }
   else {
      show_background = TRUE;
      p_cmdBackground->set_text("Background On");
   }
   p_cmdBackground->redraw();
   return FALSE;
}

int options_main(void)
{
   dialog the_dialog;
   options_dialog = &the_dialog;
   texture_button_object cmdExit("Exit", (BITMAP*)main_data[TEXTURE2].dat,
                         (FONT*)main_data[TITLE_FONT].dat, exit_options_dialog_proc);
   clear_screen_object clrObject;
   clrObject.set_draw_callback(ClearOptionsProc);

   texture_button_object cmdSoundOn((sound_on) ? "Sound On" : "Sound Off",
                         (BITMAP*)main_data[TEXTURE2].dat, (FONT*)main_data[TITLE_FONT].dat, change_sound_proc);
   p_cmdSoundOn = &cmdSoundOn;

   texture_button_object cmdMusicOn((music_on) ? "Music On" : "Music Off",
                         (BITMAP*)main_data[TEXTURE2].dat, (FONT*)main_data[TITLE_FONT].dat, change_music_proc);
   p_cmdMusicOn = &cmdMusicOn;

   texture_button_object cmdFog((show_fog) ? "Fog On" : "Fog Off",
                         (BITMAP*)main_data[TEXTURE2].dat, (FONT*)main_data[TITLE_FONT].dat, change_fog_proc);
   p_cmdFog = &cmdFog;

   texture_button_object cmdBackground((show_background) ? "Background On" : "Background Off",
                         (BITMAP*)main_data[TEXTURE2].dat, (FONT*)main_data[TITLE_FONT].dat, change_background_proc);
   p_cmdBackground = &cmdBackground;

   texture_text_object txtSoundInfo("Sound Volume", (BITMAP*)main_data[TEXTURE2].dat, font);
   texture_text_object txtSoundVolume("", (BITMAP*)main_data[TEXTURE2].dat, (FONT*)main_data[TITLE_FONT].dat);
   p_txtSoundVolume = &txtSoundVolume;
   if (sound_volume < 0 || sound_volume > 10)
      sound_volume = 8;
   p_txtSoundVolume->set_text(volume_list[sound_volume]);

   texture_text_object txtMusicInfo("Music Volume", (BITMAP*)main_data[TEXTURE2].dat, font);
   texture_text_object txtMusicVolume("", (BITMAP*)main_data[TEXTURE2].dat, (FONT*)main_data[TITLE_FONT].dat);
   p_txtMusicVolume = &txtMusicVolume;
   if (music_volume < 0 || music_volume > 10)
      music_volume = 8;
   p_txtMusicVolume->set_text(volume_list[music_volume]);

   texture_button_object cmdSoundUp(">>", (BITMAP*)main_data[TEXTURE2].dat, font, sound_up_proc);
   texture_button_object cmdSoundDn("<<", (BITMAP*)main_data[TEXTURE2].dat, font, sound_down_proc);
   texture_button_object cmdMusicUp(">>", (BITMAP*)main_data[TEXTURE2].dat, font, music_up_proc);
   texture_button_object cmdMusicDn("<<", (BITMAP*)main_data[TEXTURE2].dat, font, music_down_proc);

   char map_speed[25];
   char map_time[25];
   sprintf(map_speed, "%d", custom_map_speed);
   sprintf(map_time, "%d", custom_map_time);

   edittext_object edtMapSpeed(map_speed, 4);
   edittext_object edtMapTime(map_time, 4);
   texture_text_object txtMapSpeed("Custom Map Speed: ", (BITMAP*)main_data[TEXTURE2].dat, font);
   texture_text_object txtMapTime("Custom Map Time Limit: ", (BITMAP*)main_data[TEXTURE2].dat, font);

   texture_text_object txtInfo1("Unselecting fog and background makes the game run faster on slower", (BITMAP*)main_data[TEXTURE2].dat, font);
   texture_text_object txtInfo2("machines.  Use them if the game runs slow on yours.", (BITMAP*)main_data[TEXTURE2].dat, font);

   text_mode(-1);
   the_dialog.add(clrObject);
   the_dialog.add(cmdExit,         245, 450, 150, 25);

   the_dialog.add(cmdSoundOn,      100, 100, 150, 25);

   the_dialog.add(txtSoundInfo,    300,  85, 110, 13);
   the_dialog.add(cmdSoundDn,      300, 100,  25, 25);
   the_dialog.add(txtSoundVolume,  330, 100,  50, 25);
   the_dialog.add(cmdSoundUp,      385, 100,  25, 25);

   the_dialog.add(cmdMusicOn,      100, 200, 150, 25);

   the_dialog.add(txtMusicInfo,    300, 185, 110, 13);
   the_dialog.add(cmdMusicDn,      300, 200,  25, 25);
   the_dialog.add(txtMusicVolume,  330, 200,  50, 25);
   the_dialog.add(cmdMusicUp,      385, 200,  25, 25);

   the_dialog.add(txtMapSpeed,      90, 380, 140, 18);
   the_dialog.add(txtMapTime,       90, 400, 140, 18);
   the_dialog.add(edtMapSpeed,     250, 380, 35,  15);
   the_dialog.add(edtMapTime,      250, 400, 35,  15);

   the_dialog.add(txtInfo1,        100, 265, 400, 15);
   the_dialog.add(txtInfo2,        100, 280, 400, 15);
   the_dialog.add(cmdFog,          100, 300, 150, 25);
   the_dialog.add(cmdBackground,   350, 300, 150, 25);
   the_dialog.execute();

   int temp = atoi(edtMapSpeed.get_text());
   if (temp < 1 || temp > 9) {
      alert("Custom map speed is out of range.", "It has to be between", "1 and 9", "OK", NULL, KEY_ENTER, KEY_ESC);
   }
   else custom_map_speed = temp;

   temp = atoi(edtMapTime.get_text());
   if (temp <= 0)
      alert("Custom map time", "has to be greater than 0.", NULL, "OK", NULL, KEY_ENTER, KEY_ESC);
   else
       custom_map_time = temp;

   return FALSE;
}

int OptionsProc(void)
{
   options_main();
   game_dialog->redraw();
   return FALSE;
}

////////////////////////////////////////////
// Credits Dialog
////////////////////////////////////////////
dialog *pCreditsDlg;
#define MAX_LINES 13
char *credits1[] ={"Andrew Deren",     // programming
  						 "Rafal Wiszowaty",  // graphics
                   "Andrew Deren",
                   "Andrew Deren",     // level design
                   "Mateusz Zagata",
                   "Lukasz Zagata",
                   "Mateusz Zagata",  // music
                   "Unknown",         // sounds
                   " ",
                   "DJ Delorie",       // special thanks to
                   "Shawn Hargreaves",
                   "Robert Hohne",
                   "K. Glen Smith (ksmith@cswnet.com)"};
char *credits2[] = {"Programming",
                    "Graphics",
                    "",
                    "Level Design",
                    "",
                    "",
                    "Music",
                    "Sounds",
                    "",
                    "Special Thanks to",
                    "",
                    "",
                    ""};
int ClearCreditsProc(void)
{
   draw_custom_background("C R E D I T S");
   text_mode(-1);
   for (int i=0; i<MAX_LINES; i++) {
      textout(global_bitmap, font, credits1[i], 325, 100 + 20 * i, makecol8(255, 255, 255));
      textout_right(global_bitmap, font, credits2[i], 315, 100 + 20 * i, makecol8(255, 255, 255));
   }
   blit(global_bitmap, screen, 0, 0, 0, 0, 640, 480);
   return D_O_K;
}

int exit_credits_dialog_proc(void)
{
   pCreditsDlg->close();
   return FALSE;
}


int CreditsProc(void)
{
   texture_button_object cmdExit("Exit", (BITMAP*)main_data[TEXTURE2].dat,
                         (FONT*)main_data[TITLE_FONT].dat, exit_credits_dialog_proc);
   clear_screen_object clrObject;
   clrObject.set_draw_callback(ClearCreditsProc);

   dialog the_dialog;
   pCreditsDlg = &the_dialog;
   the_dialog.add(clrObject);
   the_dialog.add(cmdExit,         245, 450, 150, 25);
   the_dialog.execute();

   game_dialog->redraw();
   return FALSE;
}

////////////////////////////////////////////
// High Scores Dialog
////////////////////////////////////////////
dialog *pHSdialog;
char *current_highscores_episod;

int ClearHighScoresProc(void)
{
   draw_custom_background("H I G H   S C O R E S");
   int i;
   int color;
   color = makecol8(255, 0, 0);
   textprintf_centre(global_bitmap, (FONT*)main_data[TITLE_FONT].dat, 322, 57, 0, high_scores_view_name);
   textprintf_centre(global_bitmap, (FONT*)main_data[TITLE_FONT].dat, 320, 55, color, high_scores_view_name);
   textprintf(global_bitmap, font, 130, 80, color, "Name");
   textprintf(global_bitmap, font, 250, 80, color, "Score");
   textprintf(global_bitmap, font, 320, 80, color, "Level");
   for (i=0; i<10; i++) {
       if (i==high_scores_view_place) color = makecol8(255, 0, 0);
       else color = makecol8(255, 255, 255);
       textprintf(global_bitmap, font, 100, i*30+100, color, "%2d", i+1);
       textprintf(global_bitmap, font, 130, i*30+100, color, "%s", hi_scores[i].name);
       textprintf(global_bitmap, font, 250, i*30+100, color, "%6d", hi_scores[i].score);
       textprintf(global_bitmap, font, 320, i*30+100, color, "%4d", hi_scores[i].level);
   }
   blit(global_bitmap, screen, 0, 0, 0, 0, 640, 480);
   return D_O_K;
}

int exit_high_scores_dialog_proc(void)
{
   pHSdialog->close();
   return FALSE;
}

void change_episod_highscores(void)
{
   char file[256];
   current_highscores_episod = get_filename(current_highscores_episod);
   strcpy(file, current_highscores_episod);
   char *temp = index(file, '.');
   strcpy(temp, ".hsc");
   strcpy(high_scores_view_name, current_highscores_episod);
   load_highscores(file);
   high_scores_view_place = -1;
   pHSdialog->redraw();
}

int view_episod1_highscores_proc(void)
{
   current_highscores_episod = EPISOD1_DATA;
   change_episod_highscores();
   return FALSE;
}

int view_episod2_highscores_proc(void)
{
   current_highscores_episod = EPISOD2_DATA;
   change_episod_highscores();
   return FALSE;
}

int view_episod3_highscores_proc(void)
{
   current_highscores_episod = EPISOD3_DATA;
   change_episod_highscores();
   return FALSE;
}

int view_episod4_highscores_proc(void)
{
   current_highscores_episod = EPISOD4_DATA;
   change_episod_highscores();
   return FALSE;
}

int view_episod5_highscores_proc(void)
{
   current_highscores_episod = EPISOD5_DATA;
   change_episod_highscores();
   return FALSE;
}

int view_episod_custom_highscores_proc(void)
{
   char t_file_name[256];
   t_file_name[0] = '\0';
   if (file_select("Change Episod (*.gep)", t_file_name, "gep") != 0) {
      current_highscores_episod = t_file_name;
      change_episod_highscores();
   }
   return FALSE;
}

int HighScoresProc(void)
{
   texture_button_object cmdEpisod1(episod1.name, (BITMAP*)main_data[TEXTURE2].dat, font, view_episod1_highscores_proc);
   texture_button_object cmdEpisod2(episod2.name, (BITMAP*)main_data[TEXTURE2].dat, font, view_episod2_highscores_proc);
   texture_button_object cmdEpisod3(episod3.name, (BITMAP*)main_data[TEXTURE2].dat, font, view_episod3_highscores_proc);
   texture_button_object cmdEpisod4(episod4.name, (BITMAP*)main_data[TEXTURE2].dat, font, view_episod4_highscores_proc);
   texture_button_object cmdEpisod5(episod5.name, (BITMAP*)main_data[TEXTURE2].dat, font, view_episod5_highscores_proc);
   texture_button_object cmdEpisodCustom("Custom Episod", (BITMAP*)main_data[TEXTURE2].dat, font, view_episod_custom_highscores_proc);

   texture_button_object cmdExit("Exit", (BITMAP*)main_data[TEXTURE2].dat, font, exit_high_scores_dialog_proc);
   clear_screen_object clrObject;
   clrObject.set_draw_callback(ClearHighScoresProc);

   dialog the_dialog;
   pHSdialog = &the_dialog;
   the_dialog.add(clrObject);
   the_dialog.add(cmdExit,          40,  10,  95, 20);
   the_dialog.add(cmdEpisod1,       20, 450,  95, 20);
   the_dialog.add(cmdEpisod2,      120, 450,  95, 20);
   the_dialog.add(cmdEpisod3,      220, 450,  95, 20);
   the_dialog.add(cmdEpisod4,      320, 450,  95, 20);
   the_dialog.add(cmdEpisod5,      420, 450,  95, 20);
   the_dialog.add(cmdEpisodCustom, 520, 450,  95, 20);
   the_dialog.execute();

   game_dialog->redraw();
   high_scores_view_place = -1;
   return FALSE;
}

/////////////////// End of HighScores dialog ///////////////////////

int HelpGameProc(void)
{
   help_main();
   game_dialog->redraw();
   return FALSE;
}

int ExitGameProc(void)
{
   game_dialog->close();
   return FALSE;
}

int play_game_dialog(void);

int glizda_main(void)
{
   clear_screen_object clrObject;
   clrObject.set_draw_callback(draw_texture);
   frame_bitmap_object frame1((BITMAP*)main_data[FRAME4].dat);
   texture_button_object cmdPlayGame("New Game", (BITMAP*)main_data[TEXTURE2].dat, (FONT*)main_data[TITLE_FONT].dat, play_game_dialog);
   texture_button_object cmdOptions("Options", (BITMAP*)main_data[TEXTURE2].dat, (FONT*)main_data[TITLE_FONT].dat, OptionsProc);
   texture_button_object cmdCredits("Credits", (BITMAP*)main_data[TEXTURE2].dat, (FONT*)main_data[TITLE_FONT].dat, CreditsProc);
   texture_button_object cmdHighScores("View High Scores", (BITMAP*)main_data[TEXTURE2].dat, (FONT*)main_data[TITLE_FONT].dat, HighScoresProc);
   texture_button_object cmdHelp("Help", (BITMAP*)main_data[TEXTURE2].dat, (FONT*)main_data[TITLE_FONT].dat, HelpGameProc);
   texture_button_object cmdExit("Exit", (BITMAP*)main_data[TEXTURE2].dat, (FONT*)main_data[TITLE_FONT].dat, ExitGameProc);

   dialog the_dialog;
   game_dialog = &the_dialog;

   the_dialog.add(clrObject);
   the_dialog.add(frame1,          160,  100,  280, 360);
   the_dialog.add(cmdPlayGame,     200,  150,  200, 30);
   the_dialog.add(cmdOptions,      200,  200,  200, 30);
   the_dialog.add(cmdCredits,      200,  250,  200, 30);
   the_dialog.add(cmdHighScores,   200,  300,  200, 30);
   the_dialog.add(cmdHelp,         200,  350,  200, 30);
   the_dialog.add(cmdExit,         200,  400,  200, 30);
   the_dialog.execute();
   return FALSE;
}

int exit_play_dialog_proc(void)
{
   play_dialog->close();
   return FALSE;
}

int play_game_dialog(void)
{
   score_bitmap = create_bitmap(150,150);
   clear_screen_object clrObject;
   clrObject.set_draw_callback(draw_texture);

   frame_bitmap_object frame1((BITMAP*)main_data[FRAME4].dat);

   texture_button_object cmdEpisod1(episod1.name, (BITMAP*)main_data[TEXTURE2].dat, (FONT*)main_data[TITLE_FONT].dat);
   cmdEpisod1.set_click_proc(play_episod1_proc);

   texture_button_object cmdEpisod2(episod2.name, (BITMAP*)main_data[TEXTURE2].dat, (FONT*)main_data[TITLE_FONT].dat);
   cmdEpisod2.set_click_callback(play_episod2_proc);

   texture_button_object cmdEpisod3(episod3.name, (BITMAP*)main_data[TEXTURE2].dat, (FONT*)main_data[TITLE_FONT].dat);
   cmdEpisod3.set_click_callback(play_episod3_proc);

   texture_button_object cmdEpisod4(episod4.name, (BITMAP*)main_data[TEXTURE2].dat, (FONT*)main_data[TITLE_FONT].dat);
   cmdEpisod4.set_click_callback(play_episod4_proc);

   texture_button_object cmdEpisod5(episod5.name, (BITMAP*)main_data[TEXTURE2].dat, (FONT*)main_data[TITLE_FONT].dat);
   cmdEpisod5.set_click_callback(play_episod5_proc);

   texture_button_object cmdSingleMap("Custom Map", (BITMAP*)main_data[TEXTURE2].dat, (FONT*)main_data[TITLE_FONT].dat);
   cmdSingleMap.set_click_proc(PlayMapProc);

   texture_button_object cmdEpisod("Custom Episod", (BITMAP*)main_data[TEXTURE2].dat, (FONT*)main_data[TITLE_FONT].dat);
   cmdEpisod.set_click_proc(play_custom_episod_proc);

   texture_button_object cmdExit("Previous Menu", (BITMAP*)main_data[TEXTURE2].dat, (FONT*)main_data[TITLE_FONT].dat, exit_play_dialog_proc);

   dialog the_dialog;
   play_dialog = &the_dialog;

   the_dialog.add(clrObject);
   the_dialog.add(frame1,          160,   70,  280, 400);
   the_dialog.add(cmdEpisod1,      200,  110,  200, 30);
   the_dialog.add(cmdEpisod2,      200,  150,  200, 30);
   the_dialog.add(cmdEpisod3,      200,  190,  200, 30);
   the_dialog.add(cmdEpisod4,      200,  230,  200, 30);
   the_dialog.add(cmdEpisod5,      200,  270,  200, 30);
   the_dialog.add(cmdSingleMap,    200,  320,  200, 30);
   the_dialog.add(cmdEpisod,       200,  360,  200, 30);
   the_dialog.add(cmdExit,         200,  400,  200, 30);
   the_dialog.execute();
   game_dialog->redraw();
   destroy_bitmap(score_bitmap);
   return FALSE;
}
