/****************************************************
 * Project:   Glizda Game
 * FILE:      episod.cc
 * Author:    Andrew Deren
 * Date:      1/2/98
 * Length:    756
 ****************************************************/

#include <dir.h>
#include "glizda.h"
#include "agui.h"

/********************* GLOBAL VARIABLES ***********************/
char *name_getter(int, int*);
char temp_string[256];

list_object *lstNamesPtr;      // list with names of maps
dialog *episod_dialog;         // the main dialog

char e_name[26];
char e_file_name[256];
int num_maps;
MAP_INFO maps[MAX_MAPS];        // maps

/****************************************************************/

class small_map_object : public dialog_object {
protected:
   BITMAP *map_bitmap;       //map bitmap used for drawing as double buffer
   MAP_INFO *map;
public:
   small_map_object(MAP_INFO *new_map);
   ~small_map_object(void);
   virtual void msg_draw(void);  //draw object
   virtual bool msg_wantfocus(void) {return FALSE;}   //object does not want focus
   void set_map(MAP_INFO *new_map) {map = new_map;}
   MAP_INFO *get_map(void) {return map;}
};
small_map_object *map_object_ptr;

small_map_object::small_map_object(MAP_INFO *new_map)
{
   map_bitmap = create_bitmap(18*24, 18*24);    //create double buffer for drawing
   set_map(new_map);
}

small_map_object::~small_map_object(void)
{
   destroy_bitmap(map_bitmap);      //free memory taken by double buffer
}

void small_map_object::msg_draw(void)
{
   int i, j, temp;
   int selected = lstNamesPtr->selection();
   if (selected < 0 || selected >= num_maps) return;
   map = &(maps[selected]);
   if (!map) return;
   clear(map_bitmap);   //clear bitmap
   //draw all values
   for (i=0; i<18; i++) {
      for (j=0; j<18; j++) {
         temp = map->data[i][j];
         if (temp > 0 && temp <= 20) {
            //draw sprites; sprites values are taken from map.values[i][j] and they
            //correspond to array index of sprites_data file
            draw_sprite(map_bitmap, (BITMAP*)sprites_data[temp].dat, i*24, j*24);
         }
         else if (temp > 100 && temp <= 120) {
            //draw walls (object glizda cannot enter). the values are map.values[i][j]
            // - 100 and graphics are taken from walls_data
            draw_sprite(map_bitmap, (BITMAP*)walls_data[temp-100].dat, i*24, j*24);
         }  //end else if
         else if (temp > 200 && temp <= 220) {
            draw_sprite(map_bitmap, (BITMAP*)special_data[temp-200].dat, i*24, j*24);
         }
      }  //end for j
   }  //end for i

   //draw glizda
   draw_sprite(map_bitmap, (BITMAP*)glizda_data[RIGHT_OPEN].dat, map->glizda_x*24, map->glizda_y*24);
   draw_sprite(map_bitmap, (BITMAP*)glizda_data[BODY].dat, (map->glizda_x-1)*24, map->glizda_y*24);
   draw_sprite(map_bitmap, (BITMAP*)glizda_data[BODY].dat, (map->glizda_x-2)*24, map->glizda_y*24);
   draw_sprite(map_bitmap, (BITMAP*)glizda_data[TAIL_RIGHT].dat, (map->glizda_x-3)*24, map->glizda_y*24);

   //blit the bitmap to screen
   stretch_blit(map_bitmap, screen, 0, 0, map_bitmap->w, map_bitmap->h, x(), y(), w(), h());
}



edittext_object *edtEpisodNamePtr;
text_object *lblE_FileNamePtr;

int new_episod(void);
void close_episod(void);



/******************************************************************/
// file_preview_dialog stuff
#define MAX_FILES 2048
char *file_names[MAX_FILES];
int num_files;
list_object *lstFileNames;

void init_file_preview(void)
{
   num_files = 0;
   for (int i=0; i<MAX_FILES; i++)
       file_names[i] = NULL;
}

void free_files(void)
{
   int i;
   for (i=0; i<num_files; i++)
       free(file_names[i]);
}

void get_files(void)
{
   struct ffblk f;
   int done;
   done = findfirst("*.map", &f, 0);
   while (!done) {
      file_names[num_files] = (char*)malloc(sizeof(char)*strlen(f.ff_name)+1);
      strcpy(file_names[num_files], f.ff_name);
      num_files++;
      done = findnext(&f);
   }
}

// procedure for getting the names of files
char *file_name_getter(int which, int *list_size)
{
   if (which < 0) {
      *list_size = num_files;
      return NULL;
   }
   else return file_names[which];
}

int delete_file_proc(void)
{
   int selected = lstFileNames->selection();
   if (selected < 0 || selected >= num_files) return FALSE;
   delete_file(file_names[selected]);
   free_files();
   init_file_preview();
   get_files();
   lstFileNames->redraw();
   return FALSE;
}

//rename returns 0 on success rename(old_name, new_name)
int rename_file_proc(void)
{
   int selected = lstFileNames->selection();
   if (selected < 0 || selected >= num_files) return FALSE;
   char new_file[100];
   if (!input_box("Rename", "Enter new file name:", new_file))
      return FALSE;
   rename(file_names[selected], new_file);
   init_file_preview();
   get_files();
   lstFileNames->redraw();
   return FALSE;
}

/**************************** end of file dialog ***************/

// make new episod
int new_episod(void)
{
   int i;
   strcpy(e_name, "untitled episod");       // set default name of the episod
   strcpy(e_file_name, "untitled.gep");     // set default file name
   edtEpisodNamePtr->set_text(e_name);
   lblE_FileNamePtr->set_text(get_filename(e_file_name));
   num_maps = 0;                          // num maps is 0
   return FALSE;
}

// save episod to file using current e_file_name
void save_episod(void)
{
   FILE *file;
   INT version = 10;    // current version is 1.0 (10)
   if (num_maps == 0) return;      // no maps to be saved
   // open file
   file = fopen(e_file_name, "wb");
   //could not open
   if (!file) return;
   // put ID string
   fwrite("GLIZDA EPISOD", sizeof(char), strlen("GLIZDA EPISOD")+1, file);
   // put version
   fwrite(&version, sizeof(INT), 1, file);
   // put name of episod
   fwrite(edtEpisodNamePtr->get_text(), sizeof(char), 26, file);
   lblE_FileNamePtr->set_text(get_filename(e_file_name));
   // put num maps
   fwrite(&num_maps, sizeof(INT), 1, file);
   // put all maps
   fwrite(maps, sizeof(MAP_INFO), num_maps, file);
   fclose(file);
}

// open the episod
void open_episod(void)
{
   FILE *file;
   INT version;
   char temp[20];

   // try opening the file
   file = fopen(e_file_name, "rb");
   if (!file) return;     // could not open

   // get the file ID
   fread(temp, sizeof(char), strlen("GLIZDA EPISOD")+1, file);
   // check if file is glizda episod
   if (strcmp(temp, "GLIZDA EPISOD") != 0) {
      fclose(file);
      return;
   }
   // check for the right version
   fread(&version, sizeof(INT), 1, file);
   if (version != 10) {
      fclose(file);
      return;
   }
   // get the name of the episod
   fread(e_name, sizeof(char), 26, file);
   edtEpisodNamePtr->set_text(e_name);
   lblE_FileNamePtr->set_text(get_filename(e_file_name));
   // get number of maps
   fread(&num_maps, sizeof(INT), 1, file);
   // read the maps
   fread(maps, sizeof(MAP_INFO), num_maps, file);
   fclose(file);
}

// menu selection: File - Exit
int mnuExitProc(void)
{
   episod_dialog->close();
   return FALSE;
}

//help
int ProcEpisodHelp(void)
{
   static char *help_text =
           "Episod Creating Help: \n"
           "----------------------\n"
           "Episods are sequences of maps put together and "
           "are played one after another. Up to 100 maps "
           "can be stored in one episod, and for each map "
           "can be specified time limit that the map can "
           "be played, and speed of glizda. Speed of the glizda "
			  "ranges from 1 (fastest) to 10 (slowest) and time is "
			  "measured in seconds.\n\n"
           "To add a map either choose add map from map menu "
           "or click add map button. To change map properites "
           "(speed, name, time limit) select its properities from "
           "map menu or click prop button. The grab button is "
			  "when you want to change already existing map to a "
			  "different one.\n\n";
   // Object definitions - DEGUI 1.2 beta
   box_object panMain;
   centerd_text_object lblTitle("H E L P");
   textbox_object edtHelp(help_text,true,true);
   button_object cmdOK("Ok",0,true);

   // Add Objects to the dialog
   dialog the_dialog;
   the_dialog.add(panMain,140,90,260,275);
   the_dialog.add(lblTitle,264,100,28,8);
   the_dialog.add(edtHelp,145,115,250,210);
   the_dialog.add(cmdOK,240,335,60,25);
   the_dialog.center();
   the_dialog.popup();

   return FALSE;
}

//diasplay about information
int ProcEpisodAbout(void)
{
   // Object definitions - DEGUI 1.2 beta
   shadow_box_object object_1;
   bitmap_object bmpAbout((BITMAP*)main_data[ABOUT_EPISOD].dat);
   panel_ridge_object object_3;
   button_object cmdOK("OK",0,true);
   centerd_text_object lbl_GE("Glizda Editor v1.0");
   centerd_text_object lbl_AD("Andrew Deren 1997");
   centerd_text_object lbl_About("ABOUT");

   // Add Objects to the dialog
   dialog a_dialog;
   a_dialog.add(object_1,100,100,220,280);
   a_dialog.add(bmpAbout,110,130,200,175);
   a_dialog.add(object_3,105,105,210,20);
   a_dialog.add(cmdOK,190,350,36,25);
   a_dialog.add(lbl_GE,210,320,72,8);
   a_dialog.add(lbl_AD,210,335,68,8);
   a_dialog.add(lbl_About,215,110,20,8);
   a_dialog.center();
   a_dialog.popup();
   return TRUE;
}

// if current is -1 then new map is created and added at the end
// else current map is updated
int add_file_to_list(char *file_name, int current_map = -1)
{
   FILE *file;       //file
   MAP tmap;         //temp map
  	file = fopen(file_name, "rb");      //try to open file
   if (!file) {         //opening failed
     	alert("Error opening map.", NULL, NULL, "OK", NULL, KEY_ENTER, 0);
      return FALSE;
   }
  	fread(&tmap, sizeof(struct MAP), 1, file);
   fclose(file);              //close file
    //check if valid file type
   if (strcmp(tmap.IdString, "glizda") != 0) {
      //is map is invalid; ask if should be opened anyway
      alert("Error", "File of invalid type", NULL, "OK", NULL, KEY_ENTER, KEY_ESC);
      return FALSE;
   }
   //check if valid version
   if (strcmp(tmap.version, "v1.0") != 0) {
      //if invalid version ask if should be opened any way
      alert("Error!!!", " file of invalid version.", NULL, "OK", NULL, KEY_ENTER, KEY_ESC);
      return FALSE;
   }
   // get the values from single map and copy them to map info
   // structure and add that map to episod list
   int insert_map;
   if (current_map == -1) {
      insert_map = num_maps;
      if (num_maps == MAX_MAPS-1) return FALSE;
      num_maps++;
      lstNamesPtr->selection(num_maps-1);
   }
   else
       insert_map = current_map;
   strcpy(maps[insert_map].map_name, get_filename(file_name));
   maps[insert_map].num_sprites = tmap.sprites_to_eat;

   // when updating map don't change time and speed
   if (current_map != -1) {
      maps[insert_map].time_limit = 60;
      maps[insert_map].speed = 5;
   }
   maps[insert_map].glizda_x = tmap.glizda_x;
   maps[insert_map].glizda_y = tmap.glizda_y;
   memcpy(maps[insert_map].data, tmap.values, (MAP_SIZE*MAP_SIZE)*(sizeof(BYTE)));
   episod_dialog->redraw();
   return FALSE;
}

// add new map to the list
int add_map_proc(void)
{
   char file_name[256];
   file_name[0] = '\0';

   //open file dialog
   if (file_select("Open Map", file_name, "map") != 0) {
      add_file_to_list(file_name);
   }  //end if file_select
   return FALSE;
}

// take a file from the file list selection and add to the episod
int add_file_proc(void)
{
   int selected = lstFileNames->selection();
   if (selected < 0 || selected >= num_files) return FALSE;
   add_file_to_list(file_names[selected]);
   return FALSE;
}

int delete_map_proc(void)
{
   if (num_maps == 0) return FALSE;
   int which = lstNamesPtr->selection();
   MAP_INFO temp_map;
   if (which >=0 && which < num_maps) {
      for (int i=which; i<num_maps; i++)  // shift all maps above it down
         memcpy(&(maps[i]), &(maps[i+1]), sizeof(MAP_INFO));
      num_maps--;                         // decrease number of maps
   }
   if (num_maps <= which)
      lstNamesPtr->selection(which-1);
   episod_dialog->redraw();
   return FALSE;
}

int grab_map_proc(void)
{
   if (num_maps == 0) return FALSE;
   int which = lstNamesPtr->selection();
   if (which >=0 && which < num_maps) {
      char file_name[256];
      file_name[0] = '\0';
      //open file dialog
      if (file_select("Open Map", file_name, "map") != 0) {
         add_file_to_list(file_name, which);
      }  //end if file_select
   }
   return FALSE;
}


int up_map_proc(void)
{
   int which = lstNamesPtr->selection();
   if (which < 1 || which >= num_maps) return FALSE;
   MAP_INFO temp;
   memcpy(&temp, &(maps[which]), sizeof(MAP_INFO));
   memcpy(&(maps[which]), &(maps[which-1]), sizeof(MAP_INFO));
   memcpy(&(maps[which-1]), &temp, sizeof(MAP_INFO));
   lstNamesPtr->selection(which-1);
   lstNamesPtr->redraw();
   map_object_ptr->redraw();
   return FALSE;
}

int down_map_proc(void)
{
   int which = lstNamesPtr->selection();
   if (which < 0 || which >= num_maps-1) return FALSE;
   MAP_INFO temp;
   memcpy(&temp, &(maps[which]), sizeof(MAP_INFO));
   memcpy(&(maps[which]), &(maps[which+1]), sizeof(MAP_INFO));
   memcpy(&(maps[which+1]), &temp, sizeof(MAP_INFO));

   lstNamesPtr->selection(which+1);
   lstNamesPtr->redraw();
   map_object_ptr->redraw();
   return FALSE;
}

int map_properties_proc(void)
{
   int selected = lstNamesPtr->selection();
   char map_name[MAP_NAME_SIZE];
   char map_speed[4];
   char map_time[4];

   strcpy(map_name, maps[selected].map_name);
   sprintf(map_speed, "%d", maps[selected].speed);
   sprintf(map_time, "%d", maps[selected].time_limit);

   if (selected < 0 || selected > num_maps-1) return FALSE;
     // Object definitions - DEGUI 1.2 beta
   box_object object_1;
   panel_groove_object object_12;
   panel_ridge_object object_3;
   centerd_text_object object_2("Map Properties");
   text_object txtName("Name:");
   text_object txtSpeed("Speed:");
   text_object txtTime("Time:");
   edittext_object edtName(map_name, MAP_NAME_SIZE-1);
   edittext_object edtSpeed(map_speed,3);
   edittext_object edtTime(map_time,3);
   button_object cmdOk("OK",0,true);

   // Add Objects to the dialog
   dialog the_dialog;
   the_dialog.add(object_1,225,170,210,135);
   the_dialog.add(object_12,230,200,200,70);
   the_dialog.add(object_3,230,175,200,20);
   the_dialog.add(object_2,335,180,56,8);
   the_dialog.add(txtName,235,215,40,8);
   the_dialog.add(txtSpeed,235,245,48,8);
   the_dialog.add(txtTime,340,245,40,8);
   the_dialog.add(edtName,295,210,125,15);
   the_dialog.add(edtSpeed,295,240,32,13);
   the_dialog.add(edtTime,385,240,35,15);
   the_dialog.add(cmdOk,295,275,70,25);
   the_dialog.center();
   the_dialog.popup();

   int t_speed = atoi(edtSpeed.get_text());
   if (t_speed > 9 || t_speed < 1) {
      alert("The speed has to be between 1 and 9.", "Was defaulted to 5 for this map.", NULL, "OK", NULL, KEY_ENTER, KEY_ESC);
      t_speed = 5;
   }
   strcpy(maps[selected].map_name, edtName.get_text());
   maps[selected].speed = t_speed;
   maps[selected].time_limit = atoi(edtTime.get_text());
   episod_dialog->redraw();
   return FALSE;
}


// procedure for displaying the list with map names
char *name_getter(int which, int *list_size)
{
   if (which < 0) {
      *list_size = num_maps;
      return NULL;
   }
   else return maps[which].map_name;
}

// new episod
int E_ProcNew(void)
{
   int t;
   if (num_maps > 15)
      t = 1;
   else t = 0;
   new_episod();
   num_maps = t;
   lstNamesPtr->selection(0);
   episod_dialog->redraw();
//   delete_map_proc();
}

// save the episod
int E_ProcSave(void)
{
   save_episod();
}

int E_ProcSaveAs(void)
{
   char t_file_name[256];
   t_file_name[0] = '\0';
   //open file dialog
   if (file_select("Save As (*.gep)", t_file_name, "gep") != 0) {
      strcpy(e_file_name, t_file_name);
      save_episod();
   }
   episod_dialog->redraw();
}

int E_ProcOpen(void)
{
   char t_file_name[256];
   t_file_name[0] = '\0';
   if (file_select("Open (*.gep)", t_file_name, "gep") != 0) {
      strcpy(e_file_name, t_file_name);
      open_episod();
   }
   if (num_maps > 0)
      lstNamesPtr->selection(0);
   episod_dialog->redraw();
}

void selection_proc(void)
{
   lstNamesPtr->redraw();
   map_object_ptr->redraw();
}

int extract_map_proc(void)
{
   char file_name[256];
   FILE *file;       //file
   MAP tmap;         //temp map
   file_name[0] = '\0';
   int which = lstNamesPtr->selection();
   if (which < 0 || which >= num_maps)
      return FALSE;

   //open file dialog
   if (file_select("Save Map As", file_name, "map") != 0) {
      if (exists(file_name)) {
         if (!alert("Error", "File Already Exists", "Overwrite?", "Yes", "No", KEY_ENTER, KEY_ESC))
            return FALSE;
      }
     	file = fopen(file_name, "wb");      //try to open file
      if (!file) {         //opening failed
        	alert("Error opening map.", NULL, NULL, "OK", NULL, KEY_ENTER, 0);
         return FALSE;
      }
      memcpy(tmap.values, maps[which].data, (MAP_SIZE*MAP_SIZE)*(sizeof(BYTE)));
      tmap.glizda_x = maps[which].glizda_x;
      tmap.glizda_y = maps[which].glizda_y;
      tmap.sprites_to_eat = maps[which].num_sprites;
      strcpy(tmap.IdString, "glizda");  //map IdString is "glizda"
      strcpy(tmap.version, "v1.0");     //map version is "v1.0"
      fwrite(&tmap, sizeof(struct MAP), 1, file);
      //close file
      fclose(file);
   }  //end if file_select
   return FALSE;
}

int episod_screen_dump(void)
{
      RGB title_pal[256];
      get_palette(title_pal);
      //create temp bitmap
      BITMAP *tbmp = create_sub_bitmap(screen, 0, 0, SCREEN_W, SCREEN_H);
      //save the picture with game pallete
      save_pcx("episod.pcx", tbmp, title_pal);
      destroy_bitmap(tbmp);     //destroy bitmap
      return FALSE;
}


int episod_main(void)
{
   init_file_preview();
   get_files();

   char pointer[256];

   num_maps = 0;

   dialog the_dialog;
   episod_dialog = &the_dialog;
   menu mnuFile;
   menu mnuMap;
   menu mnuHelp;
   menu allegro_menu;

   // menus
   mnuFile.add("New", E_ProcNew);

   mnuFile.add("Open", E_ProcOpen);

   if (shareware)
      mnuFile.add("Save As", reg_error_dialog);
   else
      mnuFile.add("Save As", E_ProcSaveAs);

   if (shareware)
      mnuFile.add("Save", reg_error_dialog);
   else
      mnuFile.add("Save", E_ProcSave);

   mnuFile.add();
   mnuFile.add("Save Screen to File", episod_screen_dump);
   mnuFile.add();
   mnuFile.add("Exit", mnuExitProc);

   mnuMap.add("Extract", extract_map_proc);
   mnuMap.add();
   mnuMap.add("Grab Map", grab_map_proc);
   mnuMap.add("Add Map", add_map_proc);
   mnuMap.add("Delete Map", delete_map_proc);
   mnuMap.add("Move Map Up", up_map_proc);
   mnuMap.add("Move Map Down", down_map_proc);
   mnuMap.add();
   mnuMap.add("Map Properties", map_properties_proc);

   mnuHelp.add("Help", ProcEpisodHelp);
   mnuHelp.add();
   mnuHelp.add("About", ProcEpisodAbout);

   allegro_menu.add("&File", mnuFile);
   allegro_menu.add("&Map", mnuMap);
   allegro_menu.add("&Help", mnuHelp);

   menu_object menu(allegro_menu);


   list_object lstNames(name_getter);      // list with names of maps
   lstNamesPtr = &lstNames;

   lstNames.set_change_selection_proc(selection_proc);

   small_map_object map_object(NULL);
   map_object_ptr = &map_object;

   edittext_object edtEpisodName(e_name, 25);
   edtEpisodNamePtr = &edtEpisodName;

   text_object lblE_FileName(e_file_name);
   lblE_FileNamePtr = &lblE_FileName;

   // GUI Objects
   clear_screen_object clrObject;
   panel_groove_object pnlEpisodInfo;
   text_object lblMaps("Maps:");
   panel_groove_object pnlMaps;
   panel_sunken_object pnlAroundMap;
   text_object lblNumMaps("Num Maps:");

   integer_text_object txtNumMaps(&num_maps);

   text_object txtEpisodName("Episod Name:");
   centerd_text_object txtEpisodInfo("Episod Info:");

   win_button_object cmdAddEpisod("Add",0,false, add_map_proc);
   win_button_object cmdDeleteEpisod("Del",0,false, delete_map_proc);
   win_button_object cmdUpEpisod("Up",0,false, up_map_proc);
   win_button_object cmdDownEpisod("Down",0,false, down_map_proc);
   win_button_object cmdProp("Prop", 0, false, map_properties_proc);
   win_button_object cmdGrab("Grab", 0, false, grab_map_proc);

   text_object txtFileName("File Name:");

   text_object txtAbout("Glizda Episod Editor v1.00");
   text_object txtAbout2(" (c)  Andrew Deren 1997");
   panel_groove_object pnlAbout;

   panel_groove_object pnlFiles;
   list_object lstFiles(file_name_getter);
   lstFileNames = &lstFiles;
   win_button_object cmdDelFile("Del", 0, false, delete_file_proc);
   win_button_object cmdRenFile("Ren", 0, false, rename_file_proc);
   win_button_object cmdAddFile("Add", 0, false, add_file_proc);


   // Add Objects to the dialog
   the_dialog.add(clrObject);
   the_dialog.add(menu);

   the_dialog.add(pnlAroundMap,     15,   30,   375,  390);
   the_dialog.add(map_object,       17,   32,   372,  387);

   the_dialog.add(pnlAbout,         15,  430,   155,   37);
   the_dialog.add(txtAbout,         20,  435,   155,    9);
   the_dialog.add(txtAbout2,        20,  450,   155,    9);


   the_dialog.add(pnlEpisodInfo,   395,   30,   235,   80);
   the_dialog.add(txtEpisodInfo,   510,   35,    48,    8);
   the_dialog.add(txtFileName,     400,   55,    80,    8);
   the_dialog.add(lblE_FileName,   500,   55,    96,    8);
   the_dialog.add(txtEpisodName,   400,   75,    96,    8);
   the_dialog.add(edtEpisodName,   500,   70,   120,   15);
   the_dialog.add(lblNumMaps,      400,   95,    72,    8);
   the_dialog.add(txtNumMaps,      500,   95,    35,   10);


   the_dialog.add(pnlMaps,         395,   115,  235,  255);
   the_dialog.add(lblMaps,         460,   120,   40,    8);
   the_dialog.add(lstNames,        455,   135,  170,  225);

   the_dialog.add(cmdAddEpisod,    405,   135,   45,   25);
   the_dialog.add(cmdDeleteEpisod, 405,   165,   45,   25);
   the_dialog.add(cmdUpEpisod,     405,   195,   45,   25);
   the_dialog.add(cmdDownEpisod,   405,   225,   45,   25);
   the_dialog.add(cmdProp,         405,   255,   45,   25);
   the_dialog.add(cmdGrab,         405,   285,   45,   25);

   the_dialog.add(pnlFiles,        395,   375,  235,  100);
   the_dialog.add(lstFiles,        455,   380,  170,   90);
   the_dialog.add(cmdDelFile,      405,   380,   45,   25);
   the_dialog.add(cmdRenFile,      405,   410,   45,   25);
   the_dialog.add(cmdAddFile,      405,   440,   45,   25);

   new_episod();
   the_dialog.execute();
   free_files();
   return FALSE;
}



