/****************************************************
 * Project:   Glizda Game
 * FILE:      agui.cc
 * Author:    Andrew Deren
 * Date:      1/2/98
 * Length:    598
 ****************************************************/

#include "agui.h"
#include "editor.h"
#include "glizda.h"

void textout_right(BITMAP *b, FONT* f, char *string, int x, int y, int color) {
   textout(b, f, string, x-text_length(f, string), y, color);
}


/******************************* frame_object **************************/

void frame_object::msg_draw(void)
{
   switch (type) {   //what type of frame it is
   case PANEL_RAISED_FRAME:  //draw panel raised frame
      draw_3d_frame(screen,x(),y(),x2(),y2(), -1,
                   color ? color->light_shad() : degui_light_shad_color,
                   color ? color->dark_shad() : degui_dark_shad_color);
      break;
   case PANEL_SUNKEN_FRAME:  //draw sunken frame
         draw_3d_frame(screen,
                 x(),y(),x2(),y2(), -1,
                 color ? color->dark_shad() : degui_dark_shad_color,
                 color ? color->light_shad() : degui_light_shad_color);
         break;
   case PANEL_RIDGE_FRAME: //draw ridge frame
      draw_3d_frame(screen,x(),y(),x2(),y2(),-1,
                 color ? color->light_shad() : degui_light_shad_color,
                 color ? color->dark_shad() : degui_dark_shad_color);
      draw_3d_frame(screen,x()+1,y()+1,x2()-1,y2()-1,-1,
                 color ? color->dark_shad() : degui_dark_shad_color,
                 color ? color->light_shad() : degui_light_shad_color);
      break;
   case PANEL_GROOVE_FRAME:   //draw grove frame
      draw_3d_frame(screen,x(),y(),x2(),y2(),-1,
                color ? color->dark_shad() : degui_dark_shad_color,
                color ? color->light_shad() : degui_light_shad_color);
      draw_3d_frame(screen,x()+1,y()+1,x2()-1,y2()-1,-1,
                color ? color->light_shad() : degui_light_shad_color,
                color ? color->dark_shad() : degui_dark_shad_color);
      break;
   }
   _redraw = false;
}


/*********************************** bitmap_radio_object ****************/

bitmap_radio_object::bitmap_radio_object(BITMAP *b = NULL, int g = 1)
                    :radio_button_object(NULL, g, FALSE)
{
   bitmap = b; //set bitmap
}

//draw the object
void bitmap_radio_object::msg_draw(void)
{
   //fill the area under bitmap
   rectfill(screen, x(), y(), x2(), y2(),  color ? color->back() : degui_back_color);
   //stretch and draw bitmap
   stretch_sprite(screen, bitmap, x()+1, y()+1, w()-1, h()-1);
   if (selected())        //if object is selected draw sunken frame
      draw_3d_frame(screen, x(), y(), x2(), y2(), -1,
                 color ? color->dark_shad() : degui_dark_shad_color,
                 color ? color->light_shad() : degui_light_shad_color);
   else if (has_mouse())  //when mouse is over object draw raised frame
      draw_3d_frame(screen, x(), y(), x2(), y2(), -1,
                 color ? color->light_shad() : degui_light_shad_color,
                 color ? color->dark_shad() : degui_dark_shad_color);
   _redraw = FALSE;
}

/******************************* color_select_object ****************/
color_select_object::color_select_object(int c = 0)
{
   current_color = c;   //set current color
   mx = my = 0;
   sprintf(tcolor, "%3d", current_color);  //put current color in text color
   t_object = NULL;      //text object is null
}

//draw the object
void color_select_object::msg_draw(void)
{
   int i, j, c;
   c = 0;
   //draw 256 squares 10x10 pixels size aligned 16x16
   for (i=0; i<16; i++) {
      for (j=0; j<16; j++) {
         rectfill(screen, x()+i*10, y()+j*10, x()+i*10+10, y()+j*10+10, c);
         c++;
      }
   }
   //put white frame
   rect(screen, (current_color / 16)*10+x(), (current_color%16)*10+y(),
                (current_color / 16)*10+10+x(), (current_color%16)*10+10+y(),
                color ? color->back() : degui_back_color);
   _redraw = FALSE;
}

//handle the click event
void color_select_object::msg_click(void)
{
   int c;
   do {
      mx = (mouse_x - x()) / 10;
      if (mx < 0 || mx > 15) mx = -1;
      my = (mouse_y - y()) / 10;
      if (my < 0 || my > 15) my = -1;
      if (mx == -1 || my == -1)
         c = -1;
      else
         c = mx*16+my;
      if (c != current_color) {
         if (c != -1) {
            current_color = c;
            sprintf(tcolor, "%3d", current_color);
            redraw();
            if (t_object != NULL) {
               t_object->set_text(tcolor);
               t_object->redraw();
            }
         }
      }
      parent()->dialog_message(MSG_IDLE, 0);
   } while (mouse_b);
}

//displays the dialog with color selection object
int do_color_dialog(int tx = 0, int ty = 0, int c=0)
{
   dialog _color_dialog;   //the main dialog

   button_object cmdOk("OK", KEY_SPACE, TRUE);  //ok button
   color_select_object clrObj(c);               //color selection object
   text_object lblColor("Color: ");             //"color: " label
   text_object lblNumber(NULL);                 //object that displays current color number
   panel_raised_object mainPanel;               //panel under the dialog

   clrObj.set_text_object(&lblNumber);          //get color object text object
                                                //so that it can display which color
                                                //is selected

   //add objects to dialog
   _color_dialog.add(mainPanel, tx, ty, 180, 205);
   _color_dialog.add(lblColor, tx+10, ty+5, 40, 10);
   _color_dialog.add(lblNumber, tx+10, ty+18, 40, 10);
   _color_dialog.add(cmdOk, tx+100, ty+5, 64, 20);
   _color_dialog.add(clrObj, tx+10, ty+35, 160, 160);

   _color_dialog.popup();

   return clrObj.get_color(); //return color selected in dialog
}


/********************************** map_object *************************/
//constructor
MapObject::MapObject(void)
{
   mx = my = current_x = current_y = -1;  //set temp values to -1
   map_bitmap = create_bitmap(18*24, 18*24);    //create double buffer for drawing
   grid_color = 7;                   //set grid color to 7
   location = NULL;
   GridOn(TRUE);                     //set grid to be on
}

MapObject::~MapObject(void)
{
   destroy_bitmap(map_bitmap);      //free memory taken by double buffer
}

//when mouse enters object
void MapObject::msg_gotmouse(void)
{
   mx = (mouse_x - x()) / 24;    //mx is the x square the mouse is in
   my = (mouse_y - y()) / 24;    //my is the y square the mouse is in
   current_x = mx;
   current_y = my;
   redraw();                     //redraw map
}

void MapObject::msg_lostmouse(void)
{
   mx = my = current_x = current_y = -1;  //when lost focus no square is visible
   redraw();           //redraw object
}

void MapObject::msg_idle(void)
{
   if (grid_on != chkGrid->selected()) {
      grid_on = chkGrid->selected();
      redraw();
   }
   if (!has_mouse()) return;  //if mouse is not in object exit
   mx = (mouse_x - x()) / 24; //else update square
   my = (mouse_y - y()) / 24;
   //if mouse moved to different square change it
   if (mx != current_x || my != current_y) {
     current_x = mx;
     current_y = my;
     char loc_text[25];
     sprintf(loc_text, "x: %2d  y: %2d", current_x+1, current_y+1);
     if (location) {
        location->set_text(loc_text);
        location->redraw();
     }
     redraw();
   }
}

//draw map
void MapObject::msg_draw(void)
{
   int i, j, temp;
   clear(map_bitmap);   //clear bitmap
   //draw all values
   for (i=0; i<18; i++) {
      for (j=0; j<18; j++) {
         temp = map.values[i][j];
         if (temp > 0 && temp <= 20) {
            //draw sprites; sprites values are taken from map.values[i][j] and they
            //correspond to array index of sprites_data file
            draw_sprite(map_bitmap, (BITMAP*)sprites_data[temp].dat, i*24, j*24);
         }
         else if (temp > 100 && temp <= 120) {
            //draw walls (object glizda cannot enter). the values are map.values[i][j]
            // - 100 and graphics are taken from walls_data
            draw_sprite(map_bitmap, (BITMAP*)walls_data[temp-100].dat, i*24, j*24);
         }  //end else if
         else if (temp > 200 && temp <= 220) {
            draw_sprite(map_bitmap, (BITMAP*)special_data[temp-200].dat, i*24, j*24);
         }
      }  //end for j
   }  //end for i

   //draw glizda
   draw_sprite(map_bitmap, (BITMAP*)glizda_data[RIGHT_OPEN].dat, map.glizda_x*24, map.glizda_y*24);
   draw_sprite(map_bitmap, (BITMAP*)glizda_data[BODY].dat, (map.glizda_x-1)*24, map.glizda_y*24);
   draw_sprite(map_bitmap, (BITMAP*)glizda_data[BODY].dat, (map.glizda_x-2)*24, map.glizda_y*24);
   draw_sprite(map_bitmap, (BITMAP*)glizda_data[TAIL_RIGHT].dat, (map.glizda_x-3)*24, map.glizda_y*24);
   if (grid_on) DrawGrid();   //if grid is on draw it

   //if mouse is inside the map area draw white square under the mouse
   if (current_x >= 0 && current_y >= 0)
      rect(map_bitmap, current_x*24, current_y*24, current_x*24+24, current_y*24+24, 15);

   //blit the bitmap to screen
   blit(map_bitmap, screen, 0, 0, x(), y(), map_bitmap->w, map_bitmap->h);
}

//draw the grid
void MapObject::DrawGrid(void)
{
   int i;
   for (i = 0; i<18; i++) { //draw horizontal and vertical lines
       vline(map_bitmap, i*24, 0, 18*24, grid_color);
       hline(map_bitmap, 0, i*24, 18*24, grid_color);
   }
}

//handle the click
void MapObject::msg_click(void)
{
   CopyToUndo();
   //to_put is the value to be put in the map
   //if put_g is TRUE then glizda is put taggled by holding ctrl key
   int to_put, put_g = FALSE;

   //if left mouse button pressed put sprite
   if (mouse_b & 1) to_put = current_sprite;
   //else if right mouse button pressed put wall
   else if (mouse_b & 2) to_put = current_wall + 100;
   // if alt key pressed then put special sprite
   if (key_shifts & KB_CTRL_FLAG)
      to_put = current_special + 200;

   //if ctrl key is pressed put glizda
   if (key_shifts & KB_ALT_FLAG) put_g = TRUE;

   //make the mouse not go outside the map while pressed
   set_mouse_range(x(), y(), x2()-1, y2()-1);

   //repeat loop until mouse is pressed
   do {
      //check if right value is to be put
      if (current_x >= 0 && current_x < 18 && current_y >= 0 && current_y <= 18) {
         if (put_g) {  //put glizda
            //check if glizda is not already there
            if (map.glizda_x != current_x || map.glizda_y != current_y) {
               map.glizda_x = current_x;
               map.glizda_y = current_y;
               redraw();
            }
         } //end if put_g
         else if (key_shifts & KB_SHIFT_FLAG) { //if shift pressed delete sprite or wall
            if (map.values[current_x][current_y] != 0) {
               map.values[current_x][current_y] = 0;
               redraw();
            }
         }
         else {    //put sprite or wall
            if (map.values[current_x][current_y] != to_put) {
               map.values[current_x][current_y] = to_put;
               redraw();
            }
         }
      }
      if (parent())  //make dialog keep processing IDLE messages
         parent()->dialog_message(MSG_IDLE, 0);
   } while (mouse_b);
   set_mouse_range(0, 0, SCREEN_W, SCREEN_H);   //reset mouse range
}

/************************** win_button_object *********************/
void win_button_object::msg_click(void)
{
   select();  //object was clicked
   redraw();
   do {
      //mouse is enters button area
  	   if (HasMouse() && !selected() ) {
        	 select();
          redraw();
       }
       //mouse leaves button area
       else if (!HasMouse() && selected() ) {
      	 deselect();
          redraw();
       }
       parent()->dialog_message(MSG_IDLE, 0);
   } while (mouse_b);
   if (selected()) {
      deselect();
      redraw();
      if (new_click_proc() != NULL)  new_click_proc();
   }
}

/************************** font_button_object *****************/
font_button_object::font_button_object(char *t, int (*proc)(void), FONT *f=font)
        : win_button_object("", 0, false, proc)
{
   button_font = f;
   col_has_mouse = 59;
   col_selected = 9;
   col_else = 108;
   set_text(t);
}

void font_button_object::msg_draw(void)
{
   int c;
   if (has_mouse())
      c = col_has_mouse;
   else if (selected())
      c = col_selected;
   else c = col_else;
   text_mode(-1);

   textout_centre(screen, button_font, get_text(), x()+w()/2, y(), c);
   _redraw = FALSE;
}

/************************** bitmap_button_object *****************/
bitmap_button_object::bitmap_button_object(BITMAP *b)
                     :win_button_object(NULL, 0, FALSE)
{
   pop_up = NULL;
   set_bitmap(b);
}

bitmap_button_object::bitmap_button_object(BITMAP *b, int (*proc)(void), char *t=NULL)
                     :win_button_object(NULL, 0, FALSE)
{
   pop_up = NULL;
   set_bitmap(b);
   set_click_proc(proc);
   set_text(t);
}

void bitmap_button_object::set_text(char *new_text)
{
   if (pop_up) {
      destroy_bitmap(pop_up);
      destroy_bitmap(pop_up_bg);
   }
   strcpy(text, new_text);
   int x_dim = text_length(font, text) + 8;
   int y_dim = text_height(font)+6;
   pop_up = create_bitmap(x_dim, y_dim);
   pop_up_bg = create_bitmap(x_dim, y_dim);
   rectfill(pop_up, 0, 0, pop_up->w, pop_up->h, 255);
   rect(pop_up, 0, 0, pop_up->w-1, pop_up->h-1, 0);
   text_mode(-1);
   textout(pop_up, font, text, 4, 3, 0);
}

void bitmap_button_object::msg_gotmouse(void)
{
   redraw();
   if (pop_up) {
      show_mouse(NULL);
      blit(screen, pop_up_bg, x(), y2()+2, 0, 0, pop_up_bg->w, pop_up_bg->h);
      blit(pop_up, screen, 0, 0, x(), y2()+2, pop_up->w, pop_up->w);
      show_mouse(screen);
   }
}

void bitmap_button_object::msg_lostmouse(void)
{
   redraw();
   if (pop_up) {
      show_mouse(NULL);
      blit(pop_up_bg, screen, 0, 0, x(), y2()+2, pop_up_bg->w, pop_up_bg->h);
      show_mouse(screen);
   }

}

void bitmap_button_object::msg_draw(void)
{
   //fill the area under bitmap
   rectfill(screen, x(), y(), x2(), y2(),  color ? color->back() : degui_back_color);
   //stretch and draw bitmap
   if (bitmap)
      stretch_sprite(screen, bitmap, x()+1, y()+1, w()-1, h()-1);
   if (selected())        //if object is selected draw sunken frame
      draw_3d_frame(screen, x(), y(), x2(), y2(), -1,
                 color ? color->dark_shad() : degui_dark_shad_color,
                 color ? color->light_shad() : degui_light_shad_color);
   else if (has_mouse())  //when mouse is over object draw raised frame
      draw_3d_frame(screen, x(), y(), x2(), y2(), -1,
                 color ? color->light_shad() : degui_light_shad_color,
                 color ? color->dark_shad() : degui_dark_shad_color);
   _redraw = FALSE;
}

/***************************** label_object *********************/
label_object::label_object(char *text = NULL, int o = DEFAULT_ORIENTATION)
{
   set_orientation(o);
   set_text(text);
}

label_object::~label_object(void)
{
   if (string) free(string);
}

void label_object::set_text(char *text)
{
   if (string) free(string);
   if (text) {
      string = (char*)malloc(sizeof(char)*(strlen(text)+1));
      strcpy(string, text);
   }
   else string = NULL;
   redraw();
}

void label_object::set_orientation(int o = DEFAULT_ORIENTATION)
{
   orientation = o;
   redraw();
}

void label_object::msg_draw(void)
{
   rectfill(screen, x(), y(), x2(), y2(),  color ? color->back() : degui_back_color);
   if (string) {
      if (orientation == LEFT_JUSTIFY)
         textout(screen, font, string, x(), y(), color ? color->fore() : degui_fore_color);
      else if (orientation == CENTER_JUSTIFY)
         textout(screen, font, string, x()+w()/2, y(), color ? color->fore() : degui_fore_color);
      else if (orientation == RIGHT_JUSTIFY)
         textout(screen, font, string, x2()-text_length(font, string), y(),
                         color ? color->fore() : degui_fore_color);
   } //end if
   _redraw = FALSE;
}

/****************************** texture_button_object ****************/
void texture_button_object::msg_draw(void)
{
   drawing_mode(DRAW_MODE_COPY_PATTERN, texture, x(), y());
   rectfill(screen, x(), y(), x2(), y2(), 1);
   solid_mode();
   if (selected())        //if object is selected draw sunken frame
      draw_3d_frame(screen, x(), y(), x2(), y2(), -1,
                 color ? color->dark_shad() : degui_dark_shad_color,
                 color ? color->light_shad() : degui_light_shad_color);
   else  //when mouse is over object draw raised frame
      draw_3d_frame(screen, x(), y(), x2(), y2(), -1,
                 color ? color->light_shad() : degui_light_shad_color,
                 color ? color->dark_shad() : degui_dark_shad_color);
   text_mode(-1);
   if (has_mouse())
      textout_centre(screen, b_font, text, x()+w()/2, y()+h()/2-text_height(b_font)/2,
                 color ? color->back() : degui_back_color);
   else
      textout_centre(screen, b_font, text, x()+w()/2, y()+h()/2-text_height(b_font)/2,
                 color ? color->light_shad() : degui_light_shad_color);
   _redraw = FALSE;
}

/************************* input_box **************************/
int input_box(char *title, char *message, char *input)
{
   // Object definitions - DEGUI 1.2 beta
   shadow_box_object object_1;
   panel_ridge_object object_2;
   centerd_text_object object_3(title);
   centerd_text_object object_4(message);
   edittext_object edtinput(input,100);
   button_object cmdOK("OK",0,true);
   button_object cmdCancel("Cancel",0,true);

   // Add Objects to the dialog
   dialog the_dialog;
   the_dialog.add(object_1,170,165,260,140);
   the_dialog.add(object_2,175,170,250,20);
   the_dialog.add(object_3,300,175,48,8);
   the_dialog.add(object_4,300,205,48,8);
   the_dialog.add(edtinput,180,225,240,25);
   the_dialog.add(cmdOK,215,260,70,25);
   the_dialog.add(cmdCancel,305,260,75,25);
   the_dialog.center();
   dialog_object *temp = the_dialog.popup();
   strcpy(input, edtinput.get_text());
   return TRUE;
}
/********************** texture_text_object *****************/
texture_text_object::texture_text_object(char *t, BITMAP *tex = NULL, FONT *f=font)
                    :text_object(t)
{
   set_texture(tex);
   set_font(f);
   select(); /* make sunken frame */
}

void texture_text_object::msg_draw(void)
{
   drawing_mode(DRAW_MODE_COPY_PATTERN, texture, x(), y());
   rectfill(screen, x(), y(), x2(), y2(), 1);
   solid_mode();
   if (selected())        //if object is selected draw sunken frame
      draw_3d_frame(screen, x(), y(), x2(), y2(), -1,
                 color ? color->dark_shad() : degui_dark_shad_color,
                 color ? color->light_shad() : degui_light_shad_color);
   else  //when mouse is over object draw raised frame
      draw_3d_frame(screen, x(), y(), x2(), y2(), -1,
                 color ? color->light_shad() : degui_light_shad_color,
                 color ? color->dark_shad() : degui_dark_shad_color);
   text_mode(-1);
   textout_centre(screen, object_font, text, x()+w()/2, y()+h()/2-text_height(object_font)/2,
                 makecol8(255, 255, 255));
   _redraw = FALSE;
}



/************************** slider_object ********************/
class slider_object : public dialog_object {
protected:
   int max_value;
   int current_value;
   int (*change_slider_proc)(int);
public:
   slider_object(int max, int (*proc)(int)=NULL) {
      set_change_callback(proc);
      max_value = max;
      current_value = 0;
   }
   virtual void msg_click(void);
   virtual void msg_draw(void);
   void set_change_callback(int (*proc)(int)) {change_slider_proc = proc;}
   int get_current_value(void) {return current_value;}
   void set_current_value(int c) {current_value = c; redraw();}
   int get_max_value(void) {return max_value;}
   void set_max_value(int m) {max_value = m; redraw();}
};

void slider_object::msg_click(void)
{
}

void slider_object::msg_draw(void)
{
}
