#include "IGame.h"
#include <alfont.h>
#include <vector>

GameSettings gem_settings[] = {
	{ "shadows", "Draw shadows", "no|yes", 1 },
	{ "swipe", "Mouse control", "click only|click&swipe", 1 },
	{ "sound", "Sound effects", "off|25%|50%|75%|100%", 4 },
	{ 0, 0, 0, 0 }
};

class CScore {
	private:
		int score;
		int intermediate;
		int step;

	public:
		CScore() : score(0), intermediate(0)
		{
			SetLevel(1);
		}

		~CScore() {
		}

		void Add(int points) {
			score += points;
		}

		int GetScore() {
			return score;
		}

		int GetIntermediateCScore() {
			return intermediate;
		}

		void Update() {
			if (intermediate < score) {
				intermediate += step;
				intermediate = MIN(intermediate, score);
			}
			else if (intermediate > score) {
				intermediate -= step;
				intermediate = MAX(intermediate, score);
			}
		}

		void operator=(int score) {
			this->score = score;
			intermediate = score;
		}

		void SetLevel(int level) {
			this->step = 2*level;
		}
};


class ScoreParticle {
	private:
		float x,y;
		float vx, vy;
		int score;
		int ttl;
		static const int max_ttl = 128;
		int gem;

	public:
		ScoreParticle(int x, int y, int score, int gem) {
			this->x = x;
			this->y = y;
			this->score = score;
			this->ttl = max_ttl;
			this->gem = gem;

			vx = 0;
			vy = -0.25f;
		}

		~ScoreParticle() {
		}

		void Update() {
			x += vx;
			y += vy;
			--ttl;
		}

		bool Dead() {
			return ttl <= 0;
		}

		void Draw(BITMAP *canvas, ALFONT_FONT *fnt) {
			char buf[16];
			usprintf(buf, "%d", score);
			int ww = alfont_text_length(fnt, buf) + 4;
			int hh = alfont_text_height(fnt);
			BITMAP *bmp = create_bitmap(ww,hh);
			int xx = (int)x - ww/2;
			int yy = (int)y - hh/2 - 4;

			blit(canvas, bmp, xx, yy, 0, 0, bmp->w, bmp->h);

			alfont_textprintf_aa(bmp, fnt, 2+2, 2, makecol(0,0,0), buf);

			/*
			static int col[] = {
				makecol(240,230,200),
				makecol(192,0,0),
				makecol(0,192,0),
				makecol(160,0,192),
				makecol(0,128,192),
				makecol(240,240,0),
				makecol(240,192,0)
			};
			alfont_textprintf_aa(bmp, fnt, 2, 0, col[gem], buf);
			*/
			alfont_textprintf_aa(bmp, fnt, 2, 0, makecol(230,220,160), buf);

			set_trans_blender(0,0,0,ttl*256/max_ttl);
			draw_trans_sprite(canvas, bmp, xx, yy);

			destroy_bitmap(bmp);
		}
};

class GemGame : public IGame {
	private:
		int board[8][8];
		int tmp_board[8][8];
		int sel1x, sel1y;
		int sel2x, sel2y;
		int hoverx, hovery;

		int border_width;
		int tile_size;
		int bar_height;
		int bar_height_inside;
		int board_size;
		int board_x_pos;
		int board_y_pos;
		int jewel_size;

		bool paused;
		bool mouse_button_down;
		int mx, my;

		int cursor_counter;
		int cursor_interval;

		static const int SPRITE_COUNT = 22;
		BITMAP *sprite[SPRITE_COUNT];
		enum {
			BMP_BACK = 0,
			BMP_BOARD,
			BMP_JEWEL1,
			BMP_JEWEL2,
			BMP_JEWEL3,
			BMP_JEWEL4,
			BMP_JEWEL5,
			BMP_JEWEL6,
			BMP_JEWEL7,
			BMP_CURSOR1,
			BMP_CURSOR2,
			BMP_MOUSE,
			BMP_JEWEL1_SHD,
			BMP_JEWEL2_SHD,
			BMP_JEWEL3_SHD,
			BMP_JEWEL4_SHD,
			BMP_JEWEL5_SHD,
			BMP_JEWEL6_SHD,
			BMP_JEWEL7_SHD,
			BMP_CURSOR1_SHD,
			BMP_CURSOR2_SHD,
			BMP_MOUSE_SHD
		};

		static const int SAMPLE_COUNT = 5;
		SAMPLE *sample[SAMPLE_COUNT];
		enum {
			SMP_CLICK = 0,
			SMP_LEVEL,
			SMP_BADMOVE,
			SMP_DROP,
			SMP_LINEUP
		};

		enum {
			STATE_IDLE,
			STATE_SWAPPING,
			STATE_SWAPPING_BACK,
			STATE_SLIDING_DOWN,
			STATE_CLEARING_TRIPLES,
			STATE_DROPPING_COLUMNS
		};

		int state;
		int state_counter;
		int state_interval;

		ALFONT_FONT *fnt_txt;
		FONT *fnt_logo;
		PALETTE fnt_pal;

		int score_x, score1_y, score2_y;
		int moves_left;
		int last_move_x;
		int last_move_y;
		int last_move_x2;
		int last_move_y2;

		CScore score;
		int score_per_move;
		int bonus_level;
		int removed_gems;
		int next_bonus_limit;
		int prev_bonus_limit;

		std::vector<ScoreParticle *> scoreParticles;

	private:
		void PlaySample(int i) {
			if (sample[i] && gem_settings[2].value > 0) {
				play_sample(sample[i], gem_settings[2].value*255/4, 128, 1000, 0);
			}
		}

		void DrawGradient(BITMAP *canvas, int col[4]) {
			int r1 = getr(col[0]);
			int g1 = getg(col[0]);
			int b1 = getb(col[0]);
			int r2 = getr(col[1]);
			int g2 = getg(col[1]);
			int b2 = getb(col[1]);
			int r3 = getr(col[2]);
			int g3 = getg(col[2]);
			int b3 = getb(col[2]);
			int r4 = getr(col[3]);
			int g4 = getg(col[3]);
			int b4 = getb(col[3]);

			float w = canvas->w;
			float h = canvas->h;

			float dx, dy;
			for (int j=canvas->ct; j<canvas->cb; j++) {
				dy = (float)j/(float)h;
				for (int i=canvas->cl; i<canvas->cr; i++) {
					dx = (float)i/(float)w;

					int r5 = (int)(r1 + dx*(r2-r1));
					int r6 = (int)(r3 + dx*(r4-r3));
					int r  = (int)(r5 + dy*(r6-r5));
					int g5 = (int)(g1 + dx*(g2-g1));
					int g6 = (int)(g3 + dx*(g4-g3));
					int g  = (int)(g5 + dy*(g6-g5));
					int b5 = (int)(b1 + dx*(b2-b1));
					int b6 = (int)(b3 + dx*(b4-b3));
					int b  = (int)(b5 + dy*(b6-b5));

					putpixel(canvas, i, j, makecol(r,g,b));
				}
			}
		}

		void DrawBackground(BITMAP *canvas) {
			blit(sprite[BMP_BACK], canvas,
				 0, 0,
				 0, 0,
				 640, 480);
		}

		void DrawMouse(BITMAP *canvas) {
			set_alpha_blender();
			if (gem_settings[0].value == 1) {
				draw_trans_sprite(canvas, sprite[BMP_MOUSE_SHD], mx+5, my+2);
			}
			draw_trans_sprite(canvas, sprite[BMP_MOUSE], mx, my);
		}

		void DrawFrame(BITMAP *canvas, int x1, int y1, int x2, int y2, int col[4]) {
			int col1 = col[0];
			int col2 = col[1];
			int col3 = col[2];
			int col4 = col[3];

			hline(canvas, x1, y1, x2, col2);
			vline(canvas, x1, y1, y2, col2);
			hline(canvas, x1+1, y1+1, x2-1, col1);
			vline(canvas, x1+1, y1+1, y2-1, col1);

			hline(canvas, x1, y2, x2, col4);
			vline(canvas, x2, y1, y2, col4);
			hline(canvas, x1+1, y2-1, x2-1, col3);
			vline(canvas, x2-1, y1+1, y2-1, col3);
		}

		void DrawRaisedFrame(BITMAP *canvas, int x1, int y1, int x2, int y2) {
			int col[] = {
				makecol(250,240,220),
				makecol(80,70,50),
				makecol(120,110,90),
				makecol(40,30,10)
			};
			DrawFrame(canvas, x1, y1, x2, y2, col);
		}

		void DrawSunkenFrame(BITMAP *canvas, int x1, int y1, int x2, int y2) {
			int col[] = {
				makecol(120,110,90),
				makecol(40,30,10),
				makecol(250,240,220),
				makecol(80,70,50)
			};
			DrawFrame(canvas, x1, y1, x2, y2, col);
		}

		void DrawBoard(BITMAP *canvas) {
			blit(sprite[BMP_BOARD], canvas,
				 0, 0,
				 board_x_pos - border_width,
				 board_y_pos - border_width,
				 sprite[BMP_BOARD]->w,
				 sprite[BMP_BOARD]->h);

			// frame
			DrawRaisedFrame(canvas,
					  board_x_pos - border_width,
					  board_y_pos - border_width,
					  board_x_pos + board_size + border_width,
					  board_y_pos + board_size + border_width);
		}

		void DrawJewels(BITMAP *canvas) {
			set_alpha_blender();
			int cur1x,cur1y;
			int cur2x,cur2y;

			for (int y=0; y<8; y++) {
				for (int x=0; x<8; x++) {
					if (board[x][y] == -1) {
						continue;
					}

					BITMAP *spr = sprite[board[x][y] + BMP_JEWEL1];

					int xx = board_x_pos + border_width + x*tile_size + tile_size/2 - spr->w/2;
					int yy = board_y_pos + border_width + y*tile_size + tile_size/2 - spr->h/2;

					if (state == STATE_SLIDING_DOWN) {
						xx = board_x_pos + border_width + x*tile_size + tile_size/2 - spr->w/2;
						int y1 = board_y_pos + border_width + (y-8)*tile_size + tile_size/2 - spr->h/2;
						yy = y1 + (yy - y1)*state_counter/state_interval;
					}
					else if (state == STATE_SWAPPING || state == STATE_SWAPPING_BACK) {
						int x1 = x;
						int y1 = y;

						int x2 = x;
						int y2 = y;

						// is this jewel selected?
						if (x == sel1x && y == sel1y) {
							x2 = sel2x;
							y2 = sel2y;
						}
						else if (x == sel2x && y == sel2y) {
							x2 = sel1x;
							y2 = sel1y;
						}

						int xx1 = board_x_pos + border_width + x1*tile_size + tile_size/2 - spr->w/2;
						int yy1 = board_y_pos + border_width + y1*tile_size + tile_size/2 - spr->h/2;
						int xx2 = board_x_pos + border_width + x2*tile_size + tile_size/2 - spr->w/2;
						int yy2 = board_y_pos + border_width + y2*tile_size + tile_size/2 - spr->h/2;

						xx = xx1 + (xx2 - xx1)*state_counter/state_interval;
						yy = yy1 + (yy2 - yy1)*state_counter/state_interval;
					}
					if (state == STATE_DROPPING_COLUMNS) {
						if (tmp_board[x][y] != -1) {
							int y1 = board_y_pos + border_width + (y-1)*tile_size + tile_size/2 - spr->h/2;
							yy = y1 + (yy - y1)*state_counter/state_interval;
						}
					}

					if (gem_settings[0].value == 1) {
						spr = sprite[board[x][y] + BMP_JEWEL1_SHD];
						draw_trans_sprite(canvas, spr, xx + 5, yy + 4);
					}

					spr = sprite[board[x][y] + BMP_JEWEL1];
					draw_trans_sprite(canvas, spr, xx, yy);

					if (x == sel1x && y == sel1y) {
						cur1x = xx;
						cur1y = yy;
					}
					else if (x == sel2x && y == sel2y) {
						cur2x = xx;
						cur2y = yy;
					}
				}
			}
			DrawCursor(canvas, cur1x, cur1y);
			DrawCursor(canvas, cur2x, cur2y);
		}

		void DrawCursor(BITMAP *canvas, int x, int y) {
			int index_offset = 0;
			if (cursor_counter > cursor_interval/2) {
				index_offset = 1;
			}

			if (x < 0 || y < 0) {
				return;
			}

			BITMAP *spr;
			int xx, yy;

			if (gem_settings[0].value == 1) {
				spr = sprite[BMP_CURSOR1_SHD + index_offset];
				xx = x - (spr->w - sprite[BMP_JEWEL1]->w)/2;
				yy = y - (spr->h - sprite[BMP_JEWEL1]->h)/2;
				draw_trans_sprite(canvas, spr, xx + 6, yy + 4);
			}

			spr = sprite[BMP_CURSOR1 + index_offset];
			xx = x - (spr->w - sprite[BMP_JEWEL1]->w)/2;
			yy = y - (spr->h - sprite[BMP_JEWEL1]->h)/2;
			draw_trans_sprite(canvas, spr, xx, yy);
		}

		void DrawBar(BITMAP *canvas) {
			int x1 = board_x_pos - border_width;
			int y1 = 2*board_y_pos + board_size;

			int w = board_size + 2*border_width;
			int h = bar_height;

			int x2 = x1 + w;
			int y2 = y1 + h;

			float bonus_progress = (float)(removed_gems - prev_bonus_limit) / (float)(next_bonus_limit - prev_bonus_limit);
			if (bonus_progress > 1.0f) {
				bonus_progress = 1.0f;
			}

			int w2 = (int)(bonus_progress*w);

			if (bonus_progress < 1.0f) {
				rectfill(canvas, x1 + w2, y1, x2, y2, makecol(230,220,200));
				for (int i=0; i<h; i+=2) {
					hline(canvas, x1+w2, y1+i, x2, makecol(210,200,180));
				}
			}

			if (bonus_progress > 0.0f) {
				rectfill(canvas, x1, y1, x1 + w2, y2, makecol(0,190,0));
				for (int i=0; i<h; i+=2) {
					hline(canvas, x1, y1+i, x1+w2, makecol(0,170,0));
				}

				int col[] = {
					makecol(0,240,0),
					makecol(0,120,0),
					makecol(0,160,0),
					makecol(0, 30,0)
				};
				DrawFrame(canvas, x1+2, y1+2, x1 + w2, y2-2, col);
			}

			DrawSunkenFrame(canvas, x1, y1, x2, y2);
		}

		void DrawCScore(BITMAP *canvas) {
			if (fnt_txt) {
				alfont_textprintf_aa(canvas, fnt_txt, score_x+1, score1_y+1, 0, "%d", score.GetIntermediateCScore());
				alfont_textprintf_aa(canvas, fnt_txt, score_x, score1_y, makecol(0,192,0), "%d", score.GetIntermediateCScore());

				alfont_textprintf_aa(canvas, fnt_txt, score_x+1, score2_y+1, 0, "%dx", bonus_level);
				alfont_textprintf_aa(canvas, fnt_txt, score_x, score2_y, makecol(0,192,0), "%dx", bonus_level);
			}
		}

		void DrawDebug(BITMAP *) {
			/*
			int dy = 8;
			int y = -dy;
			int x = 620;

			textprintf_right_ex(canvas, font, x, y+=dy, -1, 0, "state: %d", state);
			textprintf_right_ex(canvas, font, x, y+=dy, -1, 0, "moves left: %d", moves_left);
			textprintf_right_ex(canvas, font, x, y+=dy, -1, 0, "last move: %d,%d", last_move_x, last_move_y);
			textprintf_right_ex(canvas, font, x, y+=dy, -1, 0, "last move: %d,%d", last_move_x2, last_move_y2);
			*/
		}

		int OnMouseMove(int mx, int my) {
			this->mx = mx;
			this->my = my;

			hoverx = hovery = -1;
			for (int y=0; y<8; y++) {
				for (int x=0; x<8; x++) {
					int x1 = board_x_pos + border_width + x*tile_size;
					int y1 = board_y_pos + border_width + y*tile_size;
					int x2 = x1 + tile_size;
					int y2 = y1 + tile_size;

					if (mx >= x1 && mx < x2 && my >= y1 && my < y2) {
						hoverx = x;
						hovery = y;
						return 0;
					}
				}
			}

			return 0;
		}

		int OnMouseButton(int mb) {
			if (mb) {
				if (mouse_button_down == false) {
					if (hoverx != -1 && hovery != -1) {
						if (sel1x == -1 && sel1y == -1) {
							if (sel1x == hoverx && sel1y == hovery) {
								//deselect if already selected
								sel1x = sel1y = -1;
							}
							else {
								// select 1st jewel
								sel1x = hoverx;
								sel1y = hovery;
							}
							PlaySample(SMP_CLICK);
						}
						else if (ABS(hoverx - sel1x) + ABS(hovery - sel1y) == 1) {
							sel2x = hoverx;
							sel2y = hovery;
							PlaySample(SMP_CLICK);

							// swap jewels
							int tmp = board[sel1x][sel1y];
							board[sel1x][sel1y] = board[sel2x][sel2y];
							board[sel2x][sel2y] = tmp;

							// start animation
							state = STATE_SWAPPING;
							state_counter = 0;
							state_interval = 10;
						}
						else if (hoverx == sel1x && hovery == sel1y) {
							// deselect 1st jewel
							sel1x = -1;
							sel1y = -1;
							PlaySample(SMP_CLICK);
						}
						else {
							// reselect 1st jewel
							sel1x = hoverx;
							sel1y = hovery;
							PlaySample(SMP_CLICK);
						}
					}

					mouse_button_down = true;
				}
				else if (gem_settings[1].value == 1) {
					if (hoverx != -1 && hovery != -1) {
						if (ABS(hoverx - sel1x) + ABS(hovery - sel1y) == 1) {
							sel2x = hoverx;
							sel2y = hovery;
							PlaySample(SMP_CLICK);

							// swap jewels
							int tmp = board[sel1x][sel1y];
							board[sel1x][sel1y] = board[sel2x][sel2y];
							board[sel2x][sel2y] = tmp;

							// start animation
							state = STATE_SWAPPING;
							state_counter = 0;
							state_interval = 10;
						}
					}
				}
			}
			else {
				mouse_button_down = false;
			}

			return 0;
		}

		bool LoadBitmaps() {
			char path[1024];

			static char *sprite_name[] = {
				0,
				"gameboard_marble.bmp",
				"jewel1.tga",
				"jewel2.tga",
				"jewel3.tga",
				"jewel4.tga",
				"jewel5.tga",
				"jewel6.tga",
				"jewel7.tga",
				"cursor_frame1.tga",
				"cursor_frame2.tga",
				"mouse_cursor.tga",
				0,
				0,
				0,
				0,
				0,
				0,
				0,
				0,
				0,
				0
			};

			for (int i=0; i<SPRITE_COUNT; i++) {
				if (sprite_name[i] == 0) {
					continue;
				}

				usprintf(path, "%s/gfx/%s", game_path, sprite_name[i]);
				sprite[i] = load_bitmap(path, 0);
				if (!sprite[i]) {
					DestroyBitmaps();
					return false;
				}
			}

			// create background
			sprite[BMP_BACK] = create_bitmap(640, 480);
			set_clip_rect(sprite[BMP_BACK], 0, 0, 2*board_y_pos + board_size, 479);
			int col[] = {
				makecol(240,230,210),
				makecol(140,130,110),
				makecol(120,110,100),
				makecol(220,210,190)
			};

			set_clip_rect(sprite[BMP_BACK], 0, 0, 639, 479);
			DrawGradient(sprite[BMP_BACK], col);
			for (int i=0; i<480; i+=2) {
				hline(sprite[BMP_BACK], 0, i, 639, makecol(170,160,140));
			}
			DrawBoard(sprite[BMP_BACK]);

			// draw logo
			if (fnt_logo) {
				set_palette(fnt_pal);

				int x = 540;
				int y1 = 16;
				int y2 = 64;

				for (int i=2; i>=0; i--) {
					int w = i+1;
					int bc = makecol(64*i+ 50, 64*i + 40, 64*i + 20);
					textout_centre_ex(sprite[BMP_BACK], fnt_logo, "gem", x-w, y1, bc, -1);
					textout_centre_ex(sprite[BMP_BACK], fnt_logo, "game", x-w, y2, bc, -1);
					textout_centre_ex(sprite[BMP_BACK], fnt_logo, "gem", x+w, y1, bc, -1);
					textout_centre_ex(sprite[BMP_BACK], fnt_logo, "game", x+w, y2, bc, -1);
					textout_centre_ex(sprite[BMP_BACK], fnt_logo, "gem", x, y1-w, bc, -1);
					textout_centre_ex(sprite[BMP_BACK], fnt_logo, "game", x, y2-w, bc, -1);
					textout_centre_ex(sprite[BMP_BACK], fnt_logo, "gem", x, y1+w, bc, -1);
					textout_centre_ex(sprite[BMP_BACK], fnt_logo, "game", x, y2+w, bc, -1);
				}


				textout_centre_ex(sprite[BMP_BACK], fnt_logo, "gem", x, y1, -1, -1);
				textout_centre_ex(sprite[BMP_BACK], fnt_logo, "game", x, y2, -1, -1);
			}

			// draw instructions
			int x1 = 440;
			int x2 = 630;
			int y1 = 128;
			int y2 = 384;
			int panel_col = makecol(230,220,200);
			rectfill(sprite[BMP_BACK], x1, y1, x2, y2, panel_col);

			int dy = 24;
			int y = y1 + dy - 8;
			int x = (x1+x2)/2;
			int tcol = makecol(0,192,0);
			static char *help_text[] = {
				"How to play:",
				"",
				"Swap adjacent pairs",
				"of gems to form lines",
				"of at least three",
				"gems.",
				"",
				"Create cascades for",
				"maximum points.",
				0
			};
			if (fnt_txt) {
				for (int i=0; help_text[i]; i++, y+=dy) {
					alfont_textout_centre_aa(sprite[BMP_BACK], fnt_txt, help_text[i], x+2, y+2, 0);
					alfont_textout_centre_aa(sprite[BMP_BACK], fnt_txt, help_text[i], x, y, tcol);
				}
			}
			for (int i=y1; i<y2; i+=2) {
				for (int j=x1; j<x2; j++) {
					int c = getpixel(sprite[BMP_BACK], j, i);
					if (c == panel_col) {
						putpixel(sprite[BMP_BACK], j, i, makecol(210,200,180));
					}
					else if (c == tcol) {
						putpixel(sprite[BMP_BACK], j, i, makecol(0,170,0));
					}
				}
			}

			DrawRaisedFrame(sprite[BMP_BACK], x1, y1, x2, y2);

			// Draw score panel
			x1 = 440;
			x2 = 630;
			y1 = 400;
			y2 = 460;
			int h = y2 - y1;
			rectfill(sprite[BMP_BACK], x1, y1, x2, y2, panel_col);
			for (int i=0; i<h; i+=2) {
				hline(sprite[BMP_BACK], x1, y1+i, x2, makecol(210,200,180));
			}
			DrawRaisedFrame(sprite[BMP_BACK], x1, y1, x2, y2);

			tcol = makecol(210,200,180);
			panel_col = makecol(60,50,30);
			x1 = 530;
			x2 = x2 - 8;
			y1 = y1 + 4;
			static char *score_text[] = {
				"score",
				"bonus"
			};
			if (fnt_txt) {
				for (int j=0; j<2; j++) {
					y1 = y1 + 4;
					y2 = y1 + 20;
					h = y2 - y1;
					alfont_textout_right_aa(sprite[BMP_BACK], fnt_txt, score_text[j], x1-24+1, y1-5, 0);
					alfont_textout_right_aa(sprite[BMP_BACK], fnt_txt, score_text[j], x1-24, y1-6, tcol);
					rectfill(sprite[BMP_BACK], x1, y1, x2, y2, panel_col);
					for (int i=0; i<h; i+=2) {
						hline(sprite[BMP_BACK], x1, y1+i, x2, makecol(0,0,0));
					}
					DrawSunkenFrame(sprite[BMP_BACK], x1, y1, x2, y2);

					score_x = x1+4;
					if (j==0) {
						score1_y = y1-3;
					}
					else {
						score2_y = y1-3;
					}

					y1 = y2;
				}
			}

			DrawRaisedFrame(sprite[BMP_BACK], 0, 0, 639, 479);

			// normalize cursor's alpha channel
			NormalizeAlpha(sprite[BMP_CURSOR1]);
			NormalizeAlpha(sprite[BMP_CURSOR2]);

			// create shadows
			for (int i=0; i<10; i++) {
				sprite[BMP_JEWEL1_SHD+i] = CreateShadow(sprite[BMP_JEWEL1+i]);
			}

			return true;
		}

		void DestroyBitmaps() {
			for (int i=0; i<SPRITE_COUNT; i++) {
				if (sprite[i]) {
					destroy_bitmap(sprite[i]);
					sprite[i] = 0;
				}
			}
		}

		void NormalizeAlpha(BITMAP *bmp) {
			int max_a = 0;

			for (int y=0; y<bmp->h; y++) {
				for (int x=0; x<bmp->w; x++) {
					int a = geta(getpixel(bmp, x, y));
					if (a > max_a) {
						max_a = a;
					}
				}
			}

			if (max_a == 255) {
				return;
			}

			float factor = 255.0 / (float)max_a;
			for (int y=0; y<bmp->h; y++) {
				for (int x=0; x<bmp->w; x++) {
					int c = getpixel(bmp, x, y);
					int r = getr(c);
					int g = getg(c);
					int b = getb(c);
					int a = (int)(geta(c) * factor);
					putpixel(bmp, x, y, makeacol(r,g,b,a));
				}
			}
		}

		BITMAP *CreateShadow(BITMAP *src) {
			static int kernel_size = 5;
			static int kernel[][5] = {
				{ 1, 2, 4, 2, 1 },
				{ 2, 4, 8, 4, 2 },
				{ 4, 8,16, 8, 4 },
				{ 2, 4, 8, 4, 2 },
				{ 1, 2, 4, 2, 1 },
			};

			int amp = 0;
			for (int j=0; j<kernel_size; j++) {
				for (int i=0; i<kernel_size; i++) {
					amp += kernel[i][j];
				}
			}
			amp *= 2;

			int offset = (kernel_size-1)/2;
			BITMAP *bmp = create_bitmap(src->w+offset*2, src->h+offset*2);

			for (int y=0; y<bmp->h; y++) {
				for (int x=0; x<bmp->w; x++) {
					int a = 0;

					for (int j=0; j<kernel_size; j++) {
						for (int i=0; i<kernel_size; i++) {
							int xx = x - offset + i;
							int yy = y - offset + j;

							if (xx >= 0 && xx < src->w && yy >= 0 && yy < src->h) {
								int c = getpixel(src, xx, yy);
								a += geta(c) * kernel[i][j];
							}
						}
					}

					putpixel(bmp, x, y, makeacol(0,0,0,a/amp));
				}
			}

			return bmp;
		}

		bool LoadFonts() {
			char path[1024];

			append_filename(path, game_path, "fonts/depthcharge.pcx", 1024);
			fnt_logo = load_font(path, fnt_pal, 0);
			if (!fnt_logo) {
				DestroyFonts();
				return false;
			}

			alfont_init();
			alfont_text_mode(-1);
			append_filename(path, game_path, "fonts/designerblock.ttf", 1024);
			fnt_txt = alfont_load_font(path);
			if (!fnt_txt) {
				DestroyFonts();
				return false;
			}
			else {
				alfont_set_font_size(fnt_txt, 24);
			}

			return true;
		}

		void DestroyFonts() {
			if (fnt_txt) {
				alfont_destroy_font(fnt_txt);
				fnt_txt = 0;
			}
			if (fnt_logo) {
				destroy_font(fnt_logo);
				fnt_logo = 0;
			}
		}

		bool LoadSamples() {
			char path[1024];

			static char *sample_name[] = {
				"button_click.wav",
				"complete_level.wav",
				"doesnt_work.wav",
				"jewel_drop.wav",
				"they_line_up.wav"
			};

			for (int i=0; i<SAMPLE_COUNT; i++) {
				usprintf(path, "%s/sfx/%s", game_path, sample_name[i]);
				sample[i] = load_sample(path);
				if (!sample[i]) {
					DestroySamples();
					return false;
				}
			}

			return true;
		}

		void DestroySamples() {
			for (int i=0; i<SAMPLE_COUNT; i++) {
				if (sample[i]) {
					destroy_sample(sample[i]);
					sample[i] = 0;
				}
			}
		}

		void ClearTempBoard() {
			for (int y=0; y<8; y++) {
				for (int x=0; x<8; x++) {
					tmp_board[x][y] = -1;
				}
			}
		}

		void GenerateBoard() {
			for (int y=0; y<8; y++) {
				for (int x=0; x<8; x++) {
					board[x][y] = rand()%7;
				}
			}
		}

		int GetNumberOfTriples() {
			int run = 0;
			int triples = 0;

			// for each row
			for (int y=0; y<8; y++) {
				run = 1;
				for (int x=1; x<8; x++) {
					if (board[x][y] != -1 && board[x][y] == board[x-1][y]) {
						++run;
						if (run >= 3) {
							++triples;
						}
					}
					else {
						run = 1;
					}
				}
			}

			// for each column
			for (int x=0; x<8; x++) {
				run = 1;
				for (int y=1; y<8; y++) {
					if (board[x][y] != -1 && board[x][y] == board[x][y-1]) {
						++run;
						if (run >= 3) {
							++triples;
						}
					}
					else {
						run = 1;
					}
				}
			}

			return triples;
		}

		void ClearTriples() {
			int run = 0;
			int triples = 0;

			// temp board (separately for horizontal
			//and vertical runs):
			//  0 - not in row
			//  >0 - length of row
			int board_h[8][8];
			int board_v[8][8];
			for (int y=0;y<8;y++) {
				for (int x=0;x<8;x++) {
					board_h[x][y] = 0;
					board_v[x][y] = 0;
				}
			}

			// for each row
			for (int y=0; y<8; y++) {
				run = 1;
				for (int x=1; x<8; x++) {
					if (board[x][y] == board[x-1][y]) {
						++run;
						if (run >= 3) {
							for (int i=0; i<run; i++) {
								board_h[x-i][y] = run;
							}
							++triples;
						}
					}
					else {
						run = 1;
					}
				}
			}

			// for each column
			for (int x=0; x<8; x++) {
				run = 1;
				for (int y=1; y<8; y++) {
					if (board[x][y] == board[x][y-1]) {
						++run;
						if (run >= 3) {
							for (int i=0; i<run; i++) {
								board_v[x][y-i] = run;
							}
							++triples;
						}
					}
					else {
						run = 1;
					}
				}
			}

			// assign points for horizontal runs
			for (int y=0;y<8;y++) {
				for (int x=1;x<8;x++) {
					if (board_h[x][y] > 0) {
						// does this start a new run or does the
						// old one continue?
						if (board[x][y] != board[x-1][y] || x==1) {
							// new run started -> calculate points:
							//   3 - 10 pts
							//   4 - 20 pts
							//   5 - 30 pts
							// pts = (pts + 30*runs_in_this_move)*bonus
							int pts = 10*(board_h[x][y]-2);
							pts += 30*score_per_move;
							pts *= bonus_level;

							// increase global score
							score.Add(pts);
							score_per_move += 1;
							removed_gems += board_h[x][y];

							int start_x = x;
							if (x == 1 && board[x-1][y] == board[x][y]) {
								start_x = 0;
							}
							int xx = board_x_pos + border_width + start_x*tile_size + board_h[x][y]*tile_size/2;
							int yy = board_y_pos + border_width + y*tile_size + tile_size/2;
							scoreParticles.push_back(new ScoreParticle(xx,yy,pts,board[x][y]));
						}
					}
				}
			}

			// assign points for vertical runs
			for (int x=0;x<8;x++) {
				for (int y=1;y<8;y++) {
					if (board_v[x][y] > 0) {
						// does this start a new run or does the
						// old one continue?
						if (board[x][y] != board[x][y-1] || y==1) {
							// new run started -> calculate points:
							//   3 - 10 pts
							//   4 - 20 pts
							//   5 - 30 pts
							// pts = (pts + 30*runs_in_this_move)*bonus
							int pts = 10*(board_v[x][y]-2);
							pts += 30*score_per_move;
							pts *= bonus_level;

							// increase global score
							score.Add(pts);
							score_per_move += 1;
							removed_gems += board_v[x][y];

							int start_y = y;
							if (y == 1 && board[x][y-1] == board[x][y]) {
								start_y = 0;
							}
							int xx = board_x_pos + border_width + x*tile_size + tile_size/2;
							int yy = board_y_pos + border_width + start_y*tile_size + board_v[x][y]*tile_size/2;
							scoreParticles.push_back(new ScoreParticle(xx,yy,pts,board[x][y]));
						}
					}
				}
			}

			// clear gems in board for which the board_h
			// or board_v item at the same coordinates is > 0
			for (int y=0;y<8;y++) {
				for (int x=0;x<8;x++) {
					if (board_h[x][y] > 0 || board_v[x][y] > 0) {
						board[x][y] = -1;
					}
				}
			}

			// handle bonus
			if (removed_gems > next_bonus_limit) {
				PlaySample(SMP_LEVEL);
				// clear some random gems
				for (int i=0; i<8; ) {
					int x = rand()%8;
					int y = rand()%8;

					if (board[x][y] >= 0) {
						++i;
						int pts = 10*bonus_level;
						score.Add(pts);

						int xx = board_x_pos + border_width + x*tile_size + tile_size/2;
						int yy = board_y_pos + border_width + y*tile_size + tile_size/2;
						scoreParticles.push_back(new ScoreParticle(xx,yy,pts,board[x][y]));

						board[x][y] = -1;
					}
				}
				bonus_level++;
				prev_bonus_limit = next_bonus_limit;
				next_bonus_limit *= 2;
				score.SetLevel(bonus_level);
			}

		}

		void DropJewelsByOne(int x, int y) {
			for (int yy=y; yy>0; yy--) {
				board[x][yy] = board[x][yy-1];
				board[x][yy-1] = -1;
				board[x][yy-1] = rand()%7;

				tmp_board[x][yy] = board[x][yy];
				tmp_board[x][yy-1] = board[x][yy-1];
			}
		}

		int FindLastEmpty(int x) {
			for (int y=7; y>=0; y--) {
				if (board[x][y] == -1) {
					return y;
				}
			}

			return -1;
		}

		int FindFirstNonEmpty(int x) {
			for (int y=0; y<8; y++) {
				if (board[x][y] != -1) {
					return y;
				}
			}

			return -1;
		}

		void DropColumn(int x) {
			int y = FindLastEmpty(x);
			int y2 = FindFirstNonEmpty(x);

			if (y <= y2) {
				//return;
			}
			else {
				DropJewelsByOne(x, y);
			}

			if (board[x][0] == -1) {
				board[x][0] = rand()%7;
				tmp_board[x][0] = board[x][0];
			}
		}

		void DropColumns() {
			ClearTempBoard();

			// for each column
			for (int x=0; x<8; x++) {
				DropColumn(x);
			}
		}

		bool EmptySquares() {
			for (int y=0; y<8; y++) {
				for (int x=0; x<8; x++) {
					if (board[x][y] == -1) {
						return true;
					}
				}
			}

			return false;
		}

		void SwapGems(int x1, int y1, int x2, int y2) {
			int tmp = board[x1][y1];
			board[x1][y1] = board[x2][y2];
			board[x2][y2] = tmp;
		}

		int MovesLeft() {
			int moves = 0;

			// for each gem on board
			for (int y=0; y<8; y++) {
				for (int x=0; x<8; x++) {
					// make all available moves
					// and see if they're valid

					// swap with right
					if (x < 7) {
						SwapGems(x,y, x+1,y);
						if (GetNumberOfTriples() > 0) {
							last_move_x = x;
							last_move_y = y;
							last_move_x2 = x+1;
							last_move_y2 = y;
							++moves;
						}
						SwapGems(x,y, x+1,y);
					}

					// swap with left
					if (x > 0) {
						SwapGems(x,y, x-1,y);
						if (GetNumberOfTriples() > 0) {
							last_move_x = x;
							last_move_y = y;
							last_move_x2 = x-1;
							last_move_y2 = y;
							++moves;
						}
						SwapGems(x,y, x-1,y);
					}

					// swap with above
					if (y > 0) {
						SwapGems(x,y, x,y-1);
						if (GetNumberOfTriples() > 0) {
							last_move_x = x;
							last_move_y = y;
							last_move_x2 = x;
							last_move_y2 = y-1;
							++moves;
						}
						SwapGems(x,y, x,y-1);
					}

					// swap with below
					if (y < 7) {
						SwapGems(x,y, x,y+1);
						if (GetNumberOfTriples() > 0) {
							last_move_x = x;
							last_move_y = y;
							last_move_x2 = x;
							last_move_y2 = y+1;
							++moves;
						}
						SwapGems(x,y, x,y+1);
					}
				}
			}

			return moves/2;
		}

	public:
		GemGame() : IGame() {
			border_width = 4;
			tile_size = 48;
			bar_height_inside = 16;
			bar_height = 2*border_width + bar_height_inside;
			board_size = border_width*2 + tile_size*8;
			board_y_pos = (480 - (board_size + bar_height)) / 3;
			board_x_pos = board_y_pos;
			jewel_size = tile_size / 3;
			for (int i=0; i<SPRITE_COUNT; i++) {
				sprite[i] = 0;
			}
			fnt_logo = 0;
			fnt_txt = 0;
			for (int i=0; i<SAMPLE_COUNT; i++) {
				sample[i] = 0;
			}
		}

		~GemGame() {
		}

		char *GetName() {
			return "Gem Game";
		}

		char *GetAuthor() {
			return "Miran Amon and Mark Oates";
		}

		char *GetDescription() {
			return "Swap adjacent gems to form runs of three, four or five gems of the same colour. The more gems you clear in one move, the more points you score. Create cascades for maximum points.";
		}

		char *GetVersion() {
			return "0.92";
		}

		char *GetIconPath() {
			return "icon.bmp";
		}

		bool Init() {
			if (!IGame::Init()) {
				return false;
			}

			do {
				GenerateBoard();
			} while (GetNumberOfTriples() > 0);

			ClearTempBoard();

			sel1x = sel2x = hoverx = -1;
			sel1y = sel2y = hoverx = -1;

			paused = false;
			mouse_button_down = false;
			mx = 0;
			my = 0;

			cursor_counter = 0;
			cursor_interval = 20;

			state = STATE_SLIDING_DOWN;
			state_counter = 0;
			state_interval = 50;

			moves_left = MovesLeft();;

			score = 0;
			score.SetLevel(1);
			score_per_move = 0;
			bonus_level = 1;
			removed_gems = 0;
			next_bonus_limit = 64;
			prev_bonus_limit = 0;

			bool ret = true;
			ret &= LoadFonts();
			ret &= LoadBitmaps();
			ret &= LoadSamples();

			return ret;
		}

		void Deinit() {
			DestroyBitmaps();
			DestroyFonts();
			DestroySamples();
			for (std::vector<ScoreParticle *>::iterator i = scoreParticles.begin(); i != scoreParticles.end(); ) {
				delete *i;
				i = scoreParticles.erase(i);
			}
			IGame::Deinit();
		}

		void Draw(BITMAP *canvas) {
			// background
			DrawBackground(canvas);

			// jewels
			DrawJewels(canvas);

			// bar
			DrawBar(canvas);

			// score
			DrawCScore(canvas);

			// score particles
			for (std::vector<ScoreParticle *>::iterator i = scoreParticles.begin(); i != scoreParticles.end(); ++i) {
				(*i)->Draw(canvas, fnt_txt);
			}

			// mouse
			if (!paused) {
				DrawMouse(canvas);
			}

			// debug
			DrawDebug(canvas);
		}

		int Logic(int[8], int[4], float) {
			// Do all input processing and game logic here. Refer to the
			// comments in IGame.h for details on the format of the input
			// parameters and the requirements for the output.
			// The first parameter holds the status of 8 keys, the second
			// the status of the mouse (x, y, wheel, buttons) and the third
			// the time difference in milliseconds since the last Logic
			// update (note: In this example the parameters are unnamed
			// only to avoid compile warnings. In your implementation
			// you should give them proper names.)
			// ...

			score.Update();

			for (std::vector<ScoreParticle *>::iterator i = scoreParticles.begin(); i != scoreParticles.end(); ) {
				(*i)->Update();

				if ((*i)->Dead()) {
					delete *i;
					i = scoreParticles.erase(i);
				}
				else {
					++i;
				}
			}


			poll_mouse();
			mx = mouse_x;
			my = mouse_y;

			if (!paused) {
				switch (state) {
					case STATE_IDLE:
						// handle mouse movement
						OnMouseMove(mouse_x, mouse_y);

						// handle mouse buttons
						OnMouseButton(mouse_b);
						break;

					case STATE_SWAPPING:
						++state_counter;
						if (state_counter > state_interval) {
							state_counter = 0;
							if (GetNumberOfTriples() == 0) {
								// swap jewels back
								int tmp = board[sel1x][sel1y];
								board[sel1x][sel1y] = board[sel2x][sel2y];
								board[sel2x][sel2y] = tmp;

								PlaySample(SMP_BADMOVE);

								// start animation
								state = STATE_SWAPPING_BACK;
								state_counter = 0;
								state_interval = 10;
							}
							else {
								state = STATE_CLEARING_TRIPLES;
							}
						}
						break;

					case STATE_SWAPPING_BACK:
						++state_counter;
						if (state_counter > state_interval) {
							state_counter = 0;
							state = STATE_IDLE;
							// deselect everything
							sel1x = sel1y = -1;
							sel2x = sel2y = -1;
						}
						break;

					case STATE_SLIDING_DOWN:
						++state_counter;
						if (state_counter > state_interval) {
							state_counter = 0;
							state = STATE_CLEARING_TRIPLES;
						}
						break;

					case STATE_CLEARING_TRIPLES:
						// deselect everything
						sel1x = sel1y = -1;
						sel2x = sel2y = -1;
						if (GetNumberOfTriples() == 0) {
							moves_left = MovesLeft();
							state = STATE_IDLE;
							score_per_move = 0;
						}
						else {
							PlaySample(SMP_LINEUP);
							ClearTriples();
							DropColumns();
							state = STATE_DROPPING_COLUMNS;
							state_counter = 0;
							state_interval = 10;
						}
						break;

					case STATE_DROPPING_COLUMNS:
						++state_counter;
						if (state_counter > state_interval) {
							if (EmptySquares()) {
								//PlaySample(SMP_DROP);
								DropColumns();
								state = STATE_DROPPING_COLUMNS;
								state_interval = 10;
							}
							else {
								state = STATE_CLEARING_TRIPLES;
							}
							state_counter = 0;
						}
						break;
				};
			}

			++cursor_counter;
			cursor_counter %= cursor_interval;

			if (moves_left == 0) {
				paused = true;
				return 1;
			}
			else {
				return 0;
			}
		}

		int Score() {
			// Return -1 if this game doesn't have points, otherwise the last
			// valid score where more points is better.
			return score.GetScore();
		}

		bool UseSystemMouse() {
			return false;
		}

		void Pause() {
			paused = true;
		}

		void Resume() {
			paused = false;
		}

		// Should return to a NULL terminated array of GameSettings structures.
		// For example you could define the array like this:
		//
		//	GameSettings my_settings[] = {
		//		{ "name1", "option 1", "first|second|third", 1 },
		//		{ "name2", "option 2", "first|second|third", 0 },
		//		{ "name3", "option 3", "first|second|third", 2 },
		//		{ 0, 0, 0, 0 }
		//	};
		//
		// and then return my_settings in this function.
		GameSettings* GetSettings() {
			return gem_settings;
		}
};



extern "C" DLLEXPORT IGame* GetPlugin() {
	return new GemGame;
}
