#ifndef INCLUDED_BSP_H
#define INCLUDED_BSP_H



#include <allegro.h>



/* MIN_Z is the closest a polygon can get to the camera before it is clipped.
 * It cannot be zero, since that would create a division by zero when
 * projecting.
 *
 * MAX_Z is the farthest a polygon can be before it is clipped. It should be
 * unnecessary to use this. By setting it to 0 (which is less than MIN_Z), we
 * disable far clipping altogether.
 */
#define MIN_Z 0.01
#define MAX_Z 0

#define N_CLIPPING_PLANES 5



typedef struct VERTEX
{
	/* World coordinates. */
	float x, y, z;

	/* Coordinates transformed into camera space, set while drawing. */
	float xt, yt, zt;
}
VERTEX;



typedef struct POLYGON_VERTEX
{
	VERTEX *vtx; /* This is a reference. Do not free it! */
	float u;
	float v;
}
POLYGON_VERTEX;



typedef struct POLYGON
{
	unsigned char n_vtx;

	BITMAP *texture;

	POLYGON_VERTEX *p;

	/* Polygon normal. */
	float xn, yn, zn;
}
POLYGON;



typedef struct BSP_NODE
{
	POLYGON *poly; /* This is merely a reference. Do not free it! */

	struct BSP_NODE *pos;
	struct BSP_NODE *neg;
}
BSP_NODE;



typedef struct BSP_TREE
{
	BSP_NODE *root;

	/* To delete a temporary node, first free the node, then set the pointer
	 * to NULL. 'tempnode' below points to an array of handles (a handle
	 * being a pointer to a pointer), enabling the actual pointers to be set.
	 * Note: in this special case, we must not free recursively. Subsequent
	 * BSP_TEMPORARY structs will take care of descendants. Also note that
	 * the tree has to be freed in reverse order, or else pointers will be
	 * set to NULL after the memory holding the pointers has already been
	 * freed.
	 */
	int n_tempnodes;
	int max_tempnodes;
	BSP_NODE ***tempnode;

	/* Extra vertices that must be generated when a polygon is split. */
	int n_vtx;
	int n_pvtx; /* Permanent vertices. */
	int n_vtx_pages;
	VERTEX **vtx;

	/* Extra polygons that must be generated when a polygon is split. */
	int n_poly;
	int n_ppoly; /* Permanent polygons. */
	int n_poly_pages;
	POLYGON **poly;

	/* When more than 256 vertices are created, an extra vertex page is set
	 * up. This avoids realloc(), which moves memory around and breaks
	 * pointer links. The same applies to polygons.
	 */

	/* V3D_f structures for the clip3d_f() and polygon3d_f() functions. */
	int n_v3d;
	int n_v3d_allocated;

	V3D_f **v3d, **v3dc, **v3dt;
	int *out;
}
BSP_TREE;



/* bspprep.c */

void prepare_polygon(POLYGON *poly);



/* bsp.c */

BSP_TREE *create_bsp_tree(void);
void add_to_bsp_tree(BSP_TREE *tree, POLYGON *poly, BSP_NODE **tempnode);
void finalise_bsp_tree(BSP_TREE *tree, int temporary);
void remove_temporary(BSP_TREE *tree);
void destroy_bsp_tree(BSP_TREE *tree);

/* When calling add_to_bsp_tree(), pass NULL for tempnode. The parameter is
 * there for use by add_temp_to_bsp_tree() below.
 */

#define add_temp_to_bsp_tree(tree, poly) add_to_bsp_tree(tree, poly, &(tree)->root)



/* bspdraw.c */

#define draw_prepare_vertex(camera, vtx) \
	apply_matrix_f((camera), (vtx)->x, (vtx)->y, (vtx)->z, &(vtx)->xt, &(vtx)->yt, &(vtx)->zt)

void draw_prepare_bsp_tree(MATRIX_f *camera, BSP_TREE *tree);

/* To add temporary polygons, call add_temp_to_bsp_tree(). When
 * finished, call finalise_bsp_tree() with temporary = 1. The tree may then
 * be drawn as indicated below. Call remove_temporary() afterwards.
 */

/* Before calling this function, you must:
 * 1. Make sure you have called draw_prepare_vertex on all vertices.
 * 2. Call draw_prepare_bsp_tree() on the tree.
 * 3. Call set_projection_viewport() appropriately. This may be done only
 *    once if the viewport never changes.
 * Steps 1 and 2 need not be repeated if the camera matrix remains the same.
 */
void draw_bsp_tree(BITMAP *bmp, MATRIX_f *camera, float xc, float yc, float zc, BSP_TREE *tree);



#endif /* INCLUDED_BSP_H */
