#include <string.h>
#include "main.h"
#include "timeloop.h"
#include "game.h"
#include "sfxplay.h"



/* Actually one more than the maximum skip that can be used. Must be a power
 * of two. Compile with -DDEBUGMODE=1 to check you don't exceed this.
 */
#define MAX_SKIP 16



volatile int true_time;
int logic_time;

static volatile int max_true_time;

static int timeloop_active = 0;

static int timeloop_speed;



unsigned char timeloop_scancode[MAX_PLAYERS][N_KEYS];

static volatile unsigned char timeloop_held[MAX_SKIP][MAX_PLAYERS][N_KEYS];
static int oldkey, newkey;

/* Usage: ckey[p][k], where p is the player and k is a K_* constant. */
volatile unsigned char (*ckey)[N_KEYS];



static void timeloop_handler(void)
{
	if (true_time < max_true_time) {
		int p, k;
		true_time++;
		for (p = 0; p < MAX_PLAYERS; p++)
			for (k = 0; k < N_KEYS; k++)
				timeloop_held[newkey][p][k] = key[timeloop_scancode[p][k]];
		newkey = (newkey + 1) & (MAX_SKIP - 1);
	}
}
END_OF_STATIC_FUNCTION(timeloop_handler);



void init_timeloop(void)
{
	LOCK_VARIABLE(true_time);
	LOCK_VARIABLE(max_true_time);
	LOCK_VARIABLE(timeloop_scancode);
	LOCK_VARIABLE(timeloop_held);
	LOCK_VARIABLE(oldkey);
	LOCK_VARIABLE(newkey);
	LOCK_VARIABLE(ckey);
	LOCK_FUNCTION(timeloop_handler);

	memset(timeloop_scancode, 0, MAX_PLAYERS * N_KEYS);
}



/* Notes for timeloop():
 *
 * On call to (*update)(), logic_time corresponds to the frame that has just
 * been drawn (if there was time to draw it). The first frame is 0, and will
 * always be drawn before any updates occur.
 *
 * Recursive structures such as menus can be set up using recursive calls to
 * timeloop(). Proper protocol is to call timeloop() from the end of the
 * (*update)() function. logic_time will be incremented before the inner
 * (*update)() function is called, and the inner (*draw)() function will be
 * used immediately (with the new logic_time) if time permits. When the inner
 * (*update)() function returns 1, logic_time will again be incremented
 * before the outer (*update)() function is called, and the outer (*draw)()
 * function will be used immediately (with the new logic_time) if time
 * permits.
 */
int timeloop(
	int speed,
	int max_skip,
	void (*draw)(void *data),
	int (*update)(void *data),
	void *data
)
{
	int rv;

	ASSERT(max_skip < MAX_SKIP);

	if (timeloop_active++) {
		logic_time++;
		oldkey = (oldkey + 1) & (MAX_SKIP - 1);

		while (logic_time < true_time) {
			sfx_poll();
			ckey = timeloop_held[oldkey];
			rv = (*update)(data);

			if (rv) {
				timeloop_active--;
				return rv;
			}

			logic_time++;
			oldkey = (oldkey + 1) & (MAX_SKIP - 1);
		}
	} else {
		timeloop_speed = speed;

		true_time = logic_time = 0;
		max_true_time = 0;
		oldkey = 0;
		newkey = 0;

		install_int_ex(&timeloop_handler, speed);
	}

	for (;;) {
		max_true_time = logic_time + max_skip;

		sfx_poll();
		(*draw)(data);

		while (logic_time >= true_time) {
			sfx_poll();
			yield_timeslice();
		}

		do {
			sfx_poll();
			ckey = timeloop_held[oldkey];
			rv = (*update)(data);

			if (rv) {
				if (--timeloop_active == 0)
					remove_int(timeloop_handler);
				return rv;
			}

			logic_time++;
			oldkey = (oldkey + 1) & (MAX_SKIP - 1);
		} while (logic_time < true_time);
	}
}



/* new_timeloop() can be used as an alternative to timeloop() when recursing.
 * For the inner functions, logic_time will count from 0. logic_time will
 * again be reset to 0 when the outer function resumes. There will be
 * discontinuities; before and after calling this function you can execute
 * time-consuming algorithms, and the computer will not try to catch up. This
 * function also allows the speed to be changed, whereas sub-timeloops will
 * normally inherit the speed of the caller.
 */
int new_timeloop(
	int speed,
	int max_skip,
	void (*draw)(void *data),
	int (*update)(void *data),
	void *data
)
{
	int rv;
	int parent_speed;

	ASSERT(timeloop_active);
	ASSERT(max_skip < MAX_SKIP);

	parent_speed = timeloop_speed;
	timeloop_speed = speed;
	install_int_ex(&timeloop_handler, speed);

	max_true_time = 0;
	true_time = 0;
	logic_time = -1;
	oldkey = -1;
	newkey = 0;

	rv = timeloop(speed, max_skip, draw, update, data);

	max_true_time = 0;
	true_time = 0;
	logic_time = -1;
	oldkey = -1;
	newkey = 0;

	timeloop_speed = parent_speed;
	install_int_ex(&timeloop_handler, parent_speed);

	return rv;
}

