#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "mathpi.h"
#include "main.h"
#include "screen.h"
#include "game.h"
#include "player.h"
#include "tubby.h"
#include "timeloop.h"
#include "sfx.h"
#include "sfxplay.h"



#define GAME_END_TIME (2 * GAME_FPS)



static const float player_start_x[MAX_PLAYERS]   = {-1     , 1     , 0     , 0     };
static const float player_start_y[MAX_PLAYERS]   = { 0     , 0     ,-1     , 1     };
static const float player_start_phi[MAX_PLAYERS] = { 1*M_PI, 3*M_PI, 0*M_PI, 2*M_PI};



SPRITE *add_sprite(
	GAME *game,
	int flags,
	SPRITE_UPDATE_PROC update,
	float x, float y, float z,
	float r, float zr,
	SPRITE_SHOOT_PROC shoot,
	SPRITE_DESTROY_PROC destroy,
	void *data,
	BITMAP *texture
)
{
	SPRITE *sprite = malloc(sizeof(*sprite));

	if (!sprite) {
		if (destroy)
			(*destroy)(data);
		return NULL;
	}

	sprite->flags = flags;

	sprite->update = update;

	sprite->x = x;
	sprite->y = y;
	sprite->z = z;

	sprite->r = r;
	sprite->zr = zr;

	sprite->shoot = shoot;

	sprite->destroy = destroy;
	sprite->data = data;

	sprite->poly.n_vtx = 4;
	sprite->poly.texture = texture;
	sprite->poly.p = sprite->polyvtx;
	sprite->polyvtx[0].vtx = &sprite->vtx[0];
	sprite->polyvtx[1].vtx = &sprite->vtx[1];
	sprite->polyvtx[2].vtx = &sprite->vtx[2];
	sprite->polyvtx[3].vtx = &sprite->vtx[3];
	sprite->polyvtx[0].u = 0.01;
	sprite->polyvtx[0].v = 0.01;
	sprite->polyvtx[1].u = 0.01;
	sprite->polyvtx[1].v = texture->h - 0.01;
	sprite->polyvtx[2].u = texture->w - 0.01;
	sprite->polyvtx[2].v = texture->h - 0.01;
	sprite->polyvtx[3].u = texture->w - 0.01;
	sprite->polyvtx[3].v = 0.01;

	sprite->next = game->sprite;
	game->sprite = sprite;

	return sprite;
}



static int take_screenshot;



GAME *init_game(int n_players, PLAYER_TRANS *ptrans, int n_teletubbies)
{
	GAME *game = malloc(sizeof(*game));
	int i;

	if (!game)
		return NULL;

	game->n_players = n_players;

	game->world = create_world();

	if (!game->world) {
		free(game);
		return NULL;
	}

	game->sprite = NULL;

	for (i = 0; i < n_players; i++) {
		float x, y;
		game->ptrans[i] = &ptrans[i];
		memcpy(timeloop_scancode[i], ptrans[i].k, N_KEYS);
		ptrans[i].score = 0;
		x = game->world->w * 0.5 + player_start_x[i];
		y = game->world->h * 0.5 + player_start_y[i];
		create_player(game, i, x, y, player_start_phi[i]);
	}

	while (i < MAX_PLAYERS) {
		game->ptrans[i] = NULL;
		i++;
	}

	game->n_teletubbies = 0;
	game->end_time = GAME_END_TIME;

	for (i = 0; i < n_teletubbies; i++) {
		float x, y, d;
		do {
			float xd, yd;
			x = RND(game->world->w);
			y = RND(game->world->h);
			xd = x - game->world->w * 0.5;
			yd = x - game->world->w * 0.5;
			d = xd*xd + yd*yd;
		} while (d < 3*3);
		create_teletubby(game, x, y);
	}

	take_screenshot = 0;

	return game;
}



void shut_down_game(GAME *game)
{
	destroy_world(game->world);

	{
		SPRITE *sprite = game->sprite;
		while (sprite) {
			SPRITE *next = sprite->next;
			if (sprite->destroy)
				(*sprite->destroy)(sprite->data);
			free(sprite);
			sprite = next;
		}
	}

	free(game);
}



static int all_dead(GAME *game)
{
	int i;

	for (i = 0; i < MAX_PLAYERS; i++)
		if (game->ptrans[i] && game->ptrans[i]->sprite)
			return 0;

	return 1;
}



#define VIEW_ASPECT (4.0 / 3.0)



void draw_game(void *data)
{
	GAME *game = data;
	BITMAP *bmp = get_screen();
	int p;

	for (p = 0; p < 4; p++) {
		BITMAP *sbmp;

		switch (p) {
			case 3:
				sbmp = create_sub_bitmap(bmp, bmp->w >> 1, bmp->h >> 1, bmp->w >> 1, bmp->h >> 1);
				break;
			case 2:
				sbmp = create_sub_bitmap(bmp, 0, bmp->h >> 1, bmp->w >> 1, bmp->h >> 1);
				break;
			case 1:
				sbmp = create_sub_bitmap(bmp, bmp->w >> 1, 0, bmp->w >> 1, bmp->h >> 1);
				break;
			default:
				if (game->n_players == 1)
					sbmp = bmp;
				else
					sbmp = create_sub_bitmap(bmp, 0, 0, bmp->w >> 1, bmp->h >> 1);
		}

		if (game->ptrans[p] && game->ptrans[p]->sprite) {
			CAMERA_POSITION c;
			get_player_camera(game->ptrans[p], &c);

			if (c.z >= MIN_Z) {
				float sp = sin(c.phi);
				float cp = cos(c.phi);
				float st = sin(c.theta);
				float ct = cos(c.theta);

				MATRIX_f camera = {
					{
						/* Three vectors relating to the player orientation in world space.
						 * The vectors correspond to the viewing space coordinate axes.
						 * The down vector is multiplied by VIEW_ASPECT, scaling the world
						 * in that direction; this is compensated for later by the reduced
						 * height of the screen.
						 */
						{ cp               , sp               , 0             }, /* Right vector */
						{ sp*st*VIEW_ASPECT,-cp*st*VIEW_ASPECT,-ct*VIEW_ASPECT}, /* Down vector */
						{ sp*ct            ,-cp*ct            , st            }  /* Forwards vector */
					},
					{
						/* Set to (c.x, c.y, c.z) transformed by the above camera matrix
						 * and then negated.
						 */
						- (c.x*cp + c.y*sp),
						- (c.x*sp*st - c.y*cp*st - c.z*ct) * VIEW_ASPECT,
						- (c.x*sp*ct - c.y*cp*ct + c.z*st)
					}
				};

				clear_to_color(sbmp, makecol(204,204,255));
				/*
				rectfill(sbmp, 0, 0, sbmp->w - 1, (sbmp->h >> 1) - 1, makecol(204,204,255));
				rectfill(sbmp, 0, sbmp->h >> 1, sbmp->w - 1, sbmp->h - 1, makecol(192,192,64));
				*/

				{
					SPRITE *sprite;
					for (sprite = game->sprite; sprite; sprite = sprite->next) {
						float rcp = sprite->r * cp;
						float rsp = sprite->r * sp;
						sprite->vtx[0].x = sprite->x - rcp;
						sprite->vtx[0].y = sprite->y - rsp;
						sprite->vtx[0].z = sprite->z + sprite->zr;
						sprite->vtx[1].x = sprite->x - rcp;
						sprite->vtx[1].y = sprite->y - rsp;
						sprite->vtx[1].z = sprite->z - sprite->zr;
						sprite->vtx[2].x = sprite->x + rcp;
						sprite->vtx[2].y = sprite->y + rsp;
						sprite->vtx[2].z = sprite->z - sprite->zr;
						sprite->vtx[3].x = sprite->x + rcp;
						sprite->vtx[3].y = sprite->y + rsp;
						sprite->vtx[3].z = sprite->z + sprite->zr;
						draw_prepare_vertex(&camera, &sprite->vtx[0]);
						draw_prepare_vertex(&camera, &sprite->vtx[1]);
						draw_prepare_vertex(&camera, &sprite->vtx[2]);
						draw_prepare_vertex(&camera, &sprite->vtx[3]);
						prepare_polygon(&sprite->poly);
						add_temp_to_bsp_tree(game->world->tree, &sprite->poly);
					}
				}
				finalise_bsp_tree(game->world->tree, 1);

				/* H4x0r factor: polygon routines overrun the clipping rectangle! */
				sbmp->cl++; sbmp->cr--; sbmp->ct++; sbmp->cb--;
				draw_world(sbmp, &camera, c.x, c.y, c.z, game->world);
				sbmp->cl--; sbmp->cr++; sbmp->ct--; sbmp->cb++;
				rect(sbmp, 0, 0, sbmp->w - 1, sbmp->h - 1, makecol(0,0,0));

				remove_temporary(game->world->tree);
				hline(sbmp, (sbmp->w>>1)-(sbmp->w>>3), sbmp->h>>1, (sbmp->w>>1)-(sbmp->w>>5), makecol(255,255,255));
				hline(sbmp, (sbmp->w>>1)+(sbmp->w>>5), sbmp->h>>1, (sbmp->w>>1)+(sbmp->w>>3), makecol(255,255,255));
				vline(sbmp, sbmp->w>>1, (sbmp->h>>1)-(sbmp->h>>3), (sbmp->h>>1)-(sbmp->h>>5), makecol(255,255,255));
				vline(sbmp, sbmp->w>>1, (sbmp->h>>1)+(sbmp->h>>5), (sbmp->h>>1)+(sbmp->h>>3), makecol(255,255,255));
				{
					int x = player_get_health(game->ptrans[p]->sprite);
					int w = sbmp->w - 8;
					x = (x * w + 999) / 1000;
					if (x > 0) rectfill(sbmp, 4, 4, x + 3, 7, makecol(255,0,0));
					if (x < w) rectfill(sbmp, x + 4, 4, w + 3, 7, makecol(0,0,0));
				}
				{
					int y = 12;
					textout(sbmp, font, game->ptrans[p]->name, 4, y, makecol(0,0,0));
					if (sbmp->w < 4 + FONT_W*PLAYER_NAME_MAX + FONT_W*6 + 4) y += 8;
					textprintf_right(sbmp, font, sbmp->w - 4, y, makecol(0,0,0), "%d", game->ptrans[p]->score);
				}
			} else
				clear(sbmp);
		} else
			clear(sbmp);

		if (game->n_players == 1)
			break;

		destroy_bitmap(sbmp);
	}

	if (game->n_teletubbies == 0)
		masked_stretch_blit(bmp_mission_complete, bmp,
			0, 0, bmp_mission_complete->w, bmp_mission_complete->h,
			bmp->w >> 2, (bmp->h >> 1) - (bmp->h >> 3),
			bmp->w >> 1, bmp->h >> 2);
	else if (all_dead(game))
		masked_stretch_blit(bmp_mission_failed, bmp,
			0, 0, bmp_mission_failed->w, bmp_mission_failed->h,
			bmp->w >> 2, (bmp->h >> 1) - (bmp->h >> 3),
			bmp->w >> 1, bmp->h >> 2);

	if (take_screenshot) {
		static int scrnum = 0;
		static unsigned char filename[256];
		sprintf(filename, "scr%05d.bmp", scrnum);
		while (exists(filename)) {
			scrnum++;
			sprintf(filename, "scr%05d.bmp", scrnum);
		}
		save_bmp(filename, bmp, pal);
		scrnum++;
		take_screenshot = 0;
	}

	commit_screen();
}



int update_game(void *data)
{
	GAME *game = data;

	{
		SPRITE *sprite;
		for (sprite = game->sprite; sprite; sprite = sprite->next)
			(*sprite->update)(game, sprite);
	}

	{
		SPRITE **sprite_p = &game->sprite;
		while (*sprite_p) {
			SPRITE *sprite = *sprite_p;
			if (sprite->flags & SPRITE_DEAD) {
				if (sprite->destroy)
					(*sprite->destroy)(sprite->data);
				*sprite_p = sprite->next;
				free(sprite);
			} else
				sprite_p = &sprite->next;
		}
	}

	if (all_dead(game)) {
		if (game->end_time == 0)
			return GAME_ALL_DEAD;
		else if (game->end_time-- == GAME_END_TIME)
			sfx_play(sfx_menu_error, 1, 0, 1.0);
	}

	if (game->n_teletubbies == 0) {
		if (game->end_time == 0)
			return GAME_COMPLETED;
		else if (game->end_time-- == GAME_END_TIME)
			sfx_play(sfx_menu_enter, 1, 0, 1.0);
	}

	while (keypressed()) {
		int k = readkey();
		switch (k >> 8) {
			case KEY_ESC:
				return GAME_ABORTED;
			case KEY_F1:
				take_screenshot = 1;
				break;
		}
	}

	return 0;
}

