#include <stdlib.h>
#include "main.h"
#include "screen.h"
#include "bsp.h"



#define BSP_EPSILON 0.0001



static void *critical_realloc(void *ptr, size_t size)
{
	/* Note that realloc() doesn't free on failure, but if we're aborting
	 * like this we'll be leaking memory anyway, so who cares? :)
	 */
	ptr = realloc(ptr, size);
	if (!ptr) {
		set_screen(0, 0, 0);
		allegro_message("Out of memory!\n");
		exit(1);
	}
	return ptr;
}



BSP_TREE *create_bsp_tree(void)
{
	BSP_TREE *tree = malloc(sizeof *tree);

	if (!tree)
		return NULL;

	tree->root = NULL;

	tree->n_tempnodes = 0;
	tree->max_tempnodes = 0;
	tree->tempnode = NULL;

	tree->n_vtx = 0;
	tree->n_pvtx = 0;
	tree->n_vtx_pages = 0;
	tree->vtx = NULL;

	tree->n_poly = 0;
	tree->n_ppoly = 0;
	tree->n_poly_pages = 0;
	tree->poly = NULL;

	tree->n_v3d = 0;
	tree->n_v3d_allocated = 0;

	tree->v3d = NULL;
	tree->v3dc = NULL;
	tree->v3dt = NULL;
	tree->out = NULL;

	return tree;
}



static VERTEX *bsp_tree_split_interpolate(
	BSP_TREE *tree,
	POLYGON_VERTEX *p0, float value0,
	POLYGON_VERTEX *p1, float value1,
	float *u, float *v
)
{
	float t;
	VERTEX *vtx;

	if (value0 > -BSP_EPSILON && value0 < BSP_EPSILON)
		t = 0;
	else if (value1 > -BSP_EPSILON && value1 < BSP_EPSILON)
		t = 1;
	else
		t = value0 / (value0 - value1);

	if (tree->n_vtx >= tree->n_vtx_pages << 8) {
		tree->vtx = critical_realloc(tree->vtx, (tree->n_vtx_pages + 1) * sizeof *tree->vtx);
		tree->vtx[tree->n_vtx_pages] = malloc(256 * sizeof(*tree->vtx[0]));
		if (!tree->vtx[tree->n_vtx_pages]) {
			set_screen(0, 0, 0);
			allegro_message("Out of memory!\n");
			exit(1);
		}
		tree->n_vtx_pages++;
	}

	vtx = &tree->vtx[tree->n_vtx >> 8][tree->n_vtx & 255];
	tree->n_vtx++;

	vtx->x = p0->vtx->x + (p1->vtx->x - p0->vtx->x) * t;
	vtx->y = p0->vtx->y + (p1->vtx->y - p0->vtx->y) * t;
	vtx->z = p0->vtx->z + (p1->vtx->z - p0->vtx->z) * t;

	*u = p0->u + (p1->u - p0->u) * t;
	*v = p0->v + (p1->v - p0->v) * t;

	return vtx;
}



static void bsp_tree_split_polygon(
	BSP_TREE *tree,
	POLYGON *poly,
	float *value,
	POLYGON *polypos,
	POLYGON *polyneg
)
{
	int side, startside;

	int m, n;

	polypos->n_vtx = 0;
	polyneg->n_vtx = 0;

	startside = 0;

	m = poly->n_vtx - 1;

	while (m >= 0) {
		if (value[m] >= BSP_EPSILON) {
			startside = 0;
			break;
		} else if (value[m] <= -BSP_EPSILON) {
			startside = 1;
			break;
		} else
			m--;
	}

	side = startside;

	for (n = 0; n < poly->n_vtx; n++) {
		if (side) {
			if (value[n] >= BSP_EPSILON) {
				polyneg->n_vtx++;
				side = 0;
				polypos->n_vtx++;
			}
		} else {
			if (value[n] <= -BSP_EPSILON) {
				polypos->n_vtx++;
				side = 1;
				polyneg->n_vtx++;
			}
		}

		polypos->n_vtx += 1 - side;
		polyneg->n_vtx += side;
	}

	if (polypos->n_vtx > tree->n_v3d) tree->n_v3d = polypos->n_vtx;
	if (polyneg->n_vtx > tree->n_v3d) tree->n_v3d = polyneg->n_vtx;

	polypos->p = malloc(polypos->n_vtx * sizeof *polypos->p);
	polyneg->p = malloc(polyneg->n_vtx * sizeof *polyneg->p);

	if (!polypos->p || !polyneg->p) {
		set_screen(0, 0, 0);
		allegro_message("Out of memory!\n");
		exit(1);
	}

	polypos->n_vtx = 0;
	polyneg->n_vtx = 0;

	m = poly->n_vtx - 1;

	side = startside;

	for (n = 0; n < poly->n_vtx; m = n++) {
		if (side) {
			if (value[n] >= BSP_EPSILON) {
				float u, v;

				VERTEX *newvtx = bsp_tree_split_interpolate(
					tree,
					&poly->p[m], value[m],
					&poly->p[n], value[n],
					&u, &v
				);

				polyneg->p[polyneg->n_vtx].vtx = newvtx;
				polyneg->p[polyneg->n_vtx].u = u;
				polyneg->p[polyneg->n_vtx].v = v;
				polyneg->n_vtx++;
				side = 0;
				polypos->p[polypos->n_vtx].vtx = newvtx;
				polypos->p[polypos->n_vtx].u = u;
				polypos->p[polypos->n_vtx].v = v;
				polypos->n_vtx++;
			}
		} else {
			if (value[n] <= -BSP_EPSILON) {
				float u, v;

				VERTEX *newvtx = bsp_tree_split_interpolate(
					tree,
					&poly->p[m], value[m],
					&poly->p[n], value[n],
					&u, &v
				);

				polypos->p[polypos->n_vtx].vtx = newvtx;
				polypos->p[polypos->n_vtx].u = u;
				polypos->p[polypos->n_vtx].v = v;
				polypos->n_vtx++;
				side = 1;
				polyneg->p[polyneg->n_vtx].vtx = newvtx;
				polyneg->p[polyneg->n_vtx].u = u;
				polyneg->p[polyneg->n_vtx].v = v;
				polyneg->n_vtx++;
			}
		}

		if (side)
			polyneg->p[polyneg->n_vtx++] = poly->p[n];
		else
			polypos->p[polypos->n_vtx++] = poly->p[n];
	}

	polyneg->texture = polypos->texture = poly->texture;

	polyneg->xn = polypos->xn = poly->xn;
	polyneg->yn = polypos->yn = poly->yn;
	polyneg->zn = polypos->zn = poly->zn;
}



static void add_tempnode(BSP_TREE *tree, BSP_NODE **tempnode)
{
	if (tree->n_tempnodes >= tree->max_tempnodes) {
		tree->max_tempnodes += 256;
		tree->tempnode = critical_realloc(tree->tempnode, tree->max_tempnodes * sizeof *tree->tempnode);
	}

	tree->tempnode[tree->n_tempnodes++] = tempnode;
}



void add_to_bsp_tree(BSP_TREE *tree, POLYGON *poly, BSP_NODE **tempnode)
{
	VERTEX *vtx;
	float *value;

	int side = 0;

	POLYGON *rootpoly;

	BSP_NODE *root;

	if (!tree->root) {

		/* This tree is empty. Add the polygon as the root. */

		tree->root = malloc(sizeof *tree->root);

		if (!tree->root)
			return;

		tree->root->poly = poly;
		tree->root->pos = NULL;
		tree->root->neg = NULL;

		if (poly->n_vtx > tree->n_v3d)
			tree->n_v3d = poly->n_vtx;

		if (tempnode)
			add_tempnode(tree, tempnode);

		return;
	}

	rootpoly = tree->root->poly;

	vtx = malloc(poly->n_vtx * sizeof *vtx);
	value = malloc(poly->n_vtx * sizeof *value);

	if (!vtx || !value) {
		set_screen(0, 0, 0);
		allegro_message("Out of memory!\n");
		exit(1);
	}

	/* Calculate coordinates of polygon relative to root polygon. */
	{
		int n;
		for (n = 0; n < poly->n_vtx; n++) {
			vtx[n].x = poly->p[n].vtx->x - rootpoly->p[0].vtx->x;
			vtx[n].y = poly->p[n].vtx->y - rootpoly->p[0].vtx->y;
			vtx[n].z = poly->p[n].vtx->z - rootpoly->p[0].vtx->z;
		}
	}

	/* Calculate which side each point is and how far off the side it is. */
	{
		int n;
		for (n = 0; n < poly->n_vtx; n++) {
			value[n] =
				vtx[n].x * rootpoly->xn +
				vtx[n].y * rootpoly->yn +
				vtx[n].z * rootpoly->zn;

			if (value[n] >= BSP_EPSILON)
				side |= 1;

			if (value[n] <= -BSP_EPSILON)
				side |= 2;
		}
	}

	switch (side) {
		case 3:
			/* Split this polygon into two. */
			{
				POLYGON *polypos, *polyneg;

				if (tree->n_poly + 1 >= tree->n_poly_pages << 8) {
					tree->poly = critical_realloc(tree->poly, (tree->n_poly_pages + 1) * sizeof *tree->poly);
					tree->poly[tree->n_poly_pages] = malloc(256 * sizeof(*tree->poly[0]));
					if (!tree->poly[tree->n_poly_pages]) {
						set_screen(0, 0, 0);
						allegro_message("Out of memory!\n");
						exit(1);
					}
					tree->n_poly_pages++;
				}

				polypos = &tree->poly[tree->n_poly >> 8][tree->n_poly & 255];
				tree->n_poly++;
				polyneg = &tree->poly[tree->n_poly >> 8][tree->n_poly & 255];
				tree->n_poly++;

				bsp_tree_split_polygon(
					tree, poly, value,
					polypos,
					polyneg
				);

				free(value);
				free(vtx);

				root = tree->root;
				tree->root = root->pos;
				add_to_bsp_tree(tree, polypos, tempnode ? &root->pos : NULL);
				root->pos = tree->root;
				tree->root = root->neg;
				add_to_bsp_tree(tree, polyneg, tempnode ? &root->neg : NULL);
				root->neg = tree->root;
				tree->root = root;
			}
			break;
		case 2:
			/* This polygon goes on the negative side of the tree. */
			free(value);
			free(vtx);
			root = tree->root;
			tree->root = root->neg;
			add_to_bsp_tree(tree, poly, tempnode ? &root->neg : NULL);
			root->neg = tree->root;
			tree->root = root;
			break;
		default:
			/* This polygon goes on the positive side of the tree. */
			free(value);
			free(vtx);
			root = tree->root;
			tree->root = root->pos;
			add_to_bsp_tree(tree, poly, tempnode ? &root->pos : NULL);
			root->pos = tree->root;
			tree->root = root;
	}
}



void finalise_bsp_tree(BSP_TREE *tree, int temporary)
{
	int n;

	if (tree->n_v3d_allocated >= tree->n_v3d)
		return;

	if (tree->out) free(tree->out);

	if (tree->v3dt) {
		if (tree->v3dt[0]) free(tree->v3dt[0]);
		free(tree->v3dt);
	}

	if (tree->v3dc) {
		if (tree->v3dc[0]) free(tree->v3dc[0]);
		free(tree->v3dc);
	}

	if (tree->v3d) {
		if (tree->v3d[0]) free(tree->v3d[0]);
		free(tree->v3d);
	}

	tree->v3d = malloc(tree->n_v3d * sizeof *tree->v3d);
	tree->v3dc = malloc((tree->n_v3d + N_CLIPPING_PLANES) * sizeof *tree->v3dc);
	tree->v3dt = malloc((tree->n_v3d + N_CLIPPING_PLANES) * sizeof *tree->v3dt);

	if (!tree->v3d || !tree->v3dc || !tree->v3dt) {
		set_screen(0, 0, 0);
		allegro_message("Out of memory!\n");
		exit(1);
	}

	tree->v3d[0] = malloc(tree->n_v3d * sizeof *tree->v3d[0]);
	tree->v3dc[0] = malloc((tree->n_v3d + N_CLIPPING_PLANES) * sizeof *tree->v3dc[0]);
	tree->v3dt[0] = malloc((tree->n_v3d + N_CLIPPING_PLANES) * sizeof *tree->v3dt[0]);

	if (!tree->v3d[0] || !tree->v3dc[0] || !tree->v3dt[0]) {
		set_screen(0, 0, 0);
		allegro_message("Out of memory!\n");
		exit(1);
	}

	for (n = 1; n < tree->n_v3d; n++)
		tree->v3d[n] = tree->v3d[n - 1] + 1;

	for (n = 1; n < tree->n_v3d + N_CLIPPING_PLANES; n++)
		tree->v3dc[n] = tree->v3dc[n - 1] + 1;

	for (n = 1; n < tree->n_v3d + N_CLIPPING_PLANES; n++)
		tree->v3dt[n] = tree->v3dt[n - 1] + 1;

	tree->out = malloc((tree->n_v3d + N_CLIPPING_PLANES) * sizeof *tree->out);

	if (!tree->out) {
		set_screen(0, 0, 0);
		allegro_message("Out of memory!\n");
		exit(1);
	}

	tree->n_v3d_allocated = tree->n_v3d;

	if (!temporary) {
		tree->n_pvtx = tree->n_vtx;
		tree->n_ppoly = tree->n_poly;
	}
}



void remove_temporary(BSP_TREE *tree)
{
	int i;

	while (tree->n_tempnodes) {
		tree->n_tempnodes--;
		free(*tree->tempnode[tree->n_tempnodes]);
		*tree->tempnode[tree->n_tempnodes] = NULL;
	}

	for (i = tree->n_ppoly; i < tree->n_poly; i++)
		free(tree->poly[i>>8][i&255].p);

	tree->n_vtx = tree->n_pvtx;
	tree->n_poly = tree->n_ppoly;
}



static void destroy_bsp_node(BSP_NODE *node)
{
	if (node) {
		destroy_bsp_node(node->pos);
		destroy_bsp_node(node->neg);

		free(node);
	}
}



void destroy_bsp_tree(BSP_TREE *tree)
{
	if (tree) {
		/* No need to remove temporary nodes; the recursive destruction will
		 * do that anyway. This just frees the array that holds the temporary
		 * nodes.
		 */
		if (tree->tempnode)
			free(tree->tempnode);

		if (tree->v3d) {
			if (tree->v3d[0]) free(tree->v3d[0]);
			free(tree->v3d);
		}

		if (tree->v3dc) {
			if (tree->v3dc[0]) free(tree->v3dc[0]);
			free(tree->v3dc);
		}

		if (tree->v3dt) {
			if (tree->v3dt[0]) free(tree->v3dt[0]);
			free(tree->v3dt);
		}

		if (tree->out)
			free(tree->out);

		destroy_bsp_node(tree->root);

		if (tree->poly) {
			int i;

			for (i = 0; i < tree->n_poly; i++)
				if (tree->poly[i>>8][i&255].p)
					free(tree->poly[i>>8][i&255].p);

			for (i = 0; i < tree->n_poly_pages; i++)
				free(tree->poly[i]);

			free(tree->poly);
		}

		if (tree->vtx) {
			int i;

			for (i = 0; i < tree->n_vtx_pages; i++)
				free(tree->vtx[i]);

			free(tree->vtx);
		}

		free(tree);
	}
}

