#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <allegro.h>
#include "xpm.h"
#include "area.h"
#include "items.h"
#include "bomb.h"
#include "generate.h"
#include "error.h"
#include "defines.h"
#include "levels.h"
#include "color.h"

#define DEFAULT_COLOR_DEPTH 16

#define RES_DEAD 0
#define RES_QUIT 1
#define RES_NEXTLEVEL 2

static BITMAP *dest=NULL;

static int do_vsync=0;
static int width=640;
static int height=480;
static int cdepth=0;
static int use_custom_level=0;
static int use_joystick=0;
static int joystick_ok=0;
static int mouse_ok=0;
static int allow_sound=1;

static volatile int frames=0;
static volatile int fps=0;
static volatile int pause_game=0;
static int score=0;

static volatile int ticks=0;
static void ticker()
{
  if (!pause_game) ++ticks;
}
END_OF_STATIC_FUNCTION(ticker);
static void fps_ticker()
{
  fps=frames;
  frames=0;
}
END_OF_STATIC_FUNCTION(fps_ticker);

void start_ticker()
{
  LOCK_FUNCTION(ticker);
  LOCK_FUNCTION(fps_ticker);
  LOCK_VARIABLE(ticks);
  LOCK_VARIABLE(frames);
  LOCK_VARIABLE(fps);
  LOCK_VARIABLE(pause_game);
  install_int_ex(&ticker,BPS_TO_TIMER(TICK_SPEED));
  install_int_ex(&fps_ticker,BPS_TO_TIMER(1));
  ticks=0;
}

void stop_ticker()
{
  remove_int(ticker);
  remove_int(fps_ticker);
  ticks=0;
  frames=0;
  fps=0;
}

void init()
{
  int cd=cdepth;
  if (cd==0) cd=desktop_color_depth();
  if (cd==0) cd=DEFAULT_COLOR_DEPTH;
  set_color_depth(cd);
  if (set_gfx_mode(GFX_AUTODETECT,width,height,0,0)<0) {
    exit_error("Failed to set video mode %dx%dx%d",width,height,cd);
  }
  if (install_keyboard()) exit_error("Failed to install keyboard");
  if (!use_joystick) {
    mouse_ok=(install_mouse()!=-1);
  }
  if (use_joystick) {
    joystick_ok=(install_joystick(JOY_TYPE_AUTODETECT)==0);
  }
  if (!mouse_ok && !joystick_ok) exit_error("No mouse nor joystick");
  if (allow_sound) {
    //reserve_voices(4,0);
    install_sound(DIGI_AUTODETECT,MIDI_NONE,NULL);
  }

  dest=create_bitmap(width,height);
  text_mode(-1);

  start_ticker();
}

void shutdown()
{
  stop_ticker();
  destroy_bitmap(dest);
}

ambulance::type get_ambulance_type()
{
  clear_bitmap(dest);
  int y=64;
  int dy = (3*text_height(font))/2;
  textprintf_centre(
    dest, font, SCREEN_W/2, y, CHOICE_COLOR, "Dr Lovepeace"
  );
  y += dy;
  textprintf_centre(
    dest, font, SCREEN_W/2, y, CHOICE_COLOR, "or"
  );
  y += dy;
  textprintf_centre(
    dest, font, SCREEN_W/2, y, CHOICE_COLOR, "How I learned to hate the bombs"
  );
  y += 8*dy;
  textprintf_centre(
    dest, font, SCREEN_W/2, y, CHOICE_COLOR, "Dr Lovepeace has just graduated"
  );
  y += dy;
  textprintf_centre(
    dest, font, SCREEN_W/2, y, CHOICE_COLOR, "She wants to be field medic"
  );
  y += dy;
  textprintf_centre(
    dest, font, SCREEN_W/2, y, CHOICE_COLOR,
    "Which international medical organization will she join ?"
  );

  ambulance::type type=ambulance::anar_symbol;
  const int display_width=dest->w*3/4;
  const int display_dx=(dest->w-display_width)/2;
  const int step=display_width/ambulance::num_types;
  int angle=0;
  for (;;) {
    while (ticks) {
      ++angle;
      --ticks;
    }
    const int y=270;
    rectfill(dest,0,y-8,dest->w,y+80,0);
    for (int i=0; i<ambulance::num_types; ++i) {
      int x=display_dx+i*step;
      rotate_sprite(
        dest, ambulance::get_bitmap((ambulance::type)i), x, y,
        i==type?itofix(angle):0
      );
    }
    if (do_vsync) vsync();
    blit(dest,screen,0,0,0,0,dest->w,dest->h);
    if (keypressed()) {
      switch (readkey()>>8) {
        case KEY_ESC:
        case KEY_SPACE:
        case KEY_ENTER:
          return type;
        case KEY_LEFT:
          if (type) {
            type=(ambulance::type)(type-1);
            angle=0;
          }
          break;
        case KEY_RIGHT:
          if (type<ambulance::num_types-1) {
            type=(ambulance::type)(type+1);
            angle=0;
          }
          break;
      }
    }
  }

  return type;
}


void end_screen(bool dead,int which)
{
  if (dead) {
    int white = makecol(255, 255, 255);
    int black = makecol(0, 0, 0);
    int x = SCREEN_W/2;
    int dy = text_height(font)*2;
    int y = SCREEN_H/2;
    clear_to_color(screen, white);
    for (int i=1; i<32; ++i) {
      rect(screen, i, i, SCREEN_W-i-1, SCREEN_H-i-1, black);
    }
    switch (which) {
      case 0:
        y -= dy;
        textprintf_centre(
          screen, font, x, y, black, "Dr Lovepeace died on the field."
        );
        y += dy;
        textprintf_centre(
          screen, font, x, y, black,
          "She saved %d lives before perishing under a bomb.", score
        );
        y += dy;
        textprintf_centre(
          screen, font, x, y, black,
          "Fondly remembered by her collegues."
        );
        break;
      case 1:
        textprintf_centre(screen, font, x, y, black, "Stop bombing!");
        break;
    }
  }
  else {
    int green = makecol(192, 255, 192);
    int black = makecol(0, 0, 0);
    int x = SCREEN_W/2;
    int dy = text_height(font)*2;
    int y = SCREEN_H/2;
    clear_to_color(screen, green);
    switch (which) {
      case 0:
        y -= dy/2;
        textprintf_centre(
          screen, font, x, y, black, "Dr Lovepeace just retired."
        );
        y += dy;
        textprintf_centre(
          screen, font, x, y, black,
          "She saved %d lives during her time has field medic.", score
        );
        break;
      case 1:
        textprintf_centre(screen, font, x, y, black, "Stop bombing!");
        break;
    }
  }
}

ambulance *create_ambulance(ambulance::type amb_type)
{
  ambulance *amb=new ambulance(40,40,amb_type);
  amb->set_motion_radius(32);
  amb->set_pickup_radius(48);
  return amb;
}

area *create(ambulance::type amb_type)
{
  area_generator gen(640, 480, "Test");
  gen.set_bomb_freq(20);
  gen.set_wall_count(8);
  gen.set_tree_count(10);
  gen.set_crater_count(8);
  gen.set_wounded_civilian_count(14);
  gen.set_wounded_ally_count(2);
  gen.set_wounded_enemy_count(8);
  area *map = gen.create();

  map->set_ambulance(create_ambulance(amb_type));

  return map;
}

void tick_game(area *map)
{
  int dx=0,dy=0;
  if (!use_joystick && mouse_ok) {
    poll_mouse();
    ambulance *a=map->get_ambulance();
    if (a) {
      dx=(mouse_x-a->get_x());
      dy=(mouse_y-a->get_y());
    }
  }
  if (use_joystick && joystick_ok) {
    poll_joystick();
    dx=4*(joy[0].stick[0].axis[0].pos);
    dy=4*(joy[0].stick[0].axis[0].pos);
  }
  int count=map->tick(dx,dy);
  if (count) {
    score+=count;
  }
}

int run_level(int level, ambulance::type amb_type)
{
  int ok=1;
  int ready_to_finish=0;

  pause_game=1;
  area *map=NULL;
  if (level==-1) {
    map=create(amb_type);
  }
  else {
    map=levels::get_area(level);
    map->set_ambulance(create_ambulance(amb_type));

    char msg[256];
#ifdef ALLEGRO_LINUX
    snprintf(msg,sizeof(msg),"Level %d",1+level);
#else
    sprintf(msg,"Level %d",1+level);
#endif
    map->add_message(msg,LEVEL_COLOR,TICK_SPEED*8);
    map->add_message(levels::get_text(level),LEVEL_COLOR,TICK_SPEED*8);
  }

  const ambulance *amb=map->get_ambulance();
  if (amb) position_mouse(amb->get_x(),amb->get_y());

  while (ok) {
    poll_keyboard();
    if (keypressed()) {
      int key=readkey()>>8;
      int res;
      switch(key) {
        case KEY_ESC:
          pause_game++;
          res=alert("Quit ?",NULL,NULL,"Yes","No",'y','n');
          pause_game--;
          if (res==1) {
            delete map;
            return RES_QUIT;
          }
          break;
        case KEY_ENTER:
          if (ready_to_finish) ok=0;
          break;
        case KEY_SPACE:
          pause_game=!pause_game;
          break;
#ifdef DEBUG
        case KEY_F1:
          ok=0;
          break;
#endif
      }
    }

    while (ticks) {
      tick_game(map);
      --ticks;
    }
    map->draw(dest);
    if (map->get_ambulance()) {
      circle(dest,mouse_x,mouse_y,5,makecol(255,255,255));
      circle(dest,mouse_x,mouse_y,2,makecol(255,255,255));
    }
    frames++;
    text_mode(-1);
    textprintf(dest,font,0,0,makecol(255,255,255),"%d fps",fps);
    textprintf(dest,font,dest->w-80,12,makecol(255,255,255),"%d saved",score);

    /* Messages */
    if (!map->get_ambulance()) {
      if (!ready_to_finish) {
        map->add_message("Destroyed - press enter",LOOSE_COLOR,-1);
      }
      ready_to_finish=1;
    }
    if (!map->get_pickable_count()) {
      if (!ready_to_finish) {
        map->add_message("No more to rescue - press enter",WIN_COLOR,-1);
      }
      ready_to_finish=1;
    }
    if (pause_game) {
      static int blink=0;
      blink++;
      textprintf_centre(
        dest,font,SCREEN_W/2,32,((blink/16)&1)?PAUSE_COLOR1:PAUSE_COLOR2,
        "GAME PAUSED - SPACE TO TOGGLE PAUSE"
      );
    }

    /* Screen blit */
    if (do_vsync) vsync();
    blit(dest,screen,0,0,0,0,dest->w,dest->h);
  }

  while (keypressed()) readkey();
  const int success=(map->get_ambulance()!=NULL);

  delete map;
  return success?RES_NEXTLEVEL:RES_DEAD;
}

void help()
{
  allegro_message(
    "./sh [--help] [--width <width>] [--height <height>] [--cdepth <cdepth>]\n"
    "     [--vsync] [--no-vsync] [--use-custom-level]\n"
    "     [--use_joystick] [--no-sound]\n"
  );
}

int get_number(int n,int argc,char **argv)
{
  if (n==argc) {
    help();
    exit_error("Option %s needs an argument",argv[n-1]);
  }
  char *end;
  int number=strtol(argv[n],&end,10);
  if (end && *end) {
    help();
    exit_error("Option %s needs a numeric argument",argv[n-1]);
  }
  return number;
}

void parse(int argc,char *argv[])
{
  for (int n=1;n<argc;++n) {
    char *arg=argv[n];
    if (arg[0]!='-' || arg[1]!='-') {
      help();
      exit_error("Invalid option: %s",arg);
    }
    arg+=2;
    if (!strcmp(arg,"help")) {
      help();
      exit(0);
    }
    else if (!strcmp(arg,"width")) {
      width=get_number(++n,argc,argv);
    }
    else if (!strcmp(arg,"height")) {
      height=get_number(++n,argc,argv);
    }
    else if (!strcmp(arg,"cdepth")) {
      cdepth=get_number(++n,argc,argv);
    }
    else if (!strcmp(arg,"vsync")) {
      do_vsync=1;
    }
    else if (!strcmp(arg,"no-vsync")) {
      do_vsync=0;
    }
    else if (!strcmp(arg,"use-joystick")) {
      use_joystick=1;
    }
    else if (!strcmp(arg,"no-sound")) {
      allow_sound=0;
    }
    else if (!strcmp(arg,"use-custom-level")) {
      use_custom_level=1;
    }
    else exit_error("Unknown option: --%s",arg);
  }
}

int main(int argc,char *argv[])
{
  int res=RES_QUIT;
  allegro_init();
  parse(argc,argv);
  init();

  ambulance::type amb_type = get_ambulance_type();

  if (use_custom_level) {
    res=run_level(-1, amb_type);
  }
  else {
    int level=0;
    for (;;) {
      res=run_level(level, amb_type);
      if (res==RES_DEAD || res==RES_QUIT) break;
      ++level;
    }
  }

  end_screen(res==RES_DEAD,0);
  while (keypressed()) readkey();
  readkey();
  end_screen(res==RES_DEAD,1);
  while (keypressed()) readkey();
  readkey();

  shutdown();
  return 0;
}
END_OF_MAIN();
