#include <stdlib.h>
#include <allegro.h>
#include "levels.h"

#include "generate.h"

#define TICK_SPEED 20

namespace levels {
  /*
     For example:
     levels 0/1: loin de la ligne de feu
         0: few walls, no tree, few wounded, no bomb
         1: more walls and trees
     level 2/3 plus pres de la ligne de feu
         2 some bombs
         3 more wounded
     level 4/5 small forest and town
         4 no walls only trees
         5 many walls, many craters
     6 more bombs
     7 bombs speed increase
   */
  static const int base_level_count = 10;
  static const area_generator levels[base_level_count] = {
    area_generator(
      640, 480,    /* width, height */
      "During training, Dr Lovepeace has to pick up wounded people,\n"
      "without prejudice of race, sex, religion, sexual preferences\n"
      "or political ideas.\n"
      "She must stop near a person and the van will take care of everything.",
       0,  0,      /* bombs frequency and lifetime */
       0,  0,  1,  /* wounded enemy, ally and civilian counts */
       0,  5,  5   /* crater, tree and wall count */
    ),
    area_generator(
      640, 480,    /* width, height */
      "This is Dr Lovepeace's first mission on the field.\n"
      "The battle line is still away.\n",
       0,  0,      /* bombs frequency and lifetime */
       1,  2,  1,  /* wounded enemy, ally and civilian counts */
       2,  5,  5   /* crater, tree and wall count */
    ),
    area_generator(
      640, 480,    /* width, height */
      "The battle line is now nearer, some bombs will fall and\n"
      "Dr Lovepeace must avoid them.",
      128,  TICK_SPEED*12, /* bombs frequency and lifetime */
       1,  2,  5,  /* wounded enemy, ally and civilian counts */
       5,  5,  6   /* crater, tree and wall count */
    ),
    area_generator(
      640, 480,    /* width, height */
      "Dr Lovepeace is somewhat scared :\n"
      "she will be for the first time on the front line.",
      128,  TICK_SPEED*12, /* bombs frequency and lifetime */
       4,  4,  6,  /* wounded enemy, ally and civilian counts */
       8,  5,  6   /* crater, tree and wall count */
    ),
    area_generator(
      640, 480,    /* width, height */
      "The battle has moved in a forest near a town.",
      96,  TICK_SPEED*12, /* bombs frequency and lifetime */
       6,  8,  4,  /* wounded enemy, ally and civilian counts */
       8, 60,  0   /* crater, tree and wall count */
    ),
    area_generator(
      640, 480,    /* width, height */
      "Dr Lovepeace is sent in a ruined city under fire.",
      64,  TICK_SPEED*12, /* bombs frequency and lifetime */
       8,  8,  16,  /* wounded enemy, ally and civilian counts */
      28,  0,  16   /* crater, tree and wall count */
    ),
    area_generator(
      640, 480,    /* width, height */
      "The battle is raging.",
      32,  TICK_SPEED*10, /* bombs frequency and lifetime */
       8,  8,  8,   /* wounded enemy, ally and civilian counts */
      14,  8,  8   /* crater, tree and wall count */
    ),
    area_generator(
      640, 480,    /* width, height */
      "Both camps have developped faster bombs.",
      32,  TICK_SPEED*6, /* bombs frequency and lifetime */
       8,  8,  8,   /* wounded enemy, ally and civilian counts */
      14,  8,  8   /* crater, tree and wall count */
    ),
    area_generator(
      640, 480,    /* width, height */
      "The bombs have improved again.",
      24,  TICK_SPEED*4, /* bombs frequency and lifetime */
       8,  8,  8,   /* wounded enemy, ally and civilian counts */
      14,  8,  8   /* crater, tree and wall count */
    ),
    area_generator(
      640, 480,    /* width, height */
      "During wartime, innovation only applies to weapons...",
      16,  TICK_SPEED*3, /* bombs frequency and lifetime */
       8,  8,  8,   /* wounded enemy, ally and civilian counts */
      14,  8,  8   /* crater, tree and wall count */
    ),
  };

  /* Dynamic level */
  area_generator dynamic_level(
    640, 480,    /* width, height */
    "Dr Lovepeace is now a veteran physician,\n"
    "she is sent to the most dangerous battlefields.",
     0,  0,      /* bombs frequency and lifetime */
     0,  0,  0,  /* wounded enemy, ally and civilian counts */
     0,  0,  0   /* crater, tree and wall count */
  );

  area *get_area(int level)
  {
    if (level<base_level_count) {
      return levels[level].create();
    }
    else {
      dynamic_level.set_wounded_civilian_count(8+rand()%4);
      dynamic_level.set_wounded_enemy_count(8+rand()%4);
      dynamic_level.set_wounded_ally_count(8+rand()%4);
      dynamic_level.set_crater_count(level*2);
      dynamic_level.set_tree_count(8+level+rand()%8);
      dynamic_level.set_wall_count(4+level/3+rand()%2);
      dynamic_level.set_bomb_freq(MID(2,24-level,16));
      dynamic_level.set_bomb_lifetime(TICK_SPEED*(MID(1,30/level,3)));
      return dynamic_level.create();
    }
  }

  const char *get_text(int level)
  {
    if (level<base_level_count) {
      return levels[level].get_text();
    }
    else {
      return dynamic_level.get_text();
    }
  }
}

