#include "generate.h"

#include "area.h"
#include "items.h"
#include "obstacle.h"

#include <time.h>
#include <stdlib.h>

//#define SHORT_WALLS

static inline int rand_num(int max)
{
  return (int) ((float)max*rand()/(RAND_MAX+1.0));
}


const int area_generator::border = 48;
bool area_generator::init_is_done = false;


void area_generator::init()
{
  if (!init_is_done) {
    srand(time(NULL));
    init_is_done = true;
  }
}


area_generator::area_generator(int width, int height, const char *text):
  width(width), height(height),
  bomb_freq(0), bomb_lifetime(20),
  wounded_enemy_count(0),
  wounded_ally_count(0),
  wounded_civilian_count(0),
  crater_count(0),
  tree_count(0),
  wall_count(0),
  text(text)
{
  init();
}

area_generator::area_generator(
  int width, int height, const char *text,
  int bomb_freq, int bomb_lifetime,
  int wounded_enemy_count, int wounded_ally_count, int wounded_civilian_count,
  int crater_count, int tree_count, int wall_count
):
  width(width), height(height),
  bomb_freq(bomb_freq), bomb_lifetime(bomb_lifetime),
  wounded_enemy_count(wounded_enemy_count),
  wounded_ally_count(wounded_ally_count),
  wounded_civilian_count(wounded_civilian_count),
  crater_count(crater_count),
  tree_count(tree_count),
  wall_count(wall_count),
  text(text)
{
  init();
}

area_generator::~area_generator()
{
}

typedef enum { round, rect } shape;

static item *create_bulk(shape the_shape, int radius_or_width, int height=0)
{
  item *ret = new item(0, 0);
  switch(the_shape) {
    case round:
      ret->set_obstacle(new round_obstacle(0, 0, radius_or_width, 0));
      break;
    case rect:
      ret->set_obstacle(new rect_obstacle(0, 0, radius_or_width, height, 0));
      break;
  }
  return ret;
}

static int get_h_wall_width()
{
  /* 'dummy' is used to know the wall width */
  h_wall dummy(0, 0);
  /* I know it's a rect_obstacle */
  const rect_obstacle *o = (const rect_obstacle*)(dummy.get_obstacle());
  return o->get_width();
}

static int get_v_wall_height()
{
  /* 'dummy' is used to know the wall height */
  v_wall dummy(0, 0);
  /* I know it's a rect_obstacle */
  const rect_obstacle *o = (const rect_obstacle*)(dummy.get_obstacle());
  return o->get_height();
}

static bool compute_stop_early(area *the_area, int startx, int starty)
{
  bool stop_early = false;
  item *dummy = create_bulk(round, area_generator::border);
  dummy->set_position(startx, starty);
  if (the_area->collide_with_blocked(dummy)) {
    stop_early = true;
  }
  delete dummy;
  return stop_early;
}

void area_generator::generate_h_wall(area *the_area) const
{
  const int wall_width = get_h_wall_width();

  int x1 = rand_num(width-2*border)+border;
#ifdef SHORT_WALLS
  int len = rand_num((width-border-x1)/2);
  int x2 = x1+len;
#else
  int x2 = rand_num(width-2*border)+border;
  if (x1>x2) {
    int x = x1;
    x1 = x2;
    x2 = x;
  }
#endif
  int y = rand_num(height-2*border)+border;

  h_wall::type type = (h_wall::type)rand_num(h_wall::type_count);
  item *previous_wall = 0;
  bool stop_early = compute_stop_early(the_area, x1, y);
  for (int x=x1; x<x2; x+=wall_width) {
    item *wall=new h_wall(x, y, type);
    if (the_area->collide_with_blocked(wall)) {
      if (previous_wall && stop_early) {
        the_area->destroy(previous_wall);
      }
      delete wall;
      break;
    }
    the_area->add(wall);
    previous_wall = wall;
  }
}

void area_generator::generate_v_wall(area *the_area) const
{
  const int wall_height = get_v_wall_height();

  int x = rand_num(width-2*border)+border;
  int y1 = rand_num(height-2*border)+border;
#ifdef SHORT_WALLS
  int len = rand_num((height-border-y1)/2);
  int y2 = y1+len;
#else
  int y2 = rand_num(height-2*border)+border;
  if (y1>y2) {
    int y = y1;
    y1 = y2;
    y2 = y;
  }
#endif

  v_wall::type type = (v_wall::type)rand_num(v_wall::type_count);
  item *previous_wall = 0;
  bool stop_early = compute_stop_early(the_area, x, y1);
  for (int y=y1; y<y2; y+=wall_height) {
    item *wall = new v_wall(x, y, type);
    if (the_area->collide_with_blocked(wall)) {
      if (previous_wall && stop_early) {
        the_area->destroy(previous_wall);
      }
      delete wall;
      break;
    }
    the_area->add(wall);
    previous_wall = wall;
  }
}

void area_generator::generate_trees(area *the_area) const
{
  item *dummy = create_bulk(round, border);

  for (int n=0; n<tree_count; ++n) {
    int x = rand_num(width-2*border)+border;
    int y = rand_num(height-2*border)+border;
    dummy->set_position(x, y);
    if (!the_area->collide_with_blocked(dummy)) {
      the_area->add(new tree(x, y));
    }
  }
  delete dummy;
}

void area_generator::generate_wounded(area *the_area, int max, int wtype) const
{
  wounded::type type = (wounded::type)wtype;

  for (int n=0; n<max; ) {
    int x = rand_num(width);
    int y = rand_num(height);
    item *w = new wounded(x, y, type);
    if (the_area->collide_with_blocked(w)) {
      delete w;
    }
    else {
      the_area->add(w);
      n++;
    }
  }
}

area *area_generator::create() const
{
  area *the_area = new area(width, height, bomb_freq, bomb_lifetime);

  for (int n=0; n<wall_count; ++n) {
    generate_h_wall(the_area);
    generate_v_wall(the_area);
  }
  generate_trees(the_area);
  for (int n=0; n<crater_count; ++n) {
    int x = rand_num(width);
    int y = rand_num(height);
    the_area->blast(new crater(x, y));
  }
  generate_wounded(the_area, wounded_civilian_count, wounded::civilian);
  generate_wounded(the_area, wounded_enemy_count, wounded::army0);
  generate_wounded(the_area, wounded_ally_count, wounded::army1);

  return the_area;
}

